/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.statistical.ui.editor.internal;

import org.eclipse.emf.common.util.EList;
import org.eclipse.hyades.model.statistical.SDContiguousObservation;
import org.eclipse.hyades.model.statistical.SDCounterDescriptor;
import org.eclipse.hyades.model.statistical.SDDescriptor;
import org.eclipse.hyades.model.statistical.SDDiscreteObservation;
import org.eclipse.hyades.model.statistical.SDMemberDescriptor;
import org.eclipse.hyades.model.statistical.SDRepresentation;
import org.eclipse.hyades.model.statistical.SDSnapshotObservation;
import org.eclipse.hyades.model.statistical.SDTextObservation;
import org.eclipse.hyades.models.hierarchy.TRCAgent;
import org.eclipse.hyades.statistical.ui.EditorPlugin;
import org.eclipse.hyades.statistical.ui.ImageManager;
import org.eclipse.hyades.statistical.ui.variableloader.internal.SDModifiableVariableRepresentation;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.TreeItem;

/**
 * This recursive tree operation adds a statistical model to statcon
 */
public class OpAddToTree implements StatConModelOp {
	
	private static final boolean ADD_SORTED = true;
	
	public static OpAddToTree INSTANCE = new OpAddToTree();

	static ImageManager img = EditorPlugin.img;

	private static int getIndex(TreeItem parent, String childname) {
		childname = childname.toLowerCase();
		
		TreeItem[] items = parent.getItems();
		if (ADD_SORTED) {
			int start = 0;
			if (parent.getParentItem() == null) {
				start = StatisticalModelGraphViewer.RESERVED_NODES;
			}
			for (int i = start; i < items.length; i++) {
				if (items[i].getText().toLowerCase().compareTo(childname) > 0) {
					return i;
				}
			}
		}
		return items.length;
	}
	
	public static void launch(StatisticalModelGraphViewer statcon, SDDescriptor descriptor, SDDescriptor root, boolean update) {
		StatConOpState state = new StatConOpState(new Boolean(update));
		statcon.applyStatConModelOp( new OpAddToTree(), state, descriptor, root);
	}

	private String getTreeName(SDDescriptor sd) {
		String basicname = null;
		//
		// Get the basic name for this node and its duplicates
		//
		if (sd.getDescription() != null) {
			basicname = sd.getName()+" ("+sd.getDescription()+")";
		} else {
			basicname = sd.getName();
		}
		
		if (basicname == null) {
			basicname = "";	
		}
		if (basicname.length() == 0) {
			basicname = "("+EditorPlugin.getString("ACTION_UNKNOWN")+")";	
		}
		
		return basicname;
	}	

	public void postapply(StatConOpState state, StatisticalModelGraphViewer statcon, SDDescriptor sd, SDDescriptor root_parent, boolean parents_checked, boolean parents_enabled) {
	}
		
	public StatConOpState preapply(StatConOpState state, StatisticalModelGraphViewer statcon, SDDescriptor sd, SDDescriptor root_parent, boolean parents_checked, boolean parents_enabled) {

		//ignore null nodes
		if (sd == null) return state;
		if (sd.getName() == null) return state;

		boolean update = ((Boolean)state.state).booleanValue();
		
		TreeItem item = statcon.mapDescriptorToItem(sd);
		TreeItem parent_item = statcon.mapDescriptorToItem(sd.getParent());
		
//		if (parent_item == null) return state;
		
		if (update && item != null) {
			//
			// this descriptor already exists - leave it alone and just check its children
			//
				
			//
			// update any descriptions
			//
			String tmp = getTreeName(sd);
			if (!tmp.equals(item.getText())) {
				item.setText(tmp);
			}

				
		} else {
			//
			// this descriptor does not exist or we are not updating - add it
			//
		
			String itemName = getTreeName(sd);
			
			if (parent_item == null) {
				
				//no parent item - make it a root of the agent
				TRCAgent agent = sd.getAgent();
				
				if (agent == null) return state;
				
				TreeItem parentTMP = statcon.mapAgentToItem(agent);
				item = new TreeItem(parentTMP,0,getIndex(parentTMP,itemName));
			} else {
				//parent item, create under parent
				item = new TreeItem(parent_item,0,getIndex(parent_item,itemName));		
			}
	
			//
			// Get the basic descriptor to the tree
			//
			item.setText(itemName);

			TRCAgent sdagent = sd.getAgent();
			SDDescriptor parent = sd.getParent();
			while (sdagent == null && parent != null) {
				sdagent = parent.getAgent();
				parent = parent.getParent();
			}
			
			Image statimg = null;
			
			for (int i = 0; statimg == null && i < statcon.stat_img_providers.size(); i++) {
				try {
					StatConStatisticalImageProvider provider = (StatConStatisticalImageProvider)statcon.stat_img_providers.get(i);
					statimg = provider.getImage(sd);
					if (statimg == null && sdagent != null) {
						statimg = provider.getImage(sd,sdagent);
					}
				} catch (Throwable t) {
					EditorPlugin.DBG.error("statistical image provider threw error",t);
				}
			}
			
			if (statimg == null) {
				if (sd instanceof SDCounterDescriptor) {
					item.setImage(img.getImage(ImageManager.IMG_SMODEL_COUNTERDESCRIPTOR));
				} else {
					item.setImage(img.getImage(ImageManager.IMG_SMODEL_DESCRIPTOR));
				}
			} else {
				item.setImage(statimg);
			}
			
			item.setChecked(true);
			item.setGrayed(false);

			statcon.mapDescriptorToItem(sd, item);

		}
		
		//TODO should sort the descriptor children here
		//normal sort doesn't work though, EMF doesn't like the duplicates
		//can't affect the ordering in here - need to maybe have an Op provide a comparator that deals with the ordering
//		Collections.sort(sd.getChildren(),new DescriptorComparator());

		addObservations(statcon, sd,item,update);
		addRepresentation(statcon, sd,item,update);

		return state;
	}

	private void addObservations(StatisticalModelGraphViewer statcon, SDDescriptor sd, TreeItem item, boolean update) {
		//
		// Objects related to a SnapshotObservation (graph)
		//
		EList obs = null;
		Image obs_image = null;

		//
		// Search for observations
		//
		if (sd instanceof SDMemberDescriptor) {
			SDMemberDescriptor sdmem = (SDMemberDescriptor)sd;
			
			//
			//Check to see if it has a valid observation (metric)
			//		
			obs = sdmem.getSnapshotObservation();
			if (obs != null) {
				if (obs.size() > 0) {
					obs_image = img.getImage(ImageManager.IMG_SMODEL_COUNTER);	
				} else {
					obs = null;	
				}
			}
		}		

		//
		//Set up the graph node if necessary
		//
		if (obs != null) {
			for (int i = 0; i < obs.size(); i++) {
			
				SDSnapshotObservation o = (SDSnapshotObservation)obs.get(i);
				
				//
				// These types of observations are graphable
				//
				if (o instanceof SDContiguousObservation
					|| o instanceof SDDiscreteObservation
					|| o instanceof SDTextObservation) {

					TreeItem obs_item = statcon.mapObservationToItem(o);
					
					if (!update || obs_item == null) {
						
						String itemName = sd.getName()+" ("+EditorPlugin.getString("GRAPH")+" "+(i+1)+")";
						
						obs_item = new TreeItem(item,0,0);		
			
						obs_item.setText(itemName);			
						obs_item.setImage(obs_image);
						obs_item.setGrayed(false);
						obs_item.setChecked(true);
	
						statcon.mapObservationToItem(o,obs_item);
					}
				}
			
			}
		}

	}

	private void addRepresentation(StatisticalModelGraphViewer statcon, SDDescriptor sd, TreeItem item, boolean update) {
		//
		// Objects related to a ModifiableVariableRepresentation (control)
		//
		SDModifiableVariableRepresentation rep = null;
		Image rep_image = null;

		//
		// Search for representations
		//
		if (sd instanceof SDMemberDescriptor) {
			SDMemberDescriptor sdmem = (SDMemberDescriptor)sd;

			//
			//Check to see if it has a valid modifiable representation (control)
			//
			SDRepresentation reptmp = sdmem.getRepresentation();	
			if (reptmp instanceof SDModifiableVariableRepresentation) {
				rep = (SDModifiableVariableRepresentation)reptmp;
				rep_image = rep.getImage();	
			}
		}		
		
		//
		//Set up the control node if necessary
		//
		if (rep != null) {
			TreeItem rep_item = statcon.mapRepresentationToItem(rep);
			
			if (!update || rep_item == null) {
				String itemName = sd.getName()+" ("+EditorPlugin.getString("CONTROL")+")";
				
				rep_item = new TreeItem(item,0,0);		
			
				rep_item.setText(itemName);			
				rep_item.setImage(rep_image);
				if (rep_image == null) {
					rep_item.setImage(img.getImage(ImageManager.IMG_SMODEL_MODIFIABLEREP));
				}
				rep_item.setGrayed(false);
				rep_item.setChecked(true);
	
				statcon.mapRepresentationToItem(rep,rep_item);			
			}
		}		
	}

}