/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.statistical.ui.editor.internal;

import java.util.ArrayList;
import java.util.HashMap;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.eclipse.hyades.statistical.ui.widgets.grapher.internal.Graph;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.RGB;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
/**
 * This file is a utility file that can be used by StatCon modules and by StatCon itself
 * to parse, modify and generate configuration files.
 *
 */
public class XMLConfigUtil {
	
	/**
	 * Utility method for generating a document builder which can be used to parse StatCon files.
	 * @return the DocumentBuilder which can be used to parse a StatCon file
	 * @throws Exception
	 */
	public static DocumentBuilder getDocumentBuilder() throws Exception {
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		factory.setValidating(false);
		
		DocumentBuilder builder = factory.newDocumentBuilder();
		return builder;
	}

	public static void getXmlHead(StringBuffer sb, Element e) {
		sb.append("<");
		sb.append(e.getTagName());
		sb.append(" ");
		NamedNodeMap attrs = e.getAttributes();
		for (int i = 0; i < attrs.getLength(); i++) {
			Node node = attrs.item(i);
			sb.append(node.getNodeName());
			sb.append("=\"");
			sb.append(toRawXML(node.getNodeValue()));
			sb.append("\" ");	
		}
		sb.append(">\n");
	}
	
	public static void getXmlTail(StringBuffer sb, Element e) {
		sb.append("</");
		sb.append(e.getTagName());
		sb.append(">\n");
	}

	public static String toRawXML(Node n) {
		StringBuffer sb = new StringBuffer();
		toRawXML(sb, n);
		return sb.toString();	
	}
	
	public static String toRawXML(String val) {
		StringBuffer sb = new StringBuffer();
		for (int i = 0; i < val.length(); i++) {
			char c = val.charAt(i);
			if (c == '>') {
				sb.append("&gt;");
			} else if (c == '<') {
				sb.append("&lt;");
			} else if (c == '"') {
				sb.append("&quot;");
			} else if (c == '\'') {
				sb.append("&apos;");
			} else if (c == '&') {
				sb.append("&amp;");
			} else {
				sb.append(c);	
			}
		}
		return sb.toString();			
	}
	
	private static void toRawXML(StringBuffer sb, Node n) {
		if (n.getNodeType() == Node.ELEMENT_NODE) {
			toXML(sb,(Element)n);
		} else if (n.getNodeType() == Node.TEXT_NODE) {
			String val = n.getNodeValue();
			sb.append(toRawXML(val));
		}
	}

	/**
	 * Recursively converts this element and its children to valid XML.
	 * @param e the Element to convert
	 * @return the XML string representing this element and its children
	 */
	public static String toXML(Element e) {
		StringBuffer sb = new StringBuffer();
		toXML(sb,e);
		return sb.toString();
	}
	
	private static void toXML(StringBuffer sb, Element e) {
		sb.append("<");
		sb.append(e.getTagName());
		sb.append(" ");
		NamedNodeMap attrs = e.getAttributes();
		for (int i = 0; i < attrs.getLength(); i++) {
			Node node = attrs.item(i);
			sb.append(node.getNodeName());
			sb.append("=\"");
			sb.append(toRawXML(node.getNodeValue()));
			sb.append("\" ");	
		}
		sb.append(">\n");

		ArrayList list = getAllElements(e.getChildNodes(),null);
		for (int i = 0; i < list.size(); i++) {
			Element x = (Element)list.get(i);
			toXML(sb,x);	
		}

		sb.append("</");
		sb.append(e.getTagName());
		sb.append(">\n");
	}

	/**
	 * Get the first element of a specific name from a NodeList
	 * @param list the NodeList to search
	 * @param name the tag name of the element to find
	 * @return the first element with the specified name or null if not found
	 */
	public static Element getFirstElement(NodeList list, String name) {
		for (int i = 0; i < list.getLength(); i++) {
			Object o = list.item(i);
			if (o instanceof Element) {
				Element e = (Element)o;
				
				if (name != null) {
					if (e.getTagName().equals(name)) {
						return e;
					}
				} else {
					return e;
				}
			}
		}
		
		return null;		
	}

	/**
	 * Get all the Elements with a certain name in this NodeList.
	 * @param list the list to search
	 * @param name the tag name of the elements to find
	 * @return an ArrayList (java.util.List) of the Elements with the specified tag name 
	 */
	public static ArrayList getAllElements(NodeList list, String name) {
		ArrayList elements = new ArrayList();
		
		for (int i = 0; i < list.getLength(); i++) {
			Object o = list.item(i);
			if (o instanceof Element) {
				Element e = (Element)o;
				
				if (name != null) {
					if (e.getTagName().equals(name)) {
						elements.add(o);
					}
				} else {
					elements.add(o);
				}
			}
		}
		
		return elements;		
	}

	public static void getElementAttributeMap(NodeList list, String name, String attribute, HashMap map) {
		for (int i = 0; i < list.getLength(); i++) {
			Object o = list.item(i);
			if (o instanceof Element) {
				Element e = (Element)o;
				
				if (name != null) {
					if (e.getTagName().equals(name)) {
						String attr = e.getAttribute(attribute);
						if (attr != null) {
							map.put(attr,e);	
						}
					}
				} else {
					String attr = e.getAttribute(attribute);
					if (attr != null) {
						map.put(attr,e);	
					}
				}
			}
		}
	}
	
	public static HashMap getElementAttributeMap(NodeList list, String name, String attribute) {
		HashMap map = new HashMap();
		getElementAttributeMap(list,name,attribute,map);
		return map;
	}
	
	/**
	 * Parse a double from a string (commonly an XML attribute value) and return a default 
	 * if the parse fails.
	 * @param doub the string to parse the double from
	 * @param def the default value
	 * @return the parsed value or the default if the parse failed.
	 */
	public static double parseDouble(String doub, double def) {
		try {
			return Double.parseDouble(doub);
		} catch (Exception e) {
			return def;
		}	
	}

	/**
	 * Parse a boolean from a String (commonly an XML attribute value) and return a default
	 * if the parse fails
	 * @param bool the string to parse the boolean from
	 * @param def the default value
	 * @return the parsed value or the default if the parse failed.
	 */	
	public static boolean parseBoolean(String bool, boolean def) {
		if (bool.equals("true")) return true;
		if (bool.equals("false")) return false;
		return def;
	}

	public static String generateNoDataBehaviour(int type) {
		if (type == Graph.NODATA_DO_NOTHING) {
			return "NODATA_DO_NOTHING";
		} else if (type == Graph.NODATA_DRAW_ZERO) {
			return "NODATA_DRAW_ZERO";
		} else if (type == Graph.NODATA_DRAW_PREVIOUS) {
			return "NODATA_DRAW_PREVIOUS";
		} else {
			return "NODATA_DRAW_ZERO";	
		}	
	}
	
	public static int parseNoDataBehaviour(String type, int def) {
		if (type.equals("NODATA_DO_NOTHING")) {
			return Graph.NODATA_DO_NOTHING;
		} else if (type.equals("NODATA_DRAW_ZERO")) {
			return Graph.NODATA_DRAW_ZERO;
		} else if (type.equals("NODATA_DRAW_PREVIOUS")) {
			return Graph.NODATA_DRAW_PREVIOUS;
		} else {
			return def;	
		}	
	}
	
	
	public static String generatePlottingPeriod(int type) {
		if (type == Graph.PERIOD_TYPE_TICKS) {
			return "PERIOD_TYPE_TICKS";
		} else if (type == Graph.PERIOD_TYPE_MILLIS) {
			return "PERIOD_TYPE_MILLIS";
		} else if (type == Graph.PERIOD_TYPE_PIXELS) {
			return "PERIOD_TYPE_PIXELS";
		} else if (type == Graph.PERIOD_TYPE_UNAVERAGED) {
			return "PERIOD_TYPE_UNAVERAGED";
		} else {
			return "PERIOD_TYPE_TICKS";	
		}	
	}
	
	public static int parsePlottingPeriod(String type, int def) {
		if (type.equals("PERIOD_TYPE_TICKS")) {
			return Graph.PERIOD_TYPE_TICKS;
		} else if (type.equals("PERIOD_TYPE_MILLIS")) {
			return Graph.PERIOD_TYPE_MILLIS;
		} else if (type.equals("PERIOD_TYPE_PIXELS")) {
			return Graph.PERIOD_TYPE_PIXELS;
		} else if (type.equals("PERIOD_TYPE_UNAVERAGED")) {
			return Graph.PERIOD_TYPE_UNAVERAGED;
		} else {
			return def;	
		}	
	}
	
	
	public static String generatePlottingType(int type) {
		if (type == Graph.PLOTTING_TYPE_AVERAGE) {
			return "PLOTTING_TYPE_AVERAGE";
		} else if (type == Graph.PLOTTING_TYPE_MIN) {
			return "PLOTTING_TYPE_MIN";
		} else if (type == Graph.PLOTTING_TYPE_MAX) {
			return "PLOTTING_TYPE_MAX";
		} else if (type == Graph.PLOTTING_TYPE_MIN_MAX) {
			return "PLOTTING_TYPE_MIN_MAX";
		} else if (type == Graph.PLOTTING_TYPE_MIN_MAX_AVERAGE) {
			return "PLOTTING_TYPE_MIN_MAX_AVERAGE";
		} else if (type == Graph.PLOTTING_TYPE_SUM) {
			return "PLOTTING_TYPE_SUM";
		} else if (type == Graph.PLOTTING_TYPE_NEAREST) {
			return "PLOTTING_TYPE_NEAREST";
		} else if (type == Graph.PLOTTING_TYPE_COUNT) {
			return "PLOTTING_TYPE_COUNT";
		} else if (type == Graph.PLOTTING_TYPE_STDDEV) {
			return "PLOTTING_TYPE_STDDEV";
		} else if (type == Graph.PLOTTING_TYPE_STDDEV_AVERAGE) {
			return "PLOTTING_TYPE_STDDEV_AVERAGE";
		} else if (type == Graph.PLOTTING_TYPE_GRADIENT) {
			return "PLOTTING_TYPE_GRADIENT";
		} else {
			return "PLOTTING_TYPE_AVERAGE";	
		}	
	}
	
	public static int parsePlottingType(String type, int def) {
		if (type.equals("PLOTTING_TYPE_AVERAGE")) {
			return Graph.PLOTTING_TYPE_AVERAGE;
		} else if (type.equals("PLOTTING_TYPE_MIN")) {
			return Graph.PLOTTING_TYPE_MIN;
		} else if (type.equals("PLOTTING_TYPE_MAX")) {
			return Graph.PLOTTING_TYPE_MAX;
		} else if (type.equals("PLOTTING_TYPE_MIN_MAX")) {
			return Graph.PLOTTING_TYPE_MIN_MAX;
		} else if (type.equals("PLOTTING_TYPE_MIN_MAX_AVERAGE")) {
			return Graph.PLOTTING_TYPE_MIN_MAX_AVERAGE;
		} else if (type.equals("PLOTTING_TYPE_SUM")) {
			return Graph.PLOTTING_TYPE_SUM;
		} else if (type.equals("PLOTTING_TYPE_NEAREST")) {
			return Graph.PLOTTING_TYPE_NEAREST;
		} else if (type.equals("PLOTTING_TYPE_COUNT")) {
			return Graph.PLOTTING_TYPE_COUNT;
		} else if (type.equals("PLOTTING_TYPE_STDDEV")) {
			return Graph.PLOTTING_TYPE_STDDEV;
		} else if (type.equals("PLOTTING_TYPE_STDDEV_AVERAGE")) {
			return Graph.PLOTTING_TYPE_STDDEV_AVERAGE;
		} else if (type.equals("PLOTTING_TYPE_GRADIENT")) {
			return Graph.PLOTTING_TYPE_GRADIENT;
		} else {
			return def;	
		}	
	}
	
	/**
	 * Parse an SWT line style from a String (commonly an XML attribute value) and return a default
	 * if the parse fails
	 * @param style the string to parse the SWT line style from
	 * @param def the default style
	 * @return the parsed line style or the default if the parse failed.
	 */
	public static int parseLineStyle(String style, int def) {

		if (style.equals("SOLID")) {
			return SWT.LINE_SOLID;
		} else if (style.equals("DOT")) {
			return SWT.LINE_DOT;
		} else if (style.equals("DASH")) {
			return SWT.LINE_DASH;
		} else if (style.equals("DASHDOT")) {
			return SWT.LINE_DASHDOT;
		} else if (style.equals("DASHDOTDOT")) {
			return SWT.LINE_DASHDOTDOT;
		} else {
			return def;	
		}	
	}

	/**
	 * Parse an RGB colour from a String (commonly an XML attribute value) and return a default if the parse
	 * failed. 
	 * @param col the string to parse the RGB colour from
	 * @param def the default RGB colour
	 * @return the parsed colour ot the default if the parse failed
	 */	
	public static RGB parseColor(String col, RGB def) {
		try {
			if (!col.startsWith("#")) throw new Exception();
			if (col.length() != 7) throw new Exception();

			String sr = col.substring(1,3);
			String sg = col.substring(3,5);
			String sb = col.substring(5,7);

			int r = Integer.parseInt(sr,16);
			int g = Integer.parseInt(sg,16);
			int b = Integer.parseInt(sb,16);

			return new RGB(r,g,b);
			
		} catch (Exception e) {
			return def;
		}		
	}

	/**
	 * Create a String from a colour which can be parsed by this class.
	 * @param c the Color to create a string from
	 * @return the string representing the colour
	 */	
	public static String colorToString(Color c) {
		RGB rgb = c.getRGB();

		String s = "#";
		
		if (rgb.red < 0x10) s = s+"0";
		s = s + Integer.toString(rgb.red,16);
		if (rgb.green < 0x10) s = s+"0";
		s = s + Integer.toString(rgb.green,16);
		if (rgb.blue < 0x10) s = s+"0";
		s = s + Integer.toString(rgb.blue,16);
		
		return s;
	}

	/**
	 * Create a String from a specified SWT line style which can be parsed by this class.  
	 * NOTE: Invalid styles are returned as "SOLID".
	 * @param style the SWT line style to parse
	 * @return the String represnting this line style
	 */
	public static String lineStyleToString(int style) {
		
		if (style == SWT.LINE_SOLID) {
			return "SOLID";	
		} else if (style == SWT.LINE_DOT) {
			return "DOT";	
		} else if (style == SWT.LINE_DASH) {
			return "DASH";	
		} else if (style == SWT.LINE_DASHDOT) {
			return "DASHDOT";	
		} else if (style == SWT.LINE_DASHDOTDOT) {
			return "DASHDOTDOT";	
		}
		
		return "SOLID";
	}

}