/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.statistical.ui.widgets.alert.internal;

import java.util.ArrayList;

import org.eclipse.emf.common.util.EList;
import org.eclipse.hyades.model.statistical.SDContiguousObservation;
import org.eclipse.hyades.model.statistical.SDDiscreteObservation;
import org.eclipse.hyades.model.statistical.SDSnapshotObservation;
import org.eclipse.hyades.statistical.ui.EditorPlugin;
import org.eclipse.hyades.statistical.ui.editor.internal.SDSnapshotObservationGraphSource;
import org.eclipse.hyades.statistical.ui.widgets.grapher.internal.Graph;

public class Alert extends Thread {
	//important to have it do nothing by default - need to give users a chance to
	//set the bounds before it does stuff
	
	public String name = EditorPlugin.getString("NEW_ALERT_NAME");
	public long period = 1000;
	public int calc_type = Graph.PLOTTING_TYPE_AVERAGE;
	public double lower = 0;
	public double upper = 100;
	public boolean max_trigger = true;
	public AlertAction[] trigger_actions = new AlertAction[0];
	public AlertAction[] reset_actions = new AlertAction[0];
	public SDSnapshotObservation observation;

boolean die;
//SDContiguousObservation cobs;
//SDDiscreteObservation dobs;

SDSnapshotObservationGraphSource gsource;

EList times;
EList values;

int timesPtr = 0;

boolean triggered = false;

double lastTime = Double.MIN_VALUE;

ArrayList listeners = new ArrayList();

Object LOCK = new Object();

	public void addAlertListener(AlertListener listener) {
		listeners.add(listener);
	}

	public boolean isTriggered() {
		return triggered;
	}

	public void shutdown() {
		die = true;
	}
	
	public void restart() {
		synchronized(LOCK) {
//System.out.println("ALERT RESTART!!");			
			timesPtr = 0;
			lastTime = Double.MIN_VALUE;
			triggered = false;
		}
	}
	
	public Alert() {
	}
	
	public Alert(SDContiguousObservation obs) {
		init(obs);
	}
	
	public void init(SDContiguousObservation obs) {
		times = obs.getCreationTime();
		values = obs.getValue();
		
		timesPtr = Math.max(0,times.size()-1);
		
		observation = obs;
		gsource = new SDSnapshotObservationGraphSource(obs);
		
		setDaemon(true);
		start();
	}
	
	public Alert(SDDiscreteObservation obs) {
		times = obs.getCreationTime();
		values = obs.getValue();
		
		timesPtr = Math.max(0,times.size()-1);
		
		observation = obs;
		gsource = new SDSnapshotObservationGraphSource(obs);

		start();
	}
	
	private double getValueBetween(double t1, double t2) {
		if (calc_type == Graph.PLOTTING_TYPE_AVERAGE) {
			return gsource.getAverageBetween(t1,t2);
		} else if (calc_type == Graph.PLOTTING_TYPE_COUNT) {
			return gsource.getCountBetween(t1,t2);
		} else if (calc_type == Graph.PLOTTING_TYPE_MAX) {
			return gsource.getMaxBetween(t1,t2);
		} else if (calc_type == Graph.PLOTTING_TYPE_MIN) {
			return gsource.getMinBetween(t1,t2);
		} else if (calc_type == Graph.PLOTTING_TYPE_NEAREST) {
			return gsource.getValueAt(t1);
		} else if (calc_type == Graph.PLOTTING_TYPE_STDDEV) {
			return gsource.getStandardDeviationBetween(t1,t2);
		} else if (calc_type == Graph.PLOTTING_TYPE_SUM) {
			return gsource.getSumBetween(t1,t2);
		} else {
			calc_type = Graph.PLOTTING_TYPE_AVERAGE;
			return gsource.getAverageBetween(t1,t2);
		}
	}
	
	public void run() {
		while (!die) {
			try {
				Thread.sleep(1000);	
			} catch (Exception e) {
			}
			
			try {
				synchronized(LOCK) {
					if (times.size() > timesPtr) {
						//more data - scan it for changes
	//System.out.println("Scanning observation");						
						
						double time;
						if (lastTime != Double.MIN_VALUE) {
							time = lastTime;
						} else {
							time = ((Number)times.get(timesPtr)).doubleValue();
							time -= (time % ((double)period));
						}
	//System.out.println("Starting at "+time);						
	
						double latestTimestamp = ((Number)times.get(times.size()-1)).doubleValue();
						
						double curr = getValueBetween(time,time+period);
						while (latestTimestamp > time+period) {
							//make sure we have data in the NEXT averaging period before we consider
							//this one valid
							
							if (!Double.isNaN(curr)) {
	//System.out.println("Checking "+curr+"  - "+lower+"/"+upper);						
								if (curr >= upper) {
									if (max_trigger && !triggered) {
										//Upper trigger
	//System.out.println("UPPER TRIGGER");						
										trigger(time,curr);
									} else if (!max_trigger && triggered) {
										//Upper reset
	//System.out.println("LOWER RESET");						
										reset(time,curr);
									}
								} else if (curr <= lower) {
									if (max_trigger && triggered) {
										//Lower reset
	//System.out.println("UPPER RESET");						
										reset(time,curr);
									} else if (!max_trigger && !triggered) {
										//Lower trigger
	//System.out.println("LOWER TRIGGER");						
										trigger(time,curr);
									}
								}
							}
							
							//increment the time
							time += period;
	
							curr = getValueBetween(time,time+period);
						}
						
						lastTime = time;
						
						timesPtr = times.size();
						
					}
				}//end sync
			} catch (Exception e) {
			}
		}
	}
	
	public void trigger(double time, double value) {
		triggered = true;
		
		for (int i = 0; i < listeners.size(); i++) {
			try {
				AlertListener listener = (AlertListener)listeners.get(i);
				listener.alertTriggered(this);
			} catch (Throwable t) {
				EditorPlugin.DBG.error("Error notifying alert listener of trigger",t);
			}
		}
			

		for (int n = 0; n < trigger_actions.length; n++) {
			AlertAction trigger_action = trigger_actions[n];
			trigger_action.setInformation(name,observation,value,time,true);
			trigger_action.run();
		}
	}
	
	public void reset(double time, double value) {
		triggered = false;

		for (int i = 0; i < listeners.size(); i++) {
			try {
				AlertListener listener = (AlertListener)listeners.get(i);
				listener.alertReset(this);
			} catch (Throwable t) {
				EditorPlugin.DBG.error("Error notifying alert listener of reset",t);
			}
		}
		
		for (int n = 0; n < reset_actions.length; n++) {
			AlertAction reset_action = reset_actions[n];
			reset_action.setInformation(name,observation,value,time,false);
			reset_action.run();
		}
	}
}