/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.statistical.ui.widgets.grapher.internal;

import org.eclipse.hyades.statistical.ui.widgets.zoomslider.internal.TimeZoomSlider;
import org.eclipse.hyades.statistical.ui.widgets.zoomslider.internal.ZoomIndicator;
import org.eclipse.hyades.statistical.ui.widgets.zoomslider.internal.ZoomSlider;
import org.eclipse.hyades.statistical.ui.widgets.zoomslider.internal.ZoomSliderTick;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;

public class LineGraph implements Graph {

/** use nearest data to requested points */
//public static final int TYPE_NEAREST = 0;
/** average data between requested points */
//public static final int TYPE_AVERAGE = 1;
/** sum data between requested points */
//public static final int TYPE_SUM = 2;

private static final boolean DRAW_ONLY_IF_DATA = true;
private static final boolean DRAW_LINE_FOR_DATA_POINT = true;
	
private static final int PLOTTING_TYPE_STDDEV_AVERAGE_PLUS = -9990;
private static final int PLOTTING_TYPE_STDDEV_AVERAGE_MINUS = -9991;

GraphSource source;
ZoomSlider yslider;
TimeZoomSlider xslider;

Color fg_col;
int line_width = 2;
int line_style= SWT.LINE_SOLID;

int plotting_type = Graph.PLOTTING_TYPE_AVERAGE;
int plotting_period_type = Graph.PERIOD_TYPE_TICKS;
double plotting_period_value = 1.0;

int nodata = Graph.NODATA_DRAW_ZERO;

double xoffset = 0;
double yoffset = 0;

double scaling = 1.0;

double last_drawn_value = 0.0d;

boolean use_indicators = true;

ZoomIndicator indicator;

IndicatorSource isource;

	public LineGraph(int type, GraphSource source, TimeZoomSlider xslider, ZoomSlider yslider, Color col) {
/*		if (type == TYPE_NEAREST) {
			averaging_type = Graph.PLOTTING_TYPE_NEAREST;
		} else if (type == TYPE_AVERAGE) {
			averaging_type = Graph.PLOTTING_TYPE_AVERAGE;
		} else if (type == TYPE_SUM) {
			averaging_type = Graph.PLOTTING_TYPE_SUM;
		} else {
*/		
			plotting_period_type = type;
//		}
		this.source = source;
		this.fg_col = col;
		this.xslider = xslider;
		this.yslider = yslider;
		init();
	}
	
	public LineGraph(int type, GraphSource source, TimeZoomSlider xslider, ZoomSlider yslider, Color col, boolean pixel_perfect, boolean use_indicators) {
/*		if (type == TYPE_NEAREST) {
			averaging_type = Graph.PLOTTING_TYPE_NEAREST;
		} else if (type == TYPE_AVERAGE) {
			averaging_type = Graph.PLOTTING_TYPE_AVERAGE;
		} else if (type == TYPE_SUM) {
			averaging_type = Graph.PLOTTING_TYPE_SUM;
		} else {
*/		
			plotting_period_type = type;
//		}
		this.source = source;
		this.fg_col = col;
		this.xslider = xslider;
		this.yslider = yslider;
		if (pixel_perfect) {
			plotting_period_type = Graph.PERIOD_TYPE_PIXELS;
			plotting_period_value = 1.0;
		} else {
			plotting_period_type = Graph.PERIOD_TYPE_TICKS;
			plotting_period_value = 1.0;
		}
		this.use_indicators = use_indicators;
		init();
	}

	public BasicGraphSource getGraphSource() {
		return source;	
	}
		
	private void init() {
		if (use_indicators) {
			indicator = new ZoomIndicator(yslider,last_drawn_value,fg_col);
			yslider.addZoomIndicator(indicator);
		}
	}

	public void setIndicatorSource(IndicatorSource source) {
		isource = source;
	}

	public void setUseIndicator(boolean b) {
		if (use_indicators && !b) {
			yslider.removeIndicator(indicator);
			indicator.dispose();			
		} else if (!use_indicators && b) {
			indicator = new ZoomIndicator(yslider,last_drawn_value,fg_col);
			yslider.addZoomIndicator(indicator);
		}
		use_indicators = b;
	}
	
	public double getXMin() {
		return source.getMin();		
	}

	public double getXMax() {
		return source.getMax();
	}

	public double getYMin() {
		return source.getValueMin();		
	}

	public double getYMax() {
		return source.getValueMax();		
	}

	public void setForeground(Color col) {
		fg_col = col;
		if (use_indicators) {
			indicator.setColor(col);	
		}
	}
	
	public Color getForeground() {
		return fg_col;
	}

	public void setLineWidth(int w) {
		line_width = w;
	}
	
	public int getLineWidth() {
		return line_width;
	}
	
	public void setLineStyle(int swt_line_style) {
		line_style = swt_line_style;
	}
	
	public int getLineStyle() {
		return line_style;	
	}
	
	public int getNoDataBehaviour() {
		return nodata;
	}
	public void setNoDataBehaviour(int behaviour) {
		nodata = behaviour;
	}

double recentMaxValue = Double.MIN_VALUE;	
double tmpMaxValue = 0;
	
	public double getRecentMaxValue() {
		return recentMaxValue;
	}
	
	public void paintGraph(GC gc, int x, int y, int w, int h) {

		tmpMaxValue = Double.MIN_VALUE;
		
		gc.setForeground(fg_col);
		gc.setLineWidth(line_width);	
		gc.setLineStyle(line_style);
		
//we always draw on the ticks, this way the graphs stay constant 
//since we are averaging over the same periods and thus the same
//data all the time
		ZoomSliderTick tick_start = xslider.getFirstTick();
		ZoomSliderTick tick_end = xslider.getLastTick();

		if (tick_start == null || tick_end == null) return;

//		double start = xslider.getMinVisible();
//		double end = xslider.getMaxVisible();

//		double start = xslider.pixel2Value(x);
//		double end = xslider.pixel2Value(x+w);

		double res = xslider.getIncrement();

		double start = tick_start.value - res;
		double end = tick_end.value + res;

		if (plotting_period_type == Graph.PERIOD_TYPE_TICKS) {
			res = xslider.getIncrement();
			res *= plotting_period_value;
		} else if (plotting_period_type == Graph.PERIOD_TYPE_PIXELS) {
//			res = xslider.getUnitIncrement();
			res = xslider.pixel2Value(1)-xslider.pixel2Value(0);
			res *= plotting_period_value;
		} else if (plotting_period_type == Graph.PERIOD_TYPE_MILLIS) {
			res = plotting_period_value;
			if (res < 10) res = 10;
		} else if (plotting_period_type == Graph.PERIOD_TYPE_UNAVERAGED) {
			//UNSUPPORTED
			//special case, need to iterate through the data points
			//just use 10ms periods for now though, but this is NOT CORRECT in reality
			res = 10;
		}
		
		//stops the averaging jumping around because of different start points
		start -= start % res;
		
		if (plotting_type == Graph.PLOTTING_TYPE_MIN_MAX) {
			paintGraph(gc,x,y,start,end,res,Graph.PLOTTING_TYPE_MIN);
			paintGraph(gc,x,y,start,end,res,Graph.PLOTTING_TYPE_MAX);
			
		} else if (plotting_type == Graph.PLOTTING_TYPE_MIN_MAX_AVERAGE) {
			paintGraph(gc,x,y,start,end,res,Graph.PLOTTING_TYPE_MIN);
			paintGraph(gc,x,y,start,end,res,Graph.PLOTTING_TYPE_MAX);
			//plot average last - this one will drive the indicator
			paintGraph(gc,x,y,start,end,res,Graph.PLOTTING_TYPE_AVERAGE);
			
		} else if (plotting_type == Graph.PLOTTING_TYPE_STDDEV_AVERAGE) {
			paintGraph(gc,x,y,start,end,res,PLOTTING_TYPE_STDDEV_AVERAGE_PLUS);
			paintGraph(gc,x,y,start,end,res,PLOTTING_TYPE_STDDEV_AVERAGE_MINUS);
			//plot average last - this one will drive the indicator
			paintGraph(gc,x,y,start,end,res,Graph.PLOTTING_TYPE_AVERAGE);

		} else {
			paintGraph(gc,x,y,start,end,res,plotting_type);
			
		}
	}

	public void paintGraph(GC gc, int xorigin, int yorigin, double start, double end, double res, int plot_type) {
		
		boolean found_indicator = false;
		double indicator_loc = isource.getIndicatorLocation();
		double indicator_val = last_drawn_value;
		int indicator_type = isource.getSourceType();

		boolean first = true;

		double px = 0;
		double py = 0;
		
		double last_clean_cx = 0;
		double last_clean_cy = 0;
		double last_cy = 0;
		
		//
		// X Offset
		//
//		start += xoffset;
//		end += xoffset;
		
		boolean empty = false;
		
		for (double i = start; i <= end; i+= res) {

			double pixel_x;
			double pixel_y;
			
			double cx = i;
			
			//
			// X offset
			//
			double cx_shifted = cx - xoffset;
			
			if (DRAW_ONLY_IF_DATA) {
				//dont draw past the maximum extent of the data
				if (cx_shifted > (source.getMax()+res)) continue;
				if (cx_shifted < (source.getMin()-res)) continue;
			}

			double cy = 0;
			
			if (plot_type == Graph.PLOTTING_TYPE_NEAREST) {
				//plot the closest (previous) value
				cy = source.getValueAt(cx_shifted);
			} else if (plot_type == Graph.PLOTTING_TYPE_AVERAGE) {
				cy = source.getAverageBetween(cx_shifted-res,cx_shifted);
			} else if (plot_type == Graph.PLOTTING_TYPE_SUM) {
				cy = source.getSumBetween(cx_shifted-res,cx_shifted);
			} else if (plot_type == Graph.PLOTTING_TYPE_MIN) {
				cy = source.getMinBetween(cx_shifted-res,cx_shifted);
			} else if (plot_type == Graph.PLOTTING_TYPE_MAX) {
				cy = source.getMaxBetween(cx_shifted-res,cx_shifted);
			} else if (plot_type == Graph.PLOTTING_TYPE_COUNT) {
				cy = source.getCountBetween(cx_shifted-res,cx_shifted);
			} else if (plot_type == Graph.PLOTTING_TYPE_STDDEV) {
				cy = source.getStandardDeviationBetween(cx_shifted-res,cx_shifted);
			} else if (plot_type == PLOTTING_TYPE_STDDEV_AVERAGE_PLUS) {
				cy = source.getAverageBetween(cx_shifted-res,cx_shifted) + source.getStandardDeviationBetween(cx_shifted-res,cx_shifted);
			} else if (plot_type == PLOTTING_TYPE_STDDEV_AVERAGE_MINUS) {
				cy = source.getAverageBetween(cx_shifted-res,cx_shifted) - source.getStandardDeviationBetween(cx_shifted-res,cx_shifted);
			} else if (plot_type == PLOTTING_TYPE_GRADIENT) {
				//get average
				cy = source.getAverageBetween(cx_shifted-res,cx_shifted);
			} else {
				//unrecognised type, default to average
				cy = source.getAverageBetween(cx_shifted-res,cx_shifted);
			}			
			
			//dont draw lines on the first run, we wont have valid previous pixel data
			if (first) {
				first = false;

//				double cy = source.getValueAt(cx);
	
				empty = Double.isNaN(cy);

				if (empty && (nodata == Graph.NODATA_DO_NOTHING)) {
					//draw nothing, need to start the graph all over again
					first = true;
					pixel_x = xslider.value2Pixel(cx);
					pixel_y = py;
										
				} else {
					if (empty && (nodata == Graph.NODATA_DRAW_PREVIOUS)) {
						cy = last_clean_cy;
					} else if (empty && (nodata == Graph.NODATA_DRAW_ZERO)) {
						cy = 0;
					} else {
						last_clean_cx = cx;
					}
					
					//store this for DRAW_PREVIOUS
					last_clean_cy = cy;
					
					//Y Offset
					cy += yoffset;
					
					//apply static scaling
					cy = cy * scaling;
		
					last_cy = cy;
		
					pixel_x = xslider.value2Pixel(cx);
					pixel_y = yslider.value2Pixel(cy);
				
					if (cy > tmpMaxValue) tmpMaxValue = cy;
					
					gc.drawLine(xorigin+(int)pixel_x,yorigin+(int)pixel_y,xorigin+(int)pixel_x,yorigin+(int)pixel_y);		
				}
								
			} else {

				boolean emptyPrevious = empty;
				empty = Double.isNaN(cy);

				if (empty && (nodata == Graph.NODATA_DO_NOTHING)) {
					//draw nothing, need to start the graph all over again
					first = true;
					pixel_x = xslider.value2Pixel(cx);
					pixel_y = py;

					//Indicator source type TIME - specifies the time value to show values at 
					if (indicator_type == IndicatorSource.TYPE_TIME) {
						if (indicator_loc > cx-res && indicator_loc < cx) { 	
							found_indicator = true;
							indicator_val = 0;
						}
					}
					//Indicator source type PIXEL - specifies the pixel to show values at 
					if (indicator_type == IndicatorSource.TYPE_PIXEL) {
						if (indicator_loc > px && indicator_loc < pixel_x) { 	
							found_indicator = true;
							indicator_val = 0;
						}
					}					
					
					//we had a data point just before this, but now we have none
					if (!emptyPrevious) {
						//draw a horizontal line rather than a dot for individual data values
						gc.drawLine(xorigin+(int)px,yorigin+(int)py,xorigin+(int)pixel_x,yorigin+(int)py);		
					}

					
				} else {
					if (empty && (nodata == Graph.NODATA_DRAW_PREVIOUS)) {
						cy = last_clean_cy;
					} else if (empty && (nodata == Graph.NODATA_DRAW_ZERO)) {
						cy = 0;
					}
					
					//special case for gradient
					if (plot_type == PLOTTING_TYPE_GRADIENT) {
						double deltay = cy - last_clean_cy;
						double deltax = (cx - last_clean_cx)/1000.0d;

						last_clean_cx = cx;
						
						//store this for DRAW_PREVIOUS
						last_clean_cy = cy;

						//gradient (y/x)
						cy = deltay/deltax;
						
					} else {
					
						//store this for DRAW_PREVIOUS
						last_clean_cy = cy;
					}
					
					//Y Offset
					cy += yoffset;
					
					//apply static scaling
					cy = cy * scaling;
	
					//Indicator source type TIME - specifies the time value to show values at 
					if (indicator_type == IndicatorSource.TYPE_TIME) {
						if (indicator_loc > cx-res && indicator_loc < cx) { 	
							found_indicator = true;
							indicator_val = cy;
						}
					}
	
					last_cy = cy;
		
					pixel_x = xslider.value2Pixel(cx);
					pixel_y = yslider.value2Pixel(cy);
	
					if (cy > tmpMaxValue) {
						tmpMaxValue = cy;
					}
					
					//Indicator source type PIXEL - specifies the pixel to show values at 
					if (indicator_type == IndicatorSource.TYPE_PIXEL) {
						if (indicator_loc > px && indicator_loc < pixel_x) { 	
							found_indicator = true;
							indicator_val = cy;
						}
					}
	
					gc.drawLine(xorigin+(int)px,yorigin+(int)py,xorigin+(int)pixel_x,yorigin+(int)pixel_y);
				
				}
			}
			px = pixel_x;
			py = pixel_y;
		
		}
		
		//this will be the value that appears on the indicator
		last_drawn_value = last_cy;
		
		//this is the most recent max value
		recentMaxValue = tmpMaxValue;

		if (isource == null || !found_indicator) {
			indicator.setBaseValue(yoffset);
			indicator.setValue(last_drawn_value);
		} else {
			indicator.setBaseValue(yoffset);
			indicator.setValue(indicator_val);
		}
		yslider.updateIndicators();
	}
	
	public void paintGraphNearest(GC gc, int x, int y, int w, int h) {
	}

	public int getPlottingType() {
		return plotting_type;
	}
	public void setPlottingType(int type) {
		plotting_type = type;
	}
	
	public void setPlottingPeriod(int type, double value) {
		plotting_period_type = type;
		plotting_period_value = value;
	}
	public int getPlottingPeriodType() {
		return plotting_period_type;
	}
	public double getPlottingPeriodValue() {
		return plotting_period_value;
	}
	
	public void setXOffset(double millis) {
		xoffset = millis;
	}
	public double getXOffset() {
		return xoffset;
	}
	
	public void setYOffset(double millis) {
		yoffset = millis;
	}
	public double getYOffset() {
		return yoffset;
	}
	
	
	public TimeZoomSlider getXSlider() {
		return xslider;	
	}

	public ZoomSlider getYSlider() {
		return yslider;	
	}
	public void setXSlider(TimeZoomSlider slider) {
		xslider = slider;
	}
	
	public void setYSlider(ZoomSlider slider) {
		if (use_indicators) {
			yslider.removeIndicator(indicator);
			indicator.dispose();
		}
		yslider = slider;
		if (use_indicators) {
			indicator = new ZoomIndicator(yslider,last_drawn_value,fg_col);
			yslider.addZoomIndicator(indicator);
		}
	}

	public void setStaticScaling(double mult) {
		scaling = mult;
	}
	public double getStaticScaling() {
		return scaling;	
	} 

}