/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.statistical.ui.widgets.zoomslider.internal;

import org.eclipse.swt.graphics.GC;

public class ZoomSliderTick extends java.lang.Object
{
    ZoomSlider slider;
    
    public double value;

    int pixel;
    
    public String representation;
    
    public boolean textVisible;
    
    public ZoomSliderTick()
    {
    }
    
    public ZoomSliderTick(ZoomSlider slider, double d)
    {
        this.slider = slider;
        
        value = d;
        
        pixel = (int)slider.value2Pixel(value);
        
        representation = ZoomSliderUtilities.double2String(value);
        
        textVisible = isTextVisible();
    }
    
    public int getPixel(){return pixel;}
    
    private String throughput(double value)
    {
        if (value == 0) return "";
        return String.valueOf(ZoomSliderUtilities.round(60000/value, 0.0001));
    }

    public void paintLine(GC g)
    {
        g.setForeground(slider.foregroundColor);
        
        if (slider.getOrientation() == ZoomSlider.VERTICAL)
        {
            if (textVisible)
            {
                g.drawLine(0, pixel, slider.getBounds().width, pixel);
            }
            else
            {
                g.drawLine(0, pixel, 5, pixel);
                g.drawLine(slider.getBounds().width - 5, pixel, slider.getBounds().width - 1, pixel);
            }    
        }
        else
        {
            if (textVisible)
            {
                g.drawLine(pixel, 0, pixel, slider.getBounds().height);
            }
            else
            {
                g.drawLine(pixel, 0, pixel, 5);
                g.drawLine(pixel, slider.getBounds().height - 5, pixel, slider.getBounds().height - 1);
            }    
        }
    }
    
    public void paintText(GC g)
    {
        if (textVisible)
        {
            int ascent = g.getFontMetrics().getAscent();
            
            int width = g.stringExtent(representation).x;
            
            int head = width;            
            
            int dotIndex = representation.indexOf(".");

            if (dotIndex != -1)
            {
                head = g.stringExtent(representation.substring(0, dotIndex)).x;
            }
            
            int xPos = 0;
            int yPos = 0;
            
			g.setBackground(slider.backgroundColor);
			
            if (slider.getOrientation() == ZoomSlider.VERTICAL)
            {
                xPos = slider.getDecimalPointPosition() - head;
                
                yPos = pixel - g.stringExtent(representation).y/2;
                
                g.fillRectangle(xPos - 2, yPos, width + 4, g.stringExtent(representation).y);
            }
            else
            {
                xPos = pixel - width/2;
                
                if (this.overlaps(slider.getLastVisible(), g))
                {
                    return;
                }    
                slider.setLastVisible(this);

                yPos = slider.getBounds().height/2 - slider.getFontHeight()/2 - 1;

                g.fillRectangle(xPos, yPos, width, ascent + 2);
            }                
            
			g.setForeground(slider.foregroundColor);
            g.drawString(representation, xPos, yPos-1, true);
        }    
    }
    
    protected boolean isTextVisible()
    {
        int startPoint = representation.indexOf('.');

        if (startPoint == -1)
        {
            startPoint = representation.length();
        }    
        
        int oom = slider.getIncrementOOM();
        
        if (oom >= 0) oom++;
        
        try
        {
            char significantChar = representation.charAt(startPoint - oom);
        
            if (significantChar == '0')
            {
                return true;
            }
            else if (significantChar == '5')
            {
                if (slider.getIncrement() != 5 * Math.pow(10, slider.getIncrementOOM()))
                {
                    return true;
                }
                else
                {
                    return false;
                }    
            }
            else if (significantChar == '2')
            {
                return false;
            }    
            else
            {
                return false;
            }
        }
        catch (StringIndexOutOfBoundsException e)
        {
            return true;
        }
    }
    
	public boolean overlaps(ZoomSliderTick tick, GC g)
	{
        if (tick == null)
        {
            return false;
        }
        
        if (tick.pixel > pixel)
        {
            if (tick.pixel < pixel + g.stringExtent(representation).x + 5)
            {
                return true;
            }    
        }
        else
        {
            if (tick.pixel > pixel - g.stringExtent(representation).x - 5)
            {
                return true;
            }
        }    

        return false;
	}
}
