/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LogAnalyzerSamplePage.java,v 1.10 2005/02/16 22:23:46 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.trace.sample.loganalyzer;

import java.io.File;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.trace.sample.HyadesSamplePlugin;
import org.eclipse.hyades.ui.internal.wizard.exampleproject.CreationWizard;
import org.eclipse.hyades.ui.internal.wizard.exampleproject.CreationWizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.help.WorkbenchHelp;
import org.eclipse.ui.internal.IHelpContextIds;
import org.eclipse.ui.internal.ide.IDEWorkbenchMessages;


/**
 * @author sleeloy
 *
 * This wizard page allows the user to specify the error log file to analyze, the access log file
 * to correlate and the symptom database that is used to analyze the log file.  This page also
 * allows the user to specify whether he/she wants to analyze the error log file or correlate the
 * access log.
 */
public class LogAnalyzerSamplePage extends CreationWizardPage implements Listener {

	// constants
	private static final int SIZING_TEXT_FIELD_WIDTH = 250;
	
	private boolean useDefaults = true;
	private boolean useDBDefaults = true;
	private boolean useErrorLogDefaults = true;
	private boolean analyzeLog = true;
	private boolean correlateLog = true;
	protected Text logfileFieldText;	
	protected Text symptomDBFieldText;	
	protected Text errorLogFieldText;	
	protected String initialLogFileValue;
	protected String initialSymptomDBValue;
	protected String initialErrorLogValue;
	private Label symptomDBLocationLabel;
	private Label errorLogLocationLabel;
	private Label logfileLocationLabel;
	private Button browseButton;
	private Button symptomDBBrowseButton;
	private Button errorLogBrowseButton;
	private LogAnalyzerCreationPage firstPage;
	private Button analyzeLogButton;
	private Button correlateLogButton; 
	private static final String APACHE_DEFAULT_LOG_FILE = "access.log";
	private static final String APACHE_DEFAULT_ERROR_LOG_FILE = "error.log";
	private static final String APACHE_DEFAULT_SYMPTOMDB_FILE = "symptomdb.xml";
	
	// the value the user has entered
	private String customLocationFieldValue;
	private String customSymptomDBLocationFieldValue;
	private String customErrorLogLocationFieldValue;
	
	private Text sdbFileNameText;
	private Label importSDBFileLabel;
	/**
	 * The location of the error log.
	 * @return String - represents the location of the error log.
	 */
	public String getErrorLogPath(){
		return errorLogFieldText.getText().trim();
	}
	/**
	 * The location of the access log file.
	 * @return String - represents the location of the access log file.
	 */
	public String getLogPath(){
		return logfileFieldText.getText().trim();
	}
	/**
	 * The location of the symptom database that is used to analyze the log file.
	 * @return String - represents the location of the symptom database.
	 */
	public String getSymptomDBFilePath(){
		if (useDBDefaults) return symptomDBFieldText.getText().trim();
		else return "/" + getProjectName() + "/" + sdbFileNameText.getText().trim();
	}
	
	/**
	 * Constructor.
	 * 
	 * @param creationWizard - the wizard that contains this page.
	 * @param pageName - the name of the page.
	 * @param configurationElement - the configuration element that contains information about this page.
	 */
	public LogAnalyzerSamplePage(
		CreationWizard creationWizard,
		String pageName,
		IConfigurationElement configurationElement, LogAnalyzerCreationPage firstPage) {
		super(creationWizard, pageName, configurationElement);
		customLocationFieldValue = ""; //$NON-NLS-1$
		customSymptomDBLocationFieldValue = ""; //$NON-NLS-1$
		customErrorLogLocationFieldValue = ""; //$NON-NLS-1$
		initialLogFileValue = getValue(configurationElement, "name");
		initialSymptomDBValue = getValue(configurationElement, "name");
		initialErrorLogValue = getValue(configurationElement, "name");
		setInitialSymptomDB(initialSymptomDBValue);
		setInitialErrorLog(initialErrorLogValue);
		setInitialLogFile(initialLogFileValue);
		this.firstPage = firstPage;


	}

    /**
     * Handles events. 
     */
	public void handleEvent(Event e) {
		setLogFileForSelection();
		setSymptomDBForSelection();
		setErrorLogForSelection();
	}

	private Listener logfileModifyListener = new Listener() {
		public void handleEvent(Event e) {
			setPageComplete(validatePage());
		}
	};
	
	private Listener errorLogModifyListener = new Listener() {
		public void handleEvent(Event e) {
			setPageComplete(validatePage());
		}
	};
	
	
	private Listener symptomDBModifyListener = new Listener() {
		public void handleEvent(Event e) {
			setPageComplete(validatePage());
		}
	};
	

	/**
	 * Set the log file location to the default log file location if we are set to useDefaults.
	 */
	private void setLogFileForSelection() {
		if (useDefaults) 
		logfileFieldText.setText(getDefaultLogFileForName(getProjectFieldValue()));
	}

	/**
	 * Set the Symptom Database location to the default Symptom Database location if we are set to useDefaults.
	 */
	private void setSymptomDBForSelection() {
		if (useDBDefaults) 
		symptomDBFieldText.setText(getDefaultSymptomDBForName(getProjectFieldValue()));
	}
	
	/**
	 * Set the ErrorLog location to the default Symptom Database location if we are set to useDefaults.
	 */
	private void setErrorLogForSelection() {
		if (useErrorLogDefaults) 
		errorLogFieldText.setText(getDefaultErrorLogForName(getProjectFieldValue()));
	}

	/**
	 * Returns the value of the project field
	 * with leading and trailing spaces removed.
	 * 
	 * @return the logfile in the field
	 */
	private String getProjectFieldValue() {
		if (firstPage == null)
			return ""; //$NON-NLS-1$
		String projectName = firstPage.getProjectName();
		if (projectName == null)
			return ""; //$NON-NLS-1$
		else
			return projectName.trim();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createControl(Composite parent) {
		Composite composite = new Composite(parent, SWT.NULL);
		composite.setFont(parent.getFont());
	
		initializeDialogUnits(parent);

		WorkbenchHelp.setHelp(composite, IHelpContextIds.NEW_PROJECT_WIZARD_PAGE);
	
		composite.setLayout(new GridLayout());
		composite.setLayoutData(new GridData(GridData.FILL_BOTH));
	
		createLogFileControls(composite);
		createErrorLogControls(composite);
		createSymptomDBControls(composite);
		createLogFileOptions(composite);
		setPageComplete(validatePage());
		// Show description on opening
		setErrorMessage(null);
		setMessage(null);
		setControl(composite);
		
		firstPage.getProjectText().addListener(SWT.Modify, this);
		
		
	}
	
	/**
	 * Creates the log file specification controls.
	 *
	 * @param parent the parent composite
	 */
	private final void createLogFileOptions(Composite parent) {
		Font font = parent.getFont();
		// project specification group
		Group logOptionGroup = new Group(parent, SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.numColumns = 3;
		logOptionGroup.setLayout(layout);
		logOptionGroup.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		logOptionGroup.setFont(font);
		logOptionGroup.setText(HyadesSamplePlugin.getString("OPTIONS")); //$NON-NLS-1$

		analyzeLogButton = new Button(logOptionGroup, SWT.CHECK | SWT.RIGHT);
		analyzeLogButton.setText(HyadesSamplePlugin.getString("ANALYZE_LOG_LABEL")); //$NON-NLS-1$
		analyzeLogButton.setSelection(true);
		analyzeLogButton.setFont(font);

		correlateLogButton = new Button(logOptionGroup, SWT.CHECK | SWT.RIGHT);
		correlateLogButton.setText(HyadesSamplePlugin.getString("CORRELATE_LOG_LABEL")); //$NON-NLS-1$
		correlateLogButton.setSelection(true);
		correlateLogButton.setFont(font);
		
		GridData buttonData = new GridData();
		buttonData.horizontalSpan = 3;
		analyzeLogButton.setLayoutData(buttonData);
		buttonData = new GridData();
		buttonData.horizontalSpan = 3;
		correlateLogButton.setLayoutData(buttonData);

		SelectionListener listener = new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				analyzeLog = analyzeLogButton.getSelection();
			}
		};
		analyzeLogButton.addSelectionListener(listener);
		
		SelectionListener correlateListener = new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				correlateLog = correlateLogButton.getSelection();
			}
		};
		correlateLogButton.addSelectionListener(correlateListener);
	}

	
	/**
	 * Creates the log file specification controls.
	 *
	 * @param parent the parent composite
	 */
	private final void createLogFileControls(Composite parent) {
		Font font = parent.getFont();
		// project specification group
		Group logFileGroup = new Group(parent, SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.numColumns = 3;
		logFileGroup.setLayout(layout);
		logFileGroup.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		logFileGroup.setFont(font);
		logFileGroup.setText(HyadesSamplePlugin.getString("LOG_FILE_LABEL")); //$NON-NLS-1$

		final Button useDefaultsButton = new Button(logFileGroup, SWT.CHECK | SWT.RIGHT);
		useDefaultsButton.setText(IDEWorkbenchMessages.getString("WizardNewProjectCreationPage.useDefaultLabel")); //$NON-NLS-1$
		useDefaultsButton.setSelection(useDefaults);
		useDefaultsButton.setFont(font);

		GridData buttonData = new GridData();
		buttonData.horizontalSpan = 3;
		useDefaultsButton.setLayoutData(buttonData);

		createUserSpecifiedProjectLocationGroup(logFileGroup, !useDefaults);

		SelectionListener listener = new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				useDefaults = useDefaultsButton.getSelection();
				browseButton.setEnabled(!useDefaults);
				logfileFieldText.setEnabled(!useDefaults);
				logfileLocationLabel.setEnabled(!useDefaults);
				if (useDefaults) {			
					customLocationFieldValue = logfileFieldText.getText();
					setLogFileForSelection();
				} else {
					logfileFieldText.setText(customLocationFieldValue);
				}
			}
		};
		useDefaultsButton.addSelectionListener(listener);
	}
	/**
	 * Creates the Error Log specification controls.
	 *
	 * @param parent the parent composite
	 */
	private final void createErrorLogControls(Composite parent) {
		Font font = parent.getFont();
		// project specification group
		Group errorLogGroup = new Group(parent, SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.numColumns = 3;
		errorLogGroup.setLayout(layout);
		errorLogGroup.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		errorLogGroup.setFont(font);
		errorLogGroup.setText(HyadesSamplePlugin.getString("ERRORLOG_FILE_LABEL")); //$NON-NLS-1$

		final Button useDefaultsButton = new Button(errorLogGroup, SWT.CHECK | SWT.RIGHT);
		useDefaultsButton.setText(IDEWorkbenchMessages.getString("WizardNewProjectCreationPage.useDefaultLabel")); //$NON-NLS-1$
		useDefaultsButton.setSelection(useErrorLogDefaults);
		useDefaultsButton.setFont(font);

		GridData buttonData = new GridData();
		buttonData.horizontalSpan = 3;
		useDefaultsButton.setLayoutData(buttonData);

		createUserSpecifiedErrorLogLocationGroup(errorLogGroup, !useDefaults);

		SelectionListener listener = new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				useErrorLogDefaults = useDefaultsButton.getSelection();
				errorLogBrowseButton.setEnabled(!useErrorLogDefaults);
				errorLogFieldText.setEnabled(!useErrorLogDefaults);
				errorLogLocationLabel.setEnabled(!useErrorLogDefaults);
				if (useErrorLogDefaults) {			
					customErrorLogLocationFieldValue = errorLogFieldText.getText();
					setErrorLogForSelection();
				} else {
					errorLogFieldText.setText(customErrorLogLocationFieldValue);
				}
			}
		};
		useDefaultsButton.addSelectionListener(listener);
	}

	/**
	 * Creates the Symptom Database specification controls.
	 *
	 * @param parent the parent composite
	 */
	private final void createSymptomDBControls(Composite parent) {
		Font font = parent.getFont();
		// project specification group
		Group symptomDBGroup = new Group(parent, SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.numColumns = 3;
		symptomDBGroup.setLayout(layout);
		symptomDBGroup.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		symptomDBGroup.setFont(font);
		symptomDBGroup.setText(HyadesSamplePlugin.getString("SYMPTOMDB_FILE_LABEL")); //$NON-NLS-1$

		final Button useDefaultsButton = new Button(symptomDBGroup, SWT.CHECK | SWT.RIGHT);
		useDefaultsButton.setText(IDEWorkbenchMessages.getString("WizardNewProjectCreationPage.useDefaultLabel")); //$NON-NLS-1$
		useDefaultsButton.setSelection(useDBDefaults);
		useDefaultsButton.setFont(font);

		GridData buttonData = new GridData();
		buttonData.horizontalSpan = 3;
		useDefaultsButton.setLayoutData(buttonData);
			
		GridData labelData = new GridData();
		labelData.horizontalSpan = 3;
				
		
		//		location label
		importSDBFileLabel = new Label(symptomDBGroup,SWT.NONE);
		importSDBFileLabel.setText(HyadesSamplePlugin.getString("IMPORT_SYMPTOM_DB")); //$NON-NLS-1$
		importSDBFileLabel.setFont(font);
		importSDBFileLabel.setLayoutData(labelData);
		
		if (useDefaults) importSDBFileLabel.setVisible(false);
	

		createUserSpecifiedSymptomDBLocationGroup(symptomDBGroup, !useDefaults);

		//		for the name of the sdb in the workspace
		createSDBFileNameControls(parent);	
		
		SelectionListener listener = new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				useDBDefaults = useDefaultsButton.getSelection();
				symptomDBBrowseButton.setEnabled(!useDBDefaults);
				symptomDBFieldText.setEnabled(!useDBDefaults);
				symptomDBLocationLabel.setEnabled(!useDBDefaults);
				importSDBFileLabel.setVisible(!useDBDefaults);
				
				if (useDBDefaults) {			
					customSymptomDBLocationFieldValue = symptomDBFieldText.getText();
					setSymptomDBForSelection();
				} else {
					symptomDBFieldText.setText(customSymptomDBLocationFieldValue);
				}
			}
		};
		useDefaultsButton.addSelectionListener(listener);
	}

	/**
	 * Creates the project location specification controls.
	 *
	 * @param projectGroup the parent composite
	 * @param boolean - the initial enabled state of the widgets created
	 */
	private void createUserSpecifiedProjectLocationGroup(Composite projectGroup, boolean enabled) {
	
		Font font = projectGroup.getFont();

		// location label
		logfileLocationLabel = new Label(projectGroup,SWT.NONE);
		logfileLocationLabel.setText(HyadesSamplePlugin.getString("LOG_FILE_TEXT_LABEL")); //$NON-NLS-1$
		logfileLocationLabel.setEnabled(enabled);
		logfileLocationLabel.setFont(font);

		// project location entry field
		logfileFieldText = new Text(projectGroup, SWT.BORDER);
		GridData data = new GridData(GridData.FILL_HORIZONTAL);
		data.widthHint = SIZING_TEXT_FIELD_WIDTH;
		logfileFieldText.setLayoutData(data);
		logfileFieldText.setEnabled(enabled);
		logfileFieldText.setFont(font);

		// browse button
		browseButton = new Button(projectGroup, SWT.PUSH);
		browseButton.setText(IDEWorkbenchMessages.getString("WizardNewProjectCreationPage.browseLabel")); //$NON-NLS-1$
		browseButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent event) {
				handleLocationBrowseButtonPressed();
			}
		});

		browseButton.setEnabled(enabled);
		browseButton.setFont(font);
		setButtonLayoutData(browseButton);

		// Set the initial value first before listener
		// to avoid handling an event during the creation.
		if(initialLogFileValue == null)
			logfileFieldText.setText(Platform.getLocation().toOSString());
		else
		logfileFieldText.setText(initialLogFileValue);
		logfileFieldText.addListener(SWT.Modify, logfileModifyListener);
	}
	
	/**
	 * Creates the Symptom Database location specification controls.
	 *
	 * @param symptomDBGroup the parent composite
	 * @param boolean - the initial enabled state of the widgets created
	 */
	private void createUserSpecifiedSymptomDBLocationGroup(Composite symptomDBGroup, boolean enabled) {
	
		Font font = symptomDBGroup.getFont();

		// location label
		symptomDBLocationLabel = new Label(symptomDBGroup,SWT.NONE);
		symptomDBLocationLabel.setText(HyadesSamplePlugin.getString("LOG_FILE_TEXT_LABEL")); //$NON-NLS-1$
		symptomDBLocationLabel.setEnabled(enabled);
		symptomDBLocationLabel.setFont(font);

		// project location entry field
		symptomDBFieldText = new Text(symptomDBGroup, SWT.BORDER);
		GridData data = new GridData(GridData.FILL_HORIZONTAL);
		data.widthHint = SIZING_TEXT_FIELD_WIDTH;
		symptomDBFieldText.setLayoutData(data);
		symptomDBFieldText.setEnabled(enabled);
		symptomDBFieldText.setFont(font);

		// browse button
		symptomDBBrowseButton = new Button(symptomDBGroup, SWT.PUSH);
		symptomDBBrowseButton.setText(IDEWorkbenchMessages.getString("WizardNewProjectCreationPage.browseLabel")); //$NON-NLS-1$
		symptomDBBrowseButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent event) {
				handleSymptomDBLocationBrowseButtonPressed();
			}
		});

		symptomDBBrowseButton.setEnabled(enabled);
		symptomDBBrowseButton.setFont(font);
		setButtonLayoutData(symptomDBBrowseButton);

		// Set the initial value first before listener
		// to avoid handling an event during the creation.
		if(initialSymptomDBValue == null)
			symptomDBFieldText.setText(Platform.getLocation().toOSString());
		else
		symptomDBFieldText.setText(initialSymptomDBValue);
		symptomDBFieldText.addListener(SWT.Modify, logfileModifyListener);
	}
	
	private final void createSDBFileNameControls(Composite parent)
	{
		
		
		//this field is hidden, because by default in the workspace
		//the sdb will be named symptomdb.trcdbxmi
		//however, if in the future, we decide to allow the user to change this, 
		//then set visible to true

		sdbFileNameText = new Text(parent, SWT.BORDER);
		int i = initialSymptomDBValue.lastIndexOf("/") + 1;
		sdbFileNameText.setText(initialSymptomDBValue.substring(i,initialSymptomDBValue.length()));
		sdbFileNameText.setVisible(false);
		
		

	}
	
	
	/**
	 * Creates the Error Log file location specification controls.
	 *
	 * @param errorLogGroup the parent composite
	 * @param boolean - the initial enabled state of the widgets created
	 */
	private void createUserSpecifiedErrorLogLocationGroup(Composite errorLogGroup, boolean enabled) {
	
		Font font = errorLogGroup.getFont();

		// location label
		errorLogLocationLabel = new Label(errorLogGroup,SWT.NONE);
		errorLogLocationLabel.setText(HyadesSamplePlugin.getString("LOG_FILE_TEXT_LABEL")); //$NON-NLS-1$
		errorLogLocationLabel.setEnabled(enabled);
		errorLogLocationLabel.setFont(font);

		// project location entry field
		errorLogFieldText = new Text(errorLogGroup, SWT.BORDER);
		GridData data = new GridData(GridData.FILL_HORIZONTAL);
		data.widthHint = SIZING_TEXT_FIELD_WIDTH;
		errorLogFieldText.setLayoutData(data);
		errorLogFieldText.setEnabled(enabled);
		errorLogFieldText.setFont(font);

		// browse button
		errorLogBrowseButton = new Button(errorLogGroup, SWT.PUSH);
		errorLogBrowseButton.setText(IDEWorkbenchMessages.getString("WizardNewProjectCreationPage.browseLabel")); //$NON-NLS-1$
		errorLogBrowseButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent event) {
				handleErrorLogLocationBrowseButtonPressed();
			}
		});

		errorLogBrowseButton.setEnabled(enabled);
		errorLogBrowseButton.setFont(font);
		setButtonLayoutData(errorLogBrowseButton);

		// Set the initial value first before listener
		// to avoid handling an event during the creation.
		if(initialErrorLogValue == null)
			errorLogFieldText.setText(Platform.getLocation().toOSString());
		else
		errorLogFieldText.setText(initialErrorLogValue);
		errorLogFieldText.addListener(SWT.Modify, logfileModifyListener);
	}
	


	/**
	 *	Open an appropriate directory browser
	 */
	private void handleLocationBrowseButtonPressed() {
		FileDialog dialog = new FileDialog(logfileFieldText.getShell());
		dialog.setFileName(logfileFieldText.getText());
	
		String dirName = getProjectFieldValue();
		if (!dirName.equals("")) { //$NON-NLS-1$
			File path = new File(dirName);
			if (path.exists())
				dialog.setFilterPath(new Path(dirName).toOSString());
		}
	
		String selectedFile = dialog.open();
		if (selectedFile != null) {
			customLocationFieldValue = selectedFile;
			logfileFieldText.setText(customLocationFieldValue);
		}
	}
	/**
	 *	Open an appropriate directory browser
	 */
	private void handleErrorLogLocationBrowseButtonPressed() {
		FileDialog dialog = new FileDialog(errorLogFieldText.getShell());
		dialog.setFileName(errorLogFieldText.getText());
	
		String dirName = getProjectFieldValue();
		if (!dirName.equals("")) { //$NON-NLS-1$
			File path = new File(dirName);
			if (path.exists())
				dialog.setFilterPath(new Path(dirName).toOSString());
		}
	
		String selectedFile = dialog.open();
		if (selectedFile != null) {
			customErrorLogLocationFieldValue = selectedFile;
			errorLogFieldText.setText(customErrorLogLocationFieldValue);
		}
	}
			
	/**
	 *	Open an appropriate directory browser
	 */
	private void handleSymptomDBLocationBrowseButtonPressed() {
		FileDialog dialog = new FileDialog(symptomDBFieldText.getShell());
		dialog.setFileName(symptomDBFieldText.getText());
	
		String dirName = getProjectFieldValue();
		if (!dirName.equals("")) { //$NON-NLS-1$
			File path = new File(dirName);
			if (path.exists())
				dialog.setFilterPath(new Path(dirName).toOSString());
		}
	
		String selectedFile = dialog.open();
		if (selectedFile != null) {
			customSymptomDBLocationFieldValue = selectedFile;
			symptomDBFieldText.setText(customSymptomDBLocationFieldValue);
		}
	}

	
	/**
	 * Sets the initial log file that this page will use when
	 * created. The name is ignored if the createControl(Composite)
	 * method has already been called. Leading and trailing spaces
	 * in the name are ignored.
	 * 
	 * @param name of the log file for this page
	 */
	public void setInitialLogFile(String name) {
		if (name == null)
		initialLogFileValue = null;
		else{
			initialLogFileValue = name.trim();
			initialLogFileValue = getDefaultLogFileForName(initialLogFileValue);
		}
	}	
	
	/**
		 * sets the access log text field specified by the user
		 * called by log analyzer creation page
		 * @param name
		 */
		public void setAccessLog(String name)
		{
			String temp = name + File.separator + APACHE_DEFAULT_LOG_FILE;
			if (logfileFieldText != null) logfileFieldText.setText(new Path(temp).toString());
		}
	/**
	 * Sets the initial Error Log file that this page will use when
	 * created. The name is ignored if the createControl(Composite)
	 * method has already been called. Leading and trailing spaces
	 * in the name are ignored.
	 * 
	 * @param name of the log file for this page
	 */
	public void setInitialErrorLog(String name) {
		if (name == null)
		initialErrorLogValue = null;
		else{
			initialErrorLogValue = name.trim();
			initialErrorLogValue = getDefaultErrorLogForName(initialErrorLogValue);
		}
	}
	
	/**
	 * sets the error log text field specified by the user
	 * called by log analyzer creation page
	 * @param name
	 */
	public void setErrorLog(String name)
	{
		String temp = name + File.separator + APACHE_DEFAULT_ERROR_LOG_FILE;
		if (errorLogFieldText != null) errorLogFieldText.setText(new Path(temp).toString());
	}
		


	
	/**
	 * Sets the initial Symptom Database file that this page will use when
	 * created. The name is ignored if the createControl(Composite)
	 * method has already been called. Leading and trailing spaces
	 * in the name are ignored.
	 * 
	 * @param name of the log file for this page
	 */
	public void setInitialSymptomDB(String name) {
		if (name == null)
		initialSymptomDBValue = null;
		else{
			initialSymptomDBValue = name.trim();
			initialSymptomDBValue = getDefaultSymptomDBForName(initialSymptomDBValue);
		}
	}	
	
	/**
	 * Get the defualt Log file location for the provided name.
	 * @return String
	 * @param String
	 */ 
	private String getDefaultLogFileForName(String nameValue) {
		IPath defaultPath = Platform.getLocation().append(nameValue);
		return defaultPath.toOSString() + File.separator + APACHE_DEFAULT_LOG_FILE;
//		String projectVal = getProjectFieldValue();
//		if (projectVal.equals("")) projectVal = nameValue;
//		return "/" + projectVal + "/" + APACHE_DEFAULT_LOG_FILE;
	}

	/**
	 * Get the default Symptom Database file location for the provided name.
	 * @return String
	 * @param String
	 */ 
	private String getDefaultSymptomDBForName(String nameValue) {
		IPath defaultPath = Platform.getLocation().append(nameValue);
		String projectVal = getProjectFieldValue();
		if (projectVal.equals("")) projectVal = nameValue;
		return "/" + projectVal + "/" + APACHE_DEFAULT_SYMPTOMDB_FILE;
	}

	/**
	 * Get the default Error file location for the provided name.
	 * @return String
	 * @param String
	 */ 
	private String getDefaultErrorLogForName(String nameValue) {
		IPath defaultPath = Platform.getLocation().append(nameValue);
		return defaultPath.toOSString() + File.separator + APACHE_DEFAULT_ERROR_LOG_FILE;
//		String projectVal = getProjectFieldValue();
//		if (projectVal.equals("")) projectVal = nameValue;
//		return "/" + projectVal + "/" + APACHE_DEFAULT_ERROR_LOG_FILE;
	}
	
	/**
	 * @see org.eclipse.jface.dialogs.IDialogPage#setVisible(boolean)
	 */
	public void setVisible(boolean visible) {
		this.getControl().setVisible(visible);
	}

	/**
	 * Returns the value of a given attribute in a configuration
	 * element.  If the configuration element doesn't have such attribute then
	 * this method returns an empty string.
	 * @param configurationElement
	 * @param attribute
	 * @return A not null String
	 */
	private String getValue(IConfigurationElement configurationElement, String attribute)
	{
		String value = configurationElement.getAttribute(attribute);
		if (value == null)
			return "";
			
		return value;
	}
	
	/**
	 * Returns whether this page's controls currently all contain valid 
	 * values.
	 *
	 * @return <code>true</code> if all controls are valid, and
	 *   <code>false</code> if at least one is invalid
	 */
	protected boolean validatePage() {

		String logFieldText = logfileFieldText.getText().trim();
		if (logFieldText.equals("")) { //$NON-NLS-1$
			setErrorMessage(null);
			setMessage(HyadesSamplePlugin.getString("_ERROR_LOG_FILE_IS_EMPTY")); //$NON-NLS-1$
			return false;
		}

		if (!useDefaults){
			File file = new File(logFieldText);
			if (!file.isFile()) {
				setErrorMessage(HyadesSamplePlugin.getString("_ERROR_LOG_FILE_IS_INVALID")); //$NON-NLS-1$
				return false;
			}
		}
		String symptomDBFieldString = symptomDBFieldText.getText().trim();
		if (symptomDBFieldString.equals("")) { //$NON-NLS-1$
			setErrorMessage(null);
			setMessage(HyadesSamplePlugin.getString("_ERROR_SYMPTOMDB_IS_EMPTY")); //$NON-NLS-1$
			return false;
		}
		
		if (!useDBDefaults)
		{
			if(!(symptomDBFieldString.endsWith(".xml")))
			{
				setErrorMessage(null);
				setMessage(HyadesSamplePlugin.getString("_ERROR_ERRORSDB_FILE_IS_INVALID")); //$NON-NLS-1$
				return false;
			}
			
			File sfile = new File(symptomDBFieldString);
			if (!sfile.isFile()) {
				setErrorMessage(HyadesSamplePlugin.getString("_ERROR_SYMPTOMDB_IS_INVALID")); //$NON-NLS-1$
				return false;
			}
		}
		
		String errorLogFieldString = errorLogFieldText.getText().trim();
		if (errorLogFieldString.equals("")) { //$NON-NLS-1$
			setErrorMessage(null);
			setMessage(HyadesSamplePlugin.getString("_ERROR_ERRORLOG_FILE_IS_EMPTY")); //$NON-NLS-1$
			return false;
		}
	
		if (!useErrorLogDefaults){
			File sfile = new File(errorLogFieldString);
			if (!sfile.isFile()) {
				setErrorMessage(HyadesSamplePlugin.getString("_ERROR_ERRORLOG_FILE_IS_INVALID")); //$NON-NLS-1$
				return false;
			}
		}

		setErrorMessage(null);
		setMessage(null);
		return true;
	}
	
	
	

	/**
	 * Returns the analyze check box widget.
	 * @return Button - represents the analyze check box widget.
	 */
	public Button getAnalyzeLogButton() {
		return analyzeLogButton;
	}

	/**
	 * Returns the correlate check box widget.
	 * @return Button - represents the correlate check box widget.
	 */
	public Button getCorrelateLogButton() {
		return correlateLogButton;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.dialogs.WizardNewProjectCreationPage#getProjectName()
	 */
	public String getProjectName() {	
		return firstPage.getProjectName();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.internal.wizard.exampleproject.CreationWizardPage#getInitialProjectName()
	 */
	public String getInitialProjectName() {
		return firstPage.getProjectName();
	}
	
		
	public String getSDBFileName()
	{
		if (useDBDefaults) return "";
		return symptomDBFieldText.getText().trim();
	}

}
