/********************************************************************** 
 * Copyright (c) 2005 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: AutomationServer.java,v 1.3 2005/05/20 21:38:57 sschneid Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.automation.server;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.ObjectInputStream;

import org.eclipse.core.runtime.IPlatformRunnable;
import org.eclipse.hyades.automation.core.Service;

/**
 * @author Scott E. Schneider
 * 
 * The automation server provides an internal front-end to automatable published
 * services but is primarly used by the supplied automation client to service
 * the requests of external automation clients. This public class is an internal
 * class that should not be used directly by external clients; external clients
 * should use the automation client adaptation layer to automate services.
 * 
 * @provisional
 */
public class AutomationServer implements Service.Discoverable, Service.Executable, IPlatformRunnable {

    /**
     * The command that identifies the automation value
     */
    private static final String AUTOMATION_COMMAND_IDENTIFIER = "hyades.automation.command";

    /**
     * The command that identifies the automation data
     */
    private static final String AUTOMATION_DATA_IDENTIFIER = "hyades.automation.data";

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.automation.core.Service.Discoverable#discover(java.lang.String)
     */
    public boolean discover(String identifier) {
        return true; // currently not implemented
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.automation.core.Service.Executable#execute(org.eclipse.hyades.automation.core.Service)
     */
    public Object execute(Service service) {
        return service.execute();
    }

    /**
     * Returns the automation data file name
     * 
     * @return the fully-qualified automation data identifier
     */
    private String getAutomationDataFileName() {
        return System.getProperty(AutomationServer.AUTOMATION_DATA_IDENTIFIER);
    }

    /**
     * Restores the service memento
     * 
     * @return the opaque memento
     */
    private Service.Memento restoreMemento() {
        try {
            FileInputStream fileInput = new FileInputStream(this.getAutomationDataFileName());
            ObjectInputStream objectInput = new ObjectInputStream(fileInput);
            Service.Memento memento = (Service.Memento) objectInput.readObject();
            objectInput.close();
            fileInput.close();
            return memento;
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        } catch (ClassNotFoundException e) {
            e.printStackTrace();
        }
        return null;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.core.runtime.IPlatformRunnable#run(java.lang.Object)
     */
    public Object run(Object object) {

        // Retrieve the automation command specified (others might include
        // discover, information, etc)
        String command = System.getProperty(AutomationServer.AUTOMATION_COMMAND_IDENTIFIER);

        // Only execute is supported for this release
        if (command.equalsIgnoreCase("execute")) {
            Service.Memento memento = this.restoreMemento();

            /*
             * Create a server-side service proxy that will hook up the concrete
             * implementation based on extensions found with given extension --
             * execute once instantiated and return result
             */
            Service service = new ServiceProxy(memento);
            return this.execute(service);
        }

        // Unsupported commands return null (currently this return goes nowhere)
        return null;

    }

}