/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ICorrelatorData.java,v 1.4 2005/02/16 22:20:15 qiyanli Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.collection.correlation;

import org.eclipse.hyades.collection.correlation.exceptions.InsufficientBufferLengthException;

/**
 * This interface defines the concept of a correlator, a type of object capable of holding customized context
 * information.  The key property of a correlator is that it can be transformed into differet formats, such as
 * binary and XML.
 * 
 * @author Qiyan Li
 */
public interface ICorrelatorData {
    
    /**
     * Returns the application associated with this correlator where application-instance-specific data can
     * be retrieved.
     * 
     * @return  the application associated with this correlator.
     */
    public IApplication getApplication();
	
    /**
     * Converts an array of bytes into a correlator.
     * 
     * @param data      the data buffer from which the correlation information is to be read
     * @param offset    the starting position to read the correltion data
     * @param length    the length (in bytes) of the entire buffer for the correlation data
     * @return          the index of the byte right after the last byte of the correlation data read.
     * @throws InsufficientBufferLengthException    If the buffer is not long enough for holding the correlation data
     */
	public int readBinary(byte[] data, int offset, int length) throws InsufficientBufferLengthException;
	
    /**
     * Converts this correlator into an array of bytes, and saves them in the buffer.
     * 
     * @param data      the data buffer to which the correlator bytes are to be written
     * @param offset    the starting position to write the correltion data
     * @param length    the length (in bytes) of the buffer for the correlation data
     * @return          the index of the byte right after the last byte of the correlator data written.
     * @throws InsufficientBufferLengthException    If the buffer is not long enough for holding the correlation data
     */
	public int writeBinary(byte[] data, int offset, int length) throws InsufficientBufferLengthException;
    
    /**
     * Converts an XML fragment into a correlator.
     * 
     * @param data      the data buffer from which the correlation information is to be read
     * @param offset    the starting position to read the correltion data
     * @param length    the length (in characters) of the entire buffer for the correlation data
     * @return          the index of the character right after the last character of the correlation data read.
     */
    public int readXML(StringBuffer data, int offset, int length);
    
    /**
     * Converts this correlator into an XML fragment, and saves them in the string buffer.
     * 
     * @param data  the data buffer to which the correlator information is to be written
     * @return      the string buffer with the information about this correlator attached at the end.
     */
    public StringBuffer writeXML(StringBuffer data);
}
