/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TIUtils.java,v 1.3 2005/05/09 08:34:18 dguilbaud Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.uml2sd.trace.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.loaders.trace.TraceUtils;
import org.eclipse.hyades.models.hierarchy.TRCAgent;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCCollectionMode;
import org.eclipse.hyades.models.hierarchy.TRCConfiguration;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCOption;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.models.trace.TRCFullMethodInvocation;
import org.eclipse.hyades.models.trace.TRCProcess;
import org.eclipse.hyades.models.trace.TRCThread;
import org.eclipse.hyades.trace.internal.ui.PDContentProvider;
import org.eclipse.hyades.trace.ui.HyadesConstants;
import org.eclipse.hyades.trace.ui.HyadesUtil;
import org.eclipse.hyades.ui.extension.INavigatorItem;
import org.eclipse.hyades.ui.internal.navigator.INavigator;
import org.eclipse.hyades.uml2sd.trace.TraceSDPlugin;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceProcess;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceProcesses;
import org.eclipse.hyades.uml2sd.ui.actions.widgets.Criteria;
import org.eclipse.jface.viewers.IStructuredSelection;

/**
 * Utils
 */
public class TIUtils {
	
	/**
	 * This method searches for the reference of the TRCProcess 
	 * selected in the profiling monitor view
	 */
	static public void searchTRCProcesses(EObject object, TraceProcesses list, IProgressMonitor monitor) {
		
		if (monitor.isCanceled()) {
			return;
		}
		if (object instanceof TRCMonitor) {
			TRCMonitor m = (TRCMonitor)object;
			if (m.getNodes() == null ||
				m.getNodes().isEmpty()) {
				return;
			}
			for (Iterator i = m.getNodes().iterator(); i.hasNext() && !monitor.isCanceled(); ) {
				searchTRCProcesses((EObject)i.next(), list, monitor);
			}
		} else if (object instanceof TRCNode) {
			TRCNode n = (TRCNode)object;
			if (n.getProcessProxies() == null ||
				n.getProcessProxies().isEmpty()) {
				return;
			}
			for (Iterator i = n.getProcessProxies().iterator(); i.hasNext() && !monitor.isCanceled(); ) {
				searchTRCProcesses((EObject)i.next(), list, monitor);
			}
		} else if (object instanceof TRCProcessProxy) {
			TRCProcessProxy pp = (TRCProcessProxy)object;
			if (pp.getAgentProxies() == null ||
				pp.getAgentProxies().isEmpty()) {
				return;
			}
			for (Iterator i = pp.getAgentProxies().iterator(); i.hasNext() && !monitor.isCanceled(); ) {
				searchTRCProcesses((EObject)i.next(), list, monitor);
			}
		} else if (object instanceof TRCAgentProxy &&
				   ((TRCAgentProxy) object).getType().equals(HyadesConstants.PROFILE_AGENT_TYPE)) {
			TRCAgent agent = ((TRCAgentProxy) object).getAgent();
			TRCProcess process = TraceUtils.getProcess(agent);
			if (process != null) {
				EList invocations = process.getInvocations();
				if (invocations != null &&
					invocations.size() > 0 &&
					invocations.get(0) instanceof TRCFullMethodInvocation) {
					list.add(new TraceProcess(process));
				}
			}
		}
	}

	public static List getProfileAgents(TRCProcessProxy processProxy) {
		List profileAgents = new ArrayList();
		if(processProxy!=null){
			Collection agents = processProxy.getAgentProxies();
			TRCAgentProxy agentProxy = null;
			for (Iterator iterator = agents.iterator();	iterator.hasNext();	) {
				agentProxy = (TRCAgentProxy) iterator.next();
				if(HyadesConstants.PROFILE_AGENT_TYPE.equals(agentProxy.getType())) {
					profileAgents.add(agentProxy);
				}					
			}
		}			
		return profileAgents;
	}

	/**
	 * @param methodInvocation
	 * @return
	 */
	public static double getAbsoluteEntryTime(TRCFullMethodInvocation methodInvocation) {
		return getAbsoluteEntryTime(methodInvocation.getProcess().getAgent())+
			   methodInvocation.getEntryTime();
	}

	/**
	 * @param agent
	 * @return
	 */
	public static double getAbsoluteEntryTime(TRCAgent agent) {
		double delta = agent.getAgentProxy().getProcessProxy().getNode().getDeltaTime()*1E-6;
		return delta+agent.getStartTime();
	}

	/**
	 * @param process
	 * @return
	 */
	public static double getAbsoluteEntryTime(TRCProcess process) {
		return getAbsoluteEntryTime(process.getAgent())+
			   process.getStartTime();
	}

	/**
	 * @param thread
	 * @return
	 */
	public static double getAbsoluteEntryTime(TRCThread thread) {
		return getAbsoluteEntryTime(thread.getProcess().getAgent())+
			   thread.getStartTime();
	}

	/**
	 * @param methodInvocation
	 * @return
	 */
	public static double getAbsoluteExitTime(TRCFullMethodInvocation methodInvocation) {
		double start = getAbsoluteEntryTime(methodInvocation.getProcess().getAgent());
		if (methodInvocation.getExitTime() != 0) {
			return start+methodInvocation.getExitTime();
		}
		if (methodInvocation.getInvokes() == null || methodInvocation.getInvokes().size() == 0) {
			return getAbsoluteExitTime(methodInvocation.getThread());
		}
		TRCFullMethodInvocation lastCalled =
			(TRCFullMethodInvocation)methodInvocation.getInvokes().get(methodInvocation.getInvokes().size()-1);
		if (lastCalled.getExitTime() == 0) {
			return getAbsoluteExitTime(lastCalled);
		}
		return start+lastCalled.getExitTime();
	}

	/**
	 * @param agent
	 * @return
	 */
	public static double getAbsoluteExitTime(TRCAgent agent) {
		double delta = agent.getAgentProxy().getProcessProxy().getNode().getDeltaTime()*1E-6;
		return delta+agent.getStopTime();
	}

	/**
	 * @param process
	 * @return
	 */
	public static double getAbsoluteExitTime(TRCProcess process) {
		if (process.getStopTime() == 0) {
			return getAbsoluteExitTime(process.getAgent());
		}
		return getAbsoluteEntryTime(process.getAgent())+
			   process.getStopTime();
	}

	/**
	 * @param thread
	 * @return
	 */
	public static double getAbsoluteExitTime(TRCThread thread) {
		if (thread.getStopTime() == 0) {
			return getAbsoluteExitTime(thread.getProcess().getAgent());
		}
		return getAbsoluteEntryTime(thread.getProcess().getAgent())+
			   thread.getStopTime();
	}
	
	public static boolean doesAgentProxyStillExists(String agentProxyRuntimeId) {
		List monitors = PDContentProvider.getMonitors();
		for (Iterator i = monitors.iterator(); i.hasNext(); ) {
			TRCMonitor monitor = (TRCMonitor)i.next();
			EList nodes = monitor.getNodes();
			for (Iterator j = nodes.iterator(); j.hasNext(); ) {
				TRCNode node = (TRCNode)j.next();
				EList pps = node.getProcessProxies();
				for (Iterator k = pps.iterator(); k.hasNext(); ) {
					TRCProcessProxy pp = (TRCProcessProxy)k.next();
					EList aps = pp.getAgentProxies();
					for (Iterator l = aps.iterator(); l.hasNext(); ) {
						TRCAgentProxy ap = (TRCAgentProxy)l.next();
						if (agentProxyRuntimeId.equals(ap.getRuntimeId())) {
							return true;
						}
					}
				}
			}
		}
		return false;
	}

	public static boolean matchCharacter(char c1, char c2, boolean isCaseSensitive) {
		if (TraceSDPlugin.debugMatch && unitTestDebug) {
			TraceSDPlugin.debugTraceMatch("Comparing "+c1+" to "+c2); //$NON-NLS-1$ //$NON-NLS-2$
		}
		return (c1 == c2) ||
			    (!isCaseSensitive &&	Character.toLowerCase(c1) == Character.toLowerCase(c2));
	}
	
	private static boolean trailingStars(char s[], int l) {
		int i;
		for (i = l; i < s.length && s[i] == '*'; i++);
		return i == s.length;
	}
	
	private static final boolean unitTestDebug = false;
			
	public static boolean matchCriteria(String s, Criteria criteria) {
		try {
			String e = criteria.getExpression();
			char ea[] = e.toCharArray();
			if (TraceSDPlugin.debugMatch && unitTestDebug) {
				TraceSDPlugin.debugTraceMatch("Comparing "+s+" to "+e); //$NON-NLS-1$ //$NON-NLS-2$
			}
			int i, j;
			for (i = 0, j = 0; i < s.length() && j < ea.length; i++) {
				char sc = s.charAt(i), ec = ea[j];
				if (TraceSDPlugin.debugMatch && unitTestDebug) {
					TraceSDPlugin.debugTraceMatch("s["+i+"]="+sc+" e["+j+"]="+ec); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
				}
				if (ec == '*') {
					if (j+1 == ea.length) {
						// trailing * -> everything matches
						if (TraceSDPlugin.debugMatch && unitTestDebug) {
							TraceSDPlugin.debugTraceMatch("#1a true"); //$NON-NLS-1$
						}
						return true;
					}
					if (ea[j+1] == '?') {
						 // swap * and ?, it's easier this way...
						ec = ea[j] = '?';
						ea[j+1] = '*';
					}
				}
				if (ec == '*') {
					int k;
					for (k = 1; i+k-1 < s.length() && j+k < ea.length; k++) {
						if (ea[j+k] == '*' || ea[j+k] == '?') {
							j += k;
							i += k - 2;
							break; // another wildcard -> next loop
						}
						if (!matchCharacter(s.charAt(i+k-1), ea[j+k], criteria.isCaseSenstiveSelected())) {
							break; // i will be incremented at the beginning of the next loop
						}
					}
					if (TraceSDPlugin.debugMatch && unitTestDebug) {
						TraceSDPlugin.debugTraceMatch("After '*': s["+i+"]="+sc+" e["+j+"]="+ec); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
					}
					if (i+k-1 == s.length() && trailingStars(ea, j+k)) {
						if (TraceSDPlugin.debugMatch && unitTestDebug) {
							TraceSDPlugin.debugTraceMatch("#1b true"); //$NON-NLS-1$
						}
						return true;
					}
				} else if (ec == '?') {
					if (i+1 < s.length() && j+1 < ea.length) {
						j++;
					} else {
						// trailing ? -> one char matches
						if (TraceSDPlugin.debugMatch && unitTestDebug) {
							TraceSDPlugin.debugTraceMatch(i+1 == s.length() && trailingStars(ea, j+1) ?"#3 true":"#3 false"); //$NON-NLS-1$ //$NON-NLS-2$
						}
						return i+1 == s.length() && trailingStars(ea, j+1);
					}
				} else if (!matchCharacter(sc, ec, criteria.isCaseSenstiveSelected())) {
					if (TraceSDPlugin.debugMatch && unitTestDebug) {
						TraceSDPlugin.debugTraceMatch("#4 false"); //$NON-NLS-1$
					}
					return false;
				} else {
					j++;
				}
			}
			if (TraceSDPlugin.debugMatch && unitTestDebug) {
				TraceSDPlugin.debugTraceMatch(i == s.length() && (j == ea.length || (j+1 == ea.length && (ea[j] == '*' || ea[j] == '?')))?"#5 true":"#5 false"); //$NON-NLS-1$ //$NON-NLS-2$
			}
			return i == s.length() && trailingStars(ea, j);
		} catch (Exception e) {
			if (TraceSDPlugin.debugMatch) {
				e.printStackTrace();
			}
			return false;
		}
	}

	/**
	   * @return true if collection mode used to profile this agent is ready for a
	   * UML2 Object Interactions diagram ... or something like this. 
	   */
	  public static boolean hasObjectInteractions( TRCAgentProxy agent_proxy )
	  {    
	    boolean collects_object_instances = false;
	    
	    switch( agent_proxy.getCollectionMode().getValue() )
	    {
	      case TRCCollectionMode.EXECUTION_FULL :
	      case TRCCollectionMode.HEAP_AND_EXECUTION_FULL :
	      case TRCCollectionMode.EXECUTION_NO_INSTANCES :
	        collects_object_instances = true;
	        break;
	        
	      case TRCCollectionMode.HEAP_FULL :
	      case TRCCollectionMode.HEAP_STATISTICS_ONLY :
	      case TRCCollectionMode.HEAP_AND_EXECUTION_STATISTICS_ONLY :
	      case TRCCollectionMode.EXECUTION_STATISTICS_ONLY :
	      case TRCCollectionMode.HEAP_FULL_AND_EXECUTION_STATISTICS_ONLY :
	      default:
	        collects_object_instances = false;
	        break;	    	
	    }
	    
	    return collects_object_instances;
	  }

	/**
		 *
		 */
		public static boolean hasInvocationData(TRCAgentProxy agentProxy) {
			boolean stackInformation = false;
			//boolean monitorMode = false;
			if (agentProxy != null &&
				agentProxy.getType().equals(HyadesConstants.PROFILE_AGENT_TYPE)) {
				EList configs = agentProxy.getConfigurations();
				if (configs == null) {
					return false;
				}
				for (Iterator i = configs.iterator(); i.hasNext(); ) {
					TRCConfiguration config = (TRCConfiguration)i.next();
					EList options = config.getOptions();
					if (options == null) {
						continue;
					}
					for (Iterator io = options.iterator(); io.hasNext(); ) {
						TRCOption option = (TRCOption)io.next();
						if (option.getKey().equals("STACK_INFORMATION") && //$NON-NLS-1$
							!option.getValue().equals("none")) { //$NON-NLS-1$
							stackInformation = true;
	//					} else if (option.getKey().equals("MONITOR_MODE") && //$NON-NLS-1$
	//							   option.getValue().equals("all")) { //$NON-NLS-1$
	//						monitorMode = true;
						}
					}
				}
			}
			return /*hasObjectInteractions(agentProxy) &&*/ stackInformation;// && monitorMode;
		}

	/**
	 * This method searches for the reference of the TRCProcess 
	 * selected in the profiling monitor view
	 */
	static public void searchTRCAgentProxies(EObject object, List list) {
		
		if (object instanceof TRCMonitor) {
			TRCMonitor m = (TRCMonitor)object;
			if (m.getNodes() == null ||
				m.getNodes().isEmpty()) {
				return;
			}
			for (Iterator i = m.getNodes().iterator(); i.hasNext(); ) {
				searchTRCAgentProxies((EObject)i.next(), list);
			}
		} else if (object instanceof TRCNode) {
			TRCNode n = (TRCNode)object;
			if (n.getProcessProxies() == null ||
				n.getProcessProxies().isEmpty()) {
				return;
			}
			for (Iterator i = n.getProcessProxies().iterator(); i.hasNext(); ) {
				searchTRCAgentProxies((EObject)i.next(), list);
			}
		} else if (object instanceof TRCProcessProxy) {
			TRCProcessProxy pp = (TRCProcessProxy)object;
			if (pp.getAgentProxies() == null ||
				pp.getAgentProxies().isEmpty()) {
				return;
			}
			for (Iterator i = pp.getAgentProxies().iterator(); i.hasNext(); ) {
				searchTRCAgentProxies((EObject)i.next(), list);
			}
		} else if (object instanceof TRCAgentProxy) {
			TRCAgentProxy agentProxy = (TRCAgentProxy)object;
			if (/*hasInvocationData(agentProxy) &&*/ !list.contains(agentProxy)) {
				list.add(agentProxy);
			}
		} else if (TraceSDPlugin.debugEvents) {
			TraceSDPlugin.debugTraceEvents("WARNING! searchTRCAgentProxies(): Found a "+object); //$NON-NLS-1$
		}
	}

	/**
	 * This method looks into the selection of the active navigator, usually PDProjectExplorer...
	 * Every selected TRCAgentProxy (or INavigatorItem under it) is collected in the list parameter.
	 * Selection of a TRCMonitor, TRCNode or TRCProcessProxy implies collection of all the
	 * "Profiling" agents under it.
	 * @return
	 */
	public static List collectMultiSelectionInNavigator() {
		INavigator nav = HyadesUtil.getActiveNavigator();
	    IStructuredSelection selection = nav == null ? null : nav.getStructuredSelection();
	    List list = new ArrayList();
	    if (selection != null) {
		    for (Iterator i = selection.iterator(); i.hasNext(); ) {
		    	Object obj = i.next();
	            if( obj instanceof INavigatorItem ) obj = ((INavigatorItem)obj).getData();
	            if (obj instanceof EObject) {
	            	searchTRCAgentProxies((EObject)obj, list);
	            } else if (TraceSDPlugin.debugEvents) {
	            	TraceSDPlugin.debugTraceEvents("WARNING! collectMultiSelectionInNavigator(): Found a "+obj.getClass().getName()+" "+obj); //$NON-NLS-1$ //$NON-NLS-2$
	            }
		    }
	    }
		return list;
	}

//	private static void testIt(String s, String e, boolean c, boolean ev) {
//		Criteria cr = new Criteria();
//		cr.setCaseSenstiveSelected(c); cr.setExpression(e);
//		boolean res = matchCriteria(s, cr);
//		if (res == ev) {
//			System.out.println("OK : "+s+" "+e+" ["+c+"]="+res+" (expecting "+ev+")");
//		} else {
//			System.out.println(" KO: "+s+" "+e+" ["+c+"]="+res+" (expecting "+ev+")");
//		}
//	}
//
//	public static void main(String args[]) {
//		TraceSDPlugin.debugMatch = true;
//		testIt("ThreadRun:1234", "Th*Run", false, false);
//		testIt("ThreadRun:1234", "Th*Run:123?", false, true);
//		testIt("ThreadRun:1234", "Th*Run?123?", false, true);
//		testIt("ThreadRun:1234", "Th*Run:1?3??", false, false);
//		testIt("ThreadRun:1234", "Th*Run:1?3?*", false, true);
//		testIt("ThreadRun", "Th*Run", true, true);
//		testIt("ThreadRun", "Th*Run", false, true);
//		testIt("ThreadRun", "ThreadRun*****************", true, true);
//		testIt("ThreadRun", "ThreadRu*********?*******", true, true);
//		testIt("ThreadRun", "ThreadRun*********?*******", true, false);
//		testIt("ThrennnadRun", "*n", false, true);
//		testIt("ThrennnadRun", "*N", true, false);
//		testIt("", "*", true, true);
//		testIt("TTT", "*", true, true);
//		testIt("TxxxT", "T???T", true, true);
//		testIt("TxxxT", "T?*?T", true, true);
//		testIt("TxxxT", "T??T", true, false);
//		testIt("T", "?", true, true);
//		testIt("TT", "*T", true, true);
//		testIt("TT", "?T", true, true);
//		testIt("TT", "T*", true, true);
//		testIt("TT", "T?", true, true);
//		testIt("TT", "**", true, true);
//		testIt("TT", "??", true, true);
//		testIt("GetChannels", "*Server*", false, false);
//		testIt("UmtsServer", "*Server", true, true);
//		testIt("UmtsServer", "*Server*", true, true);
//		testIt("UmtsServer", "*server", false, true);
//		testIt("UmtsServer", "*server*", false, true);
//	}
			
	
}
