package org.eclipse.hyades.logging.java;

import java.util.List;
import java.util.ListIterator;
import java.util.logging.Level;
import java.util.logging.LogRecord;

import org.eclipse.hyades.internal.logging.core.Constants;
import org.eclipse.hyades.logging.core.IExternalizableToXml;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.ComponentIdentification;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.ExtendedDataElement;
import org.eclipse.hyades.logging.events.cbe.MsgCatalogToken;
import org.eclipse.hyades.logging.events.cbe.MsgDataElement;
import org.eclipse.hyades.logging.events.cbe.ReportSituation;
import org.eclipse.hyades.logging.events.cbe.Situation;
import org.eclipse.hyades.logging.events.cbe.impl.EventFactoryContext;
import org.eclipse.hyades.logging.events.cbe.util.EventFormatter;
import org.eclipse.hyades.logging.events.cbe.util.EventHelpers;

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: CommonBaseEventLogRecord.java,v 1.32 2005/04/28 13:21:28 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Extension of the Java Logging <code>java.util.logging.LogRecord</code>
 * class used to encapsulate a Common Base Event v1.0.1 or
 * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code> within a
 * <code>java.util.logging.LogRecord</code>.
 * <p>
 * This extension class is required to log <code>java.lang.Object</code>
 * instances, such as
 * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>
 * instances through the Java Logging infrastructure.
 * <p>
 * <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code>
 * instances are created via the the single argument constructor that accepts a
 * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>
 * reference. Alternatively,
 * <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code>
 * instances may be created via the the double argument constructor that accepts
 * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code> and
 * <code>java.util.logging.Level</code> references. The following
 * <code>java.util.logging.LogRecord</code> properties are mapped to the
 * parameter Common Base Event log record based on its internal state or the
 * <code>java.util.logging.LogRecord</code> default values (see the associated
 * setter APIs for each property for mapping details):
 * <p>
 * <table border="1" cellpadding="10">
 * <tr>
 * <th>LogRecord Property</th>
 * <th>Value</th>
 * </tr>
 * <tr>
 * <td>level</td>
 * <td><code>SEVERE</code> if the parameter Common Base Event's severity is
 * set and is greater or equal to 50, <code>WARNING</code> if the parameter
 * Common Base Event's severity is set and is between 30 and 49 (inclusive) or
 * <code>INFO</code> if the parameter Common Base Event's severity is set and
 * is less than or equal to 29 or <code>OFF</code> if the parameter Common
 * Base Event's severity is <b>not </b> set.</td>
 * <td>sequenceNumber</td>
 * <td>A new unique value in increasing order within a JVM as initialized by
 * the <code>java.util.logging.LogRecord</code> super class.</td>
 * <td>threadID</td>
 * <td>A new unique value for the current thread as initialized by the
 * <code>java.util.logging.LogRecord</code> super class.</td>
 * <td>millis</td>
 * <td>The current time in milliseconds as initialized by the
 * <code>java.util.logging.LogRecord</code> super class if the parameter
 * Common Base Event's creationTime is <b>not </b> set.</td>
 * </tr>
 * </table>
 * <p>
 * NOTE: If the parameter Common Base Event log record is <code>null</code>,
 * a <code>java.util.logging.LogRecord</code> will be constructed with a
 * <code>OFF</code> level, <code>null</code> message and no Common Base
 * Event capabilities.
 * <p>
 * NOTE: Using any of the deprecated constructors results in resorting to the
 * <code>java.util.logging.LogRecord</code> programming model (e.g. no Common
 * Base Event capabilities).
 * <p>
 * NOTE: If the parameter Common Base Event log record does not contain either 
 * a <code>Situation</code> and/or <code>SourceComponentId</code> property, 
 * default properties will be constructed to ensure a well-formed 
 * Common Base Event.
 * <p>
 * <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code>
 * instances map <code>java.util.logging.LogRecord</code> properties in the
 * encapsulated Common Base Event using the following mapping:
 * <p>
 * <table border="1" cellpadding="10">
 * <tr>
 * <th>LogRecord Property</th>
 * <th>CommonBaseEvent Property</th>
 * </tr>
 * <tr>
 * <td>level</td>
 * <td>ExtendedDataElement[0...n].Name = "CommonBaseEventLogRecord:level" <br>
 * ExtendedDataElement[0...n].Type = "noValue" <br>
 * ExtendedDataElement[0...n].Children[0].Name = "name" <br>
 * ExtendedDataElement[0...n].Children[0].Type = "string" <br>
 * ExtendedDataElement[0...n].Children[0].Values[0] = &lt;level's name&gt; <br>
 * ExtendedDataElement[0...n].Children[1].Name = "value" <br>
 * ExtendedDataElement[0...n].Children[0].Type = "int" <br>
 * ExtendedDataElement[0...n].Children[0].Values[0] = &lt;level's numerical
 * value&gt; <br>
 * </td>
 * </tr>
 * <tr>
 * <td>loggerName</td>
 * <td>ExtendedDataElement[0...n].Name = "CommonBaseEventLogRecord:loggerName"
 * <br>
 * ExtendedDataElement[0...n].Type = "string" <br>
 * ExtendedDataElement[0...n].Values[0] = &lt;logger's name&gt; <br>
 * </td>
 * </tr>
 * <tr>
 * <td>msg</td>
 * <td>MsgDataElement.MsgCatalogId = &lt;msg&gt; <br>
 * </td>
 * </tr>
 * <tr>
 * <td>millis</td>
 * <td>creationTime = &lt;millis&gt; <br>
 * </td>
 * </tr>
 * <tr>
 * <td>parameters*</td>
 * <td>MsgDataElement.MsgCatalogTokens = &lt;string representation of
 * parameters&gt; <br>
 * </td>
 * </tr>
 * <tr>
 * <td>resourceBundle*</td>
 * <td>[not mapped] <br>
 * </td>
 * </tr>
 * <tr>
 * <td>resourceBundleName</td>
 * <td>MsgDataElement.MsgCatalog = &lt;resource bundle name&gt; <br>
 * <td>MsgDataElement.MsgCatalogType = "Java" <br>
 * </td>
 * </tr>
 * <tr>
 * <td>sequenceNumber</td>
 * <td>ExtendedDataElement[0...n].Name =
 * "CommonBaseEventLogRecord:sequenceNumber" <br>
 * ExtendedDataElement[0...n].Type = "long" <br>
 * ExtendedDataElement[0...n].Values[0] = &lt;sequence number&gt; <br>
 * </td>
 * </tr>
 * <tr>
 * <td>sourceClassName</td>
 * <td>ExtendedDataElement[0...n].Name =
 * "CommonBaseEventLogRecord:sourceClassName" <br>
 * ExtendedDataElement[0...n].Type = "string" <br>
 * ExtendedDataElement[0...n].Values[0] = &lt;source class name&gt; <br>
 * </td>
 * </tr>
 * <tr>
 * <td>sourceMethodName</td>
 * <td>ExtendedDataElement[0...n].Name =
 * "CommonBaseEventLogRecord:sourceMethodName" <br>
 * ExtendedDataElement[0...n].Type = "string" <br>
 * ExtendedDataElement[0...n].Values[0] = &lt;source method name&gt; <br>
 * </td>
 * </tr>
 * <tr>
 * <td>threadID</td>
 * <td>ExtendedDataElement[0...n].Name = "CommonBaseEventLogRecord:threadID"
 * <br>
 * ExtendedDataElement[0...n].Type = "int" <br>
 * ExtendedDataElement[0...n].Values[0] = &lt;thread ID&gt; <br>
 * </td>
 * </tr>
 * <tr>
 * <td>thrown*</td>
 * <td>ExtendedDataElement[0...n].Name = "CommonBaseEventLogRecord:stackTrace" <br>
 * ExtendedDataElement[0...n].Type = "stringArray" <br>
 * ExtendedDataElement[0...n].Values[0...n] =
 * [&lt; <code>java.lang.Throwable</code>'s class name&gt;[: &lt;
 * <code>java.lang.Throwable</code>'s localized message&gt;]]<br>
 * ExtendedDataElement[0...n].Values[(n + 1)...m] =
 * &lt; <code>java.lang.Throwable</code>'s stackTraceElement[0...(m - (n + 1))]&gt;]<br>
 * <br>
 * ExtendedDataElement[0...n].Children[0...m].Name = "Cause" <br>
 * ExtendedDataElement[0...n].Children[0...n].Type = "stringArray" <br>
 * ExtendedDataElement[0...n].Children[0...m].Values[0...n] =
 * &lt;cause's class name&gt;[: &lt;cause's localized message&gt;]<br>
 * ExtendedDataElement[0...n].Children[0...m].Values[(n + 1)...m] =
 * &lt;cause's stackTraceElement[0...(m - (n + 1))]&gt;</li>
 * <li>[children[0] = &lt;cause's cause&gt;]<br>
 * </td>
 * </tr>
 * </table>
 * <p>
 * <b>* </b> <code>java.util.logging.LogRecord</code> property is stored in
 * <code>java.util.logging.LogRecord</code> super class to eliminate loss of
 * data.
 * <p>
 * If the Extended Data Element's <code>values</code> property is
 * larger than 1024 characters, the string is segmented into a String array of
 * 1024-character elements and set on the Extended Data Element. However, if the
 * Extended Data Element's <code>values</code> property is 1024 or less
 * characters, the string is set directly on the first element a String array
 * and set on the Extended Data Element. As such, the <code>type</code>
 * property is set to <code>stringArray</code>.
 * <p>
 * Names identifying the associated
 * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
 * properties in the encapsulated Common Base Event for the
 * <code>java.util.logging.LogRecord</code> properties are provided as static
 * string constants. These name properties are prefixed by the
 * <code>CommonBaseEventLogRecord:</code> namespace (see
 * <code>EXTENDED_DATA_ELEMENT_NAME_NAMESPACE</code>).
 * <p>
 * NOTE: When setting associated
 * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
 * properties in the encapsulated Common Base Event, multiple like-named
 * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code> are
 * permitted but no there is no guarantee which
 * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code> is
 * retrieved when calling this class' getter APIs. That is, the predefined names
 * for the
 * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
 * (e.g. static string constants) are intended for use only by
 * <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code>.
 * <p>
 * NOTE: The Java Logging classes must be on the CLASSPATH at run-time to
 * utilize this extension class.
 * <p>
 * 
 * 
 * @author Paul E. Slauenwhite
 * @version April 28, 2005
 * @since April 1, 2003
 * @see java.util.logging.LogRecord
 * @see org.eclipse.hyades.logging.events.cbe.CommonBaseEvent
 * @see org.eclipse.hyades.logging.core.IExternalizableToXml
 */
public final class CommonBaseEventLogRecord extends LogRecord implements IExternalizableToXml {

    /**
     * Stream-Unique IDentifier (SUID) of this class.
     * <p>
     * NOTE:  This value MUST not be modified unless an incompatible 
     *        change is introduced.  
     * <p>
     * See the 'Java Object Serialization Specification' for more details.
     * <p>
     * Compatible changes include:
     * <ul>
     * <li>Adding, modifying access or modifying <code>static</code>/<code>transient</code> to non-<code>static</code>/non-<code>transient</code> of fields.</li>
     * <li>Adding or removing classes.</li>
     * <li>Adding or removing <code>writeObject</code>/<code>readObject</code> methods.</li>
     * <li>Adding <code>java.io.Serializable</code> inheritance.</li>
     * </ul>
     * <p>
     * Incompatible changes include:
     * <ul>
     * <li>Deleting, modifying non-<code>static</code>/non-<code>transient</code> to <code>static</code>/<code>transient</code> or modifying the declared type of fields.</li>
     * <li>Modifying the hierarchy or interchanging/removing <code>java.io.Serializable</code>/<code>java.io.Externalizable</code> inheritance of the class.</li>
     * <li>Modifying <code>writeObject</code>/<code>readObject</code> methods.</li>
     * <li>Adding <code>writeReplace</code>/<code>readResolve</code> methods.</li>
     * </ul>
     */
    private static final long serialVersionUID = -2592508067175423183L;

    /**
     * Instance variable which holds a reference to a
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>.
     */
    private CommonBaseEvent commonBaseEvent = null;

    /**
     * For performance reasons, the instance variable which holds the
     * encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * 's child which contains the <code>java.util.logging.LogRecord</code>'s
     * level's name.
     */
    private ExtendedDataElement levelNameExtendedDataElement = null;

    /**
     * For performance reasons, the instance variable which holds the
     * encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * 's child which contains the <code>java.util.logging.LogRecord</code>'s
     * level's value.
     */
    private ExtendedDataElement levelValueExtendedDataElement = null;

    /**
     * For performance reasons, the instance variable which holds the
     * encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * which contains the <code>java.util.logging.LogRecord</code>'s logger
     * name.
     */
    private ExtendedDataElement loggerNameExtendedDataElement = null;

    /**
     * For performance reasons, the instance variable which holds the
     * encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * which contains the <code>java.util.logging.LogRecord</code>'s sequence
     * number.
     */
    private ExtendedDataElement sequenceNumberExtendedDataElement = null;

    /**
     * For performance reasons, the instance variable which holds the
     * encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * which contains the <code>java.util.logging.LogRecord</code>'s source
     * class name.
     */
    private ExtendedDataElement sourceClassNameExtendedDataElement = null;

    /**
     * For performance reasons, the instance variable which holds the
     * encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * which contains the <code>java.util.logging.LogRecord</code>'s source
     * method name.
     */
    private ExtendedDataElement sourceMethodNameExtendedDataElement = null;

    /**
     * For performance reasons, the instance variable which holds the
     * encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * which contains the <code>java.util.logging.LogRecord</code>'s thread
     * ID.
     */
    private ExtendedDataElement threadIDExtendedDataElement = null;

    /**
     * For performance reasons, a flag denoting if the instance variable which
     * holds the encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * which contains the <code>java.util.logging.LogRecord</code>'s thrown
     * has been set.
     */
    private boolean isThrownSet = false;

    /**
     * Instance variable which holds a reference to a
     * <code>org.eclipse.hyades.logging.events.cbe.EventFactory</code> for
     * generating event instances.
     */
    private final static EventFactory EVENT_FACTORY = EventFactoryContext.getInstance().getSimpleEventFactoryHome().getAnonymousEventFactory();

    /**
     * Instance variable which holds the name space property of the encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * s.
     * <p>
     * This namespace is intended for use only by
     * <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code>
     * to easily identify
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * s associated with <code>java.util.logging.LogRecord</code> properties.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_NAMESPACE = "CommonBaseEventLogRecord:";

    /**
     * Instance variable which holds the name property of the encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * which contains the <code>java.util.logging.LogRecord</code>'s level.
     * <p>
     * This name property is prefixed by the
     * <code>CommonBaseEventLogRecord:</code> namespace (see
     * <code>EXTENDED_DATA_ELEMENT_NAME_NAMESPACE</code>).
     * <p>
     * NOTE: The value of this
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * 's name property is intended for use only by
     * <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code>.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_LEVEL = EXTENDED_DATA_ELEMENT_NAME_NAMESPACE.concat("level");

    /**
     * Instance variable which holds the name property of the encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * 's child which contains the <code>java.util.logging.LogRecord</code>'s
     * level's name.
     * <p>
     * This name property is prefixed by the
     * <code>CommonBaseEventLogRecord:</code> namespace (see
     * <code>EXTENDED_DATA_ELEMENT_NAME_NAMESPACE</code>).
     * <p>
     * NOTE: The value of this
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * 's name property is intended for use only by
     * <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code>.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_LEVEL_NAME = EXTENDED_DATA_ELEMENT_NAME_NAMESPACE.concat("name");

    /**
     * Instance variable which holds the name property of the encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * 's child which contains the <code>java.util.logging.LogRecord</code>'s
     * level's value.
     * <p>
     * This name property is prefixed by the
     * <code>CommonBaseEventLogRecord:</code> namespace (see
     * <code>EXTENDED_DATA_ELEMENT_NAME_NAMESPACE</code>).
     * <p>
     * NOTE: The value of this
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * 's name property is intended for use only by
     * <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code>.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_LEVEL_VALUE = EXTENDED_DATA_ELEMENT_NAME_NAMESPACE.concat("value");

    /**
     * Instance variable which holds the name property of the encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * which contains the <code>java.util.logging.LogRecord</code>'s logger
     * name.
     * <p>
     * This name property is prefixed by the
     * <code>CommonBaseEventLogRecord:</code> namespace (see
     * <code>EXTENDED_DATA_ELEMENT_NAME_NAMESPACE</code>).
     * <p>
     * NOTE: The value of this
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * 's name property is intended for use only by
     * <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code>.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_LOGGER_NAME = EXTENDED_DATA_ELEMENT_NAME_NAMESPACE.concat("loggerName");

    /**
     * Instance variable which holds the name property of the encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * which contains the <code>java.util.logging.LogRecord</code>'s sequence
     * number.
     * <p>
     * This name property is prefixed by the
     * <code>CommonBaseEventLogRecord:</code> namespace (see
     * <code>EXTENDED_DATA_ELEMENT_NAME_NAMESPACE</code>).
     * <p>
     * NOTE: The value of this
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * 's name property is intended for use only by
     * <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code>.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_SEQUENCE_NUMBER = EXTENDED_DATA_ELEMENT_NAME_NAMESPACE.concat("sequenceNumber");

    /**
     * Instance variable which holds the name property of the encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * which contains the <code>java.util.logging.LogRecord</code>'s source
     * class name.
     * <p>
     * This name property is prefixed by the
     * <code>CommonBaseEventLogRecord:</code> namespace (see
     * <code>EXTENDED_DATA_ELEMENT_NAME_NAMESPACE</code>).
     * <p>
     * NOTE: The value of this
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * 's name property is intended for use only by
     * <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code>.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_SOURCE_CLASS_NAME = EXTENDED_DATA_ELEMENT_NAME_NAMESPACE.concat("sourceClassName");

    /**
     * Instance variable which holds the name property of the encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * which contains the <code>java.util.logging.LogRecord</code>'s source
     * method name.
     * <p>
     * This name property is prefixed by the
     * <code>CommonBaseEventLogRecord:</code> namespace (see
     * <code>EXTENDED_DATA_ELEMENT_NAME_NAMESPACE</code>).
     * <p>
     * NOTE: The value of this
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * 's name property is intended for use only by
     * <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code>.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_SOURCE_METHOD_NAME = EXTENDED_DATA_ELEMENT_NAME_NAMESPACE.concat("sourceMethodName");

    /**
     * Instance variable which holds the name property of the encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * which contains the <code>java.util.logging.LogRecord</code>'s thread
     * ID.
     * <p>
     * This name property is prefixed by the
     * <code>CommonBaseEventLogRecord:</code> namespace (see
     * <code>EXTENDED_DATA_ELEMENT_NAME_NAMESPACE</code>).
     * <p>
     * NOTE: The value of this
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * 's name property is intended for use only by
     * <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code>.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_THREAD_ID = EXTENDED_DATA_ELEMENT_NAME_NAMESPACE.concat("threadID");

    /**
     * Instance variable which holds the name property of the encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * which contains the <code>java.util.logging.LogRecord</code>'s thrown.
     * <p>
     * This name property is prefixed by the
     * <code>CommonBaseEventLogRecord:</code> namespace (see
     * <code>EXTENDED_DATA_ELEMENT_NAME_NAMESPACE</code>).
     * <p>
     * NOTE: The value of this
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * 's name property is intended for use only by
     * <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code>.
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_THROWN = EXTENDED_DATA_ELEMENT_NAME_NAMESPACE.concat("stackTrace");

    /**
     * Instance variable which holds the name property of the encapsulated
     * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code>'s
     * child
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * (s) which contain stack frames from the
     * <code>java.util.logging.LogRecord</code>'s thrown.
     * <p>
     * This name property is prefixed by the
     * <code>CommonBaseEventLogRecord:</code> namespace (see
     * <code>EXTENDED_DATA_ELEMENT_NAME_NAMESPACE</code>).
     * <p>
     * NOTE: The value of this
     * <code>org.eclipse.hyades.logging.events.cbe.ExtendedDataElement</code>
     * 's name property is intended for use only by
     * <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code>.
     * <p>
     * 
     * @deprecated As of 3.1, this property is no longer relevant (see #setThrown(Throwable)).
     */
    public final static String EXTENDED_DATA_ELEMENT_NAME_THROWN_STACK_FRAME = "Cause";

    /**
     * Single parameter constructor to create a Common Base Event log record
     * with the parameter Common Base Event.
     * <p>
     * The following <code>java.util.logging.LogRecord</code> properties are
     * mapped to the parameter Common Base Event log record based on its
     * internal state or the <code>java.util.logging.LogRecord</code> default
     * values (see the associated setter APIs for each property for mapping
     * details):
     * <p>
     * <table border="1" cellpadding="10">
     * <tr>
     * <th>LogRecord Property</th>
     * <th>Value</th>
     * </tr>
     * <tr>
     * <td>level</td>
     * <td><code>SEVERE</code> if the parameter Common Base Event's severity
     * is set and is greater or equal to 50, <code>WARNING</code> if the
     * parameter Common Base Event's severity is set and is between 30 and 49
     * (inclusive) or <code>INFO</code> if the parameter Common Base Event's
     * severity is set and is less than or equal to 29 or <code>OFF</code> if
     * the parameter Common Base Event's severity is <b>not </b> set.</td>
     * <td>sequenceNumber</td>
     * <td>A new unique value in increasing order within a JVM as initialized
     * by the <code>java.util.logging.LogRecord</code> super class.</td>
     * <td>threadID</td>
     * <td>A new unique value for the current thread as initialized by the
     * <code>java.util.logging.LogRecord</code> super class.</td>
     * <td>millis</td>
     * <td>The current time in milliseconds as initialized by the
     * <code>java.util.logging.LogRecord</code> super class if the parameter
     * Common Base Event's creationTime is <b>not </b> set.</td>
     * </tr>
     * </table>
     * <p>
     * NOTE: If the parameter Common Base Event log record is <code>null</code>,
     * a <code>java.util.logging.LogRecord</code> will be constructed with a
     * <code>OFF</code> level, <code>null</code> message and no Common Base
     * Event capabilities.
     * <p>
     * NOTE: If the parameter Common Base Event log record does not contain either 
     * a <code>Situation</code> and/or <code>SourceComponentId</code> property, 
     * default properties will be constructed to ensure a well-formed 
     * Common Base Event.
     * <p>
     * 
     * @param commonBaseEvent
     *            The Common Base Event of the newly created log record.
     */
    public CommonBaseEventLogRecord(CommonBaseEvent commonBaseEvent) {

        super(Level.OFF, null);

        if (commonBaseEvent != null) {

            this.commonBaseEvent = commonBaseEvent;

            resolveExtendedDataElementReferences();

            //NOTE: Only need to check one of the level Extended Data Element
            // references since both are mutually inclusive:
            if ((commonBaseEvent.isSetSeverity()) && (levelNameExtendedDataElement == null)) {

                short severity = commonBaseEvent.getSeverity();

                if (severity >= 50) {
                    setLevel(Level.SEVERE);
                } else if ((severity >= 30) && (severity <= 49)) {
                    setLevel(Level.WARNING);
                } else {
                    setLevel(Level.INFO);
                }
            }

            if (!commonBaseEvent.isSetCreationTime()) {
                setMillis(super.getMillis());
            }

            if (sequenceNumberExtendedDataElement == null) {
                setSequenceNumber(super.getSequenceNumber());
            }

            if (threadIDExtendedDataElement == null) {
                setThreadID(super.getThreadID());
            }
            
            //Create and populate a default report situation:
            if(commonBaseEvent.getSituation() == null){

    			//Create a new instance of a report situation:
    	        ReportSituation reportSituation = EVENT_FACTORY.createReportSituation();
    	        reportSituation.setReasoningScope("INTERNAL");
    	        reportSituation.setReportCategory("LOG");
    	
    			//Create a new instance of a situation:
    	        Situation situation = EVENT_FACTORY.createSituation();
    	        situation.setCategoryName(Situation.REPORT_SITUATION_CATEGORY);
    	        situation.setSituationType(reportSituation);

    	        commonBaseEvent.setSituation(situation);
            }
            
            //Create and populate a default source component ID:
            if(commonBaseEvent.getSourceComponentId() == null){

    			//Create a new instance of a component ID:
                ComponentIdentification sourceComponentId = EVENT_FACTORY.createComponentIdentification();
    			sourceComponentId.setLocation(Constants.LOCAL_HOST_IP_ADDRESS);
    			sourceComponentId.setLocationType(ComponentIdentification.LOCATION_TYPE_IPV4);
    			sourceComponentId.setExecutionEnvironment("Java");
    			sourceComponentId.setComponent("Logging");
    			sourceComponentId.setSubComponent("Logger");
    			sourceComponentId.setComponentIdType("Application");
    			sourceComponentId.setComponentType("Logging_Application");

    			commonBaseEvent.setSourceComponentId(sourceComponentId);
            }
        }
    }

    /**
     * Double parameter constructor to create a Common Base Event log record
     * with the parameter Common Base Event and a specified logging level.
     * <p>
     * The following <code>java.util.logging.LogRecord</code> properties are
     * mapped to the parameter Common Base Event log record based on its
     * internal state or the <code>java.util.logging.LogRecord</code> default
     * values (see the associated setter APIs for each property for mapping
     * details):
     * <p>
     * <table border="1" cellpadding="10">
     * <tr>
     * <th>LogRecord Property</th>
     * <th>Value</th>
     * </tr>
     * <tr>
     * <td>level</td>
     * <td>Based on the parameter <code>java.util.logging.Level</code>.
     * </td>
     * <td>sequenceNumber</td>
     * <td>A new unique value in increasing order within a JVM as initialized
     * by the <code>java.util.logging.LogRecord</code> super class.</td>
     * <td>threadID</td>
     * <td>A new unique value for the current thread as initialized by the
     * <code>java.util.logging.LogRecord</code> super class.</td>
     * <td>millis</td>
     * <td>The current time in milliseconds as initialized by the
     * <code>java.util.logging.LogRecord</code> super class if the parameter
     * Common Base Event's creationTime is <b>not </b> set.</td>
     * </tr>
     * </table>
     * <p>
     * NOTE: If the parameter Common Base Event log record is <code>null</code>,
     * a <code>java.util.logging.LogRecord</code> will be constructed with a
     * <code>OFF</code> level, <code>null</code> message and no Common Base
     * Event capabilities.
     * <p>
     * NOTE: If the parameter Common Base Event log record does not contain either 
     * a <code>Situation</code> and/or <code>SourceComponentId</code> property, 
     * default properties will be constructed to ensure a well-formed 
     * Common Base Event.
     * <p>
     * 
     * @param commonBaseEvent
     *            The Common Base Event of the newly created log record.
     * @param level
     *            The logging level of the newly created log record.
     */
    public CommonBaseEventLogRecord(CommonBaseEvent commonBaseEvent, Level level) {

        super(level, null);

        if (commonBaseEvent != null) {

            this.commonBaseEvent = commonBaseEvent;

            resolveExtendedDataElementReferences();

            setLevel(level);

            if (!commonBaseEvent.isSetCreationTime()) {
                setMillis(super.getMillis());
            }

            if (sequenceNumberExtendedDataElement == null) {
                setSequenceNumber(super.getSequenceNumber());
            }

            if (threadIDExtendedDataElement == null) {
                setThreadID(super.getThreadID());
            }

            //Create and populate a default report situation:
            if(commonBaseEvent.getSituation() == null){

    			//Create a new instance of a report situation:
    	        ReportSituation reportSituation = EVENT_FACTORY.createReportSituation();
    	        reportSituation.setReasoningScope("INTERNAL");
    	        reportSituation.setReportCategory("LOG");
    	
    			//Create a new instance of a situation:
    	        Situation situation = EVENT_FACTORY.createSituation();
    	        situation.setCategoryName(Situation.REPORT_SITUATION_CATEGORY);
    	        situation.setSituationType(reportSituation);

    	        commonBaseEvent.setSituation(situation);
            }
            
            //Create and populate a default source component ID:
            if(commonBaseEvent.getSourceComponentId() == null){

    			//Create a new instance of a component ID:
                ComponentIdentification sourceComponentId = EVENT_FACTORY.createComponentIdentification();
    			sourceComponentId.setLocation(Constants.LOCAL_HOST_IP_ADDRESS);
    			sourceComponentId.setLocationType(ComponentIdentification.LOCATION_TYPE_IPV4);
    			sourceComponentId.setExecutionEnvironment("Java");
    			sourceComponentId.setComponent("Logging");
    			sourceComponentId.setSubComponent("Logger");
    			sourceComponentId.setComponentIdType("Application");
    			sourceComponentId.setComponentType("Logging_Application");

    			commonBaseEvent.setSourceComponentId(sourceComponentId);
            }
        }
    }

    /**
     * Constructor to create a Common Base Event log record with the parameter
     * logging level and message.
     * <p>
     * Using this constructor will result in the construction of a
     * <code>java.util.logging.LogRecord</code> with no Common Base Event
     * capabilities.
     * <p>
     * 
     * @param level
     *            The logging level of the newly created log record.
     * @param message
     *            The log message of the newly created log record.
     * @deprecated Use <code>CommonBaseEventLogRecord(CommonBaseEvent)</code>
     *             or
     *             <code>CommonBaseEventLogRecord(CommonBaseEvent, Level)</code>.
     */
    public CommonBaseEventLogRecord(Level level, String message) {
        super(level, message);
    }

    /**
     * Constructor to create a Common Base Event log record with the parameter
     * logging level.
     * <p>
     * Using this constructor will result in the construction of a
     * <code>java.util.logging.LogRecord</code> with no Common Base Event
     * capabilities.
     * <p>
     * 
     * @param level
     *            The logging level of the newly created log record.
     * @deprecated Use <code>CommonBaseEventLogRecord(CommonBaseEvent)</code>
     *             or
     *             <code>CommonBaseEventLogRecord(CommonBaseEvent, Level)</code>.
     */
    public CommonBaseEventLogRecord(Level level) {
        super(level, null);
    }

    /**
     * Constructor to create a Common Base Event log record with the parameter
     * <code>java.util.logging.LogRecord</code>.
     * <p>
     * Using this constructor will result in the construction of a
     * <code>java.util.logging.LogRecord</code> with no Common Base Event
     * capabilities.
     * <p>
     * 
     * @param logRecord
     *            The <code>java.util.logging.LogRecord</code> of the newly
     *            created log record.
     * @deprecated Use <code>CommonBaseEventLogRecord(CommonBaseEvent)</code>
     *             or
     *             <code>CommonBaseEventLogRecord(CommonBaseEvent, Level)</code>.
     */
    public CommonBaseEventLogRecord(LogRecord logRecord) {

        super(logRecord.getLevel(), logRecord.getMessage());

        super.setMillis(logRecord.getMillis());
        super.setSequenceNumber(logRecord.getSequenceNumber());
        super.setThreadID(logRecord.getThreadID());
        super.setLoggerName(logRecord.getLoggerName());
        super.setParameters(logRecord.getParameters());
        super.setResourceBundle(logRecord.getResourceBundle());
        super.setResourceBundleName(logRecord.getResourceBundleName());
        super.setSourceClassName(logRecord.getSourceClassName());
        super.setSourceMethodName(logRecord.getSourceMethodName());
        super.setThrown(logRecord.getThrown());
    }

    /**
     * Sets value of the commonBaseEvent instance variable to the specified
     * value.
     * <p>
     * 
     * @param commonBaseEvent
     *            The new value of the commonBaseEvent instance variable.
     * @deprecated Use <code>CommonBaseEventLogRecord(CommonBaseEvent)</code>
     *             or
     *             <code>CommonBaseEventLogRecord(CommonBaseEvent, Level)</code>
     *             constructors.
     */
    public void setCommonBaseEvent(CommonBaseEvent commonBaseEvent) {

        this.commonBaseEvent = commonBaseEvent;

        if(commonBaseEvent != null){
            
            resolveExtendedDataElementReferences();
            
            //NOTE: Only need to check one of the level Extended Data Element
            // references since both are mutually inclusive:
            if ((commonBaseEvent.isSetSeverity()) && (levelNameExtendedDataElement == null)) {

                short severity = commonBaseEvent.getSeverity();

                if (severity >= 50) {
                    setLevel(Level.SEVERE);
                } else if ((severity >= 30) && (severity <= 49)) {
                    setLevel(Level.WARNING);
                } else {
                    setLevel(Level.INFO);
                }
            }

            if (!commonBaseEvent.isSetCreationTime()) {
                setMillis(super.getMillis());
            }

            if (sequenceNumberExtendedDataElement == null) {
                setSequenceNumber(super.getSequenceNumber());
            }

            if (threadIDExtendedDataElement == null) {
                setThreadID(super.getThreadID());
            }
            
            //Create and populate a default report situation:
            if(commonBaseEvent.getSituation() == null){

    			//Create a new instance of a report situation:
    	        ReportSituation reportSituation = EVENT_FACTORY.createReportSituation();
    	        reportSituation.setReasoningScope("INTERNAL");
    	        reportSituation.setReportCategory("LOG");
    	
    			//Create a new instance of a situation:
    	        Situation situation = EVENT_FACTORY.createSituation();
    	        situation.setCategoryName(Situation.REPORT_SITUATION_CATEGORY);
    	        situation.setSituationType(reportSituation);

    	        commonBaseEvent.setSituation(situation);
            }
            
            //Create and populate a default source component ID:
            if(commonBaseEvent.getSourceComponentId() == null){

    			//Create a new instance of a component ID:
                ComponentIdentification sourceComponentId = EVENT_FACTORY.createComponentIdentification();
    			sourceComponentId.setLocation(Constants.LOCAL_HOST_IP_ADDRESS);
    			sourceComponentId.setLocationType(ComponentIdentification.LOCATION_TYPE_IPV4);
    			sourceComponentId.setExecutionEnvironment("Java");
    			sourceComponentId.setComponent("Logging");
    			sourceComponentId.setSubComponent("Logger");
    			sourceComponentId.setComponentIdType("Application");
    			sourceComponentId.setComponentType("Logging_Application");

    			commonBaseEvent.setSourceComponentId(sourceComponentId);
            }
        }
    }

    /**
     * Sets the non-null parameter level filter on the encapsulated Common Base
     * Event or <code>java.util.logging.LogRecord</code> super class.
     * <p>
     * This API will throw a <code>java.lang.NullPointerException</code> is
     * the parameter level filter is <code>null</code>.
     * <p>
     * <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code>
     * instances map <code>java.util.logging.LogRecord</code> properties in
     * the encapsulated Common Base Event using the following mapping:
     * <p>
     * <table border="1" cellpadding="10">
     * <tr>
     * <th>LogRecord Property</th>
     * <th>CommonBaseEvent Property</th>
     * </tr>
     * <tr>
     * <td>level</td>
     * <td>ExtendedDataElement[0...n].Name = "CommonBaseEventLogRecord:level"
     * <br>
     * ExtendedDataElement[0...n].Type = "noValue" <br>
     * ExtendedDataElement[0...n].Children[0].Name = "name" <br>
     * ExtendedDataElement[0...n].Children[0].Type = "string" <br>
     * ExtendedDataElement[0...n].Children[0].Values[0] = &lt;level's name&gt;
     * <br>
     * ExtendedDataElement[0...n].Children[1].Name = "value" <br>
     * ExtendedDataElement[0...n].Children[0].Type = "int" <br>
     * ExtendedDataElement[0...n].Children[0].Values[0] = &lt;level's numerical
     * value&gt; <br>
     * </td>
     * </tr>
     * </table>
     * <p>
     * NOTE: If the encapsulated Common Base Event log record is
     * <code>null</code>, this API will delegate to the associated API in the
     * <code>java.util.logging.LogRecord</code> super class.
     * <p>
     * 
     * @param level
     *            The non-null level filter, for example <code>ALL</code>.
     * @see java.util.logging.LogRecord#setLevel(java.util.logging.Level)
     */
    public void setLevel(Level level) {

        if (commonBaseEvent != null) {

            //NOTE: Only need to check one of the level Extended Data Element
            // references since both are mutually inclusive:
            if (levelNameExtendedDataElement == null) {

                levelNameExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
                levelNameExtendedDataElement.setName(EXTENDED_DATA_ELEMENT_NAME_LEVEL_NAME);
                levelNameExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_STRING_VALUE);

                levelValueExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
                levelValueExtendedDataElement.setName(EXTENDED_DATA_ELEMENT_NAME_LEVEL_VALUE);
                levelValueExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_INT_VALUE);

                ExtendedDataElement levelExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
                levelExtendedDataElement.setName(EXTENDED_DATA_ELEMENT_NAME_LEVEL);
                levelExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_NO_VALUE_VALUE);

                levelExtendedDataElement.addChild(levelNameExtendedDataElement);
                levelExtendedDataElement.addChild(levelValueExtendedDataElement);

                commonBaseEvent.addExtendedDataElement(levelExtendedDataElement);
            }

            levelNameExtendedDataElement.setValues(new String[] { level.getName()});

            levelValueExtendedDataElement.setValues(new String[] { Integer.toString(level.intValue())});
        } else {
            super.setLevel(level);
        }
    }

    /**
     * Sets the log record's associated logger name on the encapsulated Common
     * Base Event or <code>java.util.logging.LogRecord</code> super class.
     * <p>
     * <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code>
     * instances map <code>java.util.logging.LogRecord</code> properties in
     * the encapsulated Common Base Event using the following mapping:
     * <p>
     * <table border="1" cellpadding="10">
     * <tr>
     * <th>LogRecord Property</th>
     * <th>CommonBaseEvent Property</th>
     * </tr>
     * <tr>
     * <td>loggerName</td>
     * <td>ExtendedDataElement[0...n].Name =
     * "CommonBaseEventLogRecord:loggerName" <br>
     * ExtendedDataElement[0...n].Type = "string" <br>
     * ExtendedDataElement[0...n].Values[0] = &lt;logger's name&gt; <br>
     * </td>
     * </tr>
     * </table>
     * <p>
     * NOTE: If the encapsulated Common Base Event log record is
     * <code>null</code>, this API will delegate to the associated API in the
     * <code>java.util.logging.LogRecord</code> super class.
     * <p>
     * 
     * @param name
     *            The log record's associated logger name.
     * @see java.util.logging.LogRecord#setLoggerName(java.lang.String)
     */
    public void setLoggerName(String name) {

        if (commonBaseEvent != null) {

            if (loggerNameExtendedDataElement == null) {

                loggerNameExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
                loggerNameExtendedDataElement.setName(EXTENDED_DATA_ELEMENT_NAME_LOGGER_NAME);
                loggerNameExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_STRING_VALUE);

                commonBaseEvent.addExtendedDataElement(loggerNameExtendedDataElement);
            }

            loggerNameExtendedDataElement.setValues(new String[] { name});
        } else {
            super.setLoggerName(name);
        }
    }

    /**
     * Sets the string log message on the encapsulated Common Base Event or
     * <code>java.util.logging.LogRecord</code> super class.
     * <p>
     * <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code>
     * instances map <code>java.util.logging.LogRecord</code> properties in
     * the encapsulated Common Base Event using the following mapping:
     * <p>
     * <table border="1" cellpadding="10">
     * <tr>
     * <th>LogRecord Property</th>
     * <th>CommonBaseEvent Property</th>
     * </tr>
     * <tr>
     * <td>msg</td>
     * <td>MsgDataElement.MsgCatalogId = &lt;msg&gt; <br>
     * </td>
     * </tr>
     * </table>
     * <p>
     * NOTE: If the encapsulated Common Base Event log record is
     * <code>null</code>, this API will delegate to the associated API in the
     * <code>java.util.logging.LogRecord</code> super class.
     * <p>
     * 
     * @param message
     *            The string log message.
     * @see java.util.logging.LogRecord#setMessage(java.lang.String)
     */
    public void setMessage(String message) {

        if (commonBaseEvent != null) {

            MsgDataElement msgDataElement = commonBaseEvent.getMsgDataElement();

            if (msgDataElement == null) {

                msgDataElement = EVENT_FACTORY.createMsgDataElement();

                commonBaseEvent.setMsgDataElement(msgDataElement);
            }

            msgDataElement.setMsgCatalogId(message);
        } else {
            super.setMessage(message);
        }
    }

    /**
     * Sets the time stamp in milliseconds on the encapsulated Common Base Event
     * or <code>java.util.logging.LogRecord</code> super class.
     * <p>
     * <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code>
     * instances map <code>java.util.logging.LogRecord</code> properties in
     * the encapsulated Common Base Event using the following mapping:
     * <p>
     * <table border="1" cellpadding="10">
     * <tr>
     * <th>LogRecord Property</th>
     * <th>CommonBaseEvent Property</th>
     * </tr>
     * <tr>
     * <td>millis</td>
     * <td>creationTime = &lt;millis&gt; <br>
     * </td>
     * </tr>
     * </table>
     * <p>
     * NOTE: If the encapsulated Common Base Event log record is
     * <code>null</code>, this API will delegate to the associated API in the
     * <code>java.util.logging.LogRecord</code> super class.
     * <p>
     * 
     * @param millis
     *            The time stamp in milliseconds.
     * @see java.util.logging.LogRecord#setMillis(long)
     */
    public void setMillis(long millis) {

        if (commonBaseEvent != null) {
            commonBaseEvent.setCreationTimeAsLong(millis);
        } else {
            super.setMillis(millis);
        }
    }

    /**
     * Sets the array of message parameters on the encapsulated Common Base
     * Event or <code>java.util.logging.LogRecord</code> super class.
     * <p>
     * <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code>
     * instances map <code>java.util.logging.LogRecord</code> properties in
     * the encapsulated Common Base Event using the following mapping:
     * <p>
     * <table border="1" cellpadding="10">
     * <tr>
     * <th>LogRecord Property</th>
     * <th>CommonBaseEvent Property</th>
     * </tr>
     * <tr>
     * <td>parameters*</td>
     * <td>MsgDataElement.MsgCatalogTokens = &lt;string representation of
     * parameters&gt; <br>
     * </td>
     * </tr>
     * </table>
     * <p>
     * <b>* </b> <code>java.util.logging.LogRecord</code> property is stored
     * in <code>java.util.logging.LogRecord</code> super class to eliminate
     * loss of data.
     * <p>
     * NOTE: This API delegates to both the associated API in the
     * <code>java.util.logging.LogRecord</code> super class and the
     * encapsulated Common Base Event.
     * <p>
     * 
     * @param parameters
     *            The array of message parameters.
     * @see java.util.logging.LogRecord#setParameters(java.lang.Object[])
     */
    public void setParameters(Object[] parameters) {

        super.setParameters(parameters);

        if (commonBaseEvent != null) {

            MsgDataElement msgDataElement = commonBaseEvent.getMsgDataElement();

            if (msgDataElement == null) {

                msgDataElement = EVENT_FACTORY.createMsgDataElement();

                commonBaseEvent.setMsgDataElement(msgDataElement);
            } else {
                msgDataElement.clearMsgCatalogTokens();
            }

            if (parameters != null) {

                MsgCatalogToken msgCatalogToken = null;

                for (int counter = 0; counter < parameters.length; counter++) {

                    msgCatalogToken = EVENT_FACTORY.createMsgCatalogToken();
                    msgCatalogToken.setValue(String.valueOf(parameters[counter]));

                    msgDataElement.addMsgCatalogToken(msgCatalogToken);
                }
            }
        }
    }

    /**
     * Sets the name of the resource bundle on the encapsulated Common Base
     * Event or <code>java.util.logging.LogRecord</code> super class.
     * <p>
     * <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code>
     * instances map <code>java.util.logging.LogRecord</code> properties in
     * the encapsulated Common Base Event using the following mapping:
     * <p>
     * <table border="1" cellpadding="10">
     * <tr>
     * <th>LogRecord Property</th>
     * <th>CommonBaseEvent Property</th>
     * </tr>
     * <tr>
     * <td>resourceBundleName</td>
     * <td>MsgDataElement.MsgCatalog = &lt;resource bundle name&gt; <br>
     * <td>MsgDataElement.MsgCatalogType = "Java" <br>
     * </td>
     * </tr>
     * </table>
     * <p>
     * NOTE: If the encapsulated Common Base Event log record is
     * <code>null</code>, this API will delegate to the associated API in the
     * <code>java.util.logging.LogRecord</code> super class.
     * <p>
     * 
     * @param name
     *            The name of the resource bundle.
     * @see java.util.logging.LogRecord#setResourceBundleName(java.lang.String)
     */
    public void setResourceBundleName(String name) {

        if (commonBaseEvent != null) {

            MsgDataElement msgDataElement = commonBaseEvent.getMsgDataElement();

            if (msgDataElement == null) {

                msgDataElement = EVENT_FACTORY.createMsgDataElement();

                commonBaseEvent.setMsgDataElement(msgDataElement);
            }

            if(name != null){
                
	            msgDataElement.setMsgCatalogType("Java");	
	            msgDataElement.setMsgCatalog(name);
            }
            else{
                
	            msgDataElement.setMsgCatalogType(null);	
	            msgDataElement.setMsgCatalog(null);
            }
        } else {
            super.setResourceBundleName(name);
        }
    }

    /**
     * Sets the unique sequence number on the encapsulated Common Base Event or
     * <code>java.util.logging.LogRecord</code> super class.
     * <p>
     * <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code>
     * instances map <code>java.util.logging.LogRecord</code> properties in
     * the encapsulated Common Base Event using the following mapping:
     * <p>
     * <table border="1" cellpadding="10">
     * <tr>
     * <th>LogRecord Property</th>
     * <th>CommonBaseEvent Property</th>
     * </tr>
     * <tr>
     * <td>sequenceNumber</td>
     * <td>ExtendedDataElement[0...n].Name =
     * "CommonBaseEventLogRecord:sequenceNumber" <br>
     * ExtendedDataElement[0...n].Type = "long" <br>
     * ExtendedDataElement[0...n].Values[0] = &lt;sequence number&gt; <br>
     * </td>
     * </tr>
     * </table>
     * <p>
     * NOTE: If the encapsulated Common Base Event log record is
     * <code>null</code>, this API will delegate to the associated API in the
     * <code>java.util.logging.LogRecord</code> super class.
     * <p>
     * 
     * @param seq
     *            The unique sequence number.
     * @see java.util.logging.LogRecord#setSequenceNumber(long)
     */
    public void setSequenceNumber(long seq) {

        if (commonBaseEvent != null) {

            if (sequenceNumberExtendedDataElement == null) {

                sequenceNumberExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
                sequenceNumberExtendedDataElement.setName(EXTENDED_DATA_ELEMENT_NAME_SEQUENCE_NUMBER);
                sequenceNumberExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_LONG_VALUE);

                commonBaseEvent.addExtendedDataElement(sequenceNumberExtendedDataElement);
            }

            sequenceNumberExtendedDataElement.setValues(new String[] { Long.toString(seq)});
        } else {
            super.setSequenceNumber(seq);
        }
    }

    /**
     * Sets the name of the source class on the encapsulated Common Base Event
     * or <code>java.util.logging.LogRecord</code> super class.
     * <p>
     * <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code>
     * instances map <code>java.util.logging.LogRecord</code> properties in
     * the encapsulated Common Base Event using the following mapping:
     * <p>
     * <table border="1" cellpadding="10">
     * <tr>
     * <th>LogRecord Property</th>
     * <th>CommonBaseEvent Property</th>
     * </tr>
     * <tr>
     * <td>sourceClassName</td>
     * <td>ExtendedDataElement[0...n].Name =
     * "CommonBaseEventLogRecord:sourceClassName" <br>
     * ExtendedDataElement[0...n].Type = "string" <br>
     * ExtendedDataElement[0...n].Values[0] = &lt;source class name&gt; <br>
     * </td>
     * </tr>
     * </table>
     * <p>
     * NOTE: If the encapsulated Common Base Event log record is
     * <code>null</code>, this API will delegate to the associated API in the
     * <code>java.util.logging.LogRecord</code> super class.
     * <p>
     * 
     * @param sourceClassName
     *            The name of the source class.
     * @see java.util.logging.LogRecord#setSourceClassName(java.lang.String)
     */
    public void setSourceClassName(String sourceClassName) {

        if (commonBaseEvent != null) {

            if (sourceClassNameExtendedDataElement == null) {

                sourceClassNameExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
                sourceClassNameExtendedDataElement.setName(EXTENDED_DATA_ELEMENT_NAME_SOURCE_CLASS_NAME);
                sourceClassNameExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_STRING_VALUE);

                commonBaseEvent.addExtendedDataElement(sourceClassNameExtendedDataElement);
            }

            sourceClassNameExtendedDataElement.setValues(new String[] { sourceClassName});
        } else {
            super.setSourceClassName(sourceClassName);
        }
    }

    /**
     * Sets the name of the source method on the encapsulated Common Base Event
     * or <code>java.util.logging.LogRecord</code> super class.
     * <p>
     * <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code>
     * instances map <code>java.util.logging.LogRecord</code> properties in
     * the encapsulated Common Base Event using the following mapping:
     * <p>
     * <table border="1" cellpadding="10">
     * <tr>
     * <th>LogRecord Property</th>
     * <th>CommonBaseEvent Property</th>
     * </tr>
     * <tr>
     * <td>sourceMethodName</td>
     * <td>ExtendedDataElement[0...n].Name =
     * "CommonBaseEventLogRecord:sourceMethodName" <br>
     * ExtendedDataElement[0...n].Type = "string" <br>
     * ExtendedDataElement[0...n].Values[0] = &lt;source method name&gt; <br>
     * </td>
     * </tr>
     * </table>
     * <p>
     * NOTE: If the encapsulated Common Base Event log record is
     * <code>null</code>, this API will delegate to the associated API in the
     * <code>java.util.logging.LogRecord</code> super class.
     * <p>
     * 
     * @param sourceMethodName
     *            The name of the source method.
     * @see java.util.logging.LogRecord#setSourceMethodName(java.lang.String)
     */
    public void setSourceMethodName(String sourceMethodName) {

        if (commonBaseEvent != null) {

            if (sourceMethodNameExtendedDataElement == null) {

                sourceMethodNameExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
                sourceMethodNameExtendedDataElement.setName(EXTENDED_DATA_ELEMENT_NAME_SOURCE_METHOD_NAME);
                sourceMethodNameExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_STRING_VALUE);

                commonBaseEvent.addExtendedDataElement(sourceMethodNameExtendedDataElement);
            }

            sourceMethodNameExtendedDataElement.setValues(new String[] { sourceMethodName});
        } else {
            super.setSourceMethodName(sourceMethodName);
        }
    }

    /**
     * Sets the unique ID of an associated thread on the encapsulated Common
     * Base Event or <code>java.util.logging.LogRecord</code> super class.
     * <p>
     * <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code>
     * instances map <code>java.util.logging.LogRecord</code> properties in
     * the encapsulated Common Base Event using the following mapping:
     * <p>
     * <table border="1" cellpadding="10">
     * <tr>
     * <th>LogRecord Property</th>
     * <th>CommonBaseEvent Property</th>
     * </tr>
     * <tr>
     * <td>threadID</td>
     * <td>ExtendedDataElement[0...n].Name =
     * "CommonBaseEventLogRecord:threadID" <br>
     * ExtendedDataElement[0...n].Type = "int" <br>
     * ExtendedDataElement[0...n].Values[0] = &lt;thread ID&gt; <br>
     * </td>
     * </tr>
     * </table>
     * <p>
     * NOTE: If the encapsulated Common Base Event log record is
     * <code>null</code>, this API will delegate to the associated API in the
     * <code>java.util.logging.LogRecord</code> super class.
     * <p>
     * 
     * @param threadID
     *            The unique ID of an associated thread.
     * @see java.util.logging.LogRecord#setThreadID(int)
     */
    public void setThreadID(int threadID) {

        if (commonBaseEvent != null) {

            if (threadIDExtendedDataElement == null) {

                threadIDExtendedDataElement = EVENT_FACTORY.createExtendedDataElement();
                threadIDExtendedDataElement.setName(EXTENDED_DATA_ELEMENT_NAME_THREAD_ID);
                threadIDExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_INT_VALUE);

                commonBaseEvent.addExtendedDataElement(threadIDExtendedDataElement);
            }

            threadIDExtendedDataElement.setValues(new String[] { Integer.toString(threadID)});
        } else {
            super.setThreadID(threadID);
        }
    }

    /**
     * Sets an exception on the encapsulated Common Base Event or
     * <code>java.util.logging.LogRecord</code> super class.
     * <p>
     * <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code>
     * instances map <code>java.util.logging.LogRecord</code> properties in
     * the encapsulated Common Base Event using the following mapping:
     * <p>
     * <table border="1" cellpadding="10">
     * <tr>
     * <th>LogRecord Property</th>
     * <th>CommonBaseEvent Property</th>
     * </tr>
     * <tr>
     * <td>thrown*</td>
     * <td>ExtendedDataElement[0...n].Name = "CommonBaseEventLogRecord:stackTrace" <br>
     * ExtendedDataElement[0...n].Type = "stringArray" <br>
     * ExtendedDataElement[0...n].Values[0...n] =
     * [&lt; <code>java.lang.Throwable</code>'s class name&gt;[: &lt;
     * <code>java.lang.Throwable</code>'s localized message&gt;]]<br>
     * ExtendedDataElement[0...n].Values[(n + 1)...m] =
     * &lt; <code>java.lang.Throwable</code>'s stackTraceElement[0...(m - (n + 1))]&gt;]<br>
     * <br>
     * ExtendedDataElement[0...n].Children[0...m].Name = "Cause" <br>
     * ExtendedDataElement[0...n].Children[0...n].Type = "stringArray" <br>
     * ExtendedDataElement[0...n].Children[0...m].Values[0...n] =
     * &lt;cause's class name&gt;[: &lt;cause's localized message&gt;]<br>
     * ExtendedDataElement[0...n].Children[0...m].Values[(n + 1)...m] =
     * &lt;cause's stackTraceElement[0...(m - (n + 1))]&gt;</li>
     * <li>[children[0] = &lt;cause's cause&gt;]<br>
     * </td>
     * </tr>
     * </table>
     * <p>
     * <b>* </b> <code>java.util.logging.LogRecord</code> property is stored
     * in <code>java.util.logging.LogRecord</code> super class to eliminate
     * loss of data.
     * <p>
     * If the Extended Data Element's <code>values</code> property is
     * larger than 1024 characters, the string is segmented into a String array of
     * 1024-character elements and set on the Extended Data Element. However, if the
     * Extended Data Element's <code>values</code> property is 1024 or less
     * characters, the string is set directly on the first element a String array
     * and set on the Extended Data Element. As such, the <code>type</code>
     * property is set to <code>stringArray</code>.
     * <p>
     * NOTE: This API delegates to both the associated API in the
     * <code>java.util.logging.LogRecord</code> super class and the
     * encapsulated Common Base Event.
     * <p>
     * 
     * @param thrown
     *            An exception associated with this log record.
     * @see java.util.logging.LogRecord#setThrown(java.lang.Throwable)
     * @see org.eclipse.hyades.logging.events.cbe.util.EventHelpers#convertToExtendedDataElement(Throwable, String).
     */
    public void setThrown(Throwable thrown) {

        super.setThrown(thrown);

        if (commonBaseEvent != null) {

            if (isThrownSet) {
                commonBaseEvent.removeExtendedDataElements(EXTENDED_DATA_ELEMENT_NAME_THROWN);
            } else {
                isThrownSet = true;
            }

            commonBaseEvent.addExtendedDataElement(EventHelpers.convertToExtendedDataElement(thrown,EXTENDED_DATA_ELEMENT_NAME_THROWN));
        }
    }

    /**
     * Gets the value of the commonBaseEvent instance variable.
     * <p>
     * 
     * @return The value of the commonBaseEvent instance variable.
     */
    public CommonBaseEvent getCommonBaseEvent() {
        return commonBaseEvent;
    }

    /**
     * Gets the level filter from the encapsulated Common Base Event or
     * <code>java.util.logging.LogRecord</code> super class.
     * <p>
     * See the associated setter API for more details on the mapping of this
     * <code>java.util.logging.LogRecord</code> property.
     * <p>
     * 
     * @return The level filter, for example <code>ALL</code>.
     * @see java.util.logging.LogRecord#getLevel()
     */
    public Level getLevel() {

        if (commonBaseEvent != null) {

            //NOTE: Only need to check one of the level Extended Data Element
            // references since both are mutually inclusive:
            if (levelNameExtendedDataElement != null) {

                String name = ((String) (levelNameExtendedDataElement.getValues().get(0)));

                if (name != null) {

                    try {
                        return (Level.parse(name));
                    } catch (IllegalArgumentException i) {
                        //Ignore since level is a customized level.
                    }
                }

                try {
                    return (new CommonBaseEventLevel(name, Integer.parseInt(((String) (levelNameExtendedDataElement.getValues().get(0))))));
                } catch (NumberFormatException n) {
                    //Ignore since API will delegate to the super class.
                }
            }
        }

        return (super.getLevel());
    }

    /**
     * Gets the log record's associated logger name from the encapsulated Common
     * Base Event or <code>java.util.logging.LogRecord</code> super class.
     * <p>
     * See the associated setter API for more details on the mapping of this
     * <code>java.util.logging.LogRecord</code> property.
     * <p>
     * 
     * @return The log record's associated logger name.
     * @see java.util.logging.LogRecord#getLoggerName()
     */
    public String getLoggerName() {

        if (commonBaseEvent != null) {

            if (loggerNameExtendedDataElement != null) { return ((String) (loggerNameExtendedDataElement.getValues().get(0))); }
        } else {
            return (super.getLoggerName());
        }

        return null;
    }

    /**
     * Gets the string log message from the encapsulated Common Base Event or
     * <code>java.util.logging.LogRecord</code> super class.
     * <p>
     * See the associated setter API for more details on the mapping of this
     * <code>java.util.logging.LogRecord</code> property.
     * <p>
     * 
     * @return The string log message.
     * @see java.util.logging.LogRecord#getMessage()
     */
    public String getMessage() {

        if (commonBaseEvent != null) {

            MsgDataElement msgDataElement = commonBaseEvent.getMsgDataElement();

            if (msgDataElement != null) { return (msgDataElement.getMsgCatalogId()); }
        } else {
            return (super.getMessage());
        }

        return null;
    }

    /**
     * Gets the time stamp in milliseconds from the encapsulated Common Base
     * Event or <code>java.util.logging.LogRecord</code> super class.
     * <p>
     * See the associated setter API for more details on the mapping of this
     * <code>java.util.logging.LogRecord</code> property.
     * <p>
     * 
     * @return The time stamp in milliseconds.
     * @see java.util.logging.LogRecord#getMillis()
     */
    public long getMillis() {

        if (commonBaseEvent != null) { return (commonBaseEvent.getCreationTimeAsLong()); }

        return (super.getMillis());
    }

    /**
     * Gets the array of message parameters from the encapsulated Common Base
     * Event or <code>java.util.logging.LogRecord</code> super class.
     * <p>
     * See the associated setter API for more details on the mapping of this
     * <code>java.util.logging.LogRecord</code> property.
     * <p>
     * 
     * @return The array of message parameters.
     * @see java.util.logging.LogRecord#getParameters()
     */
    public Object[] getParameters() {

        Object[] parameters = super.getParameters();

        if (parameters != null) { return parameters; }

        if (commonBaseEvent != null) {

            MsgDataElement msgDataElement = commonBaseEvent.getMsgDataElement();

            if (msgDataElement != null) {

                List msgCatalogTokens = msgDataElement.getMsgCatalogTokens();
                int msgCatalogTokensCount = msgCatalogTokens.size();

                if (msgCatalogTokensCount > 0) {

                    String[] msgCatalogTokensArray = new String[msgCatalogTokensCount];

                    for (int counter = 0; counter < msgCatalogTokensCount; counter++) {
                        msgCatalogTokensArray[counter] = new String(((MsgCatalogToken) (msgCatalogTokens.get(counter))).getValue());
                    }

                    return msgCatalogTokensArray;
                }
            }
        }

        return null;
    }

    /**
     * Gets the level filter from the encapsulated Common Base Event or
     * <code>java.util.logging.LogRecord</code> super class.
     * <p>
     * See the associated setter API for more details on the mapping of this
     * <code>java.util.logging.LogRecord</code> property.
     * <p>
     * 
     * @return name of the resource bundle
     * @see java.util.logging.LogRecord#getResourceBundleName()
     */
    public String getResourceBundleName() {

        if (commonBaseEvent != null) {

            MsgDataElement msgDataElement = commonBaseEvent.getMsgDataElement();

            if (msgDataElement != null) { return (msgDataElement.getMsgCatalog()); }
        } else {
            return (super.getResourceBundleName());
        }

        return null;
    }

    /**
     * Gets the unique sequence number from the encapsulated Common Base Event
     * or <code>java.util.logging.LogRecord</code> super class.
     * <p>
     * See the associated setter API for more details on the mapping of this
     * <code>java.util.logging.LogRecord</code> property.
     * <p>
     * 
     * @return The unique sequence number.
     * @see java.util.logging.LogRecord#getSequenceNumber()
     */
    public long getSequenceNumber() {

        if (commonBaseEvent != null) {

            if (sequenceNumberExtendedDataElement != null) {

                try {
                    return (Long.parseLong(((String) (sequenceNumberExtendedDataElement.getValues().get(0)))));
                } catch (NumberFormatException n) {
                    //Ignore since API will return 0.
                }
            }
        } else {
            return (super.getSequenceNumber());
        }

        return 0;
    }

    /**
     * Gets the name of the source class from the encapsulated Common Base Event
     * or <code>java.util.logging.LogRecord</code> super class.
     * <p>
     * See the associated setter API for more details on the mapping of this
     * <code>java.util.logging.LogRecord</code> property.
     * <p>
     * 
     * @return The name of the source class.
     * @see java.util.logging.LogRecord#getSourceClassName()
     */
    public String getSourceClassName() {

        if (commonBaseEvent != null) {

            if (sourceClassNameExtendedDataElement != null) { return ((String) (sourceClassNameExtendedDataElement.getValues().get(0))); }
        } else {
            return (super.getSourceClassName());
        }

        return null;
    }

    /**
     * Gets the name of the source method from the encapsulated Common Base
     * Event or <code>java.util.logging.LogRecord</code> super class.
     * <p>
     * See the associated setter API for more details on the mapping of this
     * <code>java.util.logging.LogRecord</code> property.
     * <p>
     * 
     * @return The name of the source method.
     * @see java.util.logging.LogRecord#getSourceMethodName()
     */
    public String getSourceMethodName() {

        if (commonBaseEvent != null) {

            if (sourceMethodNameExtendedDataElement != null) { return ((String) (sourceMethodNameExtendedDataElement.getValues().get(0))); }
        } else {
            return (super.getSourceMethodName());
        }

        return null;
    }

    /**
     * Gets the unique ID of an associated thread from the encapsulated Common
     * Base Event or <code>java.util.logging.LogRecord</code> super class.
     * <p>
     * See the associated setter API for more details on the mapping of this
     * <code>java.util.logging.LogRecord</code> property.
     * <p>
     * 
     * @return The unique ID of an associated thread.
     * @see java.util.logging.LogRecord#getThreadID()
     */
    public int getThreadID() {

        if (commonBaseEvent != null) {

            if (threadIDExtendedDataElement != null) {

                try {
                    return (Integer.parseInt(((String) (threadIDExtendedDataElement.getValues().get(0)))));
                } catch (NumberFormatException n) {
                    //Ignore since API will return 0.
                }
            }
        } else {
            return (super.getThreadID());
        }

        return 0;
    }

    /**
     * Generates the formatted XML fragment representation of the encapsulated
     * Common Base Event property.
     * <p>
     * The formatted XML fragment is the representation of the encapsulated
     * Common Base Event instance variable and not the XML fragment
     * representation of the complete <code>java.util.logging.LogRecord</code>.
     * <p>
     * 
     * @return The XML fragment representation of the Common Base Event in this
     *         log record, or <code>null</code> if the value of the
     *         commonBaseEvent instance variable is set to <code>null</code>.
     */
    public String externalizeCanonicalXmlString() {

        if (commonBaseEvent != null) { return (EventFormatter.toCanonicalXMLString(commonBaseEvent, true)); }

        return null;
    }

    /**
     * Generates the formatted XML document representation of the encapsulated
     * Common Base Event property.
     * <p>
     * The formatted XML document is the representation of the encapsulated
     * Common Base Event instance variable and not the XML document
     * representation of the complete <code>java.util.logging.LogRecord</code>.
     * <p>
     * 
     * @return The XML document representation of the Common Base Event in this
     *         log record, or <code>null</code> if the value of the
     *         commonBaseEvent instance variable is set to <code>null</code>.
     */
    public String externalizeCanonicalXmlDocString() {

        if (commonBaseEvent != null) { return (EventFormatter.toCanonicalXMLDocString(commonBaseEvent, true)); }

        return null;
    }

    /**
     * Returns the textual representation of this
     * <code>CommonBaseEventLogRecord</code>.
     * <p>
     * 
     * @return Textual representation of this
     *         <code>CommonBaseEventLogRecord</code>.
     * @see java.lang.Object#toString()
     */
    public String toString() {

        if (commonBaseEvent != null) { return (commonBaseEvent.toString()); }

        return (super.toString());
    }

    /**
     * Iterates the list of Extended Data Element(s) in the class' Common Base
     * Event for Extended Data Element(s) with the
     * <code>CommonBaseEventLogRecord:</code> namespace (see
     * <code>EXTENDED_DATA_ELEMENT_NAME_NAMESPACE</code>).
     * <p>
     * References to known Extended Data Element(s) are cached for performance
     * reasons.
     */
    private void resolveExtendedDataElementReferences() {

        ListIterator listIterator = commonBaseEvent.getExtendedDataElements().listIterator();
        ExtendedDataElement extendedDataElement = null;
        String extendedDataElementName = null;

        while (listIterator.hasNext()) {

            extendedDataElement = ((ExtendedDataElement) (listIterator.next()));

            extendedDataElementName = extendedDataElement.getName().trim();

            if (extendedDataElementName.startsWith(EXTENDED_DATA_ELEMENT_NAME_NAMESPACE)) {

                //NOTE: Only need to check one of the level Extended Data
                // Element references since both are mutually inclusive:
                if ((levelNameExtendedDataElement == null) && (extendedDataElementName.equals(EXTENDED_DATA_ELEMENT_NAME_LEVEL))) {

                    List nameExtendedDataElements = extendedDataElement.getChildren(EXTENDED_DATA_ELEMENT_NAME_LEVEL_NAME);
                    List valueExtendedDataElements = extendedDataElement.getChildren(EXTENDED_DATA_ELEMENT_NAME_LEVEL_VALUE);

                    if ((nameExtendedDataElements.size() > 0) && (valueExtendedDataElements.size() > 0)) {

                        levelNameExtendedDataElement = ((ExtendedDataElement) (nameExtendedDataElements.get(0)));

                        levelValueExtendedDataElement = ((ExtendedDataElement) (valueExtendedDataElements.get(0)));
                    }
                } else if ((loggerNameExtendedDataElement == null) && (extendedDataElementName.equals(EXTENDED_DATA_ELEMENT_NAME_LOGGER_NAME))) {
                    loggerNameExtendedDataElement = extendedDataElement;
                } else if ((sequenceNumberExtendedDataElement == null) && (extendedDataElementName.equals(EXTENDED_DATA_ELEMENT_NAME_SEQUENCE_NUMBER))) {
                    sequenceNumberExtendedDataElement = extendedDataElement;
                } else if ((sourceClassNameExtendedDataElement == null) && (extendedDataElementName.equals(EXTENDED_DATA_ELEMENT_NAME_SOURCE_CLASS_NAME))) {
                    sourceClassNameExtendedDataElement = extendedDataElement;
                } else if ((sourceMethodNameExtendedDataElement == null) && (extendedDataElementName.equals(EXTENDED_DATA_ELEMENT_NAME_SOURCE_METHOD_NAME))) {
                    sourceMethodNameExtendedDataElement = extendedDataElement;
                } else if ((threadIDExtendedDataElement == null) && (extendedDataElementName.equals(EXTENDED_DATA_ELEMENT_NAME_THREAD_ID))) {
                    threadIDExtendedDataElement = extendedDataElement;
                } else if ((!isThrownSet) && (extendedDataElementName.equals(EXTENDED_DATA_ELEMENT_NAME_THROWN))) {
                    isThrownSet = true;
                }
            }
        }
    }
}