/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TRCNodeDispatcher.java,v 1.7 2005/04/22 14:52:45 bjiang Exp $
 * 
 * Contributors: 
 * IBM Corporation - initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.execution.testgen.http;

import java.lang.reflect.Constructor;
import java.util.HashMap;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.internal.execution.testgen.TestgenException;


/**
 * @author dhinson
 *
 * TRCNodeDispatcher is responsible for managing and invoking trace element (node)
 * handlers.  On creation, TRCNodeDispatcher first searches for node handlers that
 * have extended the plugin's nodeHandlers extension.  Next, for each handler kind
 * defined in HANDLER_KINDS which does not have an extension handler, a handler
 * defining plugin properties entry is sought (using a standard naming convention).
 * Failing that, a handler class using the same naming convention is searched for in
 * the local package.  If not found then a handler not found exception is thrown.
 * 
 * This design requires, at minimum, a handler be bound to each kind defined in
 * HANDLER_KINDS.  The node handler base class TRCNodeHandler can be used as a
 * default handler for any required kinds in lieu of a real implementation.  In this
 * case a TestgenHttpPlugin.properties entry should be made of the form:
 *     <node-kind>NodeHandler=org.eclipse.hyades.execution.testgen.http.TRCNodeHandler
 * for each node-kind without an implementation.
 * 
 * Note that handler binding via the nodeHandlers extension overrides any properties
 * or local package bindings, and additionally permits handlers to be defined for
 * kinds not listed in HANDLER_KINDS.  Although the later feature should be used
 * with caution as it would involve trace elements not present in
 * schema/TRCHTTPSchema.xsd.
 */
public class TRCNodeDispatcher
{
	private static final String 	DEFAULT_HANDLER_PACKAGE_NAME =
		"org.eclipse.hyades.execution.testgen.http";

	// add additional handler kinds here.  note: the handler kinds listed here
	// should be consistent with the set of trace elements defined in
	// schema/TRCHTTPSchema.xsd.		
	private static final String []	HANDLER_KINDS = {
		"TRCAnnotation",
		"TRCConnection",
		"TRCExecutorInfo",
		"TRCPacket",
		"TRCRecorderInfo"
	};
	
	private static final String	NODE_HANDLER_NAME_SUFFIX = "NodeHandler";
	private static final String	NODE_HANDLER_BASE_CLASS_NAME = "TRC" +
		NODE_HANDLER_NAME_SUFFIX;

	private HashMap map = new HashMap();
	
	public TRCNodeDispatcher (TRCContext context)
		throws TestgenException
	{
		registerHandlersFromRegistry(context);
		registerHandlersFromProperties(context);
	}

	private void registerHandlersFromRegistry (TRCContext context)
		throws TestgenException
	{	
		IExtensionPoint extPoint = Platform.getExtensionRegistry().getExtensionPoint(DEFAULT_HANDLER_PACKAGE_NAME,
			"nodeHandlers");

		IExtension[] exts = extPoint.getExtensions();
		for (int i = 0; i < exts.length; ++i) {
			IExtension ext = exts[i];
			IConfigurationElement [] confElems = ext.getConfigurationElements();
			for (int j = 0; j < confElems.length; ++j) {
				IConfigurationElement confElem = confElems[j];
				
				String nodeKind = confElem.getAttribute("nodeKind");

				TRCNodeHandler handler = null;				
				try {
					handler = (TRCNodeHandler)confElem.createExecutableExtension("class");
				}
				catch (CoreException exc) {
					exc.printStackTrace();
				}
				handler.init(context);
				registerHandler(nodeKind, handler);
			}
		}
	}
	
	private void registerHandlersFromProperties (TRCContext context)
		throws TestgenException
	{
		for (int kind = 0; kind < HANDLER_KINDS.length; ++kind) {
			String handlerKind = HANDLER_KINDS[kind];

			// if a handler is already registered for this kind then skip
			if (handlerIsRegistered(handlerKind))
				continue;
					
			String handlerClassName = handlerKind + NODE_HANDLER_NAME_SUFFIX;

			// allow handler name to be overridden by the plugin properties
			String handlerClassFQName = TestgenHttpPlugin.getResourceString(handlerClassName);
			if (handlerClassFQName.equals(handlerClassName))
				handlerClassFQName = DEFAULT_HANDLER_PACKAGE_NAME + "." + handlerClassName;

			TRCNodeHandler handler = newNodeHandler(handlerClassFQName, context);
			handler.init(context);
			registerHandler(HANDLER_KINDS[kind], handler);
		}
	}
	
	private void registerHandler (String name, TRCNodeHandler handler)
	{
//		System.err.println("registerHandler: [" + name + "] [" + handler + "]");
		map.put(name, handler);
	}

	private boolean handlerIsRegistered (String name)
	{
		return (map.get(name) == null) ? false : true;
	}
	
	private TRCNodeHandler newNodeHandler (String classFQName, TRCContext context)
		throws TestgenException
	{
			Class clas = null;
			try {
				clas = Class.forName(classFQName);
			}
			catch (Exception e) {
				throw new TestgenException(TestgenHttpPlugin.
					getResourceString("E_LOAD_NODE_HANDLER") + " " + classFQName, e);
			}

			String baseClassFQName = DEFAULT_HANDLER_PACKAGE_NAME + "." +
				NODE_HANDLER_BASE_CLASS_NAME;
			if (!classFQName.equals(baseClassFQName)) {			
				Class superClas = clas.getSuperclass();
				if (!superClas.getName().equals(baseClassFQName))
					throw new TestgenException(TestgenHttpPlugin.
						getResourceString("E_INVALID_NODE_HANDLER") + " " +
							classFQName);
			}
	
			TRCNodeHandler handler = null;
			try {
				Constructor constructor = clas.getConstructor(null);
				handler = (TRCNodeHandler)constructor.newInstance(null);
			}
			catch (Exception e) {
				throw new TestgenException(null, e);
			}
			return handler;
	}

	public void dispatch (TRCNode node)
		throws TestgenException
	{
		TRCNodeHandler handler = (TRCNodeHandler)map.get(node.getName());
		// if no handler is registered for this kind then automatically register
		// a default one.  note that this will only happen if a new trace element
		// is discovered during trace parsing that is not listed in HANDLER_KINDS. 
		if (handler == null) {
			handler = new TRCNodeHandler();
			registerHandler(node.getName(), handler);
		}
		handler.node(node);
	}
	
}
