/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: IParser.java,v 1.4 2005/02/16 22:20:55 qiyanli Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.logging.parsers;

import java.util.Hashtable;

import org.apache.commons.logging.Log;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;

/**
 * This is the interface that embodies a log file parser.  The log  
 * records are parsed from the file and CommonBaseEvent objects are
 * created to represent the data in the log records.
 * 
 * The expected flow of method calls by a user if this class would be:
 * setConfiguration();
 * preParse();
 * parse(Log); or parseNext();
 * postParse();
 * 
 * @author Dave Smith
 *
 */
public interface IParser {

	/**
	 * This method is called to provide configuration information to the parser.
	 * For example, it can be used to provide the name of the log file that is
	 * to be parsed.  Configuation parameters are passes as a Hashtable of key/value
	 * pairs where the key is the parameter name and the value is the parameter value.
	 *
	 * @param table  Hashtable containing keys and values of configuration information  
	 * @exception LogParserException thrown if the specified configuration information is incorrect or invalid for the parser.
	 */
	public void setConfiguration(Hashtable table) throws LogParserException;

	/**
	 * This method is called to perform some initialization required by the parser.
	 * For example, it can open the log file that is to be parsed or perform some
	 * preprocessing on the log file before it is parsed.  
	 *  
	 * @exception LogParserException thrown if the parser initialization fails.
	 */
	public void preParse() throws LogParserException;
		
	/**
	 * This method is called to perform the parsing of the log file and
	 * send the resulting CommonBaseEvent objects to an Apache Commons Log  
	 *
	 * @param argLogger  logger to which CommonBaseEvent objects are written  
	 * @exception LogParserException thrown if exception is detected while executing function of this superclass
	 */
	public void parse(Log argLogger) throws LogParserException;	

	/**
	 * This method parses the next set of log file records and returns an
	 * array of CommonBaseEvent objects that represent those log file records.
	 * A null array is returned when there are no more records in the file.
	 *
	 * @return the array of CBE's generated from the parsed data 
	 */
	public CommonBaseEvent[] parseNext() throws LogParserException;

	/**
	 * This method is called to perform some cleanup required after parsing of the
	 * log file is completed. For example, it can close the log file that was parsed. 
	 *  
	 * @exception LogParserException thrown if the parser cleanup fails.
	 */
	public void postParse() throws LogParserException;
		
	/**
	 * Returns the name of this parser.  The name should indicate what
	 * type of log file this parser is for.
	 *   
	 * @return String  The name of this parser
	 */
	public String getName();
	
	/**
	 * Returns the version of the parser.   
	 *
	 * @return String  The version of this parser 
	 */
	public String getVersion();
}
