package org.eclipse.hyades.logging.parsers;

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: MonitoringParser.java,v 1.11 2005/05/06 20:26:34 apnan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.Hashtable;
import java.util.Locale;

import org.eclipse.hyades.logging.adapter.util.BufferedPeriodicReader;
import org.eclipse.hyades.logging.adapter.util.MultipleFilesReader;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
/**
 * MonitoringParser is the abstract superclass parser classes that can continuously monitor and parse
 * a log file. Subclasses must use the readLine() method of this class in order to utilize the continuous
 * monitoring support. Subclasses must call setEndOfFile() method in parseNext() when it is about to return null.
 * logFile is not used by this class and is set to null.  No access is provided to the actual log file
 * except by the readLine method.  
 */
public abstract class MonitoringParser extends Parser {

    /**
     * Reader for log file to be parsed. 
     * A BufferedPeriodic Reader is used to support continuous monitoring of 
     * log files.
     */
    private BufferedPeriodicReader input = null;
    
    /**
     * Flag to indicate that there are currently no more parsed records to return.
     */
    private boolean endOfFile = true;
    
	/* Support for multiple log files */
	private boolean multi = false; // specify whether multi files are being monitored
	private RandomAccessFile tmpLog = null;

    /**
     * This function is called to provide user-specified information to the parser.
     * Currently, this superclass method handles the path and name of the log file
     * to be parsed.  Subclasses will override this method to handle additional
     * user-specified information.
     *
     * @param table  Hashtable containing keys and values of user-specified information  
     * @exception LogParserException thrown if user-specified path and name for the log file does not exist on the parsing machine
     */
    public void setConfiguration(Hashtable table) throws LogParserException {
        file_path = null;
        Integer array_size;

        /** Commented code in this method could be used to allow user specification
         **  of the size of arrays to be written to the ILogger.
         **  String array_size = null; */

		try {
			tmpFile = File.createTempFile(prefix, null);
			tmpLog = new RandomAccessFile(tmpFile, "rws");
		} catch (Exception e) {
			tmpLog = null;
		}
		
		// logFile is not used in this class so set it to null
		logFile = null;
		
        try {
            file_path = ((String) (table.get(ParserConstants.FILE_PATH_KEY)));

            if (file_path != null) {
            	/* bugzilla 91489 - get the file name and directory from the respective hash table
            	 * values instead of parsing the file_path hash table value in case the file name
            	 * is a regular expression.
            	 */
                String dirPath = ((String) (table.get(ParserConstants.DIRECTORY_KEY)));
                String fName = ((String) (table.get(ParserConstants.FILE_NAME_KEY)));

                if (fName != null && fName.length() > 0 && dirPath != null && dirPath.length() > 0) {
	                /* Must substitute platform dependent file separator character in the directory 
	                 * path variable for both UNIX and Windows platforms due to cross-platform 
	                 * communication via the Agent Controller:
	                 */
	                if (ParserConstants.FILE_SEPARATOR_CHARACTER == '/') {
	                    dirPath = dirPath.replace('\\', ParserConstants.FILE_SEPARATOR_CHARACTER);
	                }
	                else {
	                    dirPath = dirPath.replace('/', ParserConstants.FILE_SEPARATOR_CHARACTER);
	                }
                }
                else {
                	/* If the file name and directory values are not in the hash table then use
                	 * the file_path value.
                	 */
                    //Must substitute platform dependent file separator character in the file_path instance variable for both UNIX and Windows platforms due to cross-platform communication via the Agent Controller:
	                if (ParserConstants.FILE_SEPARATOR_CHARACTER == '/') {
	                    file_path = file_path.replace('\\', ParserConstants.FILE_SEPARATOR_CHARACTER);
	                }
	                else {
	                    file_path = file_path.replace('/', ParserConstants.FILE_SEPARATOR_CHARACTER);
	                }
	                
	                fName = file_path.substring(file_path.lastIndexOf(ParserConstants.FILE_SEPARATOR_CHARACTER)+1);
	                dirPath = file_path.substring(0,file_path.indexOf(fName)-1);
                }
                
                if (fName == null || dirPath == null) {
                	throw new LogParserException(ParserUtilities.getResourceString("LOG_PARSER_INVALID_FILE_NAME_ERROR_",file_path));
                }
                
                boolean continuousContext = ((Boolean)table.get(ParserConstants.CONTINUOUS_KEY)).booleanValue();
                
                File dir = new File(dirPath);
                
                if (!dir.exists() || !dir.isDirectory()) {
                	throw new LogParserException(ParserUtilities.getResourceString("LOG_PARSER_INVALID_FILE_NAME_ERROR_",file_path));
                }
        		mfr = new MultipleFilesReader(dirPath, fName, tmpLog);
        		mfr.init();

        		// If only one file matches the regex, use it as the file name
        		if(mfr.size() <= 1) {
        			multi = false;

        			// Close the temporary file since it won't be used
        			if (tmpLog != null) {
        				try {
        					tmpLog.close();
        				}
        				catch (IOException e) {
        					// Ignore this exception
        				}
        				tmpLog = null;
        			}
        			
        			if(mfr.size() == 1) {
        				file_path = mfr.getNext();  // The file name is fully qualified		
        			}
        			else if (!continuousContext) { 
        				// no match at all so throw an exception if we aren't running in continuous mode
        				throw new LogParserException(ParserUtilities.getResourceString("LOG_PARSER_INVALID_FILE_NAME_ERROR_",file_path));
        			}
        		}
        		// Otherwise, use the temporary file as the file name
        		else {
        			multi = true;

        			// Create a single log file for Parser to read
        			file_path = tmpFile.getAbsolutePath();
        			mfr.start();
        		}
        		
                /* Check for the validity of the file if we aren't running in continuous mode */
        		if (!continuousContext) {
	                File file = new File(file_path);
	
	                if (!file.isFile()) {
	                    throw new LogParserException(ParserUtilities.getResourceString("LOG_PARSER_INVALID_FILE_NAME_ERROR_",file_path));
	                }
	                else if (!file.canRead()) {
	                    throw new LogParserException(ParserUtilities.getResourceString("LOG_PARSER_CANNOT_READ_FILE_ERROR_",file_path));
	                }
        		}
                input = new BufferedPeriodicReader(file_path);
                
                Integer confBufSize = (Integer)table.get(ParserConstants.CONFIDENCE_BUFFER_SIZE_KEY);
                Integer fileFooterSize = (Integer)table.get(ParserConstants.FILE_FOOTER_SIZE_KEY);
                charset = ((String) (table.get(ParserConstants.FILE_CHARSET_KEY)));
                fileLocale = ((Locale)(table.get(ParserConstants.FILE_LOCALE_KEY)));
              
				if (charset != null) {
					input.setCharset(charset);
				}
				if (confBufSize != null) {
					input.setConfidenceBufferSize(confBufSize.intValue());
				}
				if (fileFooterSize != null) {
					input.setFileFooterSize(fileFooterSize.intValue());
				}
				
				input.setMultiFile(multi);
            }

            array_size = (Integer)table.get(ParserConstants.MESSAGE_ARRAY_SIZE_KEY);
            if (array_size != null) {
            	MessageArraySize = array_size.intValue();
            }
            
            messages = new CommonBaseEvent[MessageArraySize];
         
            for (int i = 0; i < MessageArraySize; i++) {
                messages[i] = eventFactory.createCommonBaseEvent();
            }
        }
        catch (Throwable t) {
            ParserUtilities.exceptionHandler(t, ParserUtilities.getResourceString("REMOTE_LOG_PARSER_CONFIG_PARAMETER_ERROR_"));
        }

        /** catch(NumberFormatException nfe)
         **{
         **    LogParserException lpe = new LogParserException("array_size value, " + array_size + " ,must be an integer.", nfe);
         **    lpe.fillInStackTrace();
         **    throw lpe;
         **}*/
    }

    /* utility methods */

    /**
     * Perform cleanup after parsing is complete.
     */
    public void postParse() throws LogParserException {
        try {
            if (input != null) {
                input.close();
                input = null;
            }
            
            mfr.interrupt();
            
			// Close the temporary file
			if (tmpLog != null) {
				try {
					tmpLog.close();
					tmpLog = null;
				}
				catch (IOException e) {
					// Ignore this exception
				}
			}
			// Delete the temporary file
			tmpFile.delete();
        }
        catch (Exception e) {
            // no action required
			ParserUtilities.exceptionHandler(e, "");
        }
    }

    /**
     * Reads in the next line from the log file and returns same. 
     * It returns null if end of file is reached before any more 
     * characters are written to the log file.
     * @return   the next line
     * @throws LogParserException if an error occurs while opening
     * or reading the log file.
     */
    protected String readALine() throws LogParserException {

        String msgLine = null;

        if (endOfFile) {
        	try {
        		input.prepare();
        		endOfFile = false;
        		// Get the total size of the log file
        		totalSize = input.getSize();
        	}
        	catch (Exception e) {
        		throw new LogParserException(ParserUtilities.getResourceString("LOG_PARSER_CANNOT_READ_FILE_ERROR_",file_path), e);
        	}
        }
        
        try {

	        msgLine = input.readLine();
         
			if (msgLine != null) {
				// Increment the processed size value by the length of the line read.
				processedSize += msgLine.length();
				curLineNumber++;
			}
        }
        catch (IOException i) {
        	throw new LogParserException(ParserUtilities.getResourceString("LOG_PARSER_CANNOT_READ_FILE_ERROR_",file_path), i);
        }

        return msgLine;
    }

    /**
     * Reads in the next line from the log file and returns same. 
     * It returns null if an exception occurs while reading the log
     * file or if end of file is reached before any more new characters
     * are written to the file.
     *  
     * @return   the next line
     * @deprecated as of TPTP v3.3 by readALine()
     */
    protected String readLine() {
    	try {
    		return readALine();
    	}
    	catch (LogParserException e) {
    		return null;
    	}
    }
    
    /**
     * Sets the file name to parse.
     * This must be called after setConfiguration is called
     * @param newfile name of file to parse
     * @throws LogParserException
     */
    protected void setFilename(String newfile) throws LogParserException {
    	if (input == null) {
    		throw new LogParserException(ParserUtilities.getResourceString("LOG_PARSER_NOT_CONFIGURED_ERROR_"));
    	}
    	input.setFilename(newfile);
    	return;
    }
    
    /**
     * Sets the command to convert the log file to a different format.
     * This must be called after setConfiguration is called.
     * @param converterCmd The converter command as a single string
     * @throws LogParserException
     */
    protected void setConverterCommand(String converterCmd) throws LogParserException {
    	if (input == null) {
    		throw new LogParserException(ParserUtilities.getResourceString("LOG_PARSER_NOT_CONFIGURED_ERROR_"));
    	}
    	
    	input.setConverter(converterCmd);
    	return;
    }
    
    /**
     * Sets the command to convert the log file to a different format.
     * This must be called after setConfiguration is called.
     * @param converterCmd The converter command as an array of strings
     * @throws LogParserException
     */
    protected void setConverterCommand(String [] converterCmd) throws LogParserException {
    	if (input == null) {
    		throw new LogParserException(ParserUtilities.getResourceString("LOG_PARSER_NOT_CONFIGURED_ERROR_"));
    	}
    	
    	input.setConverter(converterCmd);
    	return;
    }
    
    /**
     * Gets the current file pointer while reading the log file.
     * This must be called after setConfiguration is called
     * @return current offset within file that is being read
     */
    protected long getFilePointer() {
    	if (input != null) {
    		return input.getFilePointer();
    	}
    	return 0;
    }
    
    /**
     * Sets the end of file flag to indicate there are no more parsed records 
     * to return.  This must be called by subclasses when parseNext() returns null;
     *
     */
    protected void setEndOfFile() {
    	endOfFile = true;
    }

	/**
	 * @return Returns the endOfFile.
	 */
	public boolean isEndOfFile() {
		return endOfFile;
	}
}
