/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ModuleJ.cpp,v 1.12 2005/02/16 22:21:27 qiyanli Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/

//==============================================================================
// ModuleJ.cpp
// 9/28/99
//
//------------------------------------------------------------------------------
// Description
// Java-specific module implementation
//
//==============================================================================
#include "ModuleJ.h"
#include "JVMInsSet.h"
#include "JFileStream.h"

static CInsSetJ InsSet;

//==============================================================================
// CModuleJ implementation
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// Constructor
//

CModuleJ::CModuleJ()
:CModule(&InsSet)
{
	m_pClass = NULL;
	m_fDestroyClass = false;
	m_fAccessFlags = 0;
	m_sourceFileNamesPopulated = false;
}

//------------------------------------------------------------------------------
// Destructor
//

CModuleJ::~CModuleJ()
{
	if(m_fDestroyClass)
	{
		delete m_pClass;
	}
}

//------------------------------------------------------------------------------
// Open module as a file
//
void	
CModuleJ::Open(CSTR i_szName)
{
	CJClassBuilder*	pJClass;						// Java Class
	CJFileStream	FileStreamIn;					// File stream
	CJStream		InStream(&FileStreamIn);		// Java input stream

	m_strName = i_szName;
	pJClass = new CJClassBuilder;
	FileStreamIn.Open(i_szName, CJFileStream::ACCESS_READ);
	pJClass->Read(InStream);
	m_pClass = pJClass;
	m_fDestroyClass = true;
	FileStreamIn.Close();
	Verify();
}

//------------------------------------------------------------------------------
// IsInstrumented
// Is the module instrumented?
// The module is considered to be instrumented if it has the "Instrumented" field
//
// Returns:
//		true if module is instrumented
//		false otherwise
//
bool
CModuleJ::IsInstrumented() const
{
	CJAttribs* pAttribs = m_pClass->GetAttribs();
	for(CJAttribs::iterator iter = pAttribs->begin(); iter < pAttribs->end(); iter++)
	{
		string strName = *(*iter)->GetName();
		if(strName.compare("Instrumented") == 0)
		{
			return true;
		}
	}
	return false;
}

//------------------------------------------------------------------------------
void	
CModuleJ::SetAccessFlags(unsigned long f)
{
	m_fAccessFlags = f;
}

//------------------------------------------------------------------------------
// Open
// Open module given classfile
// Note: the classfile must be read or otherwise constructed
//       prior to passing into this routine
//
void	
CModuleJ::Open(CJClassBuilder* io_pClass, bool i_fDestroyClass)
{
	m_pClass = io_pClass;
	m_fDestroyClass = i_fDestroyClass;
	Verify();
}

//------------------------------------------------------------------------------
void	
CModuleJ::AddExtRef(CExtRef& i_ExtRef)
{
	// Add external reference to the module
	// ToDo? Verify that it is a valid Java reference?
	i_ExtRef.InjectMetaData(*this);	

}

//------------------------------------------------------------------------------
void	
CModuleJ::AddStringAttrib(CSTR i_szName, CSTR i_szValue)
{
	CJAttribs* pAttribs = m_pClass->GetAttribs();
	CStringAttribute* pStringAttrib = new CStringAttribute(m_pClass,
		                                                   CCPUtf8Info(i_szName),
		                                                   CCPUtf8Info(i_szValue));
	pAttribs->Add(pStringAttrib);
}

//------------------------------------------------------------------------------
// Module modification methods

//------------------------------------------------------------------------------
CCPFieldrefInfo*	
CModuleJ::CreateFieldRef(u2 i_u2AccFlags, CSTR i_szName, CJavaType i_jtype)
{
	CFieldInfo* pfi = m_pClass->CreateField(i_u2AccFlags, i_szName, i_jtype);
	CCPFieldrefInfo* pfri = m_pClass->CreateFieldRef(pfi);
	return pfri;
}

//------------------------------------------------------------------------------
CMethodJ*			
CModuleJ::CreateMethod(u2 i_u2AccFlags, CSTR i_szName, CSTR i_szSignature)
{
	CJMethodInfo* pmi = m_pClass->CreateMethod(i_u2AccFlags, i_szName, i_szSignature);

	CMethodJ* pmeth = new CMethodJ(this, i_szName, i_szSignature, pmi->GetCode(), i_u2AccFlags);
	CCodeAttribute* pCodeAttribute = pmi->GetCode();
	if(NULL != pCodeAttribute)
	{
		u1* pcodeBytes = pCodeAttribute->GetCode();
		u2  u2CodeLength = pCodeAttribute->GetCodeLength();
		CMethodBody *pBody = new CMethodBody(this, pcodeBytes, u2CodeLength);
		pmeth->SetBody(pBody);
	}
	pmeth->Parse();
	m_pMethods->push_back(pmeth);
	return pmeth;
}

//------------------------------------------------------------------------------
void	
CModuleJ::Parse()
{
	CJMethods* pMethods	= m_pClass->GetMethods();
	CJMethods::iterator iterjm;
	// GetClassName
	CConstPool& cp = *(m_pClass->GetConstPool());
	CCPClassInfo* pClassInfo = (CCPClassInfo*)cp[m_pClass->GetThisClass()];
	CCPUtf8Info* putf8Name = (CCPUtf8Info*)cp[pClassInfo->GetClassInd()];
	m_strName = (string)*putf8Name;
	
	// Get list of interfaces this class implements,
	// populate m_interfaceNames.

	CJInterfaces* pInterfaces = m_pClass->GetInterfaces();
	for (CJInterfaces::iterator iter = pInterfaces->begin();
		 iter != pInterfaces->end();
		 iter++)
	{
		CInterfaceInfo* this_interface = (*iter);
		u2 this_inter_index = this_interface->GetIndex();
		CCPClassInfo* this_inter_class_info = (CCPClassInfo*)cp[this_inter_index];
		u2 this_inter_name_index = this_inter_class_info->GetClassInd();
		CCPUtf8Info* this_inter_name_utf8 = (CCPUtf8Info*)cp[this_inter_name_index];
		string this_inter_name_string = (string)(*this_inter_name_utf8);
		m_interfaceNames.push_back(this_inter_name_string);
	}

	// Setup methods
	for(iterjm = pMethods->begin(); iterjm < pMethods->end(); iterjm++)
	{
		string strName = (string)(*(*iterjm)->GetName());
		string strSign = (string)(*(*iterjm)->GetDescriptor());
		u2 u2AccessFlags = (*iterjm)->GetAccessFlags();
		CCodeAttribute* pCodeAttr = (*iterjm)->GetCode();
		CMethod* pMtd = new CMethodJ(this, strName.c_str(), strSign.c_str(), pCodeAttr, u2AccessFlags);
		if(NULL != pCodeAttr)
		{
			CMethodBody *pBody = new CMethodBody(this, pCodeAttr->GetCode(), pCodeAttr->GetCodeLength());
			pMtd->SetBody(pBody);
		}
		m_pMethods->push_back(pMtd);
	}
	
	CMethods::iterator iterm;
	for(iterm =  m_pMethods->begin(); iterm < m_pMethods->end(); iterm++)
	{
		(*iterm)->Parse();
	}
}

//------------------------------------------------------------------------------
void	
CModuleJ::Emit()
{
	CModule::Emit();
}

//------------------------------------------------------------------------------
void	
CModuleJ::Emit(CJStream& i_jstream)
{
	CModule::Emit();
	m_pClass->Write(i_jstream);
}

//------------------------------------------------------------------------------
CJClassFile&
CModuleJ::GetClass()
{
	if(NULL == m_pClass)
		throw CModuleException(CModuleException::X_REASON_INTERNAL_ERROR, "No Java Class");
	return *m_pClass;
}

//------------------------------------------------------------------------------
CJClassBuilder&
CModuleJ::GetClassBuilder()
{
	if(NULL == m_pClass)
		throw CModuleException(CModuleException::X_REASON_INTERNAL_ERROR, "No Java Class");
	return *m_pClass;
}


//- Private methods ------------------------------------------------------------
void
CModuleJ::Verify()
{
	if(NULL == m_pClass)
	{
		throw CModuleException(CModuleException::X_REASON_INVALID_MODULE, "No Java Class");
	}
	//m_pClass->Verify();
	//ToDo: module verification ?
}

const vector<string>&
CModuleJ::GetSourceFileNames()
{
	// If this is the first time through here, populate m_sourceFileNames
	if (!m_sourceFileNamesPopulated) {
		CJClassFile& classFile = GetClass();
		CJAttribs* pAttribs = classFile.GetAttribs();
		for (int i = 0; i < pAttribs->size(); i++) {
			CAttributeInfo* pAttrib = (*pAttribs)[i];
			CCPUtf8Info* pNameInfo = pAttrib->GetName();
			if (((string)(*pNameInfo)).compare("SourceFile") == 0) {
				// Found the SourceFile attribute
				CSourceFileAttribute* psfa = (CSourceFileAttribute*)(pAttrib);
				CCPUtf8Info *putf8 = psfa->GetValue();
				m_sourceFileNames.push_back((string)(*putf8));
				break;
			}
		}
		m_sourceFileNamesPopulated = true;
	}

	return m_sourceFileNames;
}

//==============================================================================
// CMethodJ implementation
//

//------------------------------------------------------------------------------
// Parse
void
CMethodJ::Parse()
{

	// Java-specific parsing: make a label at every line number.
	// Also populate m_LineNumbers for mapping old to new IPs.

	if(NULL != m_pCodeAttr)
	{
		CLineNumberTableAttribute* plinenums = m_pCodeAttr->GetLineNumbers();
		if(NULL != plinenums)
		{
			CLineNumberTable& table = plinenums->GetLineNumberTable();
			u2 u2Leng = plinenums->GetTableLength();
			for(CLineNumberTable::iterator iter = table.begin(); iter != table.end(); iter++)
			{
				IP_t ip = (*iter)->GetStartPC();
				m_LineNumbers.insert(LineNumsEntry_t(ip, ip));
				// NOTE:
				// We create a 'fake' label for every line here
				// to force the parser building additional Insertion Block
				// for every line of code. This will automatically result in
				// slicing a new CInsBlock at the beginning of every line.
				m_Labels.AddLabel(ip);
			}
		}
	}

	// Generic parsing
	CMethod::Parse();

	// Exception handler table parsing
	if(NULL != m_pCodeAttr)
	{
		// Create and parse the exception table.
		m_pMtdExTable = new CMtdExTableJ(this);
		m_pMtdExTable->Parse();

		// Record the original method length for use in local variable table parsing
		m_origCodeLength = m_pCodeAttr->GetCodeLength();
	}
}

//------------------------------------------------------------------------------
// Emit
void
CMethodJ::Emit()
{
	// Generic emission
	CMethod::Emit();
	if(NULL == m_pCodeAttr)
 		return;
	// Update the line number information --------------------------------------
	CLineNumberTableAttribute* plinenums = m_pCodeAttr->GetLineNumbers();
	if(NULL != plinenums)
	{
		CLineNumberTable& table = plinenums->GetLineNumberTable();
		CLineNumberTable::iterator itrTable;
		CLineNumbers::iterator itrLines;

		for(itrTable = table.begin(); itrTable != table.end(); itrTable++)
		{
			
			IP_t ip = (*itrTable)->GetStartPC();
			itrLines = m_LineNumbers.find(ip);
			if(itrLines != m_LineNumbers.end())
				(*itrTable)->SetStartPC(itrLines->second);
		}
	}

	// Update exceptions and local variable mapping ----------------------------
	m_pMtdExTable->Emit();

	//-------
	// Loop over instructions and patch the local variable info table
	//

	CLocalVariableTableAttribute* pLclAttr = m_pCodeAttr->GetLocalVariables();
	CLocalVariableTable& LclTable = pLclAttr->GetLocalVariableTable();
	CLocalVariableTable  LclTableSav;
	CInsBlocks::iterator iterBlocks;
	IP_t ip = 0;

	if(NULL != pLclAttr)
	{
		LclTableSav = LclTable;
	}
	for(iterBlocks = m_Blocks.begin(); iterBlocks < m_Blocks.end(); iterBlocks++)
	{
		CInstructions* pins = (*iterBlocks)->GetInstructions();
		CInstructions::iterator iterIns;
		IP_t ipOrigBlk = (*iterBlocks)->GetOrigIP();

		for(iterIns = pins->begin(); iterIns != pins->end(); iterIns++)
		{
			IP_t ipOrig = (*iterIns)->GetIP();

			// Scan the local var table. Any entry that refers to the old IP of
			// this instruction gets rewritten to refer to the new IP of this instruction.
			// TODO: worry that this is slow, looping over the local var table for each insn.
			if(NULL != pLclAttr)
			{
				CLocalVariableTable::iterator itrLcl, itrLclSav;
				itrLclSav = LclTableSav.begin();
				for(itrLcl = LclTable.begin(); itrLcl != LclTable.end(); itrLcl++, itrLclSav++)
				{
					if(ipOrig == (*itrLclSav)->GetStartPC())
					{	// Original start ip found
						(*itrLcl)->SetStartPC(ip);
					}
					if(ipOrig == (*itrLclSav)->GetStartPC() + (*itrLclSav)->GetLength())
					{	// Original end ip found
						(*itrLcl)->SetLength(ip - (*itrLcl)->GetStartPC());
					}
				}
			}
			// Advance ip
			ip += (*iterIns)->GetSize(ip);
			if(ip >= 0x0000FFFF)
			{ // Code overflow
				string strProcName = m_pModule->GetName();
				strProcName += ".";
				strProcName += GetName();
				throw CModuleException(CModuleException::X_REASON_CODE_OVERRUN, strProcName.c_str());
			}
		}
	}

	// Final variable table handling: need to patch the "ends."
	// That is, those entries whose original PC+length equals the 
	// original size of the method. Patch their length values
	// so they still span to the end of the method.
	//
	// At this point in this function, ip is the new end ip of this method.
	// m_origCodeLength was recorded in the CMethod just for use right here.

	if (NULL != pLclAttr) {
		CLocalVariableTable::iterator itrLcl, itrLclSav;
		itrLclSav = LclTableSav.begin();
		IP_t endOrig = m_origCodeLength;
		for(itrLcl = LclTable.begin(); itrLcl != LclTable.end(); itrLcl++, itrLclSav++)
		{
			if(endOrig == (*itrLclSav)->GetStartPC() + (*itrLclSav)->GetLength())
			{	// end ip of this variable region is the original end ip of the method
				(*itrLcl)->SetLength(ip - (*itrLcl)->GetStartPC());
			}
		}
	}

	CalcStackDepth();
	// Replace the method body
	m_pCodeAttr->SetCode(m_pBody->GetCodeSize(), m_pBody->GiveAvayCode());

}

//------------------------------------------------------------------------------
// CalcStackDepth
//
int
CMethodJ::CalcStackDepth()
{
	int nStack = CMethod::CalcStackDepth();
	m_pCodeAttr->SetMaxStack(nStack);
	return nStack;
}

//==============================================================================
// CMtdExtableJ
//------------------------------------------------------------------------------
// For historical reasons the end of an exception block in a Java exception
// table points to the next instrucion after the protected block.
// Since the abstract exception table assumes the end of the exception block
// points to the start of the last instruction of the block, we need to recalculate
// the protected block as we parse the Java exception table.
// (We will undo this operation when we emit the table.)
//
//------------------------------------------------------------------------------
void	
CMtdExTableJ::Parse()
{
	CMethodJ* pmtd = (CMethodJ*)m_pmtd;
	CExTable& extblj = pmtd->GetCodeAttribute()->GetExTable();
	CExTable::iterator itrexj = extblj.begin();

	if(itrexj == extblj.end())
		return;		// No exceptions in this method!
	// Get exception table
	CMtdExTable* pMtdExTable = pmtd->GetExTable();
	// Patch the end of exception block addresses
	CInsBlocks* pblks = pmtd->GetInsBlocks();
	CInsBlocks::iterator itrBlks;
	CInstructions::iterator itrIns;

	for(itrBlks = pblks->begin(); itrBlks != pblks->end(); itrBlks++)
	{
		CInsBlock* pblk = *itrBlks;
		CInstructions* pins = pblk->GetInstructions();
		for(itrIns = pins->begin(); itrIns != pins->end(); itrIns++)
		{
			CInstruction* pi = *itrIns;
			IP_t ipIns = pi->GetIP();
			for(itrexj = extblj.begin(); itrexj != extblj.end(); itrexj++)
			{
				unsigned uType = itrexj->GetCatchtype();
				IP_t ipStart = itrexj->GetStartPC();
				IP_t ipEnd = itrexj->GetEndPC();
				if(ipEnd == pi->GetIP() + pi->GetSize())
				{
					ipEnd = pi->GetIP();
					CInsBlock* pblkHandler = pmtd->FindBlock(itrexj->GetHandlerPC());
					pmtd->AddException(new CMethodExceptionJ(uType, ipStart, ipEnd, pblkHandler));
				}
			}
		}
	}
}

//------------------------------------------------------------------------------
// Emit the exception table
// ToDo: Exception tables entries may be hashed to accelerate the process
void	
CMtdExTableJ::Emit()
{
	CMethodJ* pmtd = (CMethodJ*)m_pmtd;
	CLabels* plabels = pmtd->GetLabels();
	CMtdExTable* pMtdExTable = pmtd->GetExTable();
	CMtdExTable::iterator itrEx;
	CInsBlocks* pblks = pmtd->GetInsBlocks();
	CInsBlocks::iterator itrBlks;
	CInstructions::iterator itrIns;

	// Scrap the old exception table
	CExTable& extblj = pmtd->GetCodeAttribute()->GetExTable();
	extblj.clear();
	// Create new exception table
	for(itrEx = pMtdExTable->begin(); itrEx != pMtdExTable->end(); itrEx++)
	{
		bool bDone = false;
		CMethodExceptionJ* pmtdexj = (CMethodExceptionJ*)*itrEx;
		unsigned uType = pmtdexj->GetType();
		IP_t ipStart = plabels->GetLabelInstructionTarget(pmtdexj->GetStart());
		IP_t ipEnd = plabels->GetLabelInstructionTarget(pmtdexj->GetEnd());
		IP_t ipHandler = pmtdexj->GetHandler()->GetLabel();
		IP_t ipRealEnd = ipEnd;

		// Java has the end of exception block marked by the IP of
		// instruction following the last instruction in the block.
		// To folow this awkward requirement we have to scan all 
		// instructions for every exception table entry and recalculate
		// the end address.
		// Note: this implementation takes a brute force approach for simplicity.
		// The performance may be significantly improved with cashing.
		for(itrBlks = pblks->begin(); !bDone && itrBlks != pblks->end(); itrBlks++)
		{
			CInsBlock* pblk = *itrBlks;
			CInstructions* pins = pblk->GetInstructions();
			for(itrIns = pins->begin(); itrIns != pins->end(); itrIns++)
			{
				CInstruction* pi = *itrIns;
				IP_t ipIns = pi->GetIP();
				if(plabels->IsLabel(ipIns))
				{
					ipIns = plabels->GetLabelInstructionTarget(ipIns);
					if(ipIns == ipEnd)
					{
						ipRealEnd = ipIns + pi->GetSize();
						bDone = true;
						break;
					}
				}
			}
		}
		CJException jex = CJException(uType, ipStart, ipRealEnd, ipHandler); 
		extblj.push_back(jex);
	}
}

void
CMtdExTableJ::Dump(ostream &i_os) const
{
	i_os << "Exception Table (end refers to start of last instruction in region):" << endl;
	i_os << "Type\tStart\tEnd\tHandler" << endl;

	CMtdExTable::const_iterator itrEx;
	for(itrEx = begin(); itrEx != end(); itrEx++)
	{
		CMethodExceptionJ* pmtdexj = (CMethodExceptionJ*)*itrEx;
		unsigned uType = pmtdexj->GetType();
		IP_t ipStart = pmtdexj->GetStart();
		IP_t ipEnd = pmtdexj->GetEnd();
		IP_t ipHandler = pmtdexj->GetHandler()->GetLabel();

		i_os << uType << "\t" << ipStart << "\t" << ipEnd << "\t" << ipHandler << endl;
	}
}


//= End of modulej.cpp =========================================================
