package org.eclipse.hyades.logging.events.cbe.impl.tests;

import junit.framework.TestCase;

import org.eclipse.hyades.logging.events.cbe.AvailableSituation;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.SituationType;
import org.eclipse.hyades.logging.events.cbe.ValidationException;

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: AvailableSituationImplTest.java,v 1.3 2005/02/16 22:20:40 qiyanli Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * JUnit test case for the <code>AvailableSituationImpl</code> class.
 * 
 * @author Denilson Nastacio
 * @version 1.0.1
 * @since 1.0.1 
 */
public class AvailableSituationImplTest extends TestCase {

    /** Target instance for the tests. */
    private AvailableSituation sit;

    /**
     * Constructor for AvailableSituationImplTest.
     * @param testcase
     */
    public AvailableSituationImplTest(String testcase) {
        super(testcase);
    }

    /**
     * Tests whether <code>toString</code> doesn't blow up.
     */
    public void testToString() {
        assertNotNull(sit.toString());
    }

    /**
     * Tests presence of attributes in the public interface
     */
    public void testInterface() {
        sit.getAvailabilityDisposition();
        sit.getOperationDisposition();
        sit.getProcessingDisposition();
        sit.getReasoningScope();
        // setters show on other tests

        assertTrue(sit instanceof SituationType);
    }

    /**
     * Tests the serialization for <code>AvailableSituation</code>.
     */
    public void testSerialization() {
        Util.testEObjectSerialization(sit);
    }

    /**
     * Tests the <code>equals</code> operator.
     */
    public void testEquals() {
        AvailableSituation sit2 = (AvailableSituation) Util.clone(sit);
        assertEquals(sit, sit2);

        sit.setAvailabilityDisposition("a");
        sit.setOperationDisposition("z");
        sit.setProcessingDisposition("p");
        sit.setReasoningScope("r");
        sit2 = (AvailableSituation) Util.clone(sit);
        assertEquals(sit, sit2);

        sit2.setAvailabilityDisposition("difference");
        assertFalse(sit.equals(sit2));
    }

    /**
     * Tests validation for <code>AvailableSituation</code>.
     */
    public void testValidationMissingAttributes() throws ValidationException {
        Util.assertMissingAttributes(sit, "all");

        sit.setAvailabilityDisposition("a");
        sit.setOperationDisposition("o");
        sit.setProcessingDisposition("p");
        sit.setReasoningScope("r");
        sit.validate();

        sit.setAvailabilityDisposition(null);
        Util.assertMissingAttributes(sit, "availabilityDisposition");

        sit.setAvailabilityDisposition("a");
        sit.setOperationDisposition(null);
        sit.setProcessingDisposition("p");
        sit.setReasoningScope("r");
        Util.assertMissingAttributes(sit, "operationDisposition");

        sit.setAvailabilityDisposition("a");
        sit.setOperationDisposition("o");
        sit.setProcessingDisposition(null);
        sit.setReasoningScope("r");
        Util.assertMissingAttributes(sit, "processingDisposition");

        sit.setAvailabilityDisposition("a");
        sit.setOperationDisposition("o");
        sit.setProcessingDisposition("p");
        sit.setReasoningScope(null);
        Util.assertMissingAttributes(sit, "reasoningScope");
    }

    /**
     * Tests validation for <code>AvailableSituation</code>.
     */
    public void testValidationInvalidLengths() {
        sit.setAvailabilityDisposition(Util.getPaddedString(64));
        sit.setOperationDisposition(Util.getPaddedString(64));
        sit.setProcessingDisposition(Util.getPaddedString(64));
        sit.setReasoningScope(Util.getPaddedString(64));
        try {
            sit.validate();
        }
        catch (ValidationException e) {
            fail("Strings have correct length");
        }

        sit.setAvailabilityDisposition(Util.getPaddedString(65));
        sit.setOperationDisposition(Util.getPaddedString(64));
        sit.setProcessingDisposition(Util.getPaddedString(64));
        sit.setReasoningScope(Util.getPaddedString(64));
        Util.assertInvalidBoundaries(sit, "setAvailabilityDisposition");

        sit.setAvailabilityDisposition(Util.getPaddedString(64));
        sit.setOperationDisposition(Util.getPaddedString(65));
        sit.setProcessingDisposition(Util.getPaddedString(64));
        sit.setReasoningScope(Util.getPaddedString(64));
        Util.assertInvalidBoundaries(sit, "operationDisposition");

        sit.setAvailabilityDisposition(Util.getPaddedString(64));
        sit.setOperationDisposition(Util.getPaddedString(64));
        sit.setProcessingDisposition(Util.getPaddedString(65));
        sit.setReasoningScope(Util.getPaddedString(64));
        Util.assertInvalidBoundaries(sit, "processingDisposition");

        sit.setAvailabilityDisposition(Util.getPaddedString(64));
        sit.setOperationDisposition(Util.getPaddedString(64));
        sit.setProcessingDisposition(Util.getPaddedString(64));
        sit.setReasoningScope(Util.getPaddedString(65));
        Util.assertInvalidBoundaries(sit, "reasoningScope");
    }

    /**
     * 
     */
    public void testAdapter() {
        Util.assertAdapterNotification(sit);
    }

    /**
     * 
     */
    public void testUnsupportedFeature() {
        Util.assertUnsupportedFeature(sit);
    }

    /*
     * Protected methods.
     */

    /* (non-Javadoc)
     * @see junit.framework.TestCase#setUp()
     */
    protected void setUp() throws Exception {
        super.setUp();
        EventFactory eFx = EventFactory.eINSTANCE;
        sit = eFx.createAvailableSituation();

    }

}
