package org.eclipse.hyades.logging.events.cbe.impl.tests;

import java.io.IOException;
import java.util.Arrays;
import java.util.List;

import junit.framework.AssertionFailedError;
import junit.framework.TestCase;

import org.eclipse.emf.common.util.EList;
import org.eclipse.hyades.logging.core.Guid;
import org.eclipse.hyades.logging.events.cbe.AssociatedEvent;
import org.eclipse.hyades.logging.events.cbe.AssociationEngine;
import org.eclipse.hyades.logging.events.cbe.AvailableSituation;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.CompletionException;
import org.eclipse.hyades.logging.events.cbe.ComponentIdentification;
import org.eclipse.hyades.logging.events.cbe.ContentHandler;
import org.eclipse.hyades.logging.events.cbe.ContextDataElement;
import org.eclipse.hyades.logging.events.cbe.EventException;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.EventFactoryFactory;
import org.eclipse.hyades.logging.events.cbe.ExtendedDataElement;
import org.eclipse.hyades.logging.events.cbe.MsgCatalogToken;
import org.eclipse.hyades.logging.events.cbe.MsgDataElement;
import org.eclipse.hyades.logging.events.cbe.Situation;
import org.eclipse.hyades.logging.events.cbe.ValidationException;

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: CommonBaseEventImplTest.java,v 1.13 2005/03/09 17:20:56 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * JUnit test case for the <code>CommonBaseEvent</code> class.
 * 
 * @author Scott Brown
 * @author Denilson Nastacio
 * @author Paul E. Slauenwhite
 * @version 1.0.1
 * @since 1.0.1
 */
public class CommonBaseEventImplTest extends TestCase {
    
    /**
     * Target instances for all fixtures 
     */
    private CommonBaseEvent cbe;

    public CommonBaseEventImplTest(String arg0) {
        super(arg0);
    }

    protected void setUp() throws Exception {
        super.setUp();
        cbe = Util.generateEvent();
    }

    protected void tearDown() throws Exception {
        super.tearDown();
    }

    public void testEDEWithMultipleValuesWithSameContents() {

        try{
    		
            final int VALUES_COUNT = 10;
    		
            String values[] = new String[VALUES_COUNT];
            
            for (int counter = 0; counter < VALUES_COUNT; counter++) {
                values[counter] = new String("value");
            }
            
            CommonBaseEvent commonBaseEvent = EventFactory.eINSTANCE.createCommonBaseEvent();
            commonBaseEvent.addExtendedDataElement("Values",values);
          
            assertEquals(commonBaseEvent.getExtendedDataElements().size(),1);
            
            assertEquals(commonBaseEvent.getExtendedDataElements("Values").size(),1);
            
            ExtendedDataElement extendedDataElement = ((ExtendedDataElement)(commonBaseEvent.getExtendedDataElements("Values").get(0)));
             
            assertNotNull(extendedDataElement);
            
            assertEquals(extendedDataElement.getName(),"Values");
            
            assertEquals(extendedDataElement.getTypeAsInt(),ExtendedDataElement.TYPE_STRING_ARRAY_VALUE);
            
            assertEquals(extendedDataElement.getValues().size(),VALUES_COUNT);
            
            for (int counter = 0; counter < VALUES_COUNT; counter++) {                
                assertEquals(extendedDataElement.getValues().get(counter),"value");
            }
            
            assertEquals(extendedDataElement.getValuesAsStringArray().length,VALUES_COUNT);
            
            for (int counter = 0; counter < VALUES_COUNT; counter++) {                
                assertEquals(extendedDataElement.getValuesAsStringArray()[counter],"value");
            }
    	} 
        catch (AssertionFailedError a) {
            throw (a);
        } 
        catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    public void testEDEWithMultipleValuesWithSameObject() {

        try{
    		
            final int VALUES_COUNT = 10;
    		final String VALUE_STRING = "value";
            
    		String values[] = new String[VALUES_COUNT];
            
            for (int counter = 0; counter < VALUES_COUNT; counter++) {
                values[counter] = VALUE_STRING;
            }
            
            CommonBaseEvent commonBaseEvent = EventFactory.eINSTANCE.createCommonBaseEvent();
            commonBaseEvent.addExtendedDataElement("Values",values);
          
            assertEquals(commonBaseEvent.getExtendedDataElements().size(),1);
            
            assertEquals(commonBaseEvent.getExtendedDataElements("Values").size(),1);
            
            ExtendedDataElement extendedDataElement = ((ExtendedDataElement)(commonBaseEvent.getExtendedDataElements("Values").get(0)));
             
            assertNotNull(extendedDataElement);
            
            assertEquals(extendedDataElement.getName(),"Values");
            
            assertEquals(extendedDataElement.getTypeAsInt(),ExtendedDataElement.TYPE_STRING_ARRAY_VALUE);
            
            assertEquals(extendedDataElement.getValues().size(),VALUES_COUNT);
            
            for (int counter = 0; counter < VALUES_COUNT; counter++) {                
                assertEquals(extendedDataElement.getValues().get(counter),"value");
            }

            assertEquals(extendedDataElement.getValuesAsStringArray().length,VALUES_COUNT);
            
            for (int counter = 0; counter < VALUES_COUNT; counter++) {                
                assertEquals(extendedDataElement.getValuesAsStringArray()[counter],"value");
            }
        } 
        catch (AssertionFailedError a) {
            throw (a);
        } 
        catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    /**
     * Test the addExtendedDataElementWithHexValue(String, String) API.
     */
    public void testAddExtendedDataElementWithStringHexValue() {

        try {
           
            final String HEX_VALUE_STRING = "000102030405060708090A0B0C0D0E0F";         
            final byte[] HEX_VALUE_BYTES = new byte[]{0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15};
            final char[] HEX_VALUE_CHARS = HEX_VALUE_STRING.toCharArray();
            final String HEX_VALUE_NAME = "Hex Value Name";
            
            ExtendedDataElement extendedDataElement = cbe.addExtendedDataElementWithHexValue(HEX_VALUE_NAME, HEX_VALUE_STRING);

            assertEquals(extendedDataElement.getHexValue(),HEX_VALUE_STRING);
            assertEquals(extendedDataElement.getName(),HEX_VALUE_NAME);
            assertEquals(extendedDataElement.getTypeAsInt(),ExtendedDataElement.TYPE_HEX_BINARY_VALUE);
            
            assertEquals(extendedDataElement.getHexValueAsByteArray().length,HEX_VALUE_BYTES.length);

            for (int counter = 0; counter < HEX_VALUE_BYTES.length; counter++) {
                assertEquals(extendedDataElement.getHexValueAsByteArray()[counter],HEX_VALUE_BYTES[counter]);
            }            

            assertEquals(extendedDataElement.getHexValueAsCharArray().length,HEX_VALUE_CHARS.length);

            for (int counter = 0; counter < HEX_VALUE_CHARS.length; counter++) {
                assertEquals(extendedDataElement.getHexValueAsCharArray()[counter],HEX_VALUE_CHARS[counter]);
            }
            
            cbe.validate();
        } 
        catch (AssertionFailedError a) {
            throw (a);
        } 
        catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    /**
     * Test the addExtendedDataElementWithHexValue(String, byte[]) API.
     */
    public void testAddExtendedDataElementWithByteArrayHexValue() {

        try {
           
            final String HEX_VALUE_STRING = "000102030405060708090A0B0C0D0E0F";         
            final byte[] HEX_VALUE_BYTES = new byte[]{0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15};
            final char[] HEX_VALUE_CHARS = HEX_VALUE_STRING.toCharArray();
            final String HEX_VALUE_NAME = "Hex Value Name";
            
            ExtendedDataElement extendedDataElement = cbe.addExtendedDataElementWithHexValue(HEX_VALUE_NAME, HEX_VALUE_BYTES);

            assertEquals(extendedDataElement.getHexValue(),HEX_VALUE_STRING);
            assertEquals(extendedDataElement.getName(),HEX_VALUE_NAME);
            assertEquals(extendedDataElement.getTypeAsInt(),ExtendedDataElement.TYPE_HEX_BINARY_VALUE);
            
            assertEquals(extendedDataElement.getHexValueAsByteArray().length,HEX_VALUE_BYTES.length);

            for (int counter = 0; counter < HEX_VALUE_BYTES.length; counter++) {
                assertEquals(extendedDataElement.getHexValueAsByteArray()[counter],HEX_VALUE_BYTES[counter]);
            }            

            assertEquals(extendedDataElement.getHexValueAsCharArray().length,HEX_VALUE_CHARS.length);

            for (int counter = 0; counter < HEX_VALUE_CHARS.length; counter++) {
                assertEquals(extendedDataElement.getHexValueAsCharArray()[counter],HEX_VALUE_CHARS[counter]);
            }
            
            cbe.validate();
        } 
        catch (AssertionFailedError a) {
            throw (a);
        } 
        catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    public void testMultipleAnyElementsWithSameContents() {

        for (int counter = 0; counter < 5; counter++) {
            cbe.addAny(new String("anyElement"));
        }

        assertTrue(cbe.getAny().size() == 5);

        for (int counter = 0; counter < 5; counter++) {
            assertEquals(((String) (cbe.getAny().get(counter))), "anyElement");
        }
    }

    public void testMultipleAnyElementsWithSameObjects() {
       
        String anyElement = new String("anyElement");

        for (int counter = 0; counter < 5; counter++) {
            cbe.addAny(anyElement);
        }

        assertTrue(cbe.getAny().size() == 5);

        for (int counter = 0; counter < 5; counter++) {
            assertEquals(((String) (cbe.getAny().get(counter))), "anyElement");
        }
    }

    public void testMultipleAnyElementsArrayWithSameContents() {
        
        String[] anyElements = new String[5];
        
        for (int counter = 0; counter < 5; counter++) {
            anyElements[counter] = new String("anyElement");
        }

        cbe.getAny().addAll(Arrays.asList(anyElements));
        
        assertTrue(cbe.getAny().size() == 5);

        for (int counter = 0; counter < 5; counter++) {
            assertEquals(((String) (cbe.getAny().get(counter))), "anyElement");
        }
     }

    public void testMultipleAssociatedEventsWithSameContents() {

        AssociatedEvent associatedEvent = null;

        for (int counter = 0; counter < 5; counter++) {

            associatedEvent = EventFactory.eINSTANCE.createAssociatedEvent();
            associatedEvent.setAssociationEngine("associationEngine");
            associatedEvent.setResolvedEvents("resolvedEvents");

            cbe.addAssociatedEvent(associatedEvent);
        }

        assertTrue(cbe.getAssociatedEvents().size() == 5);

        for (int counter = 0; counter < 5; counter++) {
            assertEquals(((AssociatedEvent) (cbe.getAssociatedEvents().get(counter))).getAssociationEngine(), "associationEngine");
            assertEquals(((AssociatedEvent) (cbe.getAssociatedEvents().get(counter))).getResolvedEvents(), "resolvedEvents");
        }
    }

    public void testMultipleAssociatedEventsWithSameObjects() {
       
        AssociatedEvent associatedEvent = EventFactory.eINSTANCE.createAssociatedEvent();
        associatedEvent.setAssociationEngine("associationEngine");
        associatedEvent.setResolvedEvents("resolvedEvents");

        for (int counter = 0; counter < 5; counter++) {
            cbe.addAssociatedEvent(associatedEvent);
        }

        //NOTE:  The resultant EList only contains one element since the same 
        //object instance cannot be duplicated in the same EList more than once since the 
        //lifetime of the contained object is controlled by the container object.
        assertTrue(cbe.getAssociatedEvents().size() == 1);

        assertEquals(((AssociatedEvent) (cbe.getAssociatedEvents().get(0))).getAssociationEngine(), "associationEngine");
        assertEquals(((AssociatedEvent) (cbe.getAssociatedEvents().get(0))).getResolvedEvents(), "resolvedEvents");
    }

    public void testMultipleAssociatedEventsArrayWithSameContents() {
       
        AssociatedEvent[] associatedEvents = new AssociatedEvent[5];
        
        for (int counter = 0; counter < 5; counter++) {

            associatedEvents[counter] = EventFactory.eINSTANCE.createAssociatedEvent();
            associatedEvents[counter].setAssociationEngine("associationEngine");
            associatedEvents[counter].setResolvedEvents("resolvedEvents");
        }
        
        cbe.getAssociatedEvents().addAll(Arrays.asList(associatedEvents));

        assertTrue(cbe.getAssociatedEvents().size() == 5);

        for (int counter = 0; counter < 5; counter++) {
            assertEquals(((AssociatedEvent) (cbe.getAssociatedEvents().get(counter))).getAssociationEngine(), "associationEngine");
            assertEquals(((AssociatedEvent) (cbe.getAssociatedEvents().get(counter))).getResolvedEvents(), "resolvedEvents");
        }
    }
    
    public void testMultipleContextDataElementsWithSameContents() {

        ContextDataElement contextDataElement = null;

        for (int counter = 0; counter < 5; counter++) {

            contextDataElement = EventFactory.eINSTANCE.createContextDataElement();
            contextDataElement.setContextId("contextId");
            contextDataElement.setContextValue("contextValue");
            contextDataElement.setName("name");
            contextDataElement.setType("type");

            cbe.addContextDataElement(contextDataElement);
        }

        assertTrue(cbe.getContextDataElements().size() == 5);

        for (int counter = 0; counter < 5; counter++) {
            assertEquals(((ContextDataElement) (cbe.getContextDataElements().get(counter))).getContextId(), "contextId");
            assertEquals(((ContextDataElement) (cbe.getContextDataElements().get(counter))).getContextValue(), "contextValue");
            assertEquals(((ContextDataElement) (cbe.getContextDataElements().get(counter))).getName(), "name");
            assertEquals(((ContextDataElement) (cbe.getContextDataElements().get(counter))).getType(), "type");
        }
    }

    public void testMultipleContextDataElementsWithSameObjects() {
       
        ContextDataElement contextDataElement = EventFactory.eINSTANCE.createContextDataElement();
        contextDataElement.setContextId("contextId");
        contextDataElement.setContextValue("contextValue");
        contextDataElement.setName("name");
        contextDataElement.setType("type");

        for (int counter = 0; counter < 5; counter++) {
            cbe.addContextDataElement(contextDataElement);
        }

        //NOTE:  The resultant EList only contains one element since the same 
        //object instance cannot be duplicated in the same EList more than once since the 
        //lifetime of the contained object is controlled by the container object.
        assertTrue(cbe.getContextDataElements().size() == 1);

        assertEquals(((ContextDataElement) (cbe.getContextDataElements().get(0))).getContextId(), "contextId");
        assertEquals(((ContextDataElement) (cbe.getContextDataElements().get(0))).getContextValue(), "contextValue");
        assertEquals(((ContextDataElement) (cbe.getContextDataElements().get(0))).getName(), "name");
        assertEquals(((ContextDataElement) (cbe.getContextDataElements().get(0))).getType(), "type");
    }

    public void testMultipleContextDataElementsArrayWithSameContents() {
       
        ContextDataElement[] contextDataElements = new ContextDataElement[5];
        
        for (int counter = 0; counter < 5; counter++) {

            contextDataElements[counter] = EventFactory.eINSTANCE.createContextDataElement();
            contextDataElements[counter].setContextId("contextId");
            contextDataElements[counter].setContextValue("contextValue");
            contextDataElements[counter].setName("name");
            contextDataElements[counter].setType("type");
        }
        
        cbe.getContextDataElements().addAll(Arrays.asList(contextDataElements));

        assertTrue(cbe.getContextDataElements().size() == 5);

        for (int counter = 0; counter < 5; counter++) {
            assertEquals(((ContextDataElement) (cbe.getContextDataElements().get(counter))).getContextId(), "contextId");
            assertEquals(((ContextDataElement) (cbe.getContextDataElements().get(counter))).getContextValue(), "contextValue");
            assertEquals(((ContextDataElement) (cbe.getContextDataElements().get(counter))).getName(), "name");
            assertEquals(((ContextDataElement) (cbe.getContextDataElements().get(counter))).getType(), "type");
        }
    }

    public void testMultipleExtendedDataElementsWithSameContents() {

        ExtendedDataElement extendedDataElement = null;

        for (int counter = 0; counter < 5; counter++) {

            extendedDataElement = EventFactory.eINSTANCE.createExtendedDataElement();
            extendedDataElement.setName("name");
            extendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_STRING_VALUE);
            extendedDataElement.setValues(new String[] {"values"});

            cbe.addExtendedDataElement(extendedDataElement);
        }

        assertTrue(cbe.getExtendedDataElements().size() == 5);

        for (int counter = 0; counter < 5; counter++) {
            assertEquals(((ExtendedDataElement) (cbe.getExtendedDataElements().get(counter))).getName(), "name");
            assertEquals(((ExtendedDataElement) (cbe.getExtendedDataElements().get(counter))).getTypeAsInt(), ExtendedDataElement.TYPE_STRING_VALUE);
            assertEquals(((String)(((ExtendedDataElement) (cbe.getExtendedDataElements().get(counter))).getValues().get(0))), "values");
        }
    }

    public void testMultipleExtendedDataElementsWithSameObjects() {
       
        ExtendedDataElement extendedDataElement = EventFactory.eINSTANCE.createExtendedDataElement();
        extendedDataElement.setName("name");
        extendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_STRING_VALUE);
        extendedDataElement.setValues(new String[] {"values"});

        for (int counter = 0; counter < 5; counter++) {
            cbe.addExtendedDataElement(extendedDataElement);
        }

        //NOTE:  The resultant EList only contains one element since the same 
        //object instance cannot be duplicated in the same EList more than once since the 
        //lifetime of the contained object is controlled by the container object.
        assertTrue(cbe.getExtendedDataElements().size() == 1);

        assertEquals(((ExtendedDataElement) (cbe.getExtendedDataElements().get(0))).getName(), "name");
        assertEquals(((ExtendedDataElement) (cbe.getExtendedDataElements().get(0))).getTypeAsInt(), ExtendedDataElement.TYPE_STRING_VALUE);
        assertEquals(((String)(((ExtendedDataElement) (cbe.getExtendedDataElements().get(0))).getValues().get(0))), "values");
    }

    public void testMultipleExtendedDataElementsArrayWithSameContents() {
       
        ExtendedDataElement[] extendedDataElements = new ExtendedDataElement[5];
        
        for (int counter = 0; counter < 5; counter++) {

            extendedDataElements[counter] = EventFactory.eINSTANCE.createExtendedDataElement();
            extendedDataElements[counter].setName("name");
            extendedDataElements[counter].setTypeAsInt(ExtendedDataElement.TYPE_STRING_VALUE);
            extendedDataElements[counter].setValues(new String[] {"values"});
        }
        
        cbe.getExtendedDataElements().addAll(Arrays.asList(extendedDataElements));

        assertTrue(cbe.getExtendedDataElements().size() == 5);

        for (int counter = 0; counter < 5; counter++) {
            assertEquals(((ExtendedDataElement) (cbe.getExtendedDataElements().get(counter))).getName(), "name");
            assertEquals(((ExtendedDataElement) (cbe.getExtendedDataElements().get(counter))).getTypeAsInt(), ExtendedDataElement.TYPE_STRING_VALUE);
            assertEquals(((String)(((ExtendedDataElement) (cbe.getExtendedDataElements().get(counter))).getValues().get(0))), "values");
        }
    }

    /**
     * Tests whether the <code>CommonBaseEvent</code> has been properly generated. 
     */
    public void testCommonBaseEventTypes() throws Exception {
        cbe.setVersion("ABC");
        cbe.setGlobalInstanceId("Id01234567890123");
        cbe.setExtensionName("CBEEvent");
        cbe.setLocalInstanceId("MyId");
        cbe.setCreationTime("1999-05-31T13:20:00-05:00");
        cbe.setCreationTimeAsLong(987654321);
        cbe.setSeverity((short) 5);
        cbe.setMsg("ABC");
        cbe.setPriority((short) 5);
        cbe.setSequenceNumber(5);
        cbe.setRepeatCount((short) 6);
        cbe.setRepeatCount((short) 6);
        cbe.setElapsedTime((short) 60);

        ContextDataElement cde = EventFactory.eINSTANCE.createContextDataElement();
        cbe.getContextDataElements().add(cde);
        ExtendedDataElement ede = EventFactory.eINSTANCE.createExtendedDataElement();
        cbe.getExtendedDataElements().add(ede);
        AssociatedEvent ae = EventFactory.eINSTANCE.createAssociatedEvent();
        cbe.getAssociatedEvents().add(ae);
        ComponentIdentification cit1 = EventFactory.eINSTANCE.createComponentIdentification();
        cbe.setReporterComponentId(cit1);
        ComponentIdentification cit2 = EventFactory.eINSTANCE.createComponentIdentification();
        cbe.setSourceComponentId(cit2);
        MsgDataElement mde = EventFactory.eINSTANCE.createMsgDataElement();
        cbe.setMsgDataElement(mde);

        assertTrue(cbe.getVersion() instanceof String);
        assertTrue(cbe.getGlobalInstanceId() instanceof String);
        assertTrue(cbe.getExtensionName() instanceof String);
        assertTrue(cbe.getLocalInstanceId() instanceof String);
        assertTrue(cbe.getCreationTime() instanceof String);
        assertTrue(cbe.getCreationTimeAsLong() == 987654321);
        assertTrue(cbe.getSeverity() == 5);
        assertTrue(cbe.getMsg() instanceof String);
        assertTrue(cbe.getPriority() == 5);
        assertTrue(cbe.getSequenceNumber() == 5);
        assertTrue(cbe.getRepeatCount() == 6);
        assertTrue(cbe.getElapsedTime() == 60);
        assertTrue(cbe.getContextDataElements().get(0) instanceof ContextDataElement);
        assertTrue(cbe.getExtendedDataElements().get(0) instanceof ExtendedDataElement);
        assertTrue(cbe.getAssociatedEvents().get(0) instanceof AssociatedEvent);
        assertTrue(cbe.getReporterComponentId() instanceof ComponentIdentification);
        assertTrue(cbe.getSourceComponentId() instanceof ComponentIdentification);
        assertTrue(cbe.getMsgDataElement() instanceof MsgDataElement);

        Util.testSerialization(cbe);
    }

    /**
     * Tests whether the <code>CommonBaseEvent</code> checks for required properties. 
     */
    public void testValidationRequiredPrimitives() throws ValidationException {
        // First, try without a GlobalInstanceId
        CommonBaseEvent result = Util.generateEventWithoutRequiredPrimitives();
        result.setCreationTime("1999-05-31T13:20:00-05:00");

        result.validate();

        // This time don't set creationTime
        result = Util.generateEventWithoutRequiredPrimitives();
        result.setGlobalInstanceId("CE123456789012345678901234567890");
        Util.assertMissingAttributes(result, "creationTime");

        // now let's make sure a minimum required event is valid
        result = Util.generateEventWithoutRequiredPrimitives();
        result.setCreationTime("1999-05-31T13:20:00-05:00");
        result.setGlobalInstanceId("CE123456789012345678901234567890");
        result.validate();

        // need to test that elapsedTime is present if repeatCount is non-zero
        result = Util.generateEventWithoutRequiredPrimitives();
        result.setCreationTime("1999-05-31T13:20:00-05:00");
        result.setGlobalInstanceId("CE123456789012345678901234567890");
        result.setRepeatCount((short) 1);
        Util.assertMissingAttributes(result, "elapsedTime");

        result.setRepeatCount((short) 0);
        result.validate(); // elapsedTime is not required because repeatCount is 0

        result.setRepeatCount((short) 33);
        result.setElapsedTime((short) 5000);
        result.validate();

    }

    public void testValidationRequiredSourceComponentId() throws ValidationException {
        CommonBaseEvent result = EventFactory.eINSTANCE.createCommonBaseEvent();

        result.setCreationTime("1999-05-31T13:20:00-05:00");
        result.setGlobalInstanceId("CE123456789012345678901234567890");

        Situation sit = EventFactory.eINSTANCE.createSituation();
        AvailableSituation sitType = EventFactory.eINSTANCE.createAvailableSituation();
        sitType.setAvailabilityDisposition("NOT AVAILABLE");
        sitType.setOperationDisposition("STARTABLE");
        sitType.setProcessingDisposition("MGMTTASK_PROCESS");
        sitType.setReasoningScope("EXTERNAL");
        sit.setCategoryName("AvailableSituation");
        sit.setSituationType(sitType);
        result.setSituation(sit);

        Util.assertMissingAttributes(result, "sourceComponentId");

        ComponentIdentification cit2 = EventFactory.eINSTANCE.createComponentIdentification();
        cit2.setLocation("My Location 2");
        cit2.setLocationType("FQHostname");
        cit2.setComponent("My Component 2");
        cit2.setSubComponent("My sub component 2");
        cit2.setComponentIdType("My Component ID Type 2");
        cit2.setComponentType("My Component Type 2");
        result.setSourceComponentId(cit2);

        result.validate();
    }

    public void testValidationRequiredReporterComponentId() throws ValidationException {
        CommonBaseEvent result = EventFactory.eINSTANCE.createCommonBaseEvent();

        result.setCreationTime("1999-05-31T13:20:00-05:00");
        result.setGlobalInstanceId("CE123456789012345678901234567890");

        Situation sit = EventFactory.eINSTANCE.createSituation();
        AvailableSituation sitType = EventFactory.eINSTANCE.createAvailableSituation();
        sitType.setAvailabilityDisposition("NOT AVAILABLE");
        sitType.setOperationDisposition("STARTABLE");
        sitType.setProcessingDisposition("MGMTTASK_PROCESS");
        sitType.setReasoningScope("EXTERNAL");
        sit.setCategoryName("AvailableSituation");
        sit.setSituationType(sitType);
        result.setSituation(sit);

        ComponentIdentification cit1 = EventFactory.eINSTANCE.createComponentIdentification();
        cit1.setLocation("My Location");
        cit1.setLocationType("Devicename");
        cit1.setComponent("My Component");
        cit1.setSubComponent("My sub component");
        cit1.setComponentIdType("My Component ID Type");
        cit1.setComponentType("My Component Type");
        result.setReporterComponentId(cit1);

        try {
            result.validate();
            fail("Can't have ReporterComponentId if you don't have SourceComponentId");
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0203E", ve);
        }

        ComponentIdentification cit2 = EventFactory.eINSTANCE.createComponentIdentification();
        cit2.setLocation("My Location 2");
        cit2.setLocationType("Unknown");
        cit2.setComponent("My Component 2");
        cit2.setSubComponent("My sub component 2");
        cit2.setComponentIdType("My Component ID Type 2");
        cit2.setComponentType("My Component Type 2");
        result.setSourceComponentId(cit2);

        result.validate();
    }

    public void testValidationRequiredSituation() throws ValidationException {
        CommonBaseEvent result = EventFactory.eINSTANCE.createCommonBaseEvent();

        result.setCreationTime("1999-05-31T13:20:00-05:00");
        result.setGlobalInstanceId("CE123456789012345678901234567890");

        ComponentIdentification cit1 = EventFactory.eINSTANCE.createComponentIdentification();
        cit1.setLocation("My Location");
        cit1.setLocationType("OID-OSI");
        cit1.setComponent("My Component");
        cit1.setSubComponent("My sub component");
        cit1.setComponentIdType("My Component ID Type");
        cit1.setComponentType("My Component Type");
        result.setReporterComponentId(cit1);

        ComponentIdentification cit2 = EventFactory.eINSTANCE.createComponentIdentification();
        cit2.setLocation("My Location 2");
        cit2.setLocationType("Dial");
        cit2.setComponent("My Component 2");
        cit2.setSubComponent("My sub component 2");
        cit2.setComponentIdType("My Component ID Type 2");
        cit2.setComponentType("My Component Type 2");
        result.setSourceComponentId(cit2);

        Util.assertMissingAttributes(result, "situation");

        Situation sit = EventFactory.eINSTANCE.createSituation();
        AvailableSituation sitType = EventFactory.eINSTANCE.createAvailableSituation();
        sitType.setAvailabilityDisposition("NOT AVAILABLE");
        sitType.setOperationDisposition("STARTABLE");
        sitType.setProcessingDisposition("MGMTTASK_PROCESS");
        sitType.setReasoningScope("EXTERNAL");
        sit.setCategoryName("AvailableSituation");
        sit.setSituationType(sitType);
        result.setSituation(sit);

        result.validate();
    }

    /**
     * Tests the <code>equals</code> method. 
     */
    public void testEquals() {
        CommonBaseEvent cbe2 = null;

        cbe = Util.generateEvent();
        cbe.setVersion("1.0.1");
        cbe.setLocalInstanceId("local instance id");
        cbe.setSeverity((short) 0);
        cbe.setPriority((short) 10);
        cbe.setMsg("my msg");
        cbe.setRepeatCount((short) 3);
        cbe.setElapsedTime(100);
        cbe.setExtensionName("extension name");
        cbe.setSequenceNumber(987654321);

        //cannot get Util.clone() to work: CommonBaseEvent cbe2 = (CommonBaseEvent)Util.clone(cbe);
        cbe2 = Util.generateEvent();
        cbe2.setVersion("1.0.1");
        cbe2.setLocalInstanceId("local instance id");
        cbe2.setSeverity((short) 0);
        cbe2.setPriority((short) 10);
        cbe2.setMsg("my msg");
        cbe2.setRepeatCount((short) 3);
        cbe2.setElapsedTime(100);
        cbe2.setExtensionName("extension name");
        cbe2.setSequenceNumber(987654321);
        assertEquals(cbe, cbe2);
        cbe2.setSequenceNumber(123456789);
        assertFalse(cbe.equals(cbe2));

        cbe2.setSequenceNumber(987654321);
        assertEquals(cbe, cbe2);
        cbe2.addExtendedDataElement("my ede", "value");
        assertFalse(cbe.equals(cbe2));

        cbe.addExtendedDataElement("my ede", "value");
        assertEquals(cbe, cbe2);

        cbe.addExtendedDataElement("my ede", "value");
        cbe2.addExtendedDataElement("my ede", "value2");
        assertFalse(cbe.equals(cbe2));
    }

    public void testEquals1() {
        cbe = Util.generateEvent();
        cbe.setVersion("1.0.1");
        cbe.setLocalInstanceId("local instance id");
        cbe.setSeverity((short) 0);
        cbe.setPriority((short) 10);
        cbe.setMsg("my msg");
        cbe.setRepeatCount((short) 3);
        cbe.setElapsedTime(100);
        cbe.setExtensionName("extension name");
        cbe.setSequenceNumber(987654321);

        //cannot get Util.clone() to work: CommonBaseEvent cbe2 = (CommonBaseEvent)Util.clone(cbe);
        CommonBaseEvent cbe2 = Util.generateEvent();
        cbe2.setVersion("1.0.1");
        cbe2.setLocalInstanceId("local instance id");
        cbe2.setSeverity((short) 0);
        cbe2.setPriority((short) 10);
        cbe2.setMsg("my msg");
        cbe2.setRepeatCount((short) 3);
        cbe2.setElapsedTime(100);
        cbe2.setExtensionName("extension name");
        cbe2.setSequenceNumber(987654321);
        assertEquals(cbe, cbe2);

        cbe2.addExtendedDataElement("my ede", "value");
        assertFalse(cbe.equals(cbe2));
    }

    public void testSetCreationTime() throws ValidationException {
        cbe.setCreationTime("19999-05-31T13:20:00-05:00");
        try {
            cbe.validate();
            fail("year is too long to be accepted");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0200E", exc);
        }

        cbe.setCreationTime("1999-05-31T13:20:00.123Q");
        try {
            cbe.validate();
            fail("invalid UTC");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0200E", exc);
        }

        cbe.setCreationTime("1999-05-31T13:20:00.123456789012345678Z");
        cbe.validate();

        cbe.setCreationTime("1999-5-138T13:20:00-05:00");
        try {
            cbe.validate();
            fail("may does not have that many days");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0200E", exc);
        }

        cbe.setCreationTime("1999-ss-31T13:35:00-05:00");
        try {
            cbe.validate();
            fail("what kind of a month is ss");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0200E", exc);
        }

        cbe.setCreationTime("1999-05-31T27:20:09-05:00");
        try {
            cbe.validate();
            fail("27 o'clock ???");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0200E", exc);
        }

        cbe.setCreationTime("1999999-05-31T13:20:00-05:00");
        try {
            cbe.validate();
            fail("year is too long to be accepted");
        }
        catch (ValidationException exc) {
        }

        //cbe.setCreationTime("1999-05-31T13:20:00lmn");
        try {
            cbe.validate();
            //fail("bad UTC");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0200E", exc);
        }

        cbe.setCreationTime("7");
        try {
            cbe.validate();
            fail("7 ???");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0200E", exc);
        }

        cbe.setCreationTime("ABC");
        try {
            cbe.validate();
            fail("ABC ???");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0200E", exc);
        }

        cbe.setCreationTime("1999-05-31T21:20:09-05:00");
        cbe.validate();

        cbe.setCreationTime("1999,05,31T21:20:09-05:00");
        try {
            cbe.validate();
            fail("commas ???");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0200E", exc);
        }

        cbe.setCreationTime("1999 5 31T21:20:09-05:00");
        try {
            cbe.validate();
            fail("spaces ???");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0200E", exc);
        }

        cbe.setCreationTime("1999*05*31T21:20:09*05:00");
        try {
            cbe.validate();
            fail("no dashes ???");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0200E", exc);
        }

        cbe.setCreationTime("i0000000000000000000000000000000Z");
        try {
            cbe.validate();
            fail("looks like a global id ???");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0200E", exc);
        }

        cbe.setCreationTime("now is the time for all good men to come to the aid of their country");
        try {
            cbe.validate();
            fail("sentence ???");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0200E", exc);
        }

        cbe.setCreationTime("19Q9-05-31T21:20:09-05:00");
        try {
            cbe.validate();
            fail("Q ???");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0200E", exc);
        }

        cbe.setCreationTime("1999-0R-31T21:20:09-05:00");
        try {
            cbe.validate();
            fail("R ???");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0200E", exc);
        }

        cbe.setCreationTime("1999-05-S1T21:20:09-05:00");
        try {
            cbe.validate();
            fail("S ???");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0200E", exc);
        }

        cbe.setCreationTime("1999-05-31T2T:20:09-05:00");
        try {
            cbe.validate();
            fail("T ???");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0200E", exc);
        }

        cbe.setCreationTime("1999-05-31T21:2U:09-05:00");
        try {
            cbe.validate();
            fail("U ???");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0200E", exc);
        }

        cbe.setCreationTime("1999-05-31T21:20:V9-05:00");
        try {
            cbe.validate();
            fail("V ???");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0200E", exc);
        }

        cbe.setCreationTime("1999-05-31T21:20:09-0W:00");
        try {
            cbe.validate();
            fail("W ???");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0200E", exc);
        }

        cbe.setCreationTime("1999-05-31T21:20:09-05:0X");
        try {
            cbe.validate();
            fail("X ???");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0200E", exc);
        }

        cbe.setCreationTime("1999-05-31T21:20:09-05:00.YZ");
        try {
            cbe.validate();
            fail("YZ ???");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0200E", exc);
        }

        cbe.setCreationTime("1999-05-31T21:20:09-");
        try {
            cbe.validate();
            fail("can't end in minus");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0200E", exc);
        }

        cbe.setCreationTime("1999-05-31T21:20:09+");
        try {
            cbe.validate();
            fail("can't end in plus");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0200E", exc);
        }

        cbe.setCreationTime("Z1999-05-31T21:20:09-05:00");
        try {
            cbe.validate();
            fail("can't start with Z");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0200E", exc);
        }

        cbe.setCreationTime("-1999-05-31T21:20:09-05:00");
        try {
            cbe.validate();
            fail("can't start with -");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0200E", exc);
        }

        //cbe.setCreationTime("1999-05-31T21:20:09*05:00");
        //try
        //{
        //	cbe.validate();
        //	fail("offset can't start with *");
        //}
        //catch (ValidationException exc)
        //{
        //	Util.assertExceptionId("IWAT0200E", exc);
        //}

        cbe.setCreationTime("1968-05-31T21:20:09-05:00");
        try {
            cbe.validate();
            fail("year 1968 is too old");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0200E", exc);
        }

        cbe.setCreationTime("1999-05-31T21:20:09.123");
        cbe.validate();
        cbe.setCreationTime("1999-05-31T21:20:09.xyz");
        try {
            cbe.validate();
            fail("xyz is not valid subseconds");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0200E", exc);
        }

        cbe.setCreationTime("1999-05-31T21:20:09.5");
        cbe.validate();
        long long1 = cbe.getCreationTimeAsLong();
        cbe.setCreationTime("1999-05-31T21:20:09.50");
        cbe.validate();
        long long2 = cbe.getCreationTimeAsLong();
        assertEquals(long1, long2);
        cbe.setCreationTime("1999-05-31T21:20:09.500");
        cbe.validate();
        long2 = cbe.getCreationTimeAsLong();
        assertEquals(long1, long2);
        cbe.setCreationTime("1999-05-31T21:20:09.5000");
        cbe.validate();
        long2 = cbe.getCreationTimeAsLong();
        assertEquals(long1, long2);

        //cbe.setCreationTime("1999-05-31T21:20:09-13:00");
        //try
        //{
        //	cbe.validate();
        //	fail("offset too big ???");
        //}
        //catch (ValidationException exc)
        //{
        //	Util.assertExceptionId("IWAT0200E", exc);
        //}
    }

    public void testGetCreationTime() {
        cbe = EventFactory.eINSTANCE.createCommonBaseEvent();

        String instr = "2003-11-14T11:42:23.123-05:00";
        String outstr;

        outstr = cbe.getCreationTime();
        assertNull(outstr);
        cbe.setCreationTime(instr);
        outstr = cbe.getCreationTime();
        assertEquals(instr, outstr);
    }

    public void testGetGlobalInstanceId() {
        cbe = EventFactory.eINSTANCE.createCommonBaseEvent();

        String instr = "aaa";
        String outstr;

        outstr = cbe.getGlobalInstanceId();
        assertNull(outstr);
        cbe.setGlobalInstanceId(instr);
        outstr = cbe.getGlobalInstanceId();
        assertEquals(instr, outstr);
    }

    public void testSetGlobalInstanceId() {
        String instr = "a b";
        String outstr;

        cbe.setGlobalInstanceId(instr);
        outstr = cbe.getGlobalInstanceId();
        assertEquals(instr, outstr);

        cbe.setGlobalInstanceId(null);
        outstr = cbe.getGlobalInstanceId();
        assertNull(outstr);
    }

    public void testGetElapsedTime() {
        cbe = EventFactory.eINSTANCE.createCommonBaseEvent();

        long indata = 12345678;
        long outdata;

        outdata = cbe.getElapsedTime();
        assertEquals(0, outdata);
        cbe.setElapsedTime(indata);
        outdata = cbe.getElapsedTime();
        assertEquals(indata, outdata);
    }

    public void testSetElapsedTime() {
        long indata = 87654321;
        long outdata;

        cbe.setElapsedTime(indata);
        outdata = cbe.getElapsedTime();
        assertEquals(indata, outdata);
    }

    public void testUnsetElapsedTime() {
        long indata = 99999999;

        assertFalse(cbe.isSetElapsedTime());

        cbe.setElapsedTime(indata);
        assertTrue(cbe.isSetElapsedTime());

        cbe.unsetElapsedTime();
        assertFalse(cbe.isSetElapsedTime());
    }

    public void testIsSetElapsedTime() {
        long indata = 1;

        assertFalse(cbe.isSetElapsedTime());

        cbe.setElapsedTime(indata);
        assertTrue(cbe.isSetElapsedTime());

        cbe.unsetElapsedTime();
        assertFalse(cbe.isSetElapsedTime());
    }

    public void testIsSetCreationTime() {
        cbe = EventFactory.eINSTANCE.createCommonBaseEvent();
        String time = "abc";

        assertFalse(cbe.isSetCreationTime());

        cbe.setCreationTime(time);
        assertTrue(cbe.isSetCreationTime());

        cbe.unsetCreationTime();
        assertFalse(cbe.isSetCreationTime());
    }

    public void testGetExtensionName() {
        cbe = EventFactory.eINSTANCE.createCommonBaseEvent();

        String instr = "aaa";
        String outstr;

        outstr = cbe.getExtensionName();
        assertNull(outstr);
        cbe.setExtensionName(instr);
        outstr = cbe.getExtensionName();
        assertEquals(instr, outstr);
    }

    public void testSetExtensionName() {
        String instr = "a b";
        String outstr;

        cbe.setExtensionName(instr);
        outstr = cbe.getExtensionName();
        assertEquals(instr, outstr);

        cbe.setExtensionName(null);
        outstr = cbe.getExtensionName();
        assertNull(outstr);
    }

    public void testGetLocalInstanceId() {
        cbe = EventFactory.eINSTANCE.createCommonBaseEvent();

        String instr = "aaa";
        String outstr;

        outstr = cbe.getLocalInstanceId();
        assertNull(outstr);
        cbe.setLocalInstanceId(instr);
        outstr = cbe.getLocalInstanceId();
        assertEquals(instr, outstr);
    }

    public void testSetLocalInstanceId() {
        String instr = "a b";
        String outstr;

        cbe.setLocalInstanceId(instr);
        outstr = cbe.getLocalInstanceId();
        assertEquals(instr, outstr);

        cbe.setLocalInstanceId(null);
        outstr = cbe.getLocalInstanceId();
        assertNull(outstr);
    }

    public void testGetMsg() {
        cbe = EventFactory.eINSTANCE.createCommonBaseEvent();

        String instr = "aaa";
        String outstr;

        outstr = cbe.getMsg();
        assertNull(outstr);
        cbe.setMsg(instr);
        outstr = cbe.getMsg();
        assertEquals(instr, outstr);
    }

    public void testSetMsg() {
        String instr = "a b";
        String outstr;

        cbe.setMsg(instr);
        outstr = cbe.getMsg();
        assertEquals(instr, outstr);

        cbe.setMsg(null);
        outstr = cbe.getMsg();
        assertNull(outstr);
    }

    public void testGetPriority() {
        cbe = EventFactory.eINSTANCE.createCommonBaseEvent();

        short indata = 1;
        short outdata;

        outdata = cbe.getPriority();
        assertEquals(0, outdata);
        cbe.setPriority(indata);
        outdata = cbe.getPriority();
        assertEquals(indata, outdata);
    }

    public void testSetPriority() {
        short indata = 8765;
        short outdata;

        cbe.setPriority(indata);
        outdata = cbe.getPriority();
        assertEquals(indata, outdata);
    }

    public void testUnsetPriority() {
        short indata = 9999;

        assertFalse(cbe.isSetPriority());

        cbe.setPriority(indata);
        assertTrue(cbe.isSetPriority());

        cbe.unsetPriority();
        assertFalse(cbe.isSetPriority());
    }

    public void testIsSetPriority() {
        short indata = 1;

        assertFalse(cbe.isSetPriority());

        cbe.setPriority(indata);
        assertTrue(cbe.isSetPriority());

        cbe.unsetPriority();
        assertFalse(cbe.isSetPriority());
    }

    public void testGetRepeatCount() {
        cbe = EventFactory.eINSTANCE.createCommonBaseEvent();

        short indata = 1;
        short outdata;

        outdata = cbe.getRepeatCount();
        assertEquals(0, outdata);
        cbe.setRepeatCount(indata);
        outdata = cbe.getRepeatCount();
        assertEquals(indata, outdata);
    }

    public void testSetRepeatCount() {
        short indata = 8765;
        short outdata;

        cbe.setRepeatCount(indata);
        outdata = cbe.getRepeatCount();
        assertEquals(indata, outdata);
    }

    public void testUnsetRepeatCount() {
        short indata = 9999;

        assertFalse(cbe.isSetRepeatCount());

        cbe.setRepeatCount(indata);
        assertTrue(cbe.isSetRepeatCount());

        cbe.unsetRepeatCount();
        assertFalse(cbe.isSetRepeatCount());
    }

    public void testIsSetRepeatCount() {
        short indata = 1;

        assertFalse(cbe.isSetRepeatCount());

        cbe.setRepeatCount(indata);
        assertTrue(cbe.isSetRepeatCount());

        cbe.unsetRepeatCount();
        assertFalse(cbe.isSetRepeatCount());
    }

    public void testGetSequenceNumber() {
        cbe = EventFactory.eINSTANCE.createCommonBaseEvent();

        long indata = 12345678;
        long outdata;

        outdata = cbe.getSequenceNumber();
        assertEquals(0, outdata);
        cbe.setSequenceNumber(indata);
        outdata = cbe.getSequenceNumber();
        assertEquals(indata, outdata);
    }

    public void testSetSequenceNumber() {
        long indata = 87654321;
        long outdata;

        cbe.setSequenceNumber(indata);
        outdata = cbe.getSequenceNumber();
        assertEquals(indata, outdata);
    }

    public void testUnsetSequenceNumber() {
        long indata = 99999999;

        assertFalse(cbe.isSetSequenceNumber());

        cbe.setSequenceNumber(indata);
        assertTrue(cbe.isSetSequenceNumber());

        cbe.unsetSequenceNumber();
        assertFalse(cbe.isSetSequenceNumber());
    }

    public void testIsSetSequenceNumber() {
        long indata = 1;

        assertFalse(cbe.isSetSequenceNumber());

        cbe.setSequenceNumber(indata);
        assertTrue(cbe.isSetSequenceNumber());

        cbe.unsetSequenceNumber();
        assertFalse(cbe.isSetSequenceNumber());
    }

    public void testGetSeverity() {
        cbe = EventFactory.eINSTANCE.createCommonBaseEvent();

        short indata = 1;
        short outdata;

        outdata = cbe.getSeverity();
        assertEquals(0, outdata);
        cbe.setSeverity(indata);
        outdata = cbe.getSeverity();
        assertEquals(indata, outdata);
    }

    public void testSetSeverity() {
        short indata = 8765;
        short outdata;

        cbe.setSeverity(indata);
        outdata = cbe.getSeverity();
        assertEquals(indata, outdata);
    }

    public void testUnsetSeverity() {
        short indata = 9999;

        assertFalse(cbe.isSetSeverity());

        cbe.setSeverity(indata);
        assertTrue(cbe.isSetSeverity());

        cbe.unsetSeverity();
        assertFalse(cbe.isSetSeverity());
    }

    public void testIsSetSeverity() {
        short indata = 1;

        assertFalse(cbe.isSetSeverity());

        cbe.setSeverity(indata);
        assertTrue(cbe.isSetSeverity());

        cbe.unsetSeverity();
        assertFalse(cbe.isSetSeverity());
    }

    public void testGetVersion() {
        cbe = EventFactory.eINSTANCE.createCommonBaseEvent();

        String instr = "1.2.3.4";
        String outstr;

        outstr = cbe.getVersion();
        assertEquals("1.0.1", outstr);
        cbe.setVersion(instr);
        outstr = cbe.getVersion();
        assertEquals(instr, outstr);
    }

    public void testSetVersion() {
        String instr = "1.0.1";
        String outstr;

        cbe.setVersion(instr);
        outstr = cbe.getVersion();
        assertEquals(instr, outstr);

        cbe.setVersion(null);
        outstr = cbe.getVersion();
        assertNull(outstr);
    }

    public void testGetSituation() throws ValidationException {
        CommonBaseEvent gsCBE = EventFactory.eINSTANCE.createCommonBaseEvent();
        assertEquals(null, gsCBE.getSituation());

        Situation sit1 = EventFactory.eINSTANCE.createSituation();

        cbe.setSituation(sit1);
        assertEquals(sit1, cbe.getSituation());

        Situation sit2 = EventFactory.eINSTANCE.createSituation();
        sit2.setCategoryName("AvailableSituation");
        sit2.setAvailableSituation("r", "NOT AVAILABLE", "STARTABLE", "MGMTTASK_PROCESS");
        cbe.setSituation(sit2);
        assertEquals(sit2, cbe.getSituation());
    }

    public void testBasicSetSituation() {
        // tested by: testGetSituation()
    }

    public void testSetSituation() {
        // tested by: testGetSituation()
    }

    public void testGetAny() {
        // This was tested in: testAddAny()
    }

    public void testGetAssociatedEvents() {
        // This was tested in: testAddAssociatedEvents()
    }

    public void testGetContextDataElements() {
        // This was tested in: testAddContextDataElement()
    }

    /*
     * Test for EList getExtendedDataElements()
     */
    public void testGetExtendedDataElements() {
        // This was tested in: testAddExtendedDataElement...()
    }

    public void testGetMsgDataElement() throws ValidationException {
        assertEquals(null, cbe.getMsgDataElement());

        MsgDataElement mde1 = EventFactory.eINSTANCE.createMsgDataElement();

        cbe.setMsgDataElement(mde1);
        assertEquals(mde1, cbe.getMsgDataElement());

        MsgDataElement mde2 = EventFactory.eINSTANCE.createMsgDataElement();
        mde2.setMsgId("123 456");
        mde2.setMsgIdType("type 1");
        cbe.setMsgDataElement(mde2);
        assertEquals(mde2, cbe.getMsgDataElement());
    }

    public void testBasicSetMsgDataElement() {
        // This was tested in: testGetMsgDataElement()
    }

    public void testSetMsgDataElement() {
        // This was tested in: testGetMsgDataElement()
    }

    public void testGetReporterComponentId() {
        // This was tested in: testSetReporterComponentIdComponentIdentification()
    }

    public void testBasicSetReporterComponentId() {
        // This was tested in: testSetReporterComponentIdComponentIdentification()
    }

    /*
     * Test for void setReporterComponentId(ComponentIdentification)
     */
    public void testSetReporterComponentIdComponentIdentification() {
        CommonBaseEvent sciCBE = EventFactory.eINSTANCE.createCommonBaseEvent();
        assertEquals(null, sciCBE.getReporterComponentId());

        ComponentIdentification cit1 = EventFactory.eINSTANCE.createComponentIdentification();
        String p1 = "Reporter Component ID";
        String p2 = "OID-OSI";
        String p3 = "IBM DB2#7.1";
        String p4 = "My sub component";
        String p5 = "SystemName";
        String p6 = "My Component Type";

        cit1.setLocation(p1);
        cit1.setLocationType(p2);
        cit1.setComponent(p3);
        cit1.setSubComponent(p4);
        cit1.setComponentIdType(p5);
        cit1.setComponentType(p6);
        cbe.setReporterComponentId(cit1);

        assertNotNull(cbe.getReporterComponentId());
        ComponentIdentification sci = cbe.getReporterComponentId();
        assertNotNull(sci);
        assertEquals(p1, sci.getLocation());
        assertEquals(p2, sci.getLocationType());
        assertEquals(p3, sci.getComponent());
        assertEquals(p4, sci.getSubComponent());
        assertEquals(p5, sci.getComponentIdType());
        assertEquals(p6, sci.getComponentType());
    }

    public void testGetSourceComponentId() {
        CommonBaseEvent sciCBE = EventFactory.eINSTANCE.createCommonBaseEvent();
        assertEquals(null, sciCBE.getSourceComponentId());

        ComponentIdentification cit1 = EventFactory.eINSTANCE.createComponentIdentification();
        String p1 = "Source Component ID";
        String p2 = "OID-OSI";
        String p3 = "IBM DB2#7.1";
        String p4 = "My sub component";
        String p5 = "SystemName";
        String p6 = "My Component Type";

        cit1.setLocation(p1);
        cit1.setLocationType(p2);
        cit1.setComponent(p3);
        cit1.setSubComponent(p4);
        cit1.setComponentIdType(p5);
        cit1.setComponentType(p6);
        cbe.setSourceComponentId(cit1);

        assertNotNull(cbe.getSourceComponentId());
        ComponentIdentification sci = cbe.getSourceComponentId();
        assertNotNull(sci);
        assertEquals(p1, sci.getLocation());
        assertEquals(p2, sci.getLocationType());
        assertEquals(p3, sci.getComponent());
        assertEquals(p4, sci.getSubComponent());
        assertEquals(p5, sci.getComponentIdType());
        assertEquals(p6, sci.getComponentType());
    }

    /**
     * Tests whether the model has been generated in a way that 
     * allows duplicate values inside a list.
     */
    public void testDuplicateAny() {
        EventFactory eFx = EventFactory.eINSTANCE;
        CommonBaseEvent cbe = eFx.createCommonBaseEvent();
        cbe.getAny().add("1");
        cbe.getAny().add("1");
        assertEquals(2, cbe.getAny().size());
    }

    public void testBasicSetSourceComponentId() {
        // tested by: testGetSourceComponentId()
    }

    /*
     * Test for void setSourceComponentId(ComponentIdentification)
     */
    public void testSetSourceComponentIdComponentIdentification() {
        // tested by: testGetSourceComponentId()
    }

    /*
     * Test for NotificationChain eInverseRemove(InternalEObject, int, Class, NotificationChain)
     */
    public void testEInverseRemoveInternalEObjectintClassNotificationChain() {
        /*
         * generated code, no need to test: testEInverseRemoveInternalEObjectintClassNotificationChain()
         */
    }

    /*
     * Test for Object eGet(EStructuralFeature, boolean)
     */
    public void testEGetEStructuralFeatureboolean() {
        /*
         * generated code, no need to test: testEGetEStructuralFeatureboolean()
         */
    }

    /*
     * Test for boolean eIsSet(EStructuralFeature)
     */
    public void testEIsSetEStructuralFeature() {
        /*
         * generated code, no need to test: testEIsSetEStructuralFeature()
         */
    }

    /*
     * Test for void eSet(EStructuralFeature, Object)
     */
    public void testESetEStructuralFeatureObject() {
        /*
         * generated code, no need to test: testESetEStructuralFeatureObject()
         */
    }

    /*
     * Test for void eUnset(EStructuralFeature)
     */
    public void testEUnsetEStructuralFeature() {
        /*
         * generated code, no need to test: testEUnsetEStructuralFeature()
         */
    }

    /*
     * Test for String toString()
     */
    public void testToString() {
        assertNotNull("cbe with soma data filled in", cbe.toString());
        cbe = EventFactory.eINSTANCE.createCommonBaseEvent();
        assertNotNull("empty cbe", cbe.toString());
    }

    public void testComplete() throws EventException {
        ContentHandler mockHandler = new ContentHandler() {
            public void completeEvent(CommonBaseEvent event) throws CompletionException {
                event.setExtensionName("Example");
            }
        };
        EventFactory ef = EventFactoryFactory.createEventFactory(mockHandler);
        CommonBaseEvent cbe = ef.createCommonBaseEvent();
        cbe.complete();
        assertNotNull(cbe.getExtensionName());
        assertEquals("Example", cbe.getExtensionName());
    }

    /**
     * Tests whether the event throws an exception when a content handler
     * throws an exception.
     */
    public void testFailedComplete() {
        ContentHandler mockHandler = new ContentHandler() {
            public void completeEvent(CommonBaseEvent event) throws CompletionException {
                throw new CompletionException();
            }
        };
        EventFactory ef = EventFactoryFactory.createEventFactory(mockHandler);
        CommonBaseEvent cbe = ef.createCommonBaseEvent();
        try {
            cbe.complete();
            fail("Should have thrown an exception");
        }
        catch (EventException e) {
            // expected
        }
    }

    /**
     * Base fixture for thread-safety testing through JUnitPerf.
     * <P>
     * Notice that it uses a local variable for the testing since
     * JUnitPerf calls setUp/test multiple time for the same
     * TestCase instance.
     */
    public void testValidateThreaded() throws ValidationException {
        CommonBaseEvent cbe = Util.generateEvent();
        cbe.validate();
    }

    public void testValidate() throws ValidationException {
        CommonBaseEvent cbe = Util.generateEvent();

        // Make sure we start with a valid CommonBaseEvent
        cbe.validate();

        // version
        cbe.setVersion("1.0.1.0");
        try {
            cbe.validate();
            fail("Invalid version: 1.0.1.0");
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0205E", ve);
        }
        cbe.setVersion("1.0.1");
        cbe.validate();

        // localInstanceId
        cbe.setLocalInstanceId(Util.getPaddedString(129));
        try {
            cbe.validate();
            fail("Invalid LocalInstanceId");
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0207E", ve);
        }
        cbe.setLocalInstanceId("my id");
        cbe.validate();

        // globalInstanceId
        cbe.setGlobalInstanceId(Util.getAlphaString(31));
        try {
            cbe.validate();
            fail("Invalid GlobalInstanceId");
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0206E", ve);
        }
        cbe.setGlobalInstanceId(Util.getAlphaString(65));
        try {
            cbe.validate();
            fail("Invalid GlobalInstanceId");
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0206E", ve);
        }
        cbe.setGlobalInstanceId(Util.getPaddedString(42));
        try {
            cbe.validate();
            fail("Invalid GlobalInstanceId - must start with alpha character");
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0190E", ve);
        }
        cbe.setGlobalInstanceId(Util.getAlphaString(42));
        cbe.validate();
        cbe.setGlobalInstanceId(Guid.generate());
        cbe.validate();

        // creationTime
        // error cases tested here: testSetCreationTime()

        // severity
        cbe.setSeverity((short) 71);
        try {
            cbe.validate();
            fail("Invalid Severity");
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0202E", ve);
        }
        cbe.setSeverity((short) - 3);
        try {
            cbe.validate();
            fail("Invalid Severity");
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0202E", ve);
        }
        cbe.setSeverity((short) 0);
        cbe.validate();
        cbe.setSeverity((short) 50);
        cbe.validate();

        // priority
        cbe.setPriority((short) 101);
        try {
            cbe.validate();
            fail("Invalid Priority");
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0202E", ve);
        }
        cbe.setPriority((short) - 3);
        try {
            cbe.validate();
            fail("Invalid Priority");
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0202E", ve);
        }
        cbe.setPriority((short) 0);
        cbe.validate();
        cbe.setPriority((short) 100);
        cbe.validate();

        // reporterComponentId
        // This is a required property.  The validation test will be done by testValidationRequiredReporterComponentId()

        // sourceComponentId
        // This is a required property.  The validation test will be done by testValidationRequiredSourceComponentId()

        // situation
        // This is a required property.  The validation test will be done by testValidationRequiredSituation()

        // contextDataElements
        // This is an optional property.  The validation test will be done by ContextDataElementImplTest()

        // msgDataElement
        // This is an optional property.  The validation test will be done by MsgDataElementImplTest()

        // msg
        cbe.setMsg(Util.getPaddedString(1025));
        try {
            cbe.validate();
            fail("Invalid Msg");
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0207E", ve);
        }
        cbe.setMsg("howdy howdy");
        cbe.validate();

        // repeatCount
        cbe.setRepeatCount((short) 5);
        try {
            cbe.validate();
            fail("repeatCount is set, but elapsedTime is not yet set");
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0203E", ve);
        }

        // elapsedTime
        cbe.setElapsedTime(-3);
        try {
            cbe.validate();
            fail("Invalid Elapsed Time");
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0202E", ve);
        }
        cbe.setElapsedTime(0);
        cbe.validate();
        cbe.setElapsedTime(Long.MAX_VALUE);
        cbe.validate();

        //repeatCount (con't)
        cbe.setRepeatCount((short)-3);
        try {
            cbe.validate();
            fail("Invalid Repeat Count");
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0202E", ve);
        }
        cbe.setRepeatCount((short)0);
        cbe.validate();
        cbe.setRepeatCount(Short.MAX_VALUE);
        cbe.validate();

        // associatedEvents
        // This is an optional property.  The validation test will be done by AssocaitedEventsImplTest()

        // extensionName
        cbe.setExtensionName(Util.getAlphaString(65));
        try {
            cbe.validate();
            fail("Invalid extensionName");
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0207E", ve);
        }
        cbe.setExtensionName("UnCommonBaseEvent");
        cbe.validate();

        // extendedDataElements
        // This is an optional property.  The validation test will be done by ExtendedDataElementImplTest() 

        // sequenceNumber
        cbe.setSequenceNumber((short) - 3);
        try {
            cbe.validate();
            fail("Invalid Sequence Number");
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0202E", ve);
        }
        cbe.setSequenceNumber(0);
        cbe.validate();    
        cbe.setSequenceNumber(Long.MAX_VALUE);
        cbe.validate();    
   }

    public void testGetCreationTimeAsLong() throws ValidationException {
        //2003-10-30T10:42:19.452-05:00 == 1067528539452 == 2003-10-30T15:42:19.452Z

        long inDateLong = (new Long("1067528539452")).longValue();
        String inDateStr = "2003-10-30T10:42:19.452-05:00";

        cbe.setCreationTime(inDateStr);
        cbe.validate();
        long outDateLong = cbe.getCreationTimeAsLong();
        assertEquals(inDateLong, outDateLong);
    }

    public void testSetCreationTimeAsLong() throws ValidationException {
        //2003-10-30T10:42:19.452-05:00 == 1067528539452 == 2003-10-30T15:42:19.452Z

        long inDateLong = (new Long("1067528539452")).longValue();
        String inDateStr = "2003-10-30T15:42:19.452Z";
        String outDateStr;
        long outDateLong;

        cbe.setCreationTimeAsLong(inDateLong);
        cbe.validate();
        outDateStr = cbe.getCreationTime();
        assertEquals(inDateStr, outDateStr);

        cbe.setCreationTime(inDateStr);
        cbe.validate();
        outDateLong = cbe.getCreationTimeAsLong();
        assertEquals(inDateLong, outDateLong);

        inDateStr = "2003-10-30T17:42:19.452+02:00";
        cbe.setCreationTime(inDateStr);
        cbe.validate();
        outDateLong = cbe.getCreationTimeAsLong();
        assertEquals(inDateLong, outDateLong);
    }

    public void testAddAny() throws ValidationException {
        assertEquals("list should be empty", 0, cbe.getAny().size());

        String input = "sometext";
        cbe.addAny(input);
        assertEquals(1, cbe.getAny().size());
        String anyStr = (String) cbe.getAny().get(0);
        assertNotNull(anyStr);
        assertEquals(input, anyStr);

        input = "someothertext";
        cbe.addAny(input);
        assertEquals(2, cbe.getAny().size());
        anyStr = (String) cbe.getAny().get(1);
        assertNotNull(anyStr);
        assertEquals(input, anyStr);

        cbe.validate();
    }

    public void testAddAssociatedEvents() throws ValidationException {
        assertEquals("list should be empty", 0, cbe.getAssociatedEvents().size());
        String p1 = "ae name 1";
        String p2 = "ae type 1";
        String p3 = "CE123456789012345678901234567890";
        String[] p4 = { "CE12345678901234567890123456789A", "CE12345678901234567890123456789B" };
        cbe.addAssociatedEvent(p1, p2, p3, p4);
        assertEquals(1, cbe.getAssociatedEvents().size());
        AssociatedEvent ae1 = (AssociatedEvent) cbe.getAssociatedEvents().get(0);
        assertNotNull(ae1);
        AssociationEngine aei1 = ae1.getAssociationEngineInfo();
        assertEquals(p1, aei1.getName());
        assertEquals(p2, aei1.getType());
        assertEquals(p3, aei1.getId());
        String[] aer1 = ae1.getResolvedEventsAsArray();
        assertEquals(p4[0], aer1[0]);
        assertEquals(p4[1], aer1[1]);

        cbe.validate();

        String pp1 = "ae name 1";
        String pp2 = "ae type 1";
        String pp3 = "CE123456789012345678901234567891";
        String[] pp4 = { "CE12345678901234567890123456789C", "CE12345678901234567890123456789D" };
        cbe.addAssociatedEvent(pp1, pp2, pp3, pp4);
        assertEquals(2, cbe.getAssociatedEvents().size());
        AssociatedEvent ae2 = (AssociatedEvent) cbe.getAssociatedEvents().get(1);
        assertNotNull(ae2);
        AssociationEngine aei2 = ae2.getAssociationEngineInfo();
        assertEquals(pp1, aei2.getName());
        assertEquals(pp2, aei2.getType());
        assertEquals(pp3, aei2.getId());
        String[] aer2 = ae2.getResolvedEventsAsArray();
        assertEquals(pp4[0], aer2[0]);
        assertEquals(pp4[1], aer2[1]);

        cbe.validate();
    }

    public void testAddContextDataElement() throws ValidationException {
        assertEquals("list should be empty", 0, cbe.getContextDataElements().size());

        // First add one ContextDataElement.  The last two parameters, id and value, are mutually exclusive
        String p4 = Util.getAlphaString(32);
        String p3 = null;
        String p2 = "ce name 1";
        String p1 = "ce type 1";
        cbe.addContextDataElementWithId(p1, p2, p4);
        assertEquals(1, cbe.getContextDataElements().size());
        ContextDataElement cde1 = (ContextDataElement) cbe.getContextDataElements().get(0);
        assertNotNull(cde1);
        assertEquals(p4, cde1.getContextId());
        assertEquals(p3, cde1.getContextValue());
        assertEquals(p2, cde1.getName());
        assertEquals(p1, cde1.getType());

        // Add a second ContextDataElement.  The last two parameters, id and value, are mutually exclusive
        String pp4 = null;
        String pp3 = "ce value 2";
        String pp2 = "ce name 2";
        String pp1 = "ce type 2";
        cbe.addContextDataElementWithValue(pp1, pp2, pp3);
        assertEquals(2, cbe.getContextDataElements().size());
        ContextDataElement cde2 = (ContextDataElement) cbe.getContextDataElements().get(1);
        assertNotNull(cde2);
        assertEquals(pp4, cde2.getContextId());
        assertEquals(pp3, cde2.getContextValue());
        assertEquals(pp2, cde2.getName());
        assertEquals(pp1, cde2.getType());

        // Add a third ContextDataElement.  This tests that two ContextDataElements can have the same name.
        cbe.addContextDataElementWithId(p1, p2, p4);
        assertEquals(3, cbe.getContextDataElements().size());
        ContextDataElement cde3 = (ContextDataElement) cbe.getContextDataElements().get(2);
        assertNotNull(cde3);
        assertEquals(p4, cde3.getContextId());
        assertEquals(p3, cde3.getContextValue());
        assertEquals(p2, cde3.getName());
        assertEquals(p1, cde3.getType());

        cbe.validate();

        // Add a fourth ContextDataElement.  Validate should fail because both id and value are specified.
        cde1 = cbe.addContextDataElementWithValue("type", "name", "value");
        cde1.setContextId(Util.getPaddedString(32));
        assertEquals(4, cbe.getContextDataElements().size());
        ContextDataElement cde4 = (ContextDataElement) cbe.getContextDataElements().get(3);
        assertNotNull(cde4);

        try {
            cbe.validate();
            fail("Can't specify both ID and Value on a ContextDataElement");
        }
        catch (ValidationException ex) {
            Util.assertExceptionId("IWAT0417E", ex);
            Util.assertExceptionId("IWAT0204E", ex.getCause());
        }
    }

    /**
     * Test for ExtendedDataElement addExtendedDataElement(String, int, String)
     */
    public void testAddExtendedDataElementGenericWithHexBinary() throws ValidationException {
        String hexValue = "abc123";
        ExtendedDataElement ede = cbe.addExtendedDataElement("AA", ExtendedDataElement.TYPE_HEX_BINARY_VALUE, hexValue);
        assertEquals(1, cbe.getExtendedDataElements().size());

        ExtendedDataElement ede1 = (ExtendedDataElement) cbe.getExtendedDataElements().get(0);
        assertEquals("hexBinary", ede1.getType());
        assertEquals(ede, ede1);

        assertEquals(hexValue, ede.getHexValue());
    }

    /**
     * Test for ExtendedDataElement addExtendedDataElementWithHexValue(String, String)
     */
    public void testAddExtendedDataElementWithHexBinaryString() throws ValidationException {
        String hexValue = "abc123";
        ExtendedDataElement ede = cbe.addExtendedDataElementWithHexValue("AA", hexValue);
        assertEquals(1, cbe.getExtendedDataElements().size());

        ExtendedDataElement ede1 = (ExtendedDataElement) cbe.getExtendedDataElements().get(0);
        assertEquals("hexBinary", ede1.getType());
        assertEquals(ede, ede1);

        assertEquals(hexValue, ede.getHexValue());
    }

    /**
     * Test for ExtendedDataElement addExtendedDataElementWithHexValue(String, byte[])
     */
    public void testAddExtendedDataElementWithHexBinaryBytes() throws ValidationException {
        byte[] data = new byte[] { 5, 6, 7 };
        ExtendedDataElement ede = cbe.addExtendedDataElementWithHexValue("AA", data);
        assertEquals(1, cbe.getExtendedDataElements().size());

        ExtendedDataElement ede1 = (ExtendedDataElement) cbe.getExtendedDataElements().get(0);
        assertEquals("hexBinary", ede1.getType());
        assertEquals(ede, ede1);

        byte[] outData = ede1.getHexValueAsByteArray();
        assertEquals(data.length, outData.length);
        for (int i = 0; i < outData.length; i++) {
            assertEquals("mismatch: " + i, data[i], outData[i]);
        }
    }

    /**
     * Test for ExtendedDataElement addExtendedDataElement(String, int, String)
     */
    public void testAddExtendedDataElementStringintString() throws ValidationException {
        ExtendedDataElement ede;
        ExtendedDataElement ede1;
        ExtendedDataElement ede2;

        String myName = "Scott Brown";
        String myValue = "TEC Developer";

        assertEquals(0, cbe.getExtendedDataElements().size());

        ede = cbe.addExtendedDataElement(myName, ExtendedDataElement.TYPE_STRING_VALUE, myValue);
        ede1 = (ExtendedDataElement) cbe.getExtendedDataElements().get(0);
        assertNotNull(ede);
        assertNotNull(ede1);
        assertEquals(ede, ede1);
        assertEquals(myName, ede1.getName());
        assertEquals(myValue, ede1.getValuesAsString());
        assertEquals(ExtendedDataElement.TYPE_STRING, ede1.getType());

        cbe.validate();

        ede = cbe.addExtendedDataElement(myName, ExtendedDataElement.TYPE_INT_VALUE, "xxx xxx");
        ede2 = (ExtendedDataElement) cbe.getExtendedDataElements().get(1);
        assertNotNull(ede);
        assertNotNull(ede2);
        assertEquals(ede, ede2);
        assertEquals(myName, ede2.getName());
        //assertEquals("xxx xxx", ede2.getValuesAsString());
        //assertEquals(ExtendedDataElement.TYPE_STRING, ede2.getType());

        try {
            cbe.validate();
            fail("Created an ExtendedDataElement with a String value but set the type to INT");
        }
        catch (ValidationException ex) {
            Util.assertExceptionId("IWAT0417E", ex);
            Util.assertExceptionId("IWAT0192E", ex.getCause());
        }
    }

    /*
     * Test for ExtendedDataElement addExtendedDataElement(String, int, String[])
     */
    public void testAddExtendedDataElementStringintStringArray() throws ValidationException {
        ExtendedDataElement ede;
        ExtendedDataElement ede1;
        ExtendedDataElement ede2;

        String myName = "Scott Brown";
        String[] myValue = { "I am", "a", "TEC Developer" };

        assertEquals(0, cbe.getExtendedDataElements().size());

        ede = cbe.addExtendedDataElement(myName, ExtendedDataElement.TYPE_STRING_ARRAY_VALUE, myValue);
        ede1 = (ExtendedDataElement) cbe.getExtendedDataElements().get(0);
        assertNotNull(ede);
        assertNotNull(ede1);
        assertEquals(ede, ede1);
        assertEquals(myName, ede1.getName());
        assertEquals(3, ede1.getValuesAsStringArray().length);
        assertEquals(myValue[0], ede1.getValuesAsStringArray()[0]);
        assertEquals(myValue[1], ede1.getValuesAsStringArray()[1]);
        assertEquals(myValue[2], ede1.getValuesAsStringArray()[2]);
        assertEquals(ExtendedDataElement.TYPE_STRING_ARRAY, ede1.getType());

        cbe.validate();

        ede = cbe.addExtendedDataElement(myName, ExtendedDataElement.TYPE_LONG_VALUE, "123");
        ede2 = (ExtendedDataElement) cbe.getExtendedDataElements().get(1);
        assertNotNull(ede);
        assertNotNull(ede2);
        assertEquals(ede, ede2);
        assertEquals(myName, ede2.getName());
        assertEquals(123, ede2.getValuesAsLong());
        assertEquals(ExtendedDataElement.TYPE_LONG, ede2.getType());

        ede = cbe.addExtendedDataElement(myName, ExtendedDataElement.TYPE_LONG_VALUE, "xxx xxx");
        ede2 = (ExtendedDataElement) cbe.getExtendedDataElements().get(2);
        assertNotNull(ede);
        assertNotNull(ede2);
        assertEquals(ede, ede2);
        assertEquals(myName, ede2.getName());
        //assertEquals("xxx xxx", ede2.getValuesAsString());
        //assertEquals(ExtendedDataElement.TYPE_STRING, ede2.getType());

        try {
            cbe.validate();
            fail("Created an ExtendedDataElement with a String value but set the type to LONG");
        }
        catch (ValidationException ex) {
            Util.assertExceptionId("IWAT0417E", ex);
            Util.assertExceptionId("IWAT0192E", ex.getCause());
        }
    }

    /*
     * Test for ExtendedDataElement addExtendedDataElement(String, String)
     */
    public void testAddExtendedDataElementStringString() throws ValidationException {
        ExtendedDataElement ede;
        ExtendedDataElement ede1;
        ExtendedDataElement ede2;
        ExtendedDataElement ede3;

        String myName = "Scott Brown";
        String myValue = "TEC Developer";

        assertEquals(0, cbe.getExtendedDataElements().size());

        ede = cbe.addExtendedDataElement(myName, myValue);
        ede1 = (ExtendedDataElement) cbe.getExtendedDataElements().get(0);
        assertNotNull(ede);
        assertNotNull(ede1);
        assertEquals(ede, ede1);
        assertEquals(myName, ede1.getName());
        assertEquals(myValue, ede1.getValuesAsString());
        assertEquals(ExtendedDataElement.TYPE_STRING, ede1.getType());

        ede = cbe.addExtendedDataElement(myName, "xxx xxx");
        ede2 = (ExtendedDataElement) cbe.getExtendedDataElements().get(1);
        assertNotNull(ede);
        assertNotNull(ede2);
        assertEquals(ede, ede2);
        assertEquals(myName, ede2.getName());
        assertEquals("xxx xxx", ede2.getValuesAsString());
        assertEquals(ExtendedDataElement.TYPE_STRING, ede2.getType());

        cbe.validate();
    }

    /*
     * Test for ExtendedDataElement addExtendedDataElement(String, String[])
     */
    public void testAddExtendedDataElementStringStringArray() {
        ExtendedDataElement ede;
        ExtendedDataElement ede1;

        String myName = "Scott Brown";
        String[] myValue = { "TEC Developer", "bike rider" };

        assertEquals(0, cbe.getExtendedDataElements().size());

        ede = cbe.addExtendedDataElement(myName, myValue);
        ede1 = (ExtendedDataElement) cbe.getExtendedDataElements().get(0);
        assertNotNull(ede);
        assertNotNull(ede1);
        assertEquals(ede, ede1);
        assertEquals(myName, ede1.getName());
        assertEquals(myValue[0], ede1.getValuesAsStringArray()[0]);
        assertEquals(myValue[1], ede1.getValuesAsStringArray()[1]);
        assertEquals(ExtendedDataElement.TYPE_STRING_ARRAY, ede1.getType());
    }

    /*
     * Test for ExtendedDataElement addExtendedDataElement(String)
     */
    public void testAddExtendedDataElementString() {
        ExtendedDataElement ede = cbe.addExtendedDataElement("Fred");
        List edeList = cbe.getExtendedDataElements("Fred");
        ExtendedDataElement cbe_ede = (ExtendedDataElement) edeList.get(0);
        assertEquals(edeList.get(0), ede);
        assertEquals("Fred", cbe_ede.getName());
        assertEquals("Fred", ede.getName());
        assertNull(ede.getType());
        assertTrue(ede.getValues().isEmpty());
        assertNull(ede.getHexValue());
        assertTrue(ede.getChildren().isEmpty());
    }

    public void testAddExtendedDataElementWithByteArrayValue() {
        EList myEList = cbe.getExtendedDataElements();
        assertEquals(0, myEList.size());

        byte[] value = { 'a', 'b' };
        ExtendedDataElement ede = cbe.addExtendedDataElementWithByteArrayValue("myede", value);
        myEList = cbe.getExtendedDataElements();
        assertEquals(1, myEList.size());

        List edeList = cbe.getExtendedDataElements("myede");
        ExtendedDataElement cbe_ede = (ExtendedDataElement) edeList.get(0);
        assertEquals(edeList.get(0), ede);
        assertEquals("myede", cbe_ede.getName());
        assertEquals("myede", ede.getName());
        assertNotNull(ede.getType());
        assertFalse(ede.getValues().isEmpty());
        assertNull(ede.getHexValue());
        assertTrue(ede.getChildren().isEmpty());

        ExtendedDataElement tmpEde = (ExtendedDataElement) myEList.get(0);
        assertEquals(ExtendedDataElement.TYPE_BYTE_ARRAY_VALUE, tmpEde.getTypeAsInt());
        byte[] outValue = tmpEde.getValuesAsByteArray();
        assertEquals(value.length, outValue.length);

        for (int i = 0; i < value.length; i++) {
            assertEquals(value[i], outValue[i]);
        }
    }

    public void testAddExtendedDataElementWithByteValue() {
        byte value = 'a';
        ExtendedDataElement ede = cbe.addExtendedDataElementWithByteValue("myede", value);

        EList myEList = cbe.getExtendedDataElements();
        assertEquals(1, myEList.size());

        List edeList = cbe.getExtendedDataElements("myede");
        ExtendedDataElement cbe_ede = (ExtendedDataElement) edeList.get(0);
        assertEquals(edeList.get(0), ede);
        assertEquals("myede", cbe_ede.getName());
        assertEquals("myede", ede.getName());
        assertNotNull(ede.getType());
        assertFalse(ede.getValues().isEmpty());
        assertNull(ede.getHexValue());
        assertTrue(ede.getChildren().isEmpty());

        ExtendedDataElement tmpEde = (ExtendedDataElement) edeList.get(0);
        assertEquals(ExtendedDataElement.TYPE_BYTE_VALUE, tmpEde.getTypeAsInt());
        assertEquals(value, tmpEde.getValuesAsByte());
    }

    public void testAddExtendedDataElementWithDateArrayValue() {
        String[] value = { "1896-12-31T11:00:00.000Z", "3001-12-31T11:00:00.000-05:00", "2007-12-31T11:00:00.000+11:00" };

        EList myEList = cbe.getExtendedDataElements();
        assertEquals(0, myEList.size());

        ExtendedDataElement ede = cbe.addExtendedDataElementWithDateArrayValue("myede", value);
        myEList = cbe.getExtendedDataElements();
        assertTrue(myEList.size() == 1);

        List edeList = cbe.getExtendedDataElements("myede");
        ExtendedDataElement cbe_ede = (ExtendedDataElement) edeList.get(0);
        assertEquals(edeList.get(0), ede);
        assertEquals("myede", cbe_ede.getName());
        assertEquals("myede", ede.getName());
        assertNotNull(ede.getType());
        assertFalse(ede.getValues().isEmpty());
        assertNull(ede.getHexValue());
        assertTrue(ede.getChildren().isEmpty());

        ExtendedDataElement tmpEde = (ExtendedDataElement) myEList.get(0);
        assertEquals(ExtendedDataElement.TYPE_DATE_TIME_ARRAY_VALUE, tmpEde.getTypeAsInt());
        assertTrue(Arrays.asList(tmpEde.getValuesAsDateArray()).containsAll(Arrays.asList(value)));
    }

    public void testAddExtendedDataElementWithDateAsLongValue() {
        long value = System.currentTimeMillis();

        EList myEList = cbe.getExtendedDataElements();
        assertEquals(0, myEList.size());

        ExtendedDataElement ede = cbe.addExtendedDataElementWithDateAsLongValue("myede", value);
        myEList = cbe.getExtendedDataElements();
        assertEquals(1, myEList.size());

        List edeList = cbe.getExtendedDataElements("myede");
        ExtendedDataElement cbe_ede = (ExtendedDataElement) edeList.get(0);
        assertEquals(edeList.get(0), ede);
        assertEquals("myede", cbe_ede.getName());
        assertEquals("myede", ede.getName());
        assertNotNull(ede.getType());
        assertFalse(ede.getValues().isEmpty());
        assertNull(ede.getHexValue());
        assertTrue(ede.getChildren().isEmpty());

        ExtendedDataElement tmpEde = (ExtendedDataElement) myEList.get(0);
        assertEquals(ExtendedDataElement.TYPE_DATE_TIME_VALUE, tmpEde.getTypeAsInt());
        assertEquals(value, tmpEde.getValuesAsDateAsLong());
    }

    public void testAddExtendedDataElementWithDatesAsLongValue() {
        long[] value = { System.currentTimeMillis(), System.currentTimeMillis() + 100000 };

        EList myEList = cbe.getExtendedDataElements();
        assertEquals(0, myEList.size());

        ExtendedDataElement ede = cbe.addExtendedDataElementWithDatesAsLongValue("myede", value);
        myEList = cbe.getExtendedDataElements();
        assertEquals(1, myEList.size());

        List edeList = cbe.getExtendedDataElements("myede");
        ExtendedDataElement cbe_ede = (ExtendedDataElement) edeList.get(0);
        assertEquals(edeList.get(0), ede);
        assertEquals("myede", cbe_ede.getName());
        assertEquals("myede", ede.getName());
        assertNotNull(ede.getType());
        assertFalse(ede.getValues().isEmpty());
        assertNull(ede.getHexValue());
        assertTrue(ede.getChildren().isEmpty());

        ExtendedDataElement tmpEde = (ExtendedDataElement) myEList.get(0);
        assertEquals(ExtendedDataElement.TYPE_DATE_TIME_ARRAY_VALUE, tmpEde.getTypeAsInt());
        long[] outValue = tmpEde.getValuesAsDateAsLongArray();
        assertEquals(value.length, outValue.length);

        for (int i = 0; i < value.length; i++) {
            assertEquals(value[i], outValue[i]);
        }
    }

    public void testAddExtendedDataElementWithDateValue() {
        String value = "2003-11-10T12:00:00.000Z";

        EList myEList = cbe.getExtendedDataElements();
        assertEquals(0, myEList.size());

        ExtendedDataElement ede = cbe.addExtendedDataElementWithDateValue("myede", value);
        myEList = cbe.getExtendedDataElements();
        assertEquals(1, myEList.size());

        List edeList = cbe.getExtendedDataElements("myede");
        ExtendedDataElement cbe_ede = (ExtendedDataElement) edeList.get(0);
        assertEquals(edeList.get(0), ede);
        assertEquals("myede", cbe_ede.getName());
        assertEquals("myede", ede.getName());
        assertNotNull(ede.getType());
        assertFalse(ede.getValues().isEmpty());
        assertNull(ede.getHexValue());
        assertTrue(ede.getChildren().isEmpty());

        ExtendedDataElement tmpEde = (ExtendedDataElement) myEList.get(0);
        assertEquals(ExtendedDataElement.TYPE_DATE_TIME_VALUE, tmpEde.getTypeAsInt());
        assertEquals(value, tmpEde.getValuesAsDate());
    }

    public void testAddExtendedDataElementWithDoubleArrayValue() {
        double[] value = { 34.7, 65.8 };

        EList myEList = cbe.getExtendedDataElements();
        assertEquals(0, myEList.size());

        ExtendedDataElement ede = cbe.addExtendedDataElementWithDoubleArrayValue("myede", value);
        myEList = cbe.getExtendedDataElements();
        assertEquals(1, myEList.size());

        List edeList = cbe.getExtendedDataElements("myede");
        ExtendedDataElement cbe_ede = (ExtendedDataElement) edeList.get(0);
        assertEquals(edeList.get(0), ede);
        assertEquals("myede", cbe_ede.getName());
        assertEquals("myede", ede.getName());
        assertNotNull(ede.getType());
        assertFalse(ede.getValues().isEmpty());
        assertNull(ede.getHexValue());
        assertTrue(ede.getChildren().isEmpty());

        ExtendedDataElement tmpEde = (ExtendedDataElement) edeList.get(0);
        assertEquals(ExtendedDataElement.TYPE_DOUBLE_ARRAY_VALUE, tmpEde.getTypeAsInt());
        double[] outValue = tmpEde.getValuesAsDoubleArray();
        assertEquals(value.length, outValue.length);

        for (int i = 0; i < value.length; i++) {
            assertTrue(value[i] == outValue[i]);
        }
    }

    public void testAddExtendedDataElementWithDoubleValue() {
        double value = 12.5;

        EList myEList = cbe.getExtendedDataElements();
        assertEquals(0, myEList.size());

        ExtendedDataElement ede = cbe.addExtendedDataElementWithDoubleValue("myede", value);
        myEList = cbe.getExtendedDataElements();
        assertEquals(1, myEList.size());

        List edeList = cbe.getExtendedDataElements("myede");
        ExtendedDataElement cbe_ede = (ExtendedDataElement) edeList.get(0);
        assertEquals(edeList.get(0), ede);
        assertEquals("myede", cbe_ede.getName());
        assertEquals("myede", ede.getName());
        assertNotNull(ede.getType());
        assertFalse(ede.getValues().isEmpty());
        assertNull(ede.getHexValue());
        assertTrue(ede.getChildren().isEmpty());

        ExtendedDataElement tmpEde = (ExtendedDataElement) edeList.get(0);
        assertEquals(ExtendedDataElement.TYPE_DOUBLE_VALUE, tmpEde.getTypeAsInt());
        assertTrue(tmpEde.getValuesAsDouble() == value);
    }

    public void testAddExtendedDataElementWithFloatArrayValue() {
        float[] value = { 213213123, 123124144 };

        EList myEList = cbe.getExtendedDataElements();
        assertEquals(0, myEList.size());

        ExtendedDataElement ede = cbe.addExtendedDataElementWithFloatArrayValue("myede", value);
        myEList = cbe.getExtendedDataElements();
        assertEquals(1, myEList.size());

        List edeList = cbe.getExtendedDataElements("myede");
        ExtendedDataElement cbe_ede = (ExtendedDataElement) edeList.get(0);
        assertEquals(edeList.get(0), ede);
        assertEquals("myede", cbe_ede.getName());
        assertEquals("myede", ede.getName());
        assertNotNull(ede.getType());
        assertFalse(ede.getValues().isEmpty());
        assertNull(ede.getHexValue());
        assertTrue(ede.getChildren().isEmpty());

        ExtendedDataElement tmpEde = (ExtendedDataElement) myEList.get(0);
        assertEquals(ExtendedDataElement.TYPE_FLOAT_ARRAY_VALUE, tmpEde.getTypeAsInt());
        float[] outValue = tmpEde.getValuesAsFloatArray();
        assertEquals(value.length, outValue.length);

        for (int i = 0; i < value.length; i++) {
            assertTrue(value[i] == outValue[i]);
        }
    }

    public void testAddExtendedDataElementWithFloatValue() {
        float value = 'a';

        EList myEList = cbe.getExtendedDataElements();
        assertEquals(0, myEList.size());

        ExtendedDataElement ede = cbe.addExtendedDataElementWithFloatValue("myede", value);
        myEList = cbe.getExtendedDataElements();
        assertEquals(1, myEList.size());

        List edeList = cbe.getExtendedDataElements("myede");
        ExtendedDataElement cbe_ede = (ExtendedDataElement) edeList.get(0);
        assertEquals(edeList.get(0), ede);
        assertEquals("myede", cbe_ede.getName());
        assertEquals("myede", ede.getName());
        assertNotNull(ede.getType());
        assertFalse(ede.getValues().isEmpty());
        assertNull(ede.getHexValue());
        assertTrue(ede.getChildren().isEmpty());

        ExtendedDataElement tmpEde = (ExtendedDataElement) edeList.get(0);
        assertEquals(ExtendedDataElement.TYPE_FLOAT_VALUE, tmpEde.getTypeAsInt());
        assertTrue(tmpEde.getValuesAsFloat() == value);
    }

    public void testAddExtendedDataElementWithIntArrayValue() {
        int[] value = { 10, 12 };

        EList myEList = cbe.getExtendedDataElements();
        assertEquals(0, myEList.size());

        ExtendedDataElement ede = cbe.addExtendedDataElementWithIntArrayValue("myede", value);
        myEList = cbe.getExtendedDataElements();
        assertEquals(1, myEList.size());

        List edeList = cbe.getExtendedDataElements("myede");
        ExtendedDataElement cbe_ede = (ExtendedDataElement) edeList.get(0);
        assertEquals(edeList.get(0), ede);
        assertEquals("myede", cbe_ede.getName());
        assertEquals("myede", ede.getName());
        assertNotNull(ede.getType());
        assertFalse(ede.getValues().isEmpty());
        assertNull(ede.getHexValue());
        assertTrue(ede.getChildren().isEmpty());

        ExtendedDataElement tmpEde = (ExtendedDataElement) myEList.get(0);
        assertEquals(ExtendedDataElement.TYPE_INT_ARRAY_VALUE, tmpEde.getTypeAsInt());
        int[] outValue = tmpEde.getValuesAsIntArray();
        assertEquals(value.length, outValue.length);

        for (int i = 0; i < value.length; i++) {
            assertTrue(value[i] == outValue[i]);
        }
    }

    public void testAddExtendedDataElementWithIntValue() {
        int value = 123;

        EList myEList = cbe.getExtendedDataElements();
        assertEquals(0, myEList.size());

        ExtendedDataElement ede = cbe.addExtendedDataElementWithIntValue("myede", value);
        myEList = cbe.getExtendedDataElements();
        assertEquals(1, myEList.size());

        List edeList = cbe.getExtendedDataElements("myede");
        ExtendedDataElement cbe_ede = (ExtendedDataElement) edeList.get(0);
        assertEquals(edeList.get(0), ede);
        assertEquals("myede", cbe_ede.getName());
        assertEquals("myede", ede.getName());
        assertNotNull(ede.getType());
        assertFalse(ede.getValues().isEmpty());
        assertNull(ede.getHexValue());
        assertTrue(ede.getChildren().isEmpty());

        ExtendedDataElement tmpEde = (ExtendedDataElement) edeList.get(0);
        assertEquals(ExtendedDataElement.TYPE_INT_VALUE, tmpEde.getTypeAsInt());
        assertEquals(value, tmpEde.getValuesAsInt());
    }

    public void testAddExtendedDataElementWithLongArrayValue() {
        long[] value = { 10121416, 12100806 };

        EList myEList = cbe.getExtendedDataElements();
        assertEquals(0, myEList.size());

        ExtendedDataElement ede = cbe.addExtendedDataElementWithLongArrayValue("myede", value);
        myEList = cbe.getExtendedDataElements();
        assertEquals(1, myEList.size());

        List edeList = cbe.getExtendedDataElements("myede");
        ExtendedDataElement cbe_ede = (ExtendedDataElement) edeList.get(0);
        assertEquals(edeList.get(0), ede);
        assertEquals("myede", cbe_ede.getName());
        assertEquals("myede", ede.getName());
        assertNotNull(ede.getType());
        assertFalse(ede.getValues().isEmpty());
        assertNull(ede.getHexValue());
        assertTrue(ede.getChildren().isEmpty());

        ExtendedDataElement tmpEde = (ExtendedDataElement) edeList.get(0);
        assertEquals(ExtendedDataElement.TYPE_LONG_ARRAY_VALUE, tmpEde.getTypeAsInt());
        long[] outValue = tmpEde.getValuesAsLongArray();
        assertEquals(value.length, outValue.length);

        for (int i = 0; i < value.length; i++) {
            assertTrue(value[i] == outValue[i]);
        }
    }

    public void testAddExtendedDataElementWithLongValue() {
        long value = 123456789;

        EList myEList = cbe.getExtendedDataElements();
        assertEquals(0, myEList.size());

        ExtendedDataElement ede = cbe.addExtendedDataElementWithLongValue("myede", value);
        myEList = cbe.getExtendedDataElements();
        assertEquals(1, myEList.size());

        List edeList = cbe.getExtendedDataElements("myede");
        ExtendedDataElement cbe_ede = (ExtendedDataElement) edeList.get(0);
        assertEquals(edeList.get(0), ede);
        assertEquals("myede", cbe_ede.getName());
        assertEquals("myede", ede.getName());
        assertNotNull(ede.getType());
        assertFalse(ede.getValues().isEmpty());
        assertNull(ede.getHexValue());
        assertTrue(ede.getChildren().isEmpty());

        ExtendedDataElement tmpEde = (ExtendedDataElement) myEList.get(0);
        assertEquals(ExtendedDataElement.TYPE_LONG_VALUE, tmpEde.getTypeAsInt());
        assertEquals(value, tmpEde.getValuesAsLong());
    }

    public void testAddExtendedDataElementWithShortArrayValue() {
        short[] value = { 10, 12 };

        EList myEList = cbe.getExtendedDataElements();
        assertEquals(0, myEList.size());

        ExtendedDataElement ede = cbe.addExtendedDataElementWithShortArrayValue("myede", value);
        myEList = cbe.getExtendedDataElements();
        assertEquals(1, myEList.size());

        List edeList = cbe.getExtendedDataElements("myede");
        ExtendedDataElement cbe_ede = (ExtendedDataElement) edeList.get(0);
        assertEquals(edeList.get(0), ede);
        assertEquals("myede", cbe_ede.getName());
        assertEquals("myede", ede.getName());
        assertNotNull(ede.getType());
        assertFalse(ede.getValues().isEmpty());
        assertNull(ede.getHexValue());
        assertTrue(ede.getChildren().isEmpty());

        ExtendedDataElement tmpEde = (ExtendedDataElement) myEList.get(0);
        assertEquals(ExtendedDataElement.TYPE_SHORT_ARRAY_VALUE, tmpEde.getTypeAsInt());
        short[] outValue = tmpEde.getValuesAsShortArray();
        assertEquals(value.length, outValue.length);

        for (int i = 0; i < value.length; i++) {
            assertEquals(value[i], outValue[i]);
        }
    }

    public void testAddExtendedDataElementWithShortValue() {
        short value = 4;

        EList myEList = cbe.getExtendedDataElements();
        assertEquals(0, myEList.size());

        ExtendedDataElement ede = cbe.addExtendedDataElementWithShortValue("myede", value);
        myEList = cbe.getExtendedDataElements();
        assertEquals(1, myEList.size());

        List edeList = cbe.getExtendedDataElements("myede");
        ExtendedDataElement cbe_ede = (ExtendedDataElement) edeList.get(0);
        assertEquals(edeList.get(0), ede);
        assertEquals("myede", cbe_ede.getName());
        assertEquals("myede", ede.getName());
        assertNotNull(ede.getType());
        assertFalse(ede.getValues().isEmpty());
        assertNull(ede.getHexValue());
        assertTrue(ede.getChildren().isEmpty());

        ExtendedDataElement tmpEde = (ExtendedDataElement) edeList.get(0);
        assertEquals(ExtendedDataElement.TYPE_SHORT_VALUE, tmpEde.getTypeAsInt());
        assertEquals(value, tmpEde.getValuesAsShort());
    }

    /*
     * Test for MsgDataElement setMsgDataElement(String, String, MsgCatalogToken[], String, String, String, String)
     */
    public void testSetMsgDataElementStringStringMsgCatalogTokenArrayStringStringStringString() {
        String mc = "msg catalog";
        String mci = "msgcatalogid";
        String mcty = "msg catalog type";
        String mi = "msg id";
        String mit = "msg id type";
        String ml = "msg locale";

        MsgCatalogToken mct1 = EventFactory.eINSTANCE.createMsgCatalogToken();
        mct1.setValue("test 1");
        MsgCatalogToken mct2 = EventFactory.eINSTANCE.createMsgCatalogToken();
        mct2.setValue("test 2");
        MsgCatalogToken[] mcto = { mct1, mct2 };

        cbe.setMsgDataElement(mc, mci, mcto, mcty, mi, mit, ml);

        assertNotNull(cbe.getMsgDataElement());
        MsgDataElement mde = cbe.getMsgDataElement();
        assertNotNull(mde);
        assertEquals(mc, mde.getMsgCatalog());
        assertEquals(mci, mde.getMsgCatalogId());

        MsgCatalogToken mct = (MsgCatalogToken) mde.getMsgCatalogTokens().get(0);
        assertEquals("test 1", mct.getValue());
        mct = (MsgCatalogToken) mde.getMsgCatalogTokens().get(1);
        assertEquals("test 2", mct.getValue());

        assertEquals(mcty, mde.getMsgCatalogType());
        assertEquals(mi, mde.getMsgId());
        assertEquals(mit, mde.getMsgIdType());
        assertEquals(ml, mde.getMsgLocale());
    }

    /*
     * Test for MsgDataElement setMsgDataElement(String, String, String[], String, String, String, String)
     */
    public void testSetMsgDataElementStringStringStringArrayStringStringStringString() {
        String mc = "msg catalog";
        String mci = "msgcatalogid";
        String[] mcto = { "token 1", "token 2" };
        String mcty = "msg catalog type";
        String mi = "msg id";
        String mit = "msg id type";
        String ml = "msg locale";

        cbe.setMsgDataElement(mc, mci, mcto, mcty, mi, mit, ml);

        assertNotNull(cbe.getMsgDataElement());
        MsgDataElement mde = cbe.getMsgDataElement();
        assertNotNull(mde);
        assertEquals(mc, mde.getMsgCatalog());
        assertEquals(mci, mde.getMsgCatalogId());
        //assertEquals(mcto, mde.getMsgCatalogTokensAsStrings());
        assertEquals("token 1", mde.getMsgCatalogTokensAsStrings()[0]);
        assertEquals("token 2", mde.getMsgCatalogTokensAsStrings()[1]);
        assertEquals(mcty, mde.getMsgCatalogType());
        assertEquals(mi, mde.getMsgId());
        assertEquals(mit, mde.getMsgIdType());
        assertEquals(ml, mde.getMsgLocale());
    }

    /*
     * Test for List getExtendedDataElements(String)
     */
    public void testGetExtendedDataElementsString() {
        ExtendedDataElement ede1;
        ExtendedDataElement ede1a;
        ExtendedDataElement ede2;
        ExtendedDataElement ede2a;
        ExtendedDataElement ede3;
        ExtendedDataElement ede3a;
        ExtendedDataElement outede1;
        ExtendedDataElement outede2;

        ede1 = cbe.addExtendedDataElement("myEDE", "value1");
        ede1a = (ExtendedDataElement) cbe.getExtendedDataElements().get(0);
        assertEquals(ede1, ede1a);

        ede2 = cbe.addExtendedDataElement("myOtherEDE", "value2");
        ede2a = (ExtendedDataElement) cbe.getExtendedDataElements().get(1);
        assertEquals(ede2, ede2a);

        ede3 = cbe.addExtendedDataElement("myEDE", "value3");
        ede3a = (ExtendedDataElement) cbe.getExtendedDataElements().get(2);
        assertEquals(ede3, ede3a);

        List outEDEs = cbe.getExtendedDataElements("myEDE");
        assertEquals(2, outEDEs.size());
        outede1 = (ExtendedDataElement) outEDEs.get(0);
        outede2 = (ExtendedDataElement) outEDEs.get(1);

        assertEquals(ede1, outede1);
        assertEquals(ede3, outede2);
    }

    /*
     * Test for void setReporterComponentId(String, String, String, String, String, String, String)
     */
    public void testSetReporterComponentIdStringStringStringStringStringStringString() {
        String app = "my Application";
        String comp = "my component";
        String compType = "my comp type";
        String compidtype = "my comp id type";
        String subcomp = "My sub component";
        String loc = "my loc";
        String loctype = "Unknown";

        cbe.setReporterComponentId(app, comp, subcomp, compType, compidtype, loc, loctype);

        assertNotNull(cbe.getReporterComponentId());
        ComponentIdentification sci = cbe.getReporterComponentId();
        assertNotNull(sci);
        assertEquals(app, sci.getApplication());
        assertEquals(comp, sci.getComponent());
        assertEquals(compType, sci.getComponentType());
        assertEquals(compidtype, sci.getComponentIdType());
        assertEquals(subcomp, sci.getSubComponent());
        assertEquals(loc, sci.getLocation());
        assertEquals(loctype, sci.getLocationType());
    }

    /*
     * Test for void setReporterComponentId(String, String, String, String, String, String, String, String, String, String)
     */
    public void testSetReporterComponentIdStringStringStringStringStringStringStringStringStringString() {
        String app = "my Application";
        String comp = "my component";
        String compType = "my comp type";
        String compidtype = "my comp id type";
        String subcomp = "My sub component";
        String loc = "my loc";
        String loctype = "Unknown";
        String ee = "my execution environment";
        String ii = "my instance id";
        String pi = "my process id";
        String ti = "my thread id";

        cbe.setReporterComponentId(app, comp, subcomp, compType, compidtype, ee, ii, loc, loctype, pi, ti);

        assertNotNull(cbe.getReporterComponentId());
        ComponentIdentification sci = cbe.getReporterComponentId();
        assertNotNull(sci);
        assertEquals(app, sci.getApplication());
        assertEquals(comp, sci.getComponent());
        assertEquals(compType, sci.getComponentType());
        assertEquals(compidtype, sci.getComponentIdType());
        assertEquals(subcomp, sci.getSubComponent());
        assertEquals(loc, sci.getLocation());
        assertEquals(loctype, sci.getLocationType());
        assertEquals(ee, sci.getExecutionEnvironment());
        assertEquals(ii, sci.getInstanceId());
        assertEquals(pi, sci.getProcessId());
        assertEquals(ti, sci.getThreadId());
    }

    /*
     * Test for void setSourceComponentId(String, String, String, String, String, String, String, String, String, String)
     */
    public void testSetSourceComponentIdStringStringStringStringStringStringStringStringStringString() {
        String app = "my Application";
        String comp = "my component";
        String compType = "my comp type";
        String compidtype = "my comp id type";
        String subcomp = "My sub component";
        String loc = "my loc";
        String loctype = "Unknown";
        String ee = "my execution environment";
        String ii = "my instance id";
        String pi = "my process id";
        String ti = "my thread id";

        cbe.setSourceComponentId(app, comp, subcomp, compType, compidtype, ee, ii, loc, loctype, pi, ti);

        assertNotNull(cbe.getSourceComponentId());
        ComponentIdentification sci = cbe.getSourceComponentId();
        assertNotNull(sci);
        assertEquals(app, sci.getApplication());
        assertEquals(comp, sci.getComponent());
        assertEquals(compType, sci.getComponentType());
        assertEquals(compidtype, sci.getComponentIdType());
        assertEquals(subcomp, sci.getSubComponent());
        assertEquals(loc, sci.getLocation());
        assertEquals(loctype, sci.getLocationType());
        assertEquals(ee, sci.getExecutionEnvironment());
        assertEquals(ii, sci.getInstanceId());
        assertEquals(pi, sci.getProcessId());
        assertEquals(ti, sci.getThreadId());
    }

    /*
     * Test for void setSourceComponentId(String, String, String, String, String, String)
     */
    public void testSetSourceComponentIdStringStringStringStringStringString() {
        ComponentIdentification cit1 = EventFactory.eINSTANCE.createComponentIdentification();
        String app = "my Application";
        String comp = "my component";
        String compType = "my comp type";
        String compidtype = "my comp id type";
        String subcomp = "My sub component";
        String loc = "my loc";
        String loctype = "Unknown";

        cbe.setSourceComponentId(app, comp, subcomp, compType, compidtype, loc, loctype);

        assertNotNull(cbe.getSourceComponentId());
        ComponentIdentification sci = cbe.getSourceComponentId();
        assertNotNull(sci);
        assertEquals(app, sci.getApplication());
        assertEquals(comp, sci.getComponent());
        assertEquals(compType, sci.getComponentType());
        assertEquals(compidtype, sci.getComponentIdType());
        assertEquals(subcomp, sci.getSubComponent());
        assertEquals(loc, sci.getLocation());
        assertEquals(loctype, sci.getLocationType());
    }

    /**
     * 
     */
    public void testAdapter() {
        Util.assertAdapterNotification(cbe);
    }

    /**
     * 
     */
    public void testUnsupportedFeature() {
        Util.assertUnsupportedFeature(cbe);
    }

    /**
     * Base fixture for thread-safety testing through JUnitPerf.
     * <P>
     * Notice that it uses a local variable for the testing since
     * JUnitPerf calls setUp/test multiple time for the same
     * TestCase instance.
     */
    public void testEqualsThreaded() throws ValidationException {
        CommonBaseEvent cbe2 = Util.generateEvent();
        CommonBaseEvent cbe1 = Util.generateEvent();
        assertEquals(cbe1, cbe2);
    }

    /**
     * Base fixture for thread-safety testing through JUnitPerf.
     * <P>
     * Notice that it uses a local variable for the testing since
     * JUnitPerf calls setUp/test multiple time for the same
     * TestCase instance.
     */
    public void testSerializationThreaded() throws IOException {
        CommonBaseEvent cbe = Util.generateEvent();
        Util.testSerialization(cbe);
    }

    public void testJavadocSample() {
        //ContentHandler mockHandler = new MockHandler("Example");
        EventFactory ef = EventFactoryFactory.createEventFactory();
        CommonBaseEvent cbe = ef.createCommonBaseEvent("CommonBaseEvent", System.currentTimeMillis());

        cbe.setGlobalInstanceId(ef.createGlobalInstanceId());

        cbe.addContextDataElementWithValue("INTEGER", "business_tx", "123491279");

        String[] resolvedEvents = { "CE1234567890123456789012345678901A", "CE1234567890123456789012345678901B" };
        cbe.addAssociatedEvent("TEC", "summary", "CE1234567890123456789012345678901C", resolvedEvents);

        cbe.setSourceComponentId("app", "comp", "subComp", "compType", "compIdType", "loc", "Unknown");

        String[] mcToken = { "token 1", "token 2" };
        cbe.setMsgDataElement("msgCat", "msgCatId", mcToken, "msgCatType", "msgId", "msgIdType", "msgCatLoc");

        Situation sit = ef.createSituation();
        AvailableSituation sitType = EventFactory.eINSTANCE.createAvailableSituation();
        sitType.setAvailabilityDisposition("NOT AVAILABLE");
        sitType.setOperationDisposition("STARTABLE");
        sitType.setProcessingDisposition("MGMTTASK_PROCESS");
        sitType.setReasoningScope("EXTERNAL");
        sit.setCategoryName("AvailableSituation");
        sit.setSituationType(sitType);
        cbe.setSituation(sit);

        //cbe.complete();

        try {
            //System.out.println("xxxxxxxxxxx event: " + cbe);
            cbe.validate();
        }
        catch (ValidationException e) {            
            fail("should not get an exception");
        }
    }

}
