package org.eclipse.hyades.logging.events.cbe.impl.tests;

import java.io.Serializable;
import java.util.Iterator;
import java.util.List;

import javax.naming.NamingException;
import javax.naming.Reference;

import junit.framework.TestCase;

import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.CompletionException;
import org.eclipse.hyades.logging.events.cbe.ContentHandler;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.impl.EventFactoryImpl;
import org.eclipse.hyades.logging.events.cbe.impl.EventFactoryObjectFactory;

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: EventFactoryImplTest.java,v 1.5 2005/04/28 14:49:15 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * 
 * @author Denilson Nastacio
 * @version 1.0.1
 * @since 1.0.1
 */
public class EventFactoryImplTest extends TestCase implements Serializable {

    /**
     * Stream-Unique IDentifier (SUID) of this class.
     * <p>
     * NOTE:  This value MUST not be modified unless an incompatible 
     *        change is introduced.  
     * <p>
     * See the 'Java Object Serialization Specification' for more details.
     * <p>
     * Compatible changes include:
     * <ul>
     * <li>Adding, modifying access or modifying <code>static</code>/<code>transient</code> to non-<code>static</code>/non-<code>transient</code> of fields.</li>
     * <li>Adding or removing classes.</li>
     * <li>Adding or removing <code>writeObject</code>/<code>readObject</code> methods.</li>
     * <li>Adding <code>java.io.Serializable</code> inheritance.</li>
     * </ul>
     * <p>
     * Incompatible changes include:
     * <ul>
     * <li>Deleting, modifying non-<code>static</code>/non-<code>transient</code> to <code>static</code>/<code>transient</code> or modifying the declared type of fields.</li>
     * <li>Modifying the hierarchy or interchanging/removing <code>java.io.Serializable</code>/<code>java.io.Externalizable</code> inheritance of the class.</li>
     * <li>Modifying <code>writeObject</code>/<code>readObject</code> methods.</li>
     * <li>Adding <code>writeReplace</code>/<code>readResolve</code> methods.</li>
     * </ul>
     */
    private static final long serialVersionUID = -7360906523575525171L;

    /**
     * Constructor for EventFactoryImplTest.
     * @param arg0
     */
    public EventFactoryImplTest(String arg0) {
        super(arg0);
    }

    /**
     * 
     */
    public void testInterface() {
        EventFactory efi = EventFactory.eINSTANCE;
        efi.getEventPackage();
        assertNotNull(efi.createAssociatedEvent());
        assertNotNull(efi.createAssociationEngine());
        assertNotNull(efi.createAvailableSituation());
        assertNotNull(efi.createCommonBaseEvent());
        assertNotNull(efi.createComponentIdentification());
        assertNotNull(efi.createConfigureSituation());
        assertNotNull(efi.createConnectSituation());
        assertNotNull(efi.createContextDataElement());
        assertNotNull(efi.createCreateSituation());
        assertNotNull(efi.createDependencySituation());
        assertNotNull(efi.createDestroySituation());
        assertNotNull(efi.createExtendedDataElement());
        assertNotNull(efi.createFeatureSituation());
        String guid = efi.createGlobalInstanceId();
        assertTrue("Guid must start with CE: " + guid, guid.startsWith("CE"));
        assertNotNull(efi.createMsgCatalogToken());
        assertNotNull(efi.createMsgDataElement());
        assertNotNull(efi.createOtherSituation());
        assertNotNull(efi.createReportSituation());
        assertNotNull(efi.createRequestSituation());
        assertNotNull(efi.createSituation());
        assertNotNull(efi.createStartSituation());
        assertNotNull(efi.createStopSituation());

        CommonBaseEvent cbe = efi.createCommonBaseEvent("A");
        assertEquals("A", cbe.getExtensionName());
        cbe = efi.createCommonBaseEvent("A", 5);
        assertEquals("A", cbe.getExtensionName());
        assertEquals(5, cbe.getCreationTimeAsLong());
    }

    /**
     * Tests the <code>create(EClass</code> method.
     */
    public void testEPackage() {
        //        EventFactory eFx = EventFactory.eINSTANCE;
        EPackage ePck = EventFactoryImpl.getPackage();
        List l = ePck.eContents();
        Iterator i = l.iterator();
        while (i.hasNext()) {
            EClass c = (EClass) i.next();
            if (c.getName().equals("SituationType") || c.getName().equals("Serializable")) {
                try {
                    EventFactory.eINSTANCE.create(c);
                    fail(c.getName() + " is not an instantiatable class");
                }
                catch (IllegalArgumentException e) {
                    // expected
                }
            }
            else {
                EventFactory.eINSTANCE.create(c);
            }
        }
    }

    /**
     * Testing the contents of references returned by EventFactory.
     */
    public void testGetReference() {
        final String EXTENSION_NAME = "Example";
        ContentHandler mockHandler = new MockHandler(EXTENSION_NAME);
        EventFactoryImpl efi = new EventFactoryImpl();
        efi.setContentHandler(mockHandler);

        try {
            Reference ref = efi.getReference();
            assertEquals(EventFactoryObjectFactory.class.getName(), ref.getFactoryClassName());
            assertEquals(EventFactoryImpl.class.getName(), ref.getClassName());
            assertNotNull(ref.get(0));
        }
        catch (NamingException e) {
            fail("The reference should be valid.");
        }
    }

    /**
     * Testing whether the EventFactory is a JNDI citizen.
     *
     */
    public void testJndiSupport() {
        EventFactory efx = EventFactory.eINSTANCE;
        assertTrue("EventFactoryImpl must extend java.io.Serializable", efx instanceof java.io.Serializable);
        assertTrue("EventFactoryImpl must extend javax.naming.Referenceable", efx instanceof javax.naming.Referenceable);
    }

    /**
     * Tests whether the JNDI support will reject a non-serializable mediation handler. 
     */
    public void testNonSerializableHandler() {
        ContentHandler mockHandler = new ContentHandler() {
            public void completeEvent(CommonBaseEvent event) throws CompletionException {
                event.setExtensionName("Example");
            }
        };
        EventFactoryImpl efi = new EventFactoryImpl();
        efi.setContentHandler(mockHandler);
        try {
            Reference ref = efi.getReference();
            fail("Cannot accept non-serializable mediation handler");
        }
        catch (NamingException e) {
            Util.assertExceptionId("IWAT0188E", e);
        }
    }

    /*
     * Private methods. 
     */

    /**
     * 
     */
    public class MockHandler implements ContentHandler, Serializable {
        
        /**
         * Stream-Unique IDentifier (SUID) of this class.
         * <p>
         * NOTE:  This value MUST not be modified unless an incompatible 
         *        change is introduced.  
         * <p>
         * See the 'Java Object Serialization Specification' for more details.
         * <p>
         * Compatible changes include:
         * <ul>
         * <li>Adding, modifying access or modifying <code>static</code>/<code>transient</code> to non-<code>static</code>/non-<code>transient</code> of fields.</li>
         * <li>Adding or removing classes.</li>
         * <li>Adding or removing <code>writeObject</code>/<code>readObject</code> methods.</li>
         * <li>Adding <code>java.io.Serializable</code> inheritance.</li>
         * </ul>
         * <p>
         * Incompatible changes include:
         * <ul>
         * <li>Deleting, modifying non-<code>static</code>/non-<code>transient</code> to <code>static</code>/<code>transient</code> or modifying the declared type of fields.</li>
         * <li>Modifying the hierarchy or interchanging/removing <code>java.io.Serializable</code>/<code>java.io.Externalizable</code> inheritance of the class.</li>
         * <li>Modifying <code>writeObject</code>/<code>readObject</code> methods.</li>
         * <li>Adding <code>writeReplace</code>/<code>readResolve</code> methods.</li>
         * </ul>
         */
        private static final long serialVersionUID = 4726907462543421506L;

        private String extensionName;

        public MockHandler(String name) {
            extensionName = name;
        }
        public void completeEvent(CommonBaseEvent event) throws CompletionException {
            event.setExtensionName(extensionName);
        }
    }
}
