package org.eclipse.hyades.logging.events.cbe.tests;

import junit.framework.TestCase;

import org.eclipse.hyades.logging.events.cbe.EventException;
import org.eclipse.hyades.logging.events.cbe.util.EventHelpers;

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: EventExceptionTest.java,v 1.6 2005/03/10 14:51:13 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * EventExceptionTest class.
 * <p>
 * 
 * 
 * @author Denilson Nastacio
 * @author Paul E. Slauenwhite
 * @version 1.0.1
 * @since 1.0.1
 */
public class EventExceptionTest extends TestCase {
    /**
     * Constructor for EventExceptionTest.
     * @param arg0
     */
    public EventExceptionTest(String arg0) {
        super(arg0);
    }

    /*
     * Test for void EventException()
     */
    public void testEventException() {
        
        EventException e  = new EventException();
        
        assertEquals(e.getMessage(),"null");
        
        assertNull(e.getCause());
    }

    /*
     * Test for void EventException(String, String)
     */
    public void testEventExceptionStringString() {
        
        EventException e = new EventException("LOG_INVALID_JNDI_REFERENCE_EXC_","org.eclipse.hyades.logging.core.properties.plugin");

        assertExceptionMsg(e, "LOG_INVALID_JNDI_REFERENCE_EXC_", null);

        assertNull(e.getCause());
    }

    /*
     * Test for void EventException(String, String) (missing resource bundle)
     */
    public void testEventExceptionStringStringMissingResourceBundle() {
        
        EventException e = new EventException("LOG_INVALID_JNDI_REFERENCE_EXC_","MISSING_RESOURCE_BUNDLE");

        assertEquals("Can't find bundle for base name MISSING_RESOURCE_BUNDLE, locale en : LOG_INVALID_JNDI_REFERENCE_EXC_", e.getMessage());

        assertNull(e.getCause());
    }

    /*
     * Test for void EventException(String, String) (missing resource key)
     */
    public void testEventExceptionStringStringMissingResourceKey() {
        
        EventException e = new EventException("MISSING_RESOURCE_KEY","org.eclipse.hyades.logging.core.properties.plugin");

        assertEquals("Can't find resource for bundle java.util.PropertyResourceBundle, key MISSING_RESOURCE_KEY : MISSING_RESOURCE_KEY", e.getMessage());

        assertNull(e.getCause());
    }

    /*
     * Test for void EventException(String, String, Object[])
     */
    public void testEventExceptionStringStringObjectArray() {
        
        String[] parameters = new String[]{"valueOne"};
        EventException e = new EventException("LOG_INVALID_JNDI_REFERENCE_EXC_","org.eclipse.hyades.logging.core.properties.plugin", parameters);

        assertExceptionMsg(e, "LOG_INVALID_JNDI_REFERENCE_EXC_", parameters);

        assertNull(e.getCause());
    }

    /*
     * Test for void EventException(String, String, Object[]) (missing resource bundle)
     */
    public void testEventExceptionStringStringObjectArrayMissingResourceBundle() {
        
        EventException e = new EventException("LOG_INVALID_JNDI_REFERENCE_EXC_","MISSING_RESOURCE_BUNDLE", new String[]{"valueOne"});

        assertEquals("Can't find bundle for base name MISSING_RESOURCE_BUNDLE, locale en : LOG_INVALID_JNDI_REFERENCE_EXC_ {valueOne}", e.getMessage());

        assertNull(e.getCause());
    }

    /*
     * Test for void EventException(String, String, Object[]) (missing resource key)
     */
    public void testEventExceptionStringStringObjectArrayMissingResourceKey() {
        
        EventException e = new EventException("MISSING_RESOURCE_KEY","org.eclipse.hyades.logging.core.properties.plugin", new String[]{"valueOne"});

        assertEquals("Can't find resource for bundle java.util.PropertyResourceBundle, key MISSING_RESOURCE_KEY : MISSING_RESOURCE_KEY {valueOne}", e.getMessage());

        assertNull(e.getCause());
    }

   /*
    * Test for void EventException(String, String, Object[], Throwable)
    */
   public void testEventExceptionStringStringObjectArrayThrowable() {
       
       String[] parameters = new String[]{"valueOne"};
       Exception cause = new Exception("Linked Exception");
       EventException e = new EventException("LOG_INVALID_JNDI_REFERENCE_EXC_","org.eclipse.hyades.logging.core.properties.plugin", parameters, cause);

       assertEquals("IWAT0187E The reference obtained from the specified JNDI name is not a reference to an event factory.\nJNDI name: valueOne : Linked Exception", e.getMessage());

       assertEquals(cause, e.getCause());
   }

   /*
    * Test for void EventException(String, String, Object[], Throwable) (missing resource bundle)
    */
   public void testEventExceptionStringStringObjectArrayThrowableMissingResourceBundle() {
       
       Exception cause = new Exception("Linked Exception");
       EventException e = new EventException("LOG_INVALID_JNDI_REFERENCE_EXC_","MISSING_RESOURCE_BUNDLE", new String[]{"valueOne"}, cause);

       assertEquals("Can't find bundle for base name MISSING_RESOURCE_BUNDLE, locale en : LOG_INVALID_JNDI_REFERENCE_EXC_ {valueOne} : Linked Exception", e.getMessage());

       assertEquals(cause, e.getCause());
   }

   /*
    * Test for void EventException(String, String, Object[], Throwable) (missing resource key)
    */
   public void testEventExceptionStringStringObjectArrayThrowableMissingResourceKey() {
       
       Exception cause = new Exception("Linked Exception");
       EventException e = new EventException("MISSING_RESOURCE_KEY","org.eclipse.hyades.logging.core.properties.plugin", new String[]{"valueOne"},cause);

       assertEquals("Can't find resource for bundle java.util.PropertyResourceBundle, key MISSING_RESOURCE_KEY : MISSING_RESOURCE_KEY {valueOne} : Linked Exception", e.getMessage());

       assertEquals(cause, e.getCause());
   }
   
   /*
    * Test for void EventException(String)
    */
   public void testEventExceptionString() {

       EventException e = new EventException(EventHelpers.getString("LOG_INVALID_JNDI_REFERENCE_EXC_","a" ));
       
       assertExceptionMsg(e, "LOG_INVALID_JNDI_REFERENCE_EXC_", new Object[] { "a" });
       
       assertNull(e.getCause());
   }

    /*
     * Test for void EventException(String, Throwable)
     */
    public void testEventExceptionStringThrowable() {

        Exception cause = new Exception("Linked Exception");

        EventException e = new EventException(EventHelpers.getString("LOG_INVALID_JNDI_REFERENCE_EXC_","a" ), cause);
        
        assertEquals(e.getMessage(),"IWAT0187E The reference obtained from the specified JNDI name is not a reference to an event factory.\nJNDI name: a : Linked Exception");
        
        assertEquals(cause, e.getCause());
    }

    /*
     * Test for void EventException(Throwable)
     */
    public void testEventExceptionThrowable() {

        Exception linkedException = new Exception();

        EventException e = new EventException(linkedException);

        assertEquals(e.getMessage(),"null");
        
        assertEquals(linkedException, e.getCause());
    }

    /*
     * Package methods.
     */

    /**
     * Verifies whether the exception contains a message that matches the
     * resource key in the event resource bundle.
     */
    static void assertExceptionMsg(EventException e, String msgKey, Object[] params) {
        assertEquals(EventHelpers.getString(msgKey,params), e.getLocalizedMessage());
    }
}
