package org.eclipse.hyades.logging.events.cbe.tests;

import junit.framework.TestCase;

import org.eclipse.hyades.logging.events.cbe.ValidationException;
import org.eclipse.hyades.logging.events.cbe.util.EventHelpers;

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ValidationExceptionTest.java,v 1.6 2005/03/10 14:51:13 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * ValidationExceptionTest class.
 * <p>
 * 
 * 
 * @author Denilson Nastacio
 * @author Paul E. Slauenwhite
 * @version 1.0.1
 * @since 1.0.1
 */
public class ValidationExceptionTest extends TestCase {
    /**
     * Constructor for ValidationExceptionTest.
     * @param arg0
     */
    public ValidationExceptionTest(String arg0) {
        super(arg0);
    }

    /*
     * Test for void ValidationException()
     */
    public void testValidationException() {
        
        ValidationException e  = new ValidationException();
        
        assertEquals(e.getMessage(),"null");
        
        assertNull(e.getCause());
    }

    /*
     * Test for void ValidationException(String, String)
     */
    public void testValidationExceptionStringString() {
        
        ValidationException e = new ValidationException("LOG_INVALID_JNDI_REFERENCE_EXC_","org.eclipse.hyades.logging.core.properties.plugin");

        assertExceptionMsg(e, "LOG_INVALID_JNDI_REFERENCE_EXC_", null);

        assertNull(e.getCause());
    }

    /*
     * Test for void ValidationException(String, String) (missing resource bundle)
     */
    public void testValidationExceptionStringStringMissingResourceBundle() {
        
        ValidationException e = new ValidationException("LOG_INVALID_JNDI_REFERENCE_EXC_","MISSING_RESOURCE_BUNDLE");

        assertEquals("Can't find bundle for base name MISSING_RESOURCE_BUNDLE, locale en : LOG_INVALID_JNDI_REFERENCE_EXC_", e.getMessage());

        assertNull(e.getCause());
    }

    /*
     * Test for void ValidationException(String, String) (missing resource key)
     */
    public void testValidationExceptionStringStringMissingResourceKey() {
        
        ValidationException e = new ValidationException("MISSING_RESOURCE_KEY","org.eclipse.hyades.logging.core.properties.plugin");

        assertEquals("Can't find resource for bundle java.util.PropertyResourceBundle, key MISSING_RESOURCE_KEY : MISSING_RESOURCE_KEY", e.getMessage());

        assertNull(e.getCause());
    }

    /*
     * Test for void ValidationException(String, String, Object[])
     */
    public void testValidationExceptionStringStringObjectArray() {
        
        String[] parameters = new String[]{"valueOne"};
        ValidationException e = new ValidationException("LOG_INVALID_JNDI_REFERENCE_EXC_","org.eclipse.hyades.logging.core.properties.plugin", parameters);

        assertExceptionMsg(e, "LOG_INVALID_JNDI_REFERENCE_EXC_", parameters);

        assertNull(e.getCause());
    }

    /*
     * Test for void ValidationException(String, String, Object[]) (missing resource bundle)
     */
    public void testValidationExceptionStringStringObjectArrayMissingResourceBundle() {
        
        ValidationException e = new ValidationException("LOG_INVALID_JNDI_REFERENCE_EXC_","MISSING_RESOURCE_BUNDLE", new String[]{"valueOne"});

        assertEquals("Can't find bundle for base name MISSING_RESOURCE_BUNDLE, locale en : LOG_INVALID_JNDI_REFERENCE_EXC_ {valueOne}", e.getMessage());

        assertNull(e.getCause());
    }

    /*
     * Test for void ValidationException(String, String, Object[]) (missing resource key)
     */
    public void testValidationExceptionStringStringObjectArrayMissingResourceKey() {
        
        ValidationException e = new ValidationException("MISSING_RESOURCE_KEY","org.eclipse.hyades.logging.core.properties.plugin", new String[]{"valueOne"});

        assertEquals("Can't find resource for bundle java.util.PropertyResourceBundle, key MISSING_RESOURCE_KEY : MISSING_RESOURCE_KEY {valueOne}", e.getMessage());

        assertNull(e.getCause());
    }

   /*
    * Test for void ValidationException(String, String, Object[], Throwable)
    */
   public void testValidationExceptionStringStringObjectArrayThrowable() {
       
       String[] parameters = new String[]{"valueOne"};
       Exception cause = new Exception("Linked Exception");
       ValidationException e = new ValidationException("LOG_INVALID_JNDI_REFERENCE_EXC_","org.eclipse.hyades.logging.core.properties.plugin", parameters, cause);

       assertEquals("IWAT0187E The reference obtained from the specified JNDI name is not a reference to an event factory.\nJNDI name: valueOne : Linked Exception", e.getMessage());

       assertEquals(cause, e.getCause());
   }

   /*
    * Test for void ValidationException(String, String, Object[], Throwable) (missing resource bundle)
    */
   public void testValidationExceptionStringStringObjectArrayThrowableMissingResourceBundle() {
       
       Exception cause = new Exception("Linked Exception");
       ValidationException e = new ValidationException("LOG_INVALID_JNDI_REFERENCE_EXC_","MISSING_RESOURCE_BUNDLE", new String[]{"valueOne"}, cause);

       assertEquals("Can't find bundle for base name MISSING_RESOURCE_BUNDLE, locale en : LOG_INVALID_JNDI_REFERENCE_EXC_ {valueOne} : Linked Exception", e.getMessage());

       assertEquals(cause, e.getCause());
   }

   /*
    * Test for void ValidationException(String, String, Object[], Throwable) (missing resource key)
    */
   public void testValidationExceptionStringStringObjectArrayThrowableMissingResourceKey() {
       
       Exception cause = new Exception("Linked Exception");
       ValidationException e = new ValidationException("MISSING_RESOURCE_KEY","org.eclipse.hyades.logging.core.properties.plugin", new String[]{"valueOne"},cause);

       assertEquals("Can't find resource for bundle java.util.PropertyResourceBundle, key MISSING_RESOURCE_KEY : MISSING_RESOURCE_KEY {valueOne} : Linked Exception", e.getMessage());

       assertEquals(cause, e.getCause());
   }
   
   /*
    * Test for void ValidationException(String)
    */
   public void testValidationExceptionString() {

       ValidationException e = new ValidationException(EventHelpers.getString("LOG_INVALID_JNDI_REFERENCE_EXC_","a" ));
       
       assertExceptionMsg(e, "LOG_INVALID_JNDI_REFERENCE_EXC_", new Object[] { "a" });
       
       assertNull(e.getCause());
   }

    /*
     * Test for void ValidationException(String, Throwable)
     */
    public void testValidationExceptionStringThrowable() {

        Exception cause = new Exception("Linked Exception");

        ValidationException e = new ValidationException(EventHelpers.getString("LOG_INVALID_JNDI_REFERENCE_EXC_","a" ), cause);
        
        assertEquals(e.getMessage(),"IWAT0187E The reference obtained from the specified JNDI name is not a reference to an event factory.\nJNDI name: a : Linked Exception");
        
        assertEquals(cause, e.getCause());
    }

    /*
     * Test for void ValidationException(Throwable)
     */
    public void testValidationExceptionThrowable() {

        Exception linkedException = new Exception();

        ValidationException e = new ValidationException(linkedException);

        assertEquals(e.getMessage(),"null");
        
        assertEquals(linkedException, e.getCause());
    }

    /*
     * Package methods.
     */

    /**
     * Verifies whether the exception contains a message that matches the
     * resource key in the event resource bundle.
     */
    static void assertExceptionMsg(ValidationException e, String msgKey, Object[] params) {
        assertEquals(EventHelpers.getString(msgKey,params), e.getLocalizedMessage());
    }
}
