/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ExecutionComponentFactoryImpl.java,v 1.9 2005/06/03 03:18:53 sschneid Exp $
 * 
 * Contributors: 
 * IBM Rational - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.execution.local;

import java.io.IOException;
import java.util.HashMap;

import org.eclipse.hyades.execution.core.ExecutionComponentStateChangeEvent;
import org.eclipse.hyades.execution.core.IExecutionComponent;
import org.eclipse.hyades.execution.core.IExecutionComponentFactory;
import org.eclipse.hyades.execution.core.IExecutionComponentStateChangeListener;
import org.eclipse.hyades.execution.core.ISession;
import org.eclipse.hyades.execution.invocation.CallData;
import org.eclipse.hyades.execution.invocation.IRemoteObject;
import org.eclipse.hyades.execution.invocation.Marshaller;
import org.eclipse.hyades.execution.invocation.RemoteInvocationException;
import org.eclipse.hyades.execution.invocation.ReturnData;
import org.eclipse.hyades.internal.execution.local.common.BinaryCustomCommand;

/**
 * The execution component factory.
 * 
 * Each agent VM will have an instance of the factory. The factory acts as
 * it's own stub and is capable of remotely invoking the appropriate methods.
 * 
 * The factory's state is initialized when the class is loaded.
 * Something on remote agents needs to initiate the loading of this class.
 * A simple call to <code>RemoteFactory.getInstance()</code> will do.
 */
public class ExecutionComponentFactoryImpl implements IExecutionComponentFactory, IRemoteObject  {
	
	protected static HashMap instances=new HashMap();
	
	protected HashMap components=new HashMap();
	
	protected ISession sessionContext;
	
	protected ExecutionComponentFactoryImpl(ISession session){
		sessionContext=session;
		Marshaller.addInstanceToMap(getUniqueId(), this);
	}
	
	/**
     * Get the component factory in this VM. Each VM has it's own factory.
     * Each factory shares the same, well known id and provides the bootstrap
     * for remote object instantiation.
     * 
     * @return
     */
    public static IExecutionComponentFactory getInstance(final ISession session) {
    	IExecutionComponentFactory factory;
    	synchronized(instances) {
    		factory=(IExecutionComponentFactory)instances.get(session);
    		if(factory==null) {
    			factory=new ExecutionComponentFactoryImpl(session);
    			instances.put(session, factory);
    			session.addExecutionComponentStateChangeListener(
    		        new IExecutionComponentStateChangeListener() {
    		            public void stateChanged(ExecutionComponentStateChangeEvent newState) {
    		                if (newState.getState() == IExecutionComponent.DEAD) {
    		                    ExecutionComponentFactoryImpl.removeInstance(session);
    		                }
                        }
    		        }
    			);
    		}
    		return factory;
    	}
    }
	
	public ISession getSessionContext() {
		return sessionContext;
	}
	
	/**
	 * @see org.eclipse.hyades.execution.invocation.IRemoteObject#setSessionContext(org.eclipse.hyades.execution.core.ISession)
	 */
	public void setSessionContext(ISession session) {
		throw new UnsupportedOperationException("The session context of the factory should not be set via this method");
	}

	/**
	 * @see org.eclipse.hyades.execution.invocation.RemoteObject#getUniqueId()
	 */
	public Integer getUniqueId() {
		return new Integer(-1);
	}
	
	/**
	 * Remove the session execution component factory instance for the specified session
	 * 
	 * @param session the factory to remove from the instance map
	 */
	public static void removeInstance(ISession session) {
		synchronized(instances) {
			if (instances.containsKey(session)) {
			    ExecutionComponentFactoryImpl factory = (ExecutionComponentFactoryImpl) instances.get(session);
			    factory.sessionContext = null;
				instances.remove(session);
			}
		}
	}

	/** 
	 * @see org.eclipse.hyades.execution.core.IExecutionComponentFactory#createExecutionComponentByType(java.lang.String)
	 */
	public synchronized IExecutionComponent createExecutionComponentByType(String type) {
		ExecutionComponentStub stubInstance=null;
		
		/* create the local instance first */
		ClassRelationship rel=(ClassRelationship)components.get(type);
		if(rel!=null) {
			try {
				Class impl=rel.impl;
				Class stub=rel.stubClass;
				IExecutionComponent implInstance=(IExecutionComponent)impl.newInstance();
				stubInstance=(ExecutionComponentStub)stub.newInstance();
				stubInstance.setDelegate(implInstance);
				stubInstance.setSessionContext(sessionContext);
				
				Marshaller.addInstanceToMap(stubInstance.getUniqueId(), stubInstance);
				stubInstance.init();
			}
			catch(Throwable e) {
				throw new RemoteInvocationException(e);
			}
		}
		else {
			throw new RemoteInvocationException("Factory not configured for this type");
		}
		
		/* if the local instance creation was succesfull then create the remote instance */
		if(stubInstance!=null) {
			
			ReturnData invokeRtn = delegateRemoteCall(
			new Class[]{Integer.class, String.class},
				new Object[]{((IRemoteObject)stubInstance).getUniqueId(), type},
				"createExecutionComponentByType");

			Object rtnValue = invokeRtn.getReturnValue();
			if(invokeRtn.isError())
				throw new RemoteInvocationException((Throwable)invokeRtn.getReturnValue());
			if (!(rtnValue instanceof IExecutionComponent)) {
				throw new RemoteInvocationException("Return value is incorrect type");
			}		
		}
		return stubInstance;
	}
	
	/**
	 * Create a remote object and return its stub.
	 * 
	 * @param type
	 * @return
	 */
	public synchronized IRemoteObject createRemoteObjectByType(String type){
		IRemoteObject stubInstance=null;
		
		/* create the local instance first */
		ClassRelationship rel=(ClassRelationship)components.get(type);
		if(rel!=null) {
			try {
				Class impl=rel.impl;
				Class stub=rel.stubClass;
				Object implInstance=impl.newInstance();
				stubInstance=(IRemoteObject)stub.newInstance();
				stubInstance.setDelegate(implInstance);
				stubInstance.setSessionContext(sessionContext);
				
				Marshaller.addInstanceToMap(stubInstance.getUniqueId(), stubInstance);
				stubInstance.init();
			}
			catch(Throwable e) {
				throw new RemoteInvocationException(e);
			}
		}
		else {
			throw new RemoteInvocationException("Factory not configured for type \""+type+"\"");
		}
		
		/* if the local instance creation was succesfull then create the remote instance */
		if(stubInstance!=null) {
			
			ReturnData invokeRtn = delegateRemoteCall(
				new Class[]{Integer.class, String.class},
				new Object[]{((IRemoteObject)stubInstance).getUniqueId(), type},
				"createRemoteObjectByType");

			Object rtnValue = invokeRtn.getReturnValue();
			if(invokeRtn.isError())
				throw new RemoteInvocationException((Throwable)invokeRtn.getReturnValue());
			if (!(rtnValue instanceof IRemoteObject)) {
				throw new RemoteInvocationException("Return value is not an instance of IRemoteObject");
			}		
		}
		return stubInstance;
	}
	
	/**
	 * @see org.eclipse.hyades.execution.core.IExecutionComponentFactory#addExecutionComponent(java.lang.String, java.lang.String)
	 */
	public void addExecutionComponent(String type, String implClass) throws ClassNotFoundException {
		
	
		/* Locate the impl class locally */
		Class clazz=Class.forName(
			implClass, true, this.getClass().getClassLoader()/*new UpdateablePathClassLoader(getClass().getClassLoader())*/);
		
		/* If we have the impl class, lets try and find it remotely */
		ReturnData invokeRtn = delegateRemoteCall(
		new Class[]{String.class, String.class},
			new Object[]{type, implClass},
			"addExecutionComponent");

		Object rtnValue = invokeRtn.getReturnValue();
		if(invokeRtn.isError())
			throw new RemoteInvocationException((Throwable)invokeRtn.getReturnValue());
		
		/* add this class to our hashmap */
		ClassRelationship rel=new ClassRelationship();
		rel.impl=clazz;
		components.put(type, rel);
		
	}
	
	/**
	 * @see org.eclipse.hyades.execution.core.IExecutionComponentFactory#addStub(java.lang.Class, java.lang.String)
	 */
	public void addStub(String type, String stubClass) throws ClassNotFoundException {
		/* look for the stub class locally */
		Class clazz=Class.forName(
			stubClass, true, this.getClass().getClassLoader()/*new UpdateablePathClassLoader(getClass().getClassLoader())*/);
		
		/* Update the components hashmap.  */
		ClassRelationship rel=(ClassRelationship)components.get(type);
		if(rel==null) {
			throw new ClassNotFoundException("Impl class not specified with addExecutionComponent()");
		} 
		
		rel.stubClass=clazz;
	
	}
	
	/**
	 * @see org.eclipse.hyades.execution.core.IExecutionComponentFactory#addSkeleton(java.lang.Class, java.lang.String)
	 */
	public void addSkeleton(String type, String skeletonClass) throws ClassNotFoundException {
	
		/* Check the remote side for the class */
		ReturnData invokeRtn = delegateRemoteCall(
		new Class[]{String.class, String.class},
			new Object[]{type, skeletonClass},
			"addSkeleton");

		Object rtnValue = invokeRtn.getReturnValue();
		if(invokeRtn.isError())
			throw new RemoteInvocationException((Throwable)invokeRtn.getReturnValue());
	
	}


	/**
	 * @see org.eclipse.hyades.execution.invocation.RemoteObject#delegateRemoteCall()
	 */
	public ReturnData delegateRemoteCall(Class[] argTypes, Object[] callArgs, String call)
		throws RemoteInvocationException {

		// Gather the necessary data, marshal it and submit it for remote
		// invocation.
		CallData callData = new CallData(getUniqueId(), argTypes, callArgs, call);
	
		/* Create our custom command */
		BinaryCustomCommand command = new BinaryCustomCommand();
		try {
			command.setData(Marshaller.marshalMethodCall(callData));
		}
		catch ( IOException e ) {
			throw new RemoteInvocationException(e);
		}
		
		try {
			((SessionStub)sessionContext).invokeRemote(command);
			
		}
		catch(ClassCastException e) {
			
		}
	
		// Wait for notification that the remote call has returned, unmarshal
		// the return data, return the result of the remote call.
		ReturnData returnData = null;
		while (true) {
			Marshaller.waitForReturnData();
			ReturnData tmp = Marshaller.peekReturnValue();
			if ( callData.isForSameCallAs(tmp) ) {
				returnData = Marshaller.unqueueReturnValue();
				break;
			}
		}

		return returnData;
	}
	
	/**
	 * This implementation returns the factory. The factory is its own delegate.
	 * @see org.eclipse.hyades.execution.invocation.RemoteObject#getDelegate()
	 */
	public Object getDelegate() {
		return this;
	}

	/**
	 * This implementation does nothing. The factory is its own delegate.
	 * @see org.eclipse.hyades.execution.invocation.IRemoteObject#setDelegate(java.lang.Object)
	 */
	public void setDelegate(Object delegate) {}

	/**
	 * This implementation does nothing.
	 * @see org.eclipse.hyades.execution.invocation.IRemoteObject#init()
	 */
	public void init() {}
	
	protected void finalize() throws Throwable {
	    this.sessionContext = null;
	    super.finalize();
	}
	
}

class ClassRelationship {
	public Class impl;	
	public Class stubClass;
}
