/********************************************************************** 
 * Copyright (c) 2005 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: FileManagerExtendedImpl.java,v 1.4 2005/06/07 19:54:09 sschneid Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.execution.local.file;

import java.io.IOException;
import java.util.Iterator;

import org.eclipse.core.internal.runtime.Assert;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.hyades.execution.core.file.IFileLocation;
import org.eclipse.hyades.execution.core.file.IFileManager;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended;
import org.eclipse.hyades.internal.execution.core.file.FileServerCommandFactory;
import org.eclipse.hyades.internal.execution.core.file.IFileServerCommand;
import org.eclipse.hyades.internal.execution.core.file.IFileServerCommandFactory;
import org.eclipse.hyades.internal.execution.core.file.IQueryServerStatusCommand;
import org.eclipse.hyades.internal.execution.local.common.CommandElement;
import org.eclipse.hyades.internal.execution.local.common.ControlMessage;
import org.eclipse.hyades.internal.execution.local.common.ManageFileCommand;
import org.eclipse.hyades.internal.execution.local.common.ResourceLocation;
import org.eclipse.hyades.internal.execution.local.control.CommandHandler;
import org.eclipse.hyades.internal.execution.local.control.Connection;
import org.eclipse.hyades.internal.execution.local.control.Node;

/**
 * File manager extended implementation class used for file manager operations
 * and intended to be created via a factory such as the file manager factory.
 * This implements extends the legacy file manager implementation and implements
 * a new interface that extends from the legacy interface. Refer to the file
 * manager extended interface for more details and documentation. Also refer to
 * the file manager extended test within the org.eclipse.hyades.test.
 * execution.local package. Argument checking asserts need to be switched out to
 * throw appropriate exceptions such as NullPointerException and
 * IllegalArgumentException.
 * 
 * @author Scott E. Schneider
 */
class FileManagerExtendedImpl implements IFileManagerExtended {

    private static final int DEFAULT_TIMEOUT = 60000;

    /**
     * To save the cost of newing off many null progress monitor instances, one
     * is created and used and treated as immutable (the state will never be
     * read).
     */
    private static final NullProgressMonitor NULL_PROGRESS_MONITOR;

    /**
     * Initialize singleton kept instance of null progress monitor, within the
     * file manager extended implementation class to be reused for all cases
     * where a progress monitor is not specified.
     */
    static {
        NULL_PROGRESS_MONITOR = new NullProgressMonitor();
    }

    private final IFileServerCommandFactory factory;

    /**
     * Creates a new file manager extended implementation instance with the
     * given connection
     * 
     * @param connection
     *            the connection used for file manager operations
     */
    FileManagerExtendedImpl(Connection connection) {
        this.factory = FileServerCommandFactory.getInstance(this.queryFileServerLocation(connection));
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#deleteFile(
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie)
     */
    public void deleteFile(Cookie cookie) throws IOException {
        this.deleteFile(cookie, FileManagerExtendedImpl.NULL_PROGRESS_MONITOR);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#deleteFile(
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie,
     *      org.eclipse.core.runtime.IProgressMonitor)
     */
    public void deleteFile(Cookie cookie, IProgressMonitor monitor) throws IOException {
        Assert.isNotNull(cookie);
        Assert.isNotNull(monitor);
        throw new UnsupportedOperationException();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#deleteFile(
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList)
     */
    public void deleteFile(FileIdentifierList remoteIdentifiers) throws IOException {
        this.deleteFile(remoteIdentifiers, FileManagerExtendedImpl.NULL_PROGRESS_MONITOR);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#deleteFile(
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
     *      org.eclipse.core.runtime.IProgressMonitor)
     */
    public void deleteFile(FileIdentifierList remoteIdentifiers, IProgressMonitor monitor) throws IOException {
        Assert.isNotNull(remoteIdentifiers);
        Assert.isNotNull(monitor);
        IFileServerCommand command = this.factory.createDeleteFileCommand(remoteIdentifiers, monitor);
        command.execute();
        command.dispose();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.execution.core.file.IFileManager#deleteFile(java.lang.String)
     */
    public void deleteFile(String remoteName) throws IOException {
        this.deleteFile(FileIdentifierList.create(remoteName));
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#getFile(
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList)
     */
    public void getFile(FileIdentifierList localIdentifiers, FileIdentifierList remoteIdentifiers) throws IOException {
        this.getFile(localIdentifiers, remoteIdentifiers, FileManagerExtendedImpl.NULL_PROGRESS_MONITOR);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#getFile(
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
     *      org.eclipse.core.runtime.IProgressMonitor)
     */
    public void getFile(FileIdentifierList localIdentifiers, FileIdentifierList remoteIdentifiers, IProgressMonitor monitor)
            throws IOException {
        this.getFile(localIdentifiers, remoteIdentifiers, IFileManagerExtended.Option.NONE, monitor);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#getFile(
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option[])
     */
    public void getFile(FileIdentifierList localIdentifiers, FileIdentifierList remoteIdentifiers, Option[] options)
            throws IOException {
        this.getFile(localIdentifiers, remoteIdentifiers, options, FileManagerExtendedImpl.NULL_PROGRESS_MONITOR);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#getFile(
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option[],
     *      org.eclipse.core.runtime.IProgressMonitor)
     */
    public void getFile(FileIdentifierList localIdentifiers, FileIdentifierList remoteIdentifiers, Option[] options,
            IProgressMonitor monitor) throws IOException {
        Assert.isNotNull(localIdentifiers);
        Assert.isNotNull(remoteIdentifiers);
        Assert.isNotNull(options);
        Assert.isNotNull(monitor);
        Assert.isTrue(localIdentifiers.size() == remoteIdentifiers.size());
        IFileServerCommand command = this.factory.createGetFileCommand(localIdentifiers, remoteIdentifiers, options, monitor);
        command.execute();
        command.dispose();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.execution.core.file.IFileManager#getFile(java.lang.String,
     *      java.lang.String)
     */
    public void getFile(String localName, String remoteName) throws IOException {
        this.getFile(FileIdentifierList.create(localName), FileIdentifierList.create(remoteName));
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#isServerAvailable()
     */
    public boolean isServerAvailable() throws IOException {
        IQueryServerStatusCommand command = this.factory.createQueryServerStatusCommand();
        command.execute();
        command.dispose();
        return command.isServerAvailable();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#listContent(
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList)
     */
    public FileIdentifierList listContent(FileIdentifierList remoteIdentifiers) throws IOException {
        return this.listContent(remoteIdentifiers, FileManagerExtendedImpl.NULL_PROGRESS_MONITOR);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#listContent(
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
     *      org.eclipse.core.runtime.IProgressMonitor)
     */
    public FileIdentifierList listContent(FileIdentifierList remoteIdentifiers, IProgressMonitor monitor) throws IOException {
        Assert.isNotNull(remoteIdentifiers);
        Assert.isNotNull(monitor);
        throw new UnsupportedOperationException();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#modifyPermission(
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
     *      java.lang.String)
     */
    public void modifyPermission(FileIdentifierList remoteIdentifiers, String permissionDirective) throws IOException {
        this.modifyPermission(remoteIdentifiers, permissionDirective, FileManagerExtendedImpl.NULL_PROGRESS_MONITOR);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#modifyPermission(
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
     *      java.lang.String, org.eclipse.core.runtime.IProgressMonitor)
     */
    public void modifyPermission(FileIdentifierList remoteIdentifiers, String permissionDirective, IProgressMonitor monitor)
            throws IOException {
        Assert.isNotNull(remoteIdentifiers);
        Assert.isNotNull(permissionDirective);
        Assert.isNotNull(monitor);
        Assert.isTrue(permissionDirective.trim().length() > 0);
        throw new UnsupportedOperationException();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#putFile(
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie,
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList)
     */
    public void putFile(Cookie cookie, FileIdentifierList localIdentifiers, FileIdentifierList remoteIdentifiers)
            throws IOException {
        this.putFile(cookie, localIdentifiers, remoteIdentifiers, Option.NONE);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#putFile
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie,
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
     *      org.eclipse.core.runtime.IProgressMonitor)
     */
    public void putFile(Cookie cookie, FileIdentifierList localIdentifiers, FileIdentifierList remoteIdentifiers,
            IProgressMonitor monitor) throws IOException {
        this.putFile(cookie, localIdentifiers, remoteIdentifiers, Option.NONE, monitor);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#putFile(
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie,
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option[])
     */
    public void putFile(Cookie cookie, FileIdentifierList localIdentifiers, FileIdentifierList remoteIdentifiers, Option[] options)
            throws IOException {
        this.putFile(cookie, localIdentifiers, remoteIdentifiers, options, FileManagerExtendedImpl.NULL_PROGRESS_MONITOR);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#putFile(
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie,
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option[],
     *      org.eclipse.core.runtime.IProgressMonitor)
     */
    public void putFile(Cookie cookie, FileIdentifierList localIdentifiers, FileIdentifierList remoteIdentifiers, Option[] options,
            IProgressMonitor monitor) throws IOException {
        Assert.isNotNull(cookie);
        Assert.isNotNull(localIdentifiers);
        Assert.isNotNull(remoteIdentifiers);
        Assert.isNotNull(options);
        Assert.isNotNull(monitor);
        Assert.isTrue(localIdentifiers.size() == remoteIdentifiers.size());
        IFileServerCommand command = this.factory.createPutFileCommand(cookie, localIdentifiers, remoteIdentifiers, options,
                monitor);
        command.execute();
        command.dispose();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#putFile(
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList)
     */
    public void putFile(FileIdentifierList localIdentifiers, FileIdentifierList remoteIdentifiers) throws IOException {
        this.putFile(localIdentifiers, remoteIdentifiers, FileManagerExtendedImpl.NULL_PROGRESS_MONITOR);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#putFile(
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
     *      org.eclipse.core.runtime.IProgressMonitor)
     */
    public void putFile(FileIdentifierList localIdentifiers, FileIdentifierList remoteIdentifiers, IProgressMonitor monitor)
            throws IOException {
        this.putFile(localIdentifiers, remoteIdentifiers, IFileManagerExtended.Option.NONE, monitor);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#putFile(
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option[])
     */
    public void putFile(FileIdentifierList localIdentifiers, FileIdentifierList remoteIdentifiers, Option[] options)
            throws IOException {
        this.putFile(localIdentifiers, remoteIdentifiers, options, FileManagerExtendedImpl.NULL_PROGRESS_MONITOR);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#putFile(
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
     *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option[],
     *      org.eclipse.core.runtime.IProgressMonitor)
     */
    public void putFile(FileIdentifierList localIdentifiers, FileIdentifierList remoteIdentifiers, Option[] options,
            IProgressMonitor monitor) throws IOException {
        this.putFile(Cookie.NONE, localIdentifiers, remoteIdentifiers, options, monitor);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.execution.core.file.IFileManager#putFile(java.lang.String,
     *      java.lang.String)
     */
    public void putFile(String localName, String remoteName) throws IOException {
        this.putFile(FileIdentifierList.create(localName), FileIdentifierList.create(remoteName));
    }

    /**
     * Query the file server location and other information important for
     * locating and connecting to the file server, the port is variable and is
     * usually changed by setting a new port in the agent controller
     * configuration files
     * 
     * @param connection
     *            the connection to use to communicate with the agent controller
     * @return the file server location
     */
    private IFileLocation queryFileServerLocation(final Connection connection) {

        // Create a command just to receive resource location command
        ManageFileCommand command = new ManageFileCommand();
        command.setOperation(ManageFileCommand.PUT);
        command.setFilename("");
        ControlMessage message = new ControlMessage();
        message.appendCommand(command);

        // Allocate a result to store the file server location object
        final FileLocationImpl result = new FileLocationImpl();
        final Object lock = new Object();

        // Send the message to the agent controller and wait for response
        try {

            // Command is asynchronous, need to wait for response
            synchronized (lock) {

                // Send the message with anonymous implementation handler
                connection.sendMessage(message, new CommandHandler() {
                    public void incommingCommand(Node node, CommandElement command) {
                        if (command.getTag() == org.eclipse.hyades.internal.execution.local.common.Constants.RA_RESOURCE_LOCATION) {
                            ResourceLocation location = (ResourceLocation) command;
                            result.setInetAddress(node.getInetAddress());
                            result.setPort(location.getPort());
                            result.setJobKey(location.getJobKey());
                            synchronized (lock) {
                                lock.notify();
                            }
                        }
                    }
                });

                // Wait up to the timeout value
                lock.wait(FileManagerExtendedImpl.DEFAULT_TIMEOUT);

            }

        } catch (IOException e) {

            return null; // a connection is not possible, setting to null

        } catch (InterruptedException e) {

            return null; // a null location indicates connection not possible

        }
        return result;
    }

    /**
     * An adapter that adapts a legacy file manager implementation so it
     * implements the file manager extended interface
     * 
     * @author Scott E. Schneider
     */
    static class Adapter implements IFileManagerExtended {

        /**
         * The legacy implementation to delegate calls to
         */
        private final IFileManager fileManager;

        /**
         * Wrap the legacy file manager with new interface
         * 
         * @param fileManager
         *            the legacy file manager to delegate calls into
         */
        Adapter(IFileManager fileManager) {
            this.fileManager = fileManager;
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#deleteFile(
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie,
         *      org.eclipse.core.runtime.IProgressMonitor)
         */
        public void deleteFile(Cookie cookie, IProgressMonitor monitor) throws IOException {
            throw new UnsupportedOperationException();
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#deleteFile(
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList)
         */
        public void deleteFile(FileIdentifierList remoteIdentifiers) throws IOException {
            for (Iterator identifiers = remoteIdentifiers.iterator(); identifiers.hasNext();) {
                String remoteFileName = (String) identifiers.next();
                this.deleteFile(remoteFileName);
            }
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#deleteFile(
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
         *      org.eclipse.core.runtime.IProgressMonitor)
         */
        public void deleteFile(FileIdentifierList remoteIdentifiers, IProgressMonitor monitor) throws IOException {
            throw new UnsupportedOperationException();
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#deleteFile(
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie)
         */
        public void deleteFile(Cookie cookie) throws IOException {
            throw new UnsupportedOperationException();
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#getFile(
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList)
         */
        public void getFile(FileIdentifierList localIdentifiers, FileIdentifierList remoteIdentifiers) throws IOException {
            for (Iterator localIdentifier = localIdentifiers.iterator(), remoteIdentifier = remoteIdentifiers.iterator(); localIdentifier
                    .hasNext();) {
                String localFileName = (String) localIdentifier.next();
                String remoteFileName = (String) remoteIdentifier.next();
                this.getFile(localFileName, remoteFileName);
            }
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#getFile(
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option[])
         */
        public void getFile(FileIdentifierList localIdentifiers, FileIdentifierList remoteIdentifiers, Option[] options)
                throws IOException {
            throw new UnsupportedOperationException();
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#getFile(
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option[],
         *      org.eclipse.core.runtime.IProgressMonitor)
         */
        public void getFile(FileIdentifierList localIdentifiers, FileIdentifierList remoteIdentifiers, Option[] options,
                IProgressMonitor monitor) throws IOException {
            throw new UnsupportedOperationException();
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#getFile(
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
         *      org.eclipse.core.runtime.IProgressMonitor)
         */
        public void getFile(FileIdentifierList localIdentifiers, FileIdentifierList remoteIdentifiers, IProgressMonitor monitor)
                throws IOException {
            throw new UnsupportedOperationException();
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#isServerAvailable()
         */
        public boolean isServerAvailable() throws IOException {
            return true;
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#listContent(
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList)
         */
        public FileIdentifierList listContent(FileIdentifierList remoteIdentifiers) throws IOException {
            throw new UnsupportedOperationException();
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#listContent(
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
         *      org.eclipse.core.runtime.IProgressMonitor)
         */
        public FileIdentifierList listContent(FileIdentifierList remoteIdentifiers, IProgressMonitor monitor) throws IOException {
            throw new UnsupportedOperationException();
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#modifyPermission(
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
         *      java.lang.String)
         */
        public void modifyPermission(FileIdentifierList remoteIdentifiers, String permissionDirective) throws IOException {
            throw new UnsupportedOperationException();
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#modifyPermission(
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
         *      java.lang.String, org.eclipse.core.runtime.IProgressMonitor)
         */
        public void modifyPermission(FileIdentifierList remoteIdentifiers, String permissionDirective, IProgressMonitor monitor)
                throws IOException {
            throw new UnsupportedOperationException();
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#putFile(
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie,
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList)
         */
        public void putFile(Cookie cookie, FileIdentifierList localIdentifiers, FileIdentifierList remoteIdentifiers)
                throws IOException {
            throw new UnsupportedOperationException();
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#putFile(
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie,
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option[])
         */
        public void putFile(Cookie cookie, FileIdentifierList localIdentifiers, FileIdentifierList remoteIdentifiers,
                Option[] options) throws IOException {
            throw new UnsupportedOperationException();
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#putFile(
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie,
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option[],
         *      org.eclipse.core.runtime.IProgressMonitor)
         */
        public void putFile(Cookie cookie, FileIdentifierList localIdentifiers, FileIdentifierList remoteIdentifiers,
                Option[] options, IProgressMonitor monitor) throws IOException {
            throw new UnsupportedOperationException();
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#putFile(
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie,
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
         *      org.eclipse.core.runtime.IProgressMonitor)
         */
        public void putFile(Cookie cookie, FileIdentifierList localIdentifiers, FileIdentifierList remoteIdentifiers,
                IProgressMonitor monitor) throws IOException {
            throw new UnsupportedOperationException();
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#putFile(
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList)
         */
        public void putFile(FileIdentifierList localIdentifiers, FileIdentifierList remoteIdentifiers) throws IOException {
            for (Iterator localIdentifier = localIdentifiers.iterator(), remoteIdentifier = remoteIdentifiers.iterator(); localIdentifier
                    .hasNext();) {
                String localFileName = (String) localIdentifier.next();
                String remoteFileName = (String) remoteIdentifier.next();
                this.putFile(localFileName, remoteFileName);
            }
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#putFile(
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option[])
         */
        public void putFile(FileIdentifierList localIdentifiers, FileIdentifierList remoteIdentifiers, Option[] options)
                throws IOException {
            throw new UnsupportedOperationException();
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#putFile(
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option[],
         *      org.eclipse.core.runtime.IProgressMonitor)
         */
        public void putFile(FileIdentifierList localIdentifiers, FileIdentifierList remoteIdentifiers, Option[] options,
                IProgressMonitor monitor) throws IOException {
            throw new UnsupportedOperationException();
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#putFile(
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
         *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
         *      org.eclipse.core.runtime.IProgressMonitor)
         */
        public void putFile(FileIdentifierList localIdentifiers, FileIdentifierList remoteIdentifiers, IProgressMonitor monitor)
                throws IOException {
            throw new UnsupportedOperationException();
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.hyades.execution.core.file.IFileManager#getFile(java.lang.String,
         *      java.lang.String)
         */
        public void getFile(String localName, String remoteName) throws IOException {
            this.fileManager.getFile(localName, remoteName);
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.hyades.execution.core.file.IFileManager#putFile(java.lang.String,
         *      java.lang.String)
         */
        public void putFile(String localName, String remoteName) throws IOException {
            this.fileManager.putFile(localName, remoteName);
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.hyades.execution.core.file.IFileManager#deleteFile(java.lang.String)
         */
        public void deleteFile(String remoteFileName) throws IOException {
            this.fileManager.deleteFile(remoteFileName);
        }

    }

}