/********************************************************************** 
 * Copyright (c) 2005 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: FileManagerExtendedTimedImpl.java,v 1.3 2005/06/22 14:25:42 sschneid Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.execution.local.file;

import java.io.IOException;
import java.io.PrintStream;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.hyades.execution.core.timer.IPerformanceTimer;
import org.eclipse.hyades.execution.core.timer.PerformanceTimer;
import org.eclipse.hyades.internal.execution.local.control.Connection;

/**
 * Implementation of new client-side file manager, the factory is responsible
 * for createing the appropriate concrete class that abides by the file manager
 * interface.
 * 
 * @author Scott E. Schneider
 */
class FileManagerExtendedTimedImpl extends FileManagerExtendedImpl {

	/**
	 * Performance timer instance to use for all operations in this class
	 */
	private final IPerformanceTimer timer;

	/**
	 * Number of times to invoke each method for every call into a method, used
	 * strictly for obtaining a larger sample for performance measurements
	 */
	private final int iterations;

	/**
	 * Creates a new file manager extended implementation instance with the
	 * given connection, using timed mode, allowing for lightweight performance
	 * timing results to be displayed to the given printstream per file manager
	 * operation.
	 * 
	 * @param connection
	 *            the connection used for file manager operations
	 */
	FileManagerExtendedTimedImpl(Connection connection) {
		super(connection);
		this.timer = PerformanceTimer.getInstance(this.toString());
		this.iterations = 1;
	}

	/**
	 * Creates a new file manager extended implementation instance with the
	 * given connection, using timed mode, allowing for lightweight performance
	 * timing results to be displayed to the given printstream per file manager
	 * operation.
	 * 
	 * @param connection
	 *            the connection used for file manager operations
	 * @param stream
	 *            the print stream to output elasped time information to
	 */
	FileManagerExtendedTimedImpl(Connection connection, PrintStream stream) {
		super(connection);
		this.timer = PerformanceTimer.getInstance(this.toString(), stream);
		this.iterations = 1;
	}

	/**
	 * Creates a new file manager extended implementation instance with the
	 * given connection, using timed mode, allowing for lightweight performance
	 * timing results to be displayed to the given printstream per file manager
	 * operation.
	 * 
	 * @param connection
	 *            the connection used for file manager operations
	 * @param iterations
	 *            number of times to exercise each method
	 */
	FileManagerExtendedTimedImpl(Connection connection, int iterations) {
		super(connection);
		this.timer = PerformanceTimer.getInstance(this.toString());
		this.iterations = iterations;
	}

	/**
	 * Creates a new file manager extended implementation instance with the
	 * given connection, using timed mode, allowing for lightweight performance
	 * timing results to be displayed to the given printstream per file manager
	 * operation.
	 * 
	 * @param connection
	 *            the connection used for file manager operations
	 * @param stream
	 *            the print stream to output elasped time information to
	 * @param iterations
	 *            number of times to exercise each method
	 */
	FileManagerExtendedTimedImpl(Connection connection, PrintStream stream, int iterations) {
		super(connection);
		this.timer = PerformanceTimer.getInstance(this.toString(), stream);
		this.iterations = iterations;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#deleteFile(
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie,
	 *      org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void deleteFile(Cookie cookie, IProgressMonitor monitor) throws IOException {
		this.timer.start("deleteFile (using cookie to access server state)" + cookie);
		super.deleteFile(cookie, monitor);
		this.timer.stop();
		this.timer.elapsed();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#deleteFile(
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void deleteFile(FileIdentifierList remoteIdentifiers, IProgressMonitor monitor) throws IOException {
		boolean isServerAvailable = false;
		this.timer.start("deleteFile with " + remoteIdentifiers.size() + " files");
		for (int i = 1; i < this.iterations + 1; i++) {
			this.timer.start("iteration " + i);
			super.deleteFile(remoteIdentifiers, monitor);
			this.timer.stop();
			this.timer.elapsed();
		}
		this.timer.stop();
		this.timer.elapsed();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#getFile(
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option[],
	 *      org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void getFile(FileIdentifierList localIdentifiers, FileIdentifierList remoteIdentifiers, Option[] options,
			IProgressMonitor monitor) throws IOException {
	    int size = localIdentifiers.size();
		this.timer.start("getFile with " + size + " files named " + localIdentifiers.getArray()[0] + " ... " + 
		        localIdentifiers.getArray()[size-1]);
		for (int i = 1; i < this.iterations + 1; i++) {
			this.timer.start("iteration " + i);
			super.getFile(localIdentifiers, remoteIdentifiers, options, monitor);
			this.timer.stop();
			this.timer.elapsed();
		}
		this.timer.stop();
		this.timer.elapsed();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#isServerAvailable()
	 */
	public boolean isServerAvailable() throws IOException {
		boolean isServerAvailable = false;
		this.timer.start("isServerAvailable");
		for (int i = 1; i < this.iterations + 1; i++) {
			this.timer.start("iteration " + i);
			isServerAvailable = super.isServerAvailable() || isServerAvailable;
			this.timer.stop();
			this.timer.elapsed();
		}
		this.timer.stop();
		this.timer.elapsed();
		return isServerAvailable;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.execution.core.file.IFileManagerExtended#putFile(
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Cookie,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList,
	 *      org.eclipse.hyades.execution.core.file.IFileManagerExtended.Option[],
	 *      org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void putFile(Cookie cookie, FileIdentifierList localIdentifiers, FileIdentifierList remoteIdentifiers, Option[] options,
			IProgressMonitor monitor) throws IOException {
	    int size = localIdentifiers.size();
	    if (size > 0) {
	        this.timer.start("putFile with " + size + " files named " + localIdentifiers.getArray()[0] + " ... " + 
	                localIdentifiers.getArray()[size-1]);
			for (int i = 1; i < this.iterations + 1; i++) {
				this.timer.start("iteration " + i);
				super.putFile(cookie, localIdentifiers, remoteIdentifiers, options, monitor);
				this.timer.stop();
				this.timer.elapsed();
			}
			this.timer.stop();
			this.timer.elapsed();
	    }
	}

}
