/********************************************************************** 
 * Copyright (c) 2005 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: FileServerExtended.java,v 1.11 2005/06/07 19:49:40 sschneid Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.internal.execution.file;

import java.io.IOException;
import java.net.BindException;
import java.net.InetSocketAddress;
import java.nio.channels.ServerSocketChannel;
import java.nio.channels.SocketChannel;

import org.eclipse.hyades.internal.execution.core.file.SocketChannelFactory;

/**
 * The new extended file server implementation that will take advantage of
 * java.nio and other performance optimizations to increase speed of transfers
 * and flexibility of file manager commands offered. It is runnable, so it can
 * be run by assigning a particular thread -- basically the agent controller
 * could start up multiple file servers on different ports if desired.
 * 
 * @author Scott E. Schneider
 */
class FileServerExtended implements Runnable, IFileServer {

    /**
     * Default timeout for sockets (client and server)
     */
    private static final int DEFAULT_SOCKET_TIMEOUT = 5000;

    /**
     * Default shutdown request state is false until shutdown is desired
     */
    {
        this.isRunning = false;
        this.shutdown = false;
    }

    /**
     * Parameters sent by the agent controller and set in the init method
     */
    private IFileServerParameters parameters;

    /**
     * The sole server socket for this instance of the file server, a file
     * server can be instantiated more than once -- it is not a singleton, but
     * in typical use via the agent controller it is a singleton in practice,
     * every instance must have its own unique port to accept connections on
     */
    private ServerSocketChannel serverSocket;

    /**
     * Various file server specific status state
     */
    private int serverStatus, errorType, initStatus;

    /**
     * Shutdown request flag, set by the quite method, starts as false, read in
     * the run method as a signal to exit the server
     */
    private boolean shutdown;
    
    /**
     * Indicates if the server's run method has been invoked, the server can only
     * be run once and left alive, once shutdown the same instance cannot be brought
     * back up and re-used again
     */
    private boolean isRunning;

    /**
     * Constructor invoked by the agent controller to start up a new instance
     */
    FileServerExtended() {
        this.serverStatus = FileServiceConstants.RA_FS_STATUS_OK;
        this.errorType = FileServiceConstants.RA_FS_STATUS_OK;
        this.initStatus = FileServiceConstants.RA_FS_UNINITIALIZED;
    }

    /**
     * Clean-up resources appropriately and cleanly
     */
    private void cleanup() {
        try {
            this.serverSocket.close();
            this.serverSocket = null;
            this.parameters = null;
        } catch (IOException e) {
        } catch (Throwable t) {
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#finalize()
     */
    protected void finalize() {
        this.cleanup();
    }

    /**
     * Get the error type of the last triggered and stored error condition
     */
    public int getErrorType() {
        return this.errorType;
    }

    /**
     * Retrieve the file server status, used by the agent controller to verify
     * the file server has been started up successfully
     */
    public synchronized int getFileServerStatus() {
        final long timeout = 5000L;
        
        // Determine if the server is starting up but not yet initialized
        if (this.initStatus != FileServiceConstants.RA_FS_INITIALIZED && this.isRunning) {
            try {
                this.wait(timeout);
            } catch (InterruptedException e) {
                // No need to handle this exception
            }
        }
        
        // If the server is not starting up, then just return the status without wait
        return this.serverStatus;
    }

    /**
     * Retrieve the port to accept file server connections on, set by the agent
     * controller and typically this value is set and read from the agent
     * controller static configuration files
     * 
     * @return the port to use for this instance of the file server
     */
    private int getPort() {
        return this.parameters.getPort();
    }

    /**
     * Initialize method invoked by the agent controller to pass the state of
     * the file server parameters, such as port number to use for the server and
     * connection handler implementation suggestion
     */
    public void init(IFileServerParameters parameters) {
        this.parameters = parameters;
    }

    /**
     * Request to shutdown the server, the file server will shutdown after it
     * has completed the accept in the run loop and returns to the top to check
     * the condition again
     */
    public void quit() {
        this.shutdown = true;
        this.cleanup(); // force cleanup immediately
    }

    /**
     * Primary execution loop method, run as run is explicitly invoked on the
     * calling thread or run by its own thread if set to a thread and start
     * called
     */
    public void run() {
        
        // Run can only be invoked with action taken once
        synchronized (this) {
        
            /* 
             * Determine if file server is already running, if so
             * return without any further action and leave the server
             * running -- if the server is not already running, start
             * up the server
             */
            if (this.isRunning) {
                return;
            } else {
                this.isRunning = true;
            }
            
        }

        try {

            // Initialize the file server extended
            INITIALIZATION: {
                
            
	            // Continue to hold lock until initialized, then notify waiters
	            synchronized (this) {
	                
		            // Establish file server using a java.nio server socket channel
		            this.serverSocket = ServerSocketChannel.open();
		            this.serverSocket.configureBlocking(true);
		
		            // Set server socket timeout and bind to port
		            this.serverSocket.socket().setSoTimeout(DEFAULT_SOCKET_TIMEOUT);
		            this.serverSocket.socket().bind(new InetSocketAddress(this.getPort()));
		
		            // Change state to initialized and notify any waiters
		            this.initStatus = FileServiceConstants.RA_FS_INITIALIZED;
		            
		            // Potentially a thread is waiting on get file server status
		            this.notifyAll();
		            
	            }
	            
            }
        
        	// Create a server socket and begin listening for connection requests
        	CONNECTION_LISTENING: {

	            // Main file server loop to accept connections until shutdown
	            while (!this.shutdown) {
	
	                SocketChannel clientChannel = this.serverSocket.accept();
	
	                // Set socket timeout to avoid blocking forever
	                clientChannel.socket().setSoTimeout(DEFAULT_SOCKET_TIMEOUT);
	                clientChannel.socket().setTcpNoDelay(true);
	
	                // Take newly connected socket to client and send to handler
	                IConnectionHandler handler = (IConnectionHandler) this.parameters.getConnectionHandler();
	                handler.connectionAccepted(SocketChannelFactory.getInstance().create(clientChannel));
	                
	            }

            }

        } catch (BindException e) {

            // Handle bind exception, update file server status
            this.serverStatus = FileServiceConstants.RA_FS_STATUS_ERROR;
            this.errorType = FileServiceConstants.RA_BIND_EXCEPTION;
            this.initStatus = FileServiceConstants.RA_FS_INITIALIZED;
            
        } catch (IOException e) {

            // Handle exception appropriately, update file server status
            this.serverStatus = FileServiceConstants.RA_FS_STATUS_ERROR;
            this.errorType = FileServiceConstants.RA_IO_ERROR;
            this.initStatus = FileServiceConstants.RA_FS_INITIALIZED;
            
        } catch (Throwable t) {
            
            // Squelch all others but not handled specifically
            
        } finally {

            // Clean-up resources appropriately
            this.cleanup();

        }

    }

}