/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Process.java,v 1.5 2005/04/26 15:13:30 sschneid Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.internal.execution.local.control;

import java.util.Enumeration;

import org.eclipse.hyades.internal.execution.local.common.Console;

public interface Process {
	
	/**
	 * Launches the process on the node that was provided
	 * to the ProcessFactory during construction of this
	 * Process instance.  If the Process was already launched
	 * the ProcessActiveException is thrown.
	 */
	void launch() throws ProcessActiveException, NotConnectedException, NoSuchApplicationException;
	
	/**
	 * Kills this process if it is active.
	 */
	void kill(long delay) throws InactiveProcessException, NotConnectedException;
	
	/**
	 * Provides an Enumeration of all the Agents associated
	 * with this Process instance.
	 * @see Agent
	 */
	Enumeration listAgents();
	
	/**
	 * Retrieves an agent based upon it's name
	 * @see Agent
	 * @return the agent with the specified name if it exists, null otherwise
	 */
	Agent getAgent(String name);
	
	/**
	 * Provides an enumeration of all the agents of a specific type that
	 * are assciated with this Process.
	 */
	Enumeration getAgentsByType(String type);
	
	/**
	 * Adds a ProcessListener to this Process instance.
	 * @see ProcessListener
	 */
	void addProcessListener(ProcessListener listener);
	
	/**
	 * Removes a ProcessListener from this Process instance.
	 * @see ProcessListener
	 */
	void removeProcessListener(ProcessListener listener);
	
	/**
	 * Retrieves the Node that this Process instance
	 * is associated with.
	 * @see Node
	 */
	Node getNode() throws InactiveProcessException;
	
	/**
	 * Save a user friendly name so that it can be retrieved later via Process#getName()
	 */
	void setName(String name);
	
	/**
	 * Retrieves the name of this process as provided
	 * by Process#setName()
	 */
	String getName();
	
	/**
	 * Retrieves the uuid of this process if it is known, null otherwise.
	 */
	String getUUID();
	
	
	
	/**
	 * Set the exectauble name for this process.
	 */
	void setExecutable(String exe) throws ProcessActiveException;
	
	/**
	 * Get the exectable name for this process.
	 */
	String getExecutable();
	
	/**
	 * Retrieve the process ID (pid) for this process
	 */
	String getProcessId() throws InactiveProcessException;
	
	boolean isActive();
	
	/**
	 * Retrieves the console of this Process instance.
	 */
	Console getConsole();
	
	/**
	 * Adds an environment variable to this process.
	 */
	void addEnvironmentVariable(Variable variable) throws ProcessActiveException;
	
	/**
	 * Retrieves an environment variable from this process.
	 * @return the environment variable with the specified name attribute if it exists, null otherwise.
	 */
	Variable getEnvironmentVariable(String name);
	
	/**
	 * Removes an environment variable from this process.
	 */
	void removeEnvironmentVariable(Variable variable) throws ProcessActiveException;
	
	/**
	 * Provides an enumeration of each Variabe comprising
	 * the entire environment of this process.
	 * @see Variable
	 */
	Enumeration getEnvironment();
	
	/**
	 * Set the parameters for this process
	 */
	void setParameters(String parameters) throws ProcessActiveException;
	
	/**
	 * Retrieves the parameters of the process.
	 */
	String getParameters();
	
	/**
	 * Set's the home directory of the process.
	 */
	void setLocation(String location) throws ProcessActiveException;
	
    /**
     * Blocks calling thread until the agent is retrieved, if the agent
     * cannot be retrieved within the timeout period, null is returned for the
     * agent return value.
     * 
     * @param agentType
     * 			the type of agent qualified the selection
     * @param agentName
     * 			the name of the agent to wait for
     * @param timeout
     * 			the maximum time to wait for the agent to be retrieved
     * @return the agent retrieved or null if the timeout is reached before
     *         the agent is found
     * @throws NotConnectedException
     */
    Agent waitForAgent(String agentType, String agentName, int timeout)
            throws NotConnectedException, InactiveProcessException;

}

