/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: IDatapoolCell.java,v 1.4 2005/02/16 22:21:43 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.execution.runtime.datapool;

/**
 * An element of a {@link IDatapoolRecord record} within a datapool.  Each
 * cell contains a reference to the {@link IDatapoolVariable variable} that 
 * it is referenced by along with the actual value the cell represents.
 * There are a series of convienence methods for accessing the associated 
 * value as a specific type, such as a <code>String</code> or an <code>int</code>.
 * 
 */
public interface IDatapoolCell
{
	
	/**
	 * Access to the {@link IDatapoolVariable variable} that is used to reference
	 * this cell within a {@link IDatapoolRecord record}.
	 * 
	 * @return	The {@link IDatapoolVariable variable} that is used to reference
	 *			this cell within a {@link IDatapoolRecord record}.
	 */
	public IDatapoolVariable getCellVariable();

	/**
	 * Access to the {@link IDatapoolRecord record} that is the parent of this 
	 * cell.
	 * 
	 * @return	The {@link IDatapoolRecord record} that is the parent of this
	 *			cell.
	 */
	public IDatapoolRecord getCellRecord();
	
	/**
	 * The value associated with the cell is returned.
	 * 
	 * @return	The value associated with the cell is returned.
	 */
	public Object getCellValue();
	
	/**
	 * A shortcut for the <code>String</code> value associated with the cell is returned.
	 * If the value in the cell is not a <code>String</code> value then it is
	 * converted to a <code>String</code> before being returned.
	 * 
	 * @return	The <code>String</code> value associated with the cell.
	 */
	public String getStringValue();
	
	/**
	 * A shortcut for the <code>long</code> value associated with  the cell is returned.
	 * If the value in the cell is not a <code>long</code> value then an attempt
	 * is made to convert the value to a <code>long</code>.  If the value can not
	 * be converted a <code>java.lang.NumberFormatException</code> will be thrown.
	 * 
	 * @return	The <code>long</code> value associated with the cell.
	 */
	public long getLongValue();
	
	/**
	 * A shortcut for the <code>int</code> value associated with  the cell is returned.
	 * If the value in the cell is not a <code>int</code> value then an attempt
	 * is made to convert the value to a <code>int</code>.  If the value can not
	 * be converted a <code>java.lang.NumberFormatException</code> will be thrown.
	 * 
	 * @return	The <code>int</code> value associated with the cell.
	 */
	public int getIntValue();
	
	/**
	 * A shortcut for the <code>short</code> value associated with  the cell is returned.
	 * If the value in the cell is not a <code>short</code> value then an attempt
	 * is made to convert the value to a <code>short</code>.  If the value can not
	 * be converted a <code>java.lang.NumberFormatException</code> will be thrown.
	 * 
	 * @return	The <code>short</code> value associated with the cell.
	 */
	public short getShortValue();
	
	/**
	 * A shortcut for the <code>byte</code> value associated with  the cell is returned.
	 * If the value in the cell is not a <code>byte</code> value then an attempt
	 * is made to convert the value to a <code>byte</code>.  If the value can not
	 * be converted a <code>java.lang.NumberFormatException</code> will be thrown.
	 * 
	 * @return	The <code>byte</code> value associated with the cell.
	 */
	public byte getByteValue();
	
	/**
	 * A shortcut for the <code>double</code> value associated with  the cell is returned.
	 * If the value in the cell is not a <code>double</code> value then an attempt
	 * is made to convert the value to a <code>double</code>.  If the value can not
	 * be converted a <code>java.lang.NumberFormatException</code> will be thrown.
	 * 
	 * @return	The <code>double</code> value associated with the cell.
	 */
	public double getDoubleValue();
	
	/**
	 * A shortcut for the <code>float</code> value associated with  the cell is returned.
	 * If the value in the cell is not a <code>float</code> value then an attempt
	 * is made to convert the value to a <code>float</code>.  If the value can not
	 * be converted a <code>java.lang.NumberFormatException</code> will be thrown.
	 * 
	 * @return	The <code>float</code> value associated with the cell.
	 */
	public float getFloatValue();
	
	/**
	 * A shortcut for the <code>boolean</code> value associated with  the cell is returned.
	 * If the value in the cell is not a <code>boolean</code> value then an attempt
	 * is made to convert the value to a <code>boolean</code>.  If the value can not
	 * be converted an appropriate exception will be thrown.
	 * 
	 * @return	The <code>boolean</code> value associated with the cell.
	 */
	public boolean getBooleanValue();
	
	/**
	 * A shortcut for the <code>char</code> value associated with  the cell is returned.
	 * If the value in the cell is not a <code>char</code> value then an attempt
	 * is made to convert the value to a <code>char</code>.  If more then one 
	 * character is represented in the value only the first character will be
	 * returned.
	 * 
	 * @return	The <code>char</code> value associated with the cell.
	 */
	public char getCharValue();
	
}
