/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DBMap.java,v 1.4 2005/02/16 22:21:28 qiyanli Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.resources.database.internal;

import java.util.List;

import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EModelElement;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.hyades.resources.database.internal.dbmodel.Column;
import org.eclipse.hyades.resources.database.internal.dbmodel.Database;
import org.eclipse.hyades.resources.database.internal.dbmodel.Table;
import org.eclipse.hyades.resources.database.internal.impl.ClassMetadata;
/**
 * This represents a map from Ecore constructs to the corresponding RDBSchema
 * constructs. Since some Ecore constructs require more than one piece of data
 * in the map, many of the mappings are represented using data objects. Here are
 * the expected mappings:
 * 
 * <ol>
 * <li>EPackage --> RDBDatabase
 * <li>EClass --> ClassData
 * <li>EAttribute --> AttributeData
 * <li>EReference --> ReferenceData
 * </ol>
 * <p>
 * For an EClass, the mapping is to a ClassData object that has the class table,
 * a boolean to determine whether the database code generates an id for
 * instances of the class, and the proxyURI column if there is one.
 * <p>
 * For an EAttribute, the mapping is to an AttributeData object. The object has
 * either the column in the class table that holds the value, or the object has
 * the attribute table and the columns holding the id, value, and, optionally,
 * order columns.
 * <p>
 * For an EReference, the mapping is to a ReferenceData object that has the
 * reference table, the source and target columns, and, optionally, the order
 * column.
 * <p>
 * This class also includes useful query methods.
 */
public interface DBMap {
	/**
	 * Add the dbRepresentation for the given Ecore model element to the
	 * mapping.
	 * 
	 * @param element
	 *            An Ecore model construct
	 * @param dbRepresentation
	 *            The RDBSchema representation of the construct.
	 */
	void add(EModelElement element, Object dbRepresentation);

	/**
	 * Get the RDBSchema representation of the given Ecore construct.
	 */
	Object getDBRepresentation(EModelElement element);

	/**
	 * Returns the last RDBDatabase object added to the map.
	 */
	Database getDatabase();

	/**
	 * Returns a list containing class tables.
	 */
	List getClassTables();

	/**
	 * Returns a list containing reference tables.
	 */
	List getReferenceTables();

	/**
	 * Returns a list containing attribute tables.
	 */
	List getAttributeTables();

	/**
	 * Returns a list of all subclasses of the given class; the list is ordered
	 * so subclasses appear before superclasses. If there are no subclasses, an
	 * empty list is returned.
	 */
	List getAllSubclasses(EClass eClass);

	/**
	 * Returns the resource table, or null if there is no resource table.
	 */
	Table getResourceTable();

	/**
	 * Sets the resource table to the given table.
	 */
	void setResourceTable(Table table);

	/**
	 * Returns the proxy table, or null if there is no proxy table.
	 */
	Table getProxyTable();

	/**
	 * Sets the proxy table to the given table.
	 */
	void setProxyTable(Table table);

	/**
	 * Returns the id table, or null if there is no id table.
	 */
	Table getIdTable();

	/**
	 * Sets the id table to the given table.
	 */
	void setIdTable(Table table);

	/**
	 * Get the class for the given table. If the table does not correspond to a
	 * class, or cannot be found, return null.
	 */
	EClass getClass(Table table);

	/**
	 * Get the reference corresponding to the given column of a class table, or
	 * null, if the column does not correspond to a reference.
	 */
	EReference getReference(Column column);

	/**
	 * This class holds data for an EClass. The table is the class table that
	 * holds attribute values for the local, single-valued attributes. The
	 * computeId flag indicates whether the database code should generate an id
	 * or not. If the local, single-valued attributes do not have a value that
	 * can be used as an id, set computeId to true so the database code assigns
	 * an id. Otherwise, set computeId to false. The proxyURI column stores
	 * cross-resource references; if you are not storing resources in the
	 * database, you can set it to null.
	 */
	public class ClassData {
		protected Table table;
		protected boolean computeId;
		protected Column proxyURI;

		public ClassData(Table table, boolean computeId, Column proxyURI) {
			this.table = table;
			this.computeId = computeId;
			this.proxyURI = proxyURI;
		}

		public Table getTable() {
			return table;
		}
		public boolean isComputeId() {
			return computeId;
		}
		public Column getProxyURIColumn() {
			return proxyURI;
		}
	}

	/**
	 * This class holds data for an EAttribute. If the attribute is
	 * single-valued, set the value column. Otherwise, set the RDBTable to the
	 * attribute table. For many-valued attributes, the id column holds object
	 * ids, and the value column holds the actual values; also, the order column
	 * is used to order the attribute values. If the order column is null, the
	 * values are not ordered.
	 */
	public class AttributeData {
		protected Column value, order, id;
		protected Table table;

		/**
		 * Use this constructor for single-valued attributes.
		 */
		public AttributeData(Column value) {
			this.value = value;
		}

		/**
		 * Use this constructor for many-valued attributes.
		 */
		public AttributeData(Table table, Column id, Column value, Column order) {
			this.table = table;
			this.id = id;
			this.value = value;
			this.order = order;
		}

		public Table getTable() {
			return table;
		}
		public Column getIdColumn() {
			return id;
		}
		public Column getValueColumn() {
			return value;
		}
		public Column getOrderColumn() {
			return order;
		}
	}

	/**
	 * For references, the RDBTable is the table representing the reference. The
	 * source RDBColumn is the column containing ids of instances of the class
	 * that has the reference. The target RDBColumn is the column containing ids
	 * of referenced instances. The order RDBColumn is null for single-valued
	 * references. For many-valued references, it is the column to use to order
	 * the referenced instances. For many-valued references, if the order column
	 * is null, reference values are not ordered.
	 */
	public class ReferenceData {
		protected Table table;
		protected Column source, target, order;

		public ReferenceData(Table table, Column source, Column target, Column order) {
			this.table = table;
			this.source = source;
			this.target = target;
			this.order = order;
		}

		public Table getTable() {
			return table;
		}
		public Column getSourceColumn() {
			return source;
		}
		public Column getTargetColumn() {
			return target;
		}
		public Column getOrderColumn() {
			return order;
		}
	}
	public ClassMetadata getClassMetadata(EClass theClass);
} // DBMap
