/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Database.java,v 1.4 2005/02/16 22:21:28 qiyanli Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.resources.database.internal;

import java.util.Collection;
import java.util.List;
import java.util.Properties;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.resources.database.internal.extensions.DatabaseType;
import org.eclipse.hyades.resources.database.internal.extensions.JDBCHelper;
/**
 * This interface is an abstract interface to an underlying database. Here is
 * how you use it:
 * <ol>
 * <li>Create a database object using the DatabaseFactory.
 * <li>Call the open() method to prepare the database connection.
 * <li>Set the DBMap, either by creating one yourself or invoking a setDBMap
 * method that creates a default one for you.
 * <li>Invoke methods to add or get EObjects, and add or get EMF resources.
 * <li>Call the close() method to shut down the database.
 * </ol>
 */
public interface Database {
	/**
	 * This method creates a connection to the database. If create is true, it
	 * attempts to create the database and open a connection to it. If create is
	 * false, it creates a connection to an existing database.
	 * 
	 * @param create
	 *            If true, this method creates a connection to a new database.
	 *            If false, this method creates a connection to an existing
	 *            database.
	 */
	void open(boolean create) throws Exception;

	/**
	 * This method adds EObjects to the database. EObjects that are contained by
	 * the EObjects in the list are added to the database; EObjects that are
	 * referenced by the EObjects but are not contained are not added to the
	 * database.
	 * 
	 * @param objects
	 *            The EObjects to add to the database.
	 */
	void add(List objects) throws Exception;

	/**
	 * This method adds the given resource to the database. It throws a
	 * ResourceAddedException if the resource is already in the database, and it
	 * throws a NoResourceTableException if the DBMap does not have a resource
	 * table.
	 * 
	 * @param resource
	 *            The Resource to add to the database.
	 * @exception ResourceAddedException
	 * @exception NoResourceTableException
	 */
	void add(Resource resource) throws Exception;

	/**
	 * This method deletes the resource with the given URI from the resource
	 * table. It throws a NoResourceTableException if the DBMap does not have a
	 * resource table.
	 * 
	 * @param URI
	 *            The URI of the resource to delete.
	 * @exception NoResourceTableException
	 */
	void deleteResource(URI uri) throws Exception;

	/**
	 * This method returns objects that are matched by the object query. If
	 * there are no such objects, null is returned. The dbObjects parameter
	 * allows you to get the database ids of matched objects. You can set it to
	 * null if you wish.
	 * 
	 * @param query
	 *            The object query
	 */
	EObject[] getObjects(ObjectQuery query) throws Exception;

	/**
	 * This method returns referenced objects. If there are no such objects,
	 * null is returned. The dbObjects parameter allows you to get the database
	 * ids of the referenced objects. You can set it to null if you do not need
	 * the database ids.
	 * 
	 * @param query
	 *            The reference query.
	 */
	EObject[] getObjects(ReferenceQuery query) throws Exception;

	/**
	 * This method loads the given resource. If there is no resource table in
	 * the DBMap, a NoResourceTableException is thrown. If the resource is not
	 * found in the database, a ResourceNotFoundException is thrown.
	 * <p>
	 * If allObjects is true, all contained objects are restored; if allObjects
	 * is false, only the top level objects are restored.
	 * 
	 * @param resource
	 *            The resource to load
	 * @param allObjects
	 *            If true, all contained objects are restored; if false, only
	 *            the top level objects are restored.
	 * @exception NoResourceTableException
	 * @exception ResourceNotFoundException
	 */
	void load(Resource resource, boolean allObjects) throws Exception;

	/**
	 * This method is the same as the previous load method, with two additional
	 * parameters. If notLoadedClasses contains EClasses, instances of those
	 * classes are not created and added to multi-valued references whose type
	 * is one of the EClasses. This allows paging list behavior to occur.
	 * 
	 * @param resource
	 *            The resource to load
	 * @param allObjects
	 *            If true, all contained objects are restored; if false, only
	 *            the top level objects are restored.
	 * @param notLoadedClasses
	 *            Instances of these classes are not created and set to
	 *            multi-valued references--this enables paging lists to be
	 *            created instead.
	 * @exception NoResourceTableException
	 * @exception ResourceNotFoundException
	 */
	void load(Resource resource, boolean allObjects, Collection notLoadedClasses) throws Exception;

	/**
	 * This method removes all EObjects from the database.
	 */
	void clear() throws Exception;

	/**
	 * This method removes all tables from the database.
	 */
	void dropTables() throws Exception;

	/**
	 * Creates the database tables and possibly other database objects,
	 * depending on the type of the database.
	 */
	void createTables() throws Exception;

	/**
	 * Shuts down the underlying database. You must invoke this method when you
	 * are finished so the database is updated correctly.
	 */
	void close() throws Exception;

	/**
	 * Sets the DBMap used by this Database object.
	 */
	void setDBMap(DBMap map);

	/**
	 * Creates an DBMap from the given EPackage.
	 */
	void setDBMap(EPackage model);

	/**
	 * Creates a DBMap from the given list of EPackages.
	 */
	void setDBMap(List packages);

	/**
	 * Get the TypeMap. Once you have the TypeMap, you can modify the returned
	 * TypeMap.
	 */
	TypeMap getTypeMap();

	/**
	 * Get the DatabaseType of the underlining database.
	 */
	DatabaseType getType();

	/**
	 * You can turn caching on by using this method; by default, caching is off.
	 */
	void setCaching(boolean caching);

	/**
	 * This method returns true if caching is on and false if caching is off.
	 */
	boolean isCaching();

	/**
	 * This method allows you to set the maximum size of the cache. By default,
	 * it is 10000.
	 */
	void setCacheSize(int maximum);

	/**
	 * This method returns the maximum cache size.
	 */
	int getCacheSize();

	/**
	 * Forces the updates, used when caching is enabled
	 */
	void forceUpdates() throws Exception;

	void setProperties(Properties properties);

	/**
	 * @return
	 */
	Properties getProperties();
	public JDBCHelper getJDBCHelper();
	public DBMap getDBMap();

} // Database
