/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TypeMap.java,v 1.4 2005/02/16 22:21:28 qiyanli Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.resources.database.internal;

import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.hyades.resources.database.internal.dbmodel.StringType;
/**
 * This interface represents the mapping from Ecore datatypes to SQL types. You
 * can also specify the SQL type for generated database ids.
 * <p>
 * You typically use the getTypeMap() method on a database object to obtain the
 * default type map, then modify the map.
 * <p>
 * You need to update a TypeMap before the database object creates a DBMap,
 * since it uses the TypeMap when it creates the DBMap.
 */
public interface TypeMap {

	/**
	 * This method specifies the mapping from an Ecore datatype with the given
	 * name to a SQL type defined by java.sql.Types. If there is an existing
	 * mapping from the Ecore datatype, it is removed as a result of using this
	 * method.
	 */
	void add(String ecoreDatatype, int sqlType);

	/**
	 * This method specifies the mapping from an EAttribute to a SQL type as
	 * defined by java.sql.Types. This can be used to override the default type
	 * mapping for a specific EAttribute. If you specify Type.VARCHAR, the
	 * default VARCHAR maxSQLLength will be used. To use a different VARCHAR
	 * maxSQLLength than the default one, use method add(EAttribute,
	 * StringType).
	 */
	void add(EAttribute attribute, int sqlType);

	/**
	 * This method specifies the mapping from an EAttribute to one of the SQL
	 * string types. This can be used to override the default type mapping for a
	 * specific EAttribute. Use this method to specify a SQL VARCHAR type with a
	 * specific maxSQLLength.
	 */
	void add(EAttribute attribute, StringType type);

	/**
	 * Returns the SQL type for the given Ecore datatype name. If there is no
	 * mapping, -999 is returned.
	 */
	int getSQLType(String ecoreDatatype);

	/**
	 * Returns the Integer or StringType for the given EAttribute. Returns null
	 * if one has not been registered with the TypeMap.
	 * 
	 * @param attribute
	 * @return
	 */
	Object getSQLType(EAttribute attribute);

	/**
	 * Gets the maxSQLLength to use for the SQL type VARCHAR. By default, it is
	 * 160.
	 */
	int getVarCharLength();

	/**
	 * Sets the maxSQLLength to use for the SQL type VARCHAR.
	 */
	void setVarCharLength(int length);

	/**
	 * Returns the SQL type to use for database ids. By default, it is INTEGER.
	 */
	int getIdType();

	/**
	 * Sets the SQL type to use for database ids. The valid values are any of
	 * the integer types or LONG.
	 */
	void setIdType(int type);
} // TypeMap
