/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DBCommandFactory.java,v 1.4 2005/02/16 22:21:29 qiyanli Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.resources.database.internal.extensions;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleSearchQuery;
import org.eclipse.hyades.models.hierarchy.extensions.TimeBasedCorrelationQuery;
import org.eclipse.hyades.resources.database.internal.DBMap;
import org.eclipse.hyades.resources.database.internal.InternalDatabase;
import org.eclipse.hyades.resources.database.internal.ObjectQuery;
import org.eclipse.hyades.resources.database.internal.ReferenceQuery;
import org.eclipse.hyades.resources.database.internal.impl.AddResourceCommand;
import org.eclipse.hyades.resources.database.internal.impl.ClearCommand;
import org.eclipse.hyades.resources.database.internal.impl.Contains;
import org.eclipse.hyades.resources.database.internal.impl.DBCommand;
import org.eclipse.hyades.resources.database.internal.impl.DatabaseExtensions;
import org.eclipse.hyades.resources.database.internal.impl.DeleteObjects;
import org.eclipse.hyades.resources.database.internal.impl.DetectResourceCommand;
import org.eclipse.hyades.resources.database.internal.impl.DropTableCommand;
import org.eclipse.hyades.resources.database.internal.impl.GetCommpressedPathByURICommand;
import org.eclipse.hyades.resources.database.internal.impl.GetIdByURICommand;
import org.eclipse.hyades.resources.database.internal.impl.GetIdsOfMatches;
import org.eclipse.hyades.resources.database.internal.impl.GetIdsOfReferencedObjects;
import org.eclipse.hyades.resources.database.internal.impl.GetIndex;
import org.eclipse.hyades.resources.database.internal.impl.GetLargestDatabaseIdCommand;
import org.eclipse.hyades.resources.database.internal.impl.GetObjectQuery;
import org.eclipse.hyades.resources.database.internal.impl.GetReferenceCountCommand;
import org.eclipse.hyades.resources.database.internal.impl.GetReferenceQuery;
import org.eclipse.hyades.resources.database.internal.impl.GetTopLevelObjects;
import org.eclipse.hyades.resources.database.internal.impl.LoadCommand;
import org.eclipse.hyades.resources.database.internal.impl.SetAttributeCommand;
import org.eclipse.hyades.resources.database.internal.impl.SetReferenceCommand;
import org.eclipse.hyades.resources.database.internal.impl.SimpleSearchCommand;
import org.eclipse.hyades.resources.database.internal.impl.TimeBasedCorrelationCommand;
import org.eclipse.hyades.resources.database.internal.impl.UpdateCommand;
import org.eclipse.hyades.resources.database.internal.impl.UpdateLargestDatabaseId;
import org.eclipse.hyades.resources.database.internal.impl.WeakObjectCache;
import org.eclipse.hyades.resources.database.util.IDatabaseExtensions;
/**
 * Creates DBCommand objects.
 */
public class DBCommandFactory {
	/**
	 * Constructor for DBCommandFactory.
	 */
	private DBCommandFactory() {
		super();
	}
	public static final DBCommandFactory INSTANCE = new DBCommandFactory();
	/**
	 * Creates a command that adds EObjects to the database.
	 */
	public DBCommand createAddCommand(JDBCHelper helper, DBMap map, DatabaseType type, List objects, WeakObjectCache cache, Resource resource) {
		IDatabaseExtensions extensions = ((IDatabaseExtensions) DatabaseExtensions.getInstance().get(type));
		if (extensions == null)
			return null;
		return extensions.createAddCommand(helper, map, objects, cache, resource);
	}
	/**
	 * Creates a command that deletes all EObjects from the database.
	 */
	public DBCommand createClearCommand(JDBCHelper helper, DBMap map) {
		return new ClearCommand(helper, map);
	}
	/**
	 * Creates a get command to retrieve objects from the database.
	 */
	public DBCommand createGetCommand(JDBCHelper helper, DBMap map, DatabaseType type, List classesAndIds, boolean getAllObjects, WeakObjectCache cache, Collection notLoadedClasses, String uri) {
		return createGetCommand(helper, map, type, classesAndIds, getAllObjects, cache, notLoadedClasses, uri, null);
	}
	public DBCommand createGetCommand(JDBCHelper helper, DBMap map, DatabaseType type, List classesAndIds, boolean getAllObjects, WeakObjectCache cache, Collection notLoadedClasses, String uri, Resource resource) {
		IDatabaseExtensions extensions = ((IDatabaseExtensions) DatabaseExtensions.getInstance().get(type));
		if (extensions == null)
			return null;
		return extensions.createGetCommand(helper, map, classesAndIds, getAllObjects, cache, notLoadedClasses, uri, resource);
	}
	/**
	 * Creates a command to retrieve objects based on an object query.
	 */
	public DBCommand createGetObjectQuery(JDBCHelper helper, DBMap map, DatabaseType type, ObjectQuery query, WeakObjectCache cache) {
		return new GetObjectQuery(helper, map, type, query, cache);
	}
	/**
	 * Creates a command to retrieve objects based on a reference query.
	 */
	public DBCommand createGetReferenceQuery(JDBCHelper helper, DBMap map, DatabaseType type, ReferenceQuery query, Integer id, WeakObjectCache cache) {
		return new GetReferenceQuery(helper, map, type, query, id, cache);
	}
	/**
	 * Creates a command to count the number of referenced objects for the
	 * object with the given database id and reference.
	 */
	public DBCommand createGetReferenceCount(JDBCHelper helper, DBMap map, Integer id, EReference reference) {
		return new GetReferenceCountCommand(helper, map, id, reference);
	}
	/**
	 * Creates a command to compute either the indexOf or lastIndexOf for a
	 * given object, reference, and referencedObject.
	 */
	public DBCommand createGetIndex(JDBCHelper helper, DBMap map, Integer sourceId, EReference reference, Integer targetId, boolean last) {
		return new GetIndex(helper, map, sourceId, reference, targetId, last);
	}
	/**
	 * Creates a command to determine whether the referencedId is in the given
	 * reference for the given object.
	 */
	public DBCommand createContains(JDBCHelper helper, DBMap map, Integer id, EReference reference, Integer referencedId) {
		return new Contains(helper, map, id, reference, referencedId);
	}
	/**
	 * Creates a command to get the ids of objects whose attribute values match
	 * the values in the given map and that are instances of the given EClass.
	 */
	public DBCommand createGetIdsOfMatches(JDBCHelper helper, DBMap map, DatabaseType type, EClass eClass, Map values) {
		return new GetIdsOfMatches(helper, map, type, eClass, values);
	}
	/**
	 * Creates a command that executes DDL statements to create tables and other
	 * database objects.
	 */
	public DBCommand createTableCommand(JDBCHelper helper, DBMap map, DatabaseType type) {
		IDatabaseExtensions extensions = ((IDatabaseExtensions) DatabaseExtensions.getInstance().get(type));
		if (extensions == null)
			return null;
		return extensions.createTableCommand(helper, map);
	}
	/**
	 * Creates a command that executes DROP TABLE statements for each table in
	 * the database.
	 */
	public DBCommand createDropTableCommand(JDBCHelper helper, DBMap map) {
		return new DropTableCommand(helper, map);
	}
	/**
	 * Creates a command that inserts rows into the resource table for each
	 * object.
	 */
	public DBCommand createAddResourceCommand(JDBCHelper helper, DBMap map, Resource resource, WeakObjectCache cache) {
		return new AddResourceCommand(helper, map, resource, cache);
	}
	/**
	 * Creates a command that determines if the given resource is already in the
	 * resource table.
	 */
	public DBCommand createDetectResourceCommand(JDBCHelper helper, DBMap map, Resource resource) {
		return new DetectResourceCommand(helper, map, resource);
	}
	/**
	 * Creates a command that loads the resource.
	 */
	public DBCommand createLoadCommand(JDBCHelper helper, DBMap map, DatabaseType type, Resource resource, boolean allObjects, Collection notLoadedClasses, WeakObjectCache cache) {
		return new LoadCommand(helper, map, type, resource, allObjects, notLoadedClasses, cache);
	}
	/**
	 * Creates a command that deletes a resource from the resource table.
	 */
	public DBCommand createDeleteResourceCommand(JDBCHelper helper, DBMap map, DatabaseType type, URI uri) {
		return new DeleteResourceCommand(helper, map, type, uri);
	}
	/**
	 * Creates a command that deletes objects from the database.
	 */
	public DBCommand createDeleteObjects(JDBCHelper helper, DBMap map,List classesAndIds) {
		return new DeleteObjects(helper, map,classesAndIds);
	}
	/**
	 * Creates a command that sets a reference.
	 */
	public DBCommand createSetReferenceCommand(JDBCHelper helper, DBMap map, DatabaseType type, Integer sourceId, EReference reference, Integer targetId, int position) {
		return new SetReferenceCommand(helper, map, type, sourceId, reference, targetId, position);
	}
	/**
	 * Creates a command to update the database with the given list of updates.
	 */
	public DBCommand createUpdateCommand(InternalDatabase database, List updates, WeakObjectCache cache) {
		return new UpdateCommand(database, updates, cache);
	}
	/**
	 * Creates a command to set the attribute of the object with the given id to
	 * the given value.
	 */
	public DBCommand createSetAttributeCommand(JDBCHelper helper, DBMap map, Integer id, EAttribute attribute, Object value) {
		return new SetAttributeCommand(helper, map, id, attribute, value);
	}
	/**
	 * Creates a command to get the ids of referenced objects given the ids of
	 * source objects.
	 */
	public DBCommand createGetIdsOfReferencedObjects(JDBCHelper helper, DBMap map, DatabaseType type, int[] ids, EReference reference, boolean order) {
		return new GetIdsOfReferencedObjects(helper, map, type, ids, reference, order);
	}
	/**
	 * Creates a command to get the ids of top level objects of the resource
	 * with the given URI.
	 */
	public DBCommand createGetTopLevelObjects(JDBCHelper helper, DBMap map, URI uri) {
		return new GetTopLevelObjects(helper, map, uri);
	}
	/**
	 * Creates a command to get the largest database id from the id table in the
	 * database.
	 */
	public DBCommand createGetLargestDatabaseId(JDBCHelper helper, DBMap map) {
		return new GetLargestDatabaseIdCommand(helper, map);
	}
	/**
	 * Store the given integer value in the id table as the largest database id.
	 */
	public DBCommand createUpdateLargestDatabaseId(JDBCHelper helper, DBMap map, int largest) {
		return new UpdateLargestDatabaseId(helper, map, largest);
	}
	/**
	 * @param query
	 * @param targetResourceSet
	 * @return
	 */
	public DBCommand createSimpleSearchCommand(JDBCHelper helper, DBMap dbMap, SimpleSearchQuery query, ResourceSet targetResourceSet,WeakObjectCache cache,Collection notLoadedClasses) {
		return new SimpleSearchCommand(helper, dbMap, query, targetResourceSet,cache,notLoadedClasses);
	}

	public DBCommand createGetCommpressedPathByURICommand(JDBCHelper helper, DBMap dbMap, String uri) {
		return new GetCommpressedPathByURICommand(helper, dbMap, uri);
	}
	public DBCommand createGetIdByURICommand(JDBCHelper helper, DBMap dbMap, String uri) {
		return new GetIdByURICommand(helper, dbMap, uri);
	}
	public DBCommand createTimeBaseCorrelationCommand(JDBCHelper helper, DBMap dbMap, TimeBasedCorrelationQuery query, ResourceSet targetResourceSet, WeakObjectCache cache, Collection notLoadedClasses) {
		return new TimeBasedCorrelationCommand(helper, dbMap, query, targetResourceSet,cache,notLoadedClasses);
	}
	public DBCommand createSetReferenceCommand(JDBCHelper helper, DBMap dbMap, DatabaseType type, Integer sourceId, EReference reference, List targets, List positions) {
		return new SetReferenceCommand(helper, dbMap, type, sourceId, reference, targets, positions);
	}
} // DBCommandFactory
