/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: JDBCHelper.java,v 1.14 2005/04/08 14:16:07 slavescu Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.resources.database.internal.extensions;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;
import java.net.URL;
import java.net.URLClassLoader;
import java.sql.BatchUpdateException;
import java.sql.Connection;
import java.sql.Driver;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Enumeration;
import java.util.List;
import java.util.Properties;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.resources.database.internal.DBCollectedExceptions;
import org.eclipse.hyades.resources.database.internal.impl.DBHyadesResourceExtension;
import org.eclipse.hyades.resources.database.internal.impl.SQLStatement;
/**
 * This class provides basic JDBC functionality.
 */
public abstract class JDBCHelper {
	public class DriverClassLoader extends URLClassLoader {
		public DriverClassLoader(URL urls[], ClassLoader parent) {
			super(urls, parent);
		}
		//		public Class getDriverClass(String name) throws
		// ClassNotFoundException {
		//			try {
		//				return loadClass(name, true);
		//			} catch (ClassNotFoundException ex) {
		//				ex.printStackTrace();
		//			}
		//			return Class.forName(name);
		//		}
		protected Class findClass(final String name) throws ClassNotFoundException {
			return super.findClass(name);
		}
	}
	protected Connection connection;
	protected boolean debug = false;
	protected String driver;
	protected Driver driverInstance;
	protected Writer file;
	//	protected Map preparedStatements = new HashMap();
	private Properties properties;
	protected String protocol;
	protected boolean toFileOnly = false;
	protected DatabaseType type;
	protected String urlString;
	public JDBCHelper(DatabaseType type) {
		this.type = type;
		try {
			String s = System.getProperties().getProperty("JDBCHelper.debug"); // use
			debug = Boolean.valueOf(s).booleanValue();
		} catch (Exception e) {
			//ignore and leave the default cacheSize
		}
		try {
			String s = System.getProperties().getProperty("JDBCHelper.toFileOnly"); // use
			toFileOnly = Boolean.valueOf(s).booleanValue();
		} catch (Exception e) {
			//ignore and leave the default cacheSize
		}
	}
	/**
	 * Close the database connection and shutdown the database. Returns true if
	 * the database was shut down successfully and false otherwise.
	 */
	public void close() throws SQLException {
		if (!getConnection().getAutoCommit())
			getConnection().commit();
		getConnection().close();
	}
	/**
	 *  
	 */
	public void commitTransaction() throws Exception {
		if (toFileOnly)
			return;
		if (!getConnection().getAutoCommit())
			getConnection().commit();
	}
	public PreparedStatement createPreparedStatement(String statement) throws Exception {
		//		PreparedStatement preparedStatement =
		// (PreparedStatement)preparedStatements.get(statement);
		//		if(preparedStatement==null)
		//			preparedStatement = getConnection().prepareStatement(statement);
		//		return preparedStatement;
		if (toFileOnly)
			return null;
		return getConnection().prepareStatement(statement);
	}
	public Statement createStatement() throws Exception {
		if (toFileOnly)
			return null;
		return getConnection().createStatement();
	}
	public Statement createStatement(int resultSetType, int resultSetConcurrency) throws Exception {
		if (toFileOnly)
			return null;
		return getConnection().createStatement(resultSetType, resultSetConcurrency);
	}
	public Statement createStatement(int resultSetType, int resultSetConcurrency, int resultSetHoldability) throws Exception {
		if (toFileOnly)
			return null;
		return getConnection().createStatement(resultSetType, resultSetConcurrency, resultSetHoldability);
	}
	/**
	 * @param sqlStatement
	 * @param statement
	 */
	public synchronized void execute(Statement sqlStatement, String statement) throws Exception {
		if (debug == true)
			System.out.println(statement);
		if (toFileOnly == true)
			writeToFile(statement);
		else
			try {
				sqlStatement.execute(statement);
			} catch (Exception e) {
				LoadersUtils.log(e, "------------ JDBCHelper.execute sqlString=" + statement);
				sqlStatement.close();
				throw e;
			}
		sqlStatement.close();
	}
	public synchronized int[] executeBatchedPreparedStatement(PreparedStatement sqlStatement) throws Exception {
		if (debug == true)
			LoadersUtils.log(sqlStatement.toString());
		if (toFileOnly == true) {
			writeToFile(printPreparedStatementParameters(sqlStatement));
			return new int[0];
		} else {
			int[] res = null;
			try {
				res = sqlStatement.executeBatch();
			} catch (BatchUpdateException e) {
				LoadersUtils.log(e, "------------ JDBCHelper.executeBatchedPreparedStatement");
				SQLException e1 = e;
				int i = 1;
				while ((e1 = e1.getNextException()) != null) {
					LoadersUtils.log(e1, "------------ " + (i++) + "-------------");
				}
				sqlStatement.close();
				throw e;
			}
			return res;
		}
	}
	/**
	 * @param statement
	 */
	public synchronized boolean executePreparedStatement(PreparedStatement sqlStatement) throws Exception {
		boolean res = true;
		if (debug == true)
			System.out.println(sqlStatement);
		if (toFileOnly == true) {
			writeToFile(printPreparedStatementParameters(sqlStatement));
			return true;
		} else {
			try {
				res = sqlStatement.execute();
			} catch (Exception e) {
				LoadersUtils.log(e, "------------ JDBCHelper.executePreparedStatement");
				sqlStatement.close();
				throw e;
			}
			sqlStatement.close();
		}
		return res;
	}
	/**
	 * @param sqlStatement
	 * @param statement2
	 * @return
	 */
	public synchronized ResultSet executeQuery(Statement sqlStatement, String statement) throws Exception {
		ResultSet res = null;
		if (debug == true)
			System.out.println(statement);
		if (toFileOnly == true)
			writeToFile(statement);
		else {
			try {
				res = sqlStatement.executeQuery(statement);
			} catch (Exception e) {
				LoadersUtils.log(e, "------------ JDBCHelper.executeQuery sqlString=" + statement);
				sqlStatement.close();
				throw e;
			}
		}
		return res;
	}
	public synchronized void executeStatement(String statement) throws Exception {
		if (debug == true)
			System.out.println(statement);
		if (toFileOnly == true)
			writeToFile(statement);
		else {
			Statement s = getConnection().createStatement();
			execute(s, statement);
			s.close();
		}
	}
	/**
	 * @param s
	 * @param sqlStatement
	 */
	public synchronized void executeUpdate(Statement sqlStatement, String statement) throws Exception {
		if (debug == true)
			System.out.println(statement);
		if (toFileOnly == true)
			writeToFile(statement);
		else {
			try {
				sqlStatement.executeUpdate(statement);
			} catch (Exception e) {
				LoadersUtils.log(e, "------------ JDBCHelper.executeUpdate sqlString=" + statement);
				sqlStatement.close();
				throw e;
			}
			sqlStatement.close();
		}
	}
	public synchronized void executeUpdateStatement(String statement) throws Exception {
		if (debug == true)
			System.out.println(statement);
		if (toFileOnly == true)
			writeToFile(statement);
		else {
			Statement s = getConnection().createStatement();
			executeUpdate(s, statement);
			s.close();
		}
	}
	public void executeUpdateStatements(List statements) throws Exception {
		for (int i = 0, l = statements.size(); i < l; i++) {
			SQLStatement statement = (SQLStatement) statements.get(i);
			executeUpdateStatement(statement.getStatement());
		}
	}
	public void executeUpdateStatements(String[] statements) throws Exception {
		for (int i = 0; i < statements.length; i++)
			executeUpdateStatement(statements[i]);
	}
	public Connection getConnection() {
		try {
			if (connection == null || connection.isClosed()) {
//				Properties newProperties = HyadesResourceExtensions.getInstance().getProperties();
//				properties = (Properties) newProperties.clone();
//				openForExistingDB(DBHyadesResourceExtension.DB_NAME, properties);
//			} else {
//				Properties newProperties = HyadesResourceExtensions.getInstance().getProperties();
//				if (!newProperties.equals(properties)) {
//					properties = (Properties) newProperties.clone();
//				openForExistingDB(DBHyadesResourceExtension.DB_NAME, properties);
//				}
				openForExistingDB(DBHyadesResourceExtension.DB_NAME, properties);
			}
		} catch (Exception e) {
			throw new DBCollectedExceptions(e);
		}
		return connection;
	}
	/**
	 * @return Returns the database type.
	 */
	public DatabaseType getDatabaseType() {
		return type;
	}
	/**
	 * @param properties
	 * @return
	 */
	protected String getLocation(Properties properties) {
		String location = properties.getProperty("location");
		if (location == null || location.length() == 0)
			location = "";
		else
			location = "//" + location + "/";
		return location;
	}
	public Driver loadJDBCDriver() throws Exception {
		//		Class.forName(driver).newInstance();
		if (driverInstance != null)
			return driverInstance;
		return loadJDBCDriverSpecial();
	}
	protected Driver loadJDBCDriverSpecial() throws Exception {
		//		ClassLoader cl = new URLClassLoader(new URL[]{new
		// URL("file://E:/test/workspaces/defects/hyades-3.0-rdb/com.ibm.etools.resources.database.extension/lib/db2j.jar")},
		// getClass().getClassLoader());
		if (driverInstance != null)
			return driverInstance;
		DriverClassLoader cl = new DriverClassLoader(new URL[]{new URL(urlString)}, getClass().getClassLoader());
		//		Class clazz = cl.getDriverClass(driver);
		//		System.out.println("JDBCHelper.loadJDBCDriverSpecial() - before");
		//		driverInstance = (Driver) clazz.newInstance();
		//		DriverManager.registerDriver(driverInstance);
		//		System.out.println("JDBCHelper.loadJDBCDriverSpecial() - after");
		try {
			// If this is the first time the class is loaded, it registers
			// itself
			// with DriverManager
			Class driverClass = Class.forName(driver, true, cl);
			// In case we unloaded our own driver:
			Enumeration enum1 = DriverManager.getDrivers();
			while (enum1.hasMoreElements()) {
				Driver element = (Driver) enum1.nextElement();
				if (element.getClass() == driverClass) {
					driverInstance = element;
				}
			}
			if (driverInstance == null) {
				driverInstance = (Driver) driverClass.newInstance();
				DriverManager.registerDriver(driverInstance);
			}
		} catch (Exception exc) {
			throw exc;
		}
		return driverInstance;
	}
	/**
	 * Load the JDBC driver and open the connection to the named database.
	 */
	public void openForExistingDB(String name, Properties properties) throws Exception {
		loadJDBCDriver();
		if (!toFileOnly)
			connection = driverInstance.connect(protocol + getLocation(properties) + name, properties);
	}
	/**
	 * Load the JDBC driver and open the connection to the named database,
	 * creating it if it is not already created. NOTE: Currently, this only
	 * works for Cloudscape.
	 */
	public void openForNewDB(String name, Properties properties) throws Exception {
		loadJDBCDriver();
		connection = driverInstance.connect(protocol + getLocation(properties) + name + ";create=true", properties);
	}
	/**
	 * @param statement
	 * @return
	 */
	private String printPreparedStatementParameters(PreparedStatement statement) throws Exception {
		String res = statement.toString();
		boolean first = true;
		try {
			res += ": PreparedStatement.resultSetMetaData=";
			ResultSetMetaData metaData = statement.getMetaData();
			if (metaData == null) {
				res += metaData;
			} else
				for (int i = 0; i < metaData.getColumnCount(); i++) {
					if (first) {
						first = false;
					} else {
						res += ", ";
					}
					res += "col" + i + "=" + metaData.getTableName(i) + "." + metaData.getColumnLabel(i);
				}
			//			ParameterMetaData parameterMetaData = ps.getParameterMetaData();
			//			res+="\n PreparedStatement.parameterMetaData=";
			//			first=true;
			//			if(metaData==null)
			//			{
			//				res+=parameterMetaData;
			//			}
			//			else
			//				for (int i = 0; i < parameterMetaData.getParameterCount(); i++) {
			//					if(first)
			//					{
			//						first=false;
			//					}
			//					else
			//					{
			//						res+=", ";
			//					}
			//					res+="param"+i+"="+parameterMetaData.getParameterTypeName(i);
			//				}
		} catch (Exception e) {
			e.printStackTrace();
		}
		return res;
	}
	/**
	 *  
	 */
	public void rollbackTransaction() throws Exception {
		if (!getConnection().getAutoCommit())
			getConnection().rollback();
	}
	public void setAutoCommit(boolean value) throws Exception {
		if (toFileOnly)
			return;
		getConnection().setAutoCommit(value);
	}
	/**
	 * @param hold_cursors_over_commit
	 */
	public void setHoldability(int hold_cursors_over_commit) throws Exception {
		if (toFileOnly)
			return;
		try {
			getConnection().setHoldability(hold_cursors_over_commit);
		} catch (Exception e) {
			// ignore exception
		}
	}
	/**
	 * Shuts down the database. This is used by Cloudscape in embedded mode.
	 * Returns true if the shutdown was successful; false otherwise.
	 */
	public boolean shutdown() {
		boolean exceptionThrown = false;
		try {
			DriverManager.getConnection(protocol + ";shutdown=true");
		} catch (SQLException exception) {
			exceptionThrown = true;
		}
		return exceptionThrown;
	}
	/**
	 * @param sqlString
	 */
	protected void writeToFile(String sqlString) {
		try {
			if (file == null) {
				file = new FileWriter("/JDBCHelper_toFile_" + (new java.util.Date()).getTime() + ".sql");
			}
			file.write(sqlString + "\n");
			file.flush();
		} catch (IOException e) {
			e.printStackTrace();
			throw new RuntimeException(e);
		}
	}
	/**
	 * @param properties
	 */
	public void setProperties(Properties properties) {
		if(this.properties!=null && this.properties.equals(properties))
		{
			return;
		}
		else
		{
			this.properties=properties;
			if(connection!=null)
			{
				try {
					connection.close();
				} catch (SQLException e) {
					throw new DBCollectedExceptions(e);
				}
				connection=null;
				getConnection();
			}
		}
	}
} // JDBCHelper
