/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DynamicPagingListImpl.java,v 1.4 2005/02/16 22:21:31 qiyanli Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.resources.database.internal.impl;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import org.eclipse.emf.common.notify.NotificationChain;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.InternalEObject;
import org.eclipse.emf.ecore.util.EContentAdapter;
import org.eclipse.emf.ecore.util.DelegatingEcoreEList.Dynamic;
import org.eclipse.hyades.models.hierarchy.util.PerfUtil;
import org.eclipse.hyades.resources.database.internal.DBCollectedExceptions;
import org.eclipse.hyades.resources.database.internal.DBResource;
import org.eclipse.hyades.resources.database.internal.Database;
import org.eclipse.hyades.resources.database.internal.InternalDatabase;
import org.eclipse.hyades.resources.database.internal.PagingList;
import org.eclipse.hyades.resources.database.internal.QueryFactory;
import org.eclipse.hyades.resources.database.internal.ReferenceQuery;
/**
 * This class implements the PagingList interface and extends the
 * DelegatingEcoreEList.Dynamic class. This way, an instance of this class can
 * be returned in the generated and reflective EMF APIs.
 */
public class DynamicPagingListImpl extends Dynamic implements PagingList {
	protected int pagingSize, first;
	protected InternalDatabase database;
	protected EObject[] objects;
	protected boolean loaded;
	protected boolean setReferences;
	protected int size;
	protected boolean modified;
	public DynamicPagingListImpl(InternalEObject owner, EStructuralFeature eStructuralFeature, DBResource resource, int pagingSize) {
		super(owner, eStructuralFeature);
		init(pagingSize);
		this.database = (InternalDatabase) resource.getDatabase();
	}
	public DynamicPagingListImpl(InternalEObject owner, EStructuralFeature eStructuralFeature, Database database, int pagingSize) {
		super(owner, eStructuralFeature);
		init(pagingSize);
		this.database= (InternalDatabase) database;
	}
	protected void init(int pagingSize) {
		this.pagingSize = pagingSize;
		first = -1;
		//		lastIndex = -1;
		size = -1;
		loaded = false;
		setReferences = true;
		modified = false;
	}
	/**
	 * @see org.eclipse.emf.common.util.DelegatingEList#delegateList()
	 */
	protected List delegateList() {
		return this;
	}
	public void set(Object newValue) {
		throw new UnsupportedOperationException();
	}

	public void unset() {
		throw new UnsupportedOperationException();
	}
	public NotificationChain basicRemove(Object object, NotificationChain notifications) {
		throw new UnsupportedOperationException();
	}
	public NotificationChain basicAdd(Object object, NotificationChain notifications) {
		return notifications;
		//throw new UnsupportedOperationException();
	}
	protected boolean delegateContains(Object object) {
		if (!(object instanceof EObject))
			return false;
		boolean result = false;
		InternalEObject eObject = (InternalEObject) object;
		if (isContainment()) {
			result = eObject.eContainer() == owner && (hasNavigableInverse() ? eObject.eContainerFeatureID() == getInverseFeatureID() : InternalEObject.EOPPOSITE_FEATURE_BASE - eObject.eContainerFeatureID() == getFeatureID());
		}
		// We can also optimize single valued reverse.
		//
		else if (hasNavigableInverse() && !hasManyInverse())
			result = ((EObject) object).eGet(getInverseEReference()) == owner;
		if (result)
			return result;
		else {
			try {
				return getInternalDatabase().contains(getObject(), getReference(), eObject);
			} catch (Exception e) {
				throw new DBCollectedExceptions(e);
			}
		}
	}
	/**
	 * @see java.util.List#get(int)
	 */
	protected Object delegateGet(int index) {
		if (!loaded || objects == null || index >= first + objects.length)
			load(index);
		else if (index < first) {
			int start = first - pagingSize;
			if (start < 0)
				start = 0;
			else if (index >= 0 && index < start)
				start = index;
			load(start);
		}
		if (objects[index - first] == null) {
			if (size() > index) {
				load(first);
			} else
				throw new IndexOutOfBoundsException();
		}
		// TODO : MS Check if the following commented statement is still valid
		//		if (inDatabase() && !modified && ((index == size() - 1 && index -
		// lastIndex == 1) || (index == 0 && lastIndex == 1))) {
		//			Object object = objects[index - first];
		//			objects = null;
		//			loaded = false;
		//			return object;
		//		}
		//
		//		lastIndex = index;
		return objects[index - first];
	}
	/* (non-Javadoc)
	 * @see org.eclipse.emf.common.util.DelegatingEList#delegateIterator()
	 */
	protected Iterator delegateIterator() {
		//return an empty iterator for now.
		// TODO MS - implement a real PagingList iterator
		return (new ArrayList()).iterator();
	}
	
	/**
	 * Returns the value of dbResource.isInDatabase(); returns true if resource
	 * is null.
	 */
	protected boolean inDatabase() {
		DBResource resource = (DBResource) owner.eResource();
		if (resource == null)
			return true;
		else
			return resource.isInDatabase();
	}
	/**
	 * Loads pagingSize objects from the getInternalDatabase() starting with the given index.
	 * 
	 * @param index
	 */
	protected void load(int index) {
		PerfUtil p = new PerfUtil("DynamicPagingListImpl.load() index=" + index, true);
		if (getInternalDatabase().isCaching()) {
			try {
				getInternalDatabase().forceUpdates();
			} catch (Exception e) {
				throw new DBCollectedExceptions(e);
			}
		}
		loaded = true;
		ReferenceQuery query = createReferenceQuery(index, index + pagingSize - 1);
		EObject[] referenced = getObjects(query);
		if (referenced == null)
			throw new IndexOutOfBoundsException();
		for (int i = 0; i < referenced.length; i++)
			addAdapter(referenced[i]);
		first = index;
		if (objects == null) {
			if (index == 0 && referenced.length < pagingSize) {
				objects = new EObject[referenced.length];
			} else {
				objects = new EObject[pagingSize];
			}
		}
		System.arraycopy(referenced, 0, objects, 0, referenced.length);
		for (int i = referenced.length; i < objects.length; i++)
			objects[i] = null;
		p.stopAndPrintStatus();
	}
	protected EObject[] getObjects(ReferenceQuery query) {
		EObject[] referenced = null;
		try {
			referenced = getInternalDatabase().getObjects(query);
		} catch (Exception e) {
			throw new DBCollectedExceptions(e);
		}
		return referenced;
	}
	protected ReferenceQuery createReferenceQuery(int lower, int upper) {
		ReferenceQuery query = QueryFactory.INSTANCE.createReferenceQuery();
		query.setEObject(getObject());
		query.setEReference(getReference());
		query.setLower(lower);
		query.setUpper(upper);
		if (!setReferences)
			query.setReferences(setReferences);
		Set notLoadedClasses = new HashSet(1);
		notLoadedClasses.add(getReference().getEReferenceType());
		query.setNotLoadedClasses(notLoadedClasses);
		return query;
	}
	/**
	 * @see java.util.Collection#size()
	 */
	protected int delegateSize() {
		if (size == -1)
			try {
				size = getInternalDatabase().count(getObject(), getReference());
				if (size == -1)
					size = 0;
			} catch (Exception e) {
				throw new DBCollectedExceptions(e);
			}
		return size;
	}
	/**
	 * @see org.eclipse.hyades.resources.getInternalDatabase().PagingList#setPagingSize(int)
	 */
	public void setPagingSize(int value) {
		if (value == pagingSize)
			return;
		boolean increase = false;
		if (!loaded)
			objects = new EObject[value];
		else if (value < pagingSize) {
			EObject[] oldObjects = objects;
			objects = new EObject[value];
			System.arraycopy(oldObjects, 0, objects, 0, value);
		} else if (value > pagingSize)
			increase = true;
		pagingSize = value;
		if (loaded && increase) {
			objects = new EObject[value];
			load(first);
		}
	}
	/**
	 * @see org.eclipse.hyades.resources.getInternalDatabase().PagingList#getPagingSize()
	 */
	public int getPagingSize() {
		return pagingSize;
	}
	/**
	 * @see org.eclipse.hyades.resources.getInternalDatabase().PagingList#setDatabase(org.eclipse.hyades.resources.getInternalDatabase().getInternalDatabase())
	 */
	public void setDatabase(Database database) {
		this.database = (InternalDatabase) database;
	}
	/**
	 * @see org.eclipse.hyades.resources.getInternalDatabase().PagingList#getInternalDatabase()
	 */
	public Database getDatabase() {
		return getInternalDatabase();
	}
	/**
	 * @see org.eclipse.hyades.resources.getInternalDatabase().PagingList#getObject()
	 */
	public EObject getObject() {
		return owner;
	}
	/**
	 * @see org.eclipse.hyades.resources.getInternalDatabase().PagingList#getReference()
	 */
	public EReference getReference() {
		return (EReference) eStructuralFeature;
	}
	protected boolean delegateIsEmpty() {
		return size() == 0;
	}
	public void setReferences(boolean setReferences) {
		this.setReferences = setReferences;
	}
	protected void delegateAdd(Object object) {
		add(size(), object);
	}
	protected void delegateAdd(int index, Object element) {
		if (!loaded) {
			loaded = true;
			objects = new EObject[pagingSize];
		} else if (index >= first + objects.length) {
			if (objects.length < pagingSize) {
				EObject[] newObjects = new EObject[pagingSize];
				System.arraycopy(objects, 0, newObjects, 0, objects.length);
				objects = newObjects;
			}
		}
		if (first == -1 || index >= first + objects.length) {
			first = index;
			objects[0] = (EObject) element;
			for (int i = 1; i < objects.length; i++)
				objects[i] = null;
			++size;
			modified = true;
		} else if (index < first) {
			throw new UnsupportedOperationException();
			// Not supported yet; only adding at end is supported.
		} else {
			if (objects[index - first] != null)
				throw new UnsupportedOperationException();
			objects[index - first] = (EObject) element;
			++size;
			modified = true;
		}
	}
	protected Object[] delegateToArray() {
		if (objects != null && objects.length < pagingSize) {
			Object[] dbObjects = new Object[objects.length];
			System.arraycopy(objects, 0, dbObjects, 0, objects.length);
			return dbObjects;
		}
		if (size() == 0)
			return new Object[0];
		ReferenceQuery query = createReferenceQuery(0, -1);
		EObject[] dbObjects = getObjects(query);
		if (objects == null && dbObjects.length < pagingSize) {
			loaded = true;
			first = 0;
			objects = new EObject[dbObjects.length];
			System.arraycopy(dbObjects, 0, objects, 0, dbObjects.length);
			size = dbObjects.length;
		}
		return dbObjects;
	}
	protected Object[] delegateToArray(Object[] array) {
		EObject[] dbObjects = null;
		if (objects != null && objects.length < pagingSize)
			dbObjects = objects;
		if (dbObjects == null) {
			if (size() == 0)
				dbObjects = new EObject[0];
			else {
				ReferenceQuery query = createReferenceQuery(0, -1);
				dbObjects = getObjects(query);
			}
		}
		if (array.length < dbObjects.length)
			array = (Object[]) java.lang.reflect.Array.newInstance(array.getClass().getComponentType(), dbObjects.length);
		System.arraycopy(dbObjects, 0, array, 0, dbObjects.length);
		if (array.length > dbObjects.length)
			array[dbObjects.length] = null;
		if (objects == null && dbObjects.length < pagingSize) {
			loaded = true;
			first = 0;
			objects = new EObject[dbObjects.length];
			System.arraycopy(dbObjects, 0, objects, 0, dbObjects.length);
			size = dbObjects.length;
		}
		return array;
	}
	protected int delegateIndexOf(Object element) {
		if (!(element instanceof EObject))
			return -1;
		EObject eObject = (EObject) element;
		try {
			return getInternalDatabase().indexOf(getObject(), getReference(), eObject);
		} catch (Exception e) {
			throw new DBCollectedExceptions(e);
		}
	}
	protected int delegateLastIndexOf(Object element) {
		if (!(element instanceof EObject))
			return -1;
		EObject eObject = (EObject) element;
		try {
			return getInternalDatabase().lastIndexOf(getObject(), getReference(), eObject);
		} catch (Exception e) {
			throw new DBCollectedExceptions(e);
		}
	}
	/**
	 * @return
	 */
	protected InternalDatabase getInternalDatabase() {
		if(database.isCaching())
		{
			try {
				database.forceUpdates();
			} catch (Exception e) {
				throw new DBCollectedExceptions(e);
			}
		}
		return database;
	}
	protected void addAdapter(EObject object) {
		if (object == null)
			return;
		DBResource resource = (DBResource) object.eResource();
		if (resource == null)
			return;
		EContentAdapter contentAdapter = resource.getContentAdapter();
		if (contentAdapter == null)
			return;
		if (!object.eAdapters().contains(contentAdapter))
			object.eAdapters().add(contentAdapter);
	}
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.emf.common.util.DelegatingEList#delegateClear()
	 */
	protected void delegateClear() {
		init(pagingSize);
	}
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.emf.common.util.DelegatingEList#delegateRemove(int)
	 */
	protected Object delegateRemove(int index) {
		throw new UnsupportedOperationException();
	}
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.emf.common.util.DelegatingEList#delegateSet(int,
	 *      java.lang.Object)
	 */
	protected Object delegateSet(int index, Object object) {
		throw new UnsupportedOperationException();
	}
} // DynamicPagingListImpl
