/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: GetIdsOfReferencedObjects.java,v 1.4 2005/02/16 22:21:31 qiyanli Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.resources.database.internal.impl;

import java.sql.ResultSet;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.emf.ecore.EReference;
import org.eclipse.hyades.resources.database.internal.DBMap;
import org.eclipse.hyades.resources.database.internal.extensions.DatabaseType;
import org.eclipse.hyades.resources.database.internal.extensions.JDBCHelper;
/**
 * Returns the ids of referenced objects given an array of ids for the source
 * objects and the reference. If ordered is false, a list of ids is returned
 * containing all of the referenced objects; if ordered is true, then an array
 * of ids is returned for each source object.
 */
public class GetIdsOfReferencedObjects extends DBCommand {
	protected DatabaseType type;
	protected int[] ids;
	protected EReference reference;
	protected boolean order;

	public GetIdsOfReferencedObjects(JDBCHelper helper, DBMap map, DatabaseType type, int[] ids, EReference reference, boolean order) {
		super(helper, map);
		this.type = type;
		this.ids = ids;
		this.reference = reference;
		this.order = order;
	}

	public Object execute() throws Exception {
		String[] queries = createQueries();

		if (order)
			return executeOrderedQueries(queries);
		else
			return executeUnorderedQueries(queries);
	}

	protected int[][] executeOrderedQueries(String[] queries) throws Exception {
		int[][] referencedIds = new int[ids.length][];

		for (int i = 0; i < queries.length; i++) {
			if (debug)
				System.out.println(queries[i]);

			Statement queryStatement = helper.createStatement();
			ResultSet rs = helper.executeQuery(queryStatement, queries[i]);
			List idsForObject = new ArrayList();
			int currentId = -1;
			int index = -1;

			while (rs.next()) {
				int source = rs.getInt(1);
				int target = rs.getInt(2);

				if (source != currentId) {
					currentId = source;

					if (index != -1) {
						referencedIds[index] = createIds(idsForObject);
						idsForObject.clear();
					}

					index = find(currentId, ids);
				}

				idsForObject.add(new Integer(target));
			}

			rs.close();
			queryStatement.close();

			if (index != -1)
				referencedIds[index] = createIds(idsForObject);
		}

		return referencedIds;
	}

	protected List executeUnorderedQueries(String[] queries) throws Exception {
		List referencedIds = new ArrayList();

		for (int i = 0; i < queries.length; i++) {
			if (debug)
				System.out.println(queries[i]);

			Statement queryStatement = helper.createStatement();
			ResultSet rs = helper.executeQuery(queryStatement, queries[i]);

			while (rs.next()) {
				int target = rs.getInt(2);

				if (target > 0)
					referencedIds.add(new Integer(target));
			}

			rs.close();
			queryStatement.close();
		}

		return referencedIds;
	}

	protected String[] createQueries() {
		StatementFactory factory = StatementFactory.INSTANCE;
		QueryStatement query = factory.createGreedyReferenceQuery(ids, reference, dbMap, type, order);
		String queryString = query.getStatement();
		String[] queryStrings;

		if (queryString != null) {
			queryStrings = new String[1];
			queryStrings[0] = queryString;
		} else
			queryStrings = query.getStatements();

		return queryStrings;
	}

	protected int[] createIds(List idsForObject) {
		int[] ids = new int[idsForObject.size()];

		for (int i = 0, l = idsForObject.size(); i < l; i++)
			ids[i] = ((Integer) idsForObject.get(i)).intValue();

		return ids;
	}

	protected int find(int id, int[] ids) {
		for (int i = 0; i < ids.length; i++)
			if (id == ids[i])
				return i;

		return -1;
	}
} // GetIdsOfReferencedObjects
