/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: RDBHelper.java,v 1.4 2005/02/16 22:21:31 qiyanli Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.resources.database.internal.impl;

import java.sql.Types;
import java.util.List;

import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.hyades.resources.database.internal.TypeMap;
import org.eclipse.hyades.resources.database.internal.dbmodel.Column;
import org.eclipse.hyades.resources.database.internal.dbmodel.Constraint;
import org.eclipse.hyades.resources.database.internal.dbmodel.Database;
import org.eclipse.hyades.resources.database.internal.dbmodel.DbmodelFactory;
import org.eclipse.hyades.resources.database.internal.dbmodel.SQLType;
import org.eclipse.hyades.resources.database.internal.dbmodel.StringType;
import org.eclipse.hyades.resources.database.internal.dbmodel.Table;
/**
 * This class provides convenience methods for working with the database model
 * constructs.
 */
public class RDBHelper {
	protected static final int COLUMN_NAME_LENGTH = 30;
	protected static final String PRIMARY_KEY_TYPE = "PRIMARYKEY";
	protected static final String FOREIGN_KEY_TYPE = "FOREIGNKEY";
	protected static final String INDEX_TYPE = "INDEX";
	protected static final char[] vowels = {'a', 'e', 'i', 'o', 'u'};
	protected DbmodelFactory dbFactory;

	/**
	 * Constructor for RDBHelper.
	 */
	public RDBHelper() {
		super();
	}

	/**
	 * Create a table with the given name for the given database.
	 */
	public Table createTable(Database database, String name) {
		Table table = getFactory().createTable();
		table.setName(name);
		database.getTables().add(table);
		return table;
	}

	/**
	 * Get the factory for the database model.
	 */
	protected DbmodelFactory getFactory() {
		if (dbFactory == null)
			dbFactory = DbmodelFactory.eINSTANCE;

		return dbFactory;
	}

	/**
	 * Add a column to the given table with the given name.
	 */
	public Column addColumnToTable(Table table, String name) {
		Column column = getFactory().createColumn();
		column.setName(getLegalColumnName(name));
		column.setAllowNull(true);
		table.getColumns().add(column);
		return column;
	}

	/**
	 * Shortens the name by removing vowels, if possible.
	 */
	protected String getLegalColumnName(String name) {
		if (name.equals("iD")) {
			name = "emf_" + name;
		}
		if (name.length() <= COLUMN_NAME_LENGTH)
			return name;

		int excess = name.length() - COLUMN_NAME_LENGTH;
		StringBuffer newName = new StringBuffer(name);
		int index = newName.length() - 1;

		while (newName.length() > COLUMN_NAME_LENGTH && index > 0) {
			char c = newName.charAt(index);

			if (isVowel(c))
				newName.delete(index, index + 1);
			else
				--index;
		}
		return newName.toString();
	}

	protected boolean isVowel(char c) {
		for (int i = 0; i < vowels.length; i++)
			if (c == vowels[i])
				return true;

		return false;
	}

	public void setColumnType(Column column, String typeName, TypeMap typeMap) {
		int sqlType = typeMap.getSQLType(typeName);

		if (sqlType == -999)
			sqlType = Types.VARCHAR;

		setColumnTypeToSQLType(column, sqlType, typeMap);
	}

	protected void setColumnTypeToSQLType(Column column, int sqlType, TypeMap typeMap) {
		switch (sqlType) {
			case Types.INTEGER :
				SQLType type1 = dbFactory.createSQLType();
				type1.setSqlType(Types.INTEGER);
				column.setType(type1);
				break;
			case Types.SMALLINT :
				SQLType type2 = dbFactory.createSQLType();
				type2.setSqlType(Types.SMALLINT);
				column.setType(type2);
				break;
			case Types.BIGINT :
				SQLType type3 = dbFactory.createSQLType();
				type3.setSqlType(Types.BIGINT);
				column.setType(type3);
				break;
			case Types.CHAR :
				StringType type4 = dbFactory.createStringType();
				type4.setLength(1);
				type4.setSqlType(Types.CHAR);
				column.setType(type4);
				break;
			case Types.DOUBLE :
				SQLType type5 = dbFactory.createSQLType();
				type5.setSqlType(Types.DOUBLE);
				column.setType(type5);
				break;
			case Types.LONGVARCHAR :
				StringType type7 = dbFactory.createStringType();
				type7.setSqlType(Types.LONGVARCHAR);
				column.setType(type7);
				break;
			default :
				StringType type6 = dbFactory.createStringType();
				type6.setLength(typeMap.getVarCharLength());
				type6.setSqlType(Types.VARCHAR);
				column.setType(type6);
				break;
		}
	}

	public void setColumnType(Column column, EAttribute attribute, TypeMap typeMap) {
		Object type = typeMap.getSQLType(attribute);

		if (type == null)
			setColumnType(column, attribute.getEType().getName(), typeMap);
		else if (type instanceof Integer)
			setColumnTypeToSQLType(column, ((Integer) type).intValue(), typeMap);
		else if (type instanceof StringType)
			column.setType((StringType) type);
	}

	public void addPrimaryKeyToTable(Table table, Column primaryKey, String name) {
		Constraint constraint = addConstraintToTable(table, name);
		primaryKey.getConstraints().add(constraint);
	}

	protected Constraint addConstraintToTable(Table table, String name) {
		Constraint constraint = getFactory().createConstraint();
		constraint.setName(name);
		constraint.setType(PRIMARY_KEY_TYPE);
		table.getConstraints().add(constraint);
		return constraint;
	}

	/**
	 * Add an index to the given table with the given name.
	 */
	public Constraint addIndexToTable(Table table, String name) {
		Constraint constraint = getFactory().createConstraint();
		constraint.setName(name);
		constraint.setType(INDEX_TYPE);
		table.getConstraints().add(constraint);
		return constraint;
	}

	/**
	 * Return the primary key column for the given table, or null if there is no
	 * primary key column.
	 */
	public Column getPrimaryKey(Table table) {
		List columns = table.getColumns();

		for (int i = 0, l = columns.size(); i < l; i++) {
			Column column = (Column) columns.get(i);
			List constraints = column.getConstraints();

			if (!constraints.isEmpty())
				for (int j = 0, l2 = constraints.size(); j < l2; j++) {
					Constraint constraint = (Constraint) constraints.get(j);

					if (constraint.getType().equals(PRIMARY_KEY_TYPE))
						return column;
				}
		}

		return null;
	}
	public Column getColumnByName(Table table, String columnName) {
		List columns = table.getColumns();

		for (int i = 0, l = columns.size(); i < l; i++) {
			Column column = (Column) columns.get(i);
			if (column.getName().equals(columnName)) {
				return column;
			}
		}

		return null;
	}

	/**
	 * Return the Constraint for the primary key column of the given table.
	 */
	public Constraint getPrimaryKeyConstraint(Table table) {
		List constraints = table.getConstraints();

		for (int i = 0, l = constraints.size(); i < l; i++) {
			Constraint constraint = (Constraint) constraints.get(i);

			if (constraint.getType().equals(PRIMARY_KEY_TYPE))
				return constraint;
		}

		return null;
	}
} // RDBHelper
