/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: StatementFactory.java,v 1.4 2005/02/16 22:21:32 qiyanli Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.resources.database.internal.impl;

import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleSearchQuery;
import org.eclipse.hyades.resources.database.internal.DBMap;
import org.eclipse.hyades.resources.database.internal.dbmodel.Table;
import org.eclipse.hyades.resources.database.internal.extensions.DatabaseType;
import org.eclipse.hyades.resources.database.internal.extensions.InsertObjects;
import org.eclipse.hyades.resources.database.internal.extensions.JDBCHelper;
/**
 * This class creates SQLStatement objects.
 */
public class StatementFactory {
	/**
	 * Constructor for StatementFactory.
	 */
	private StatementFactory() {
		super();
	}

	public static final StatementFactory INSTANCE = new StatementFactory();

	/**
	 * A table statement generates a CREATE TABLE SQL statement from an
	 * RDBTable.
	 */
	public SQLStatement createTableStatement(DatabaseType dbType, DBMap map, Table table) {
		return new TableStatement(dbType, map, table);
	}

	/**
	 * An InsertObjects statement generates an insert statement that inserts
	 * rows into a class table.
	 */
	public InsertStatement createInsertStatement(DatabaseType dbType, Table table, List columns) {
		return new InsertStatement(dbType, table, columns);
	}

	/**
	 * An InsertObjects statement generates an insert statement that inserts
	 * rows into a class table.
	 */
	public InsertStatement createInsertObjects(DatabaseType type, ClassMetadata metadata, TypeConverter converter, Map objectsToIds, Map manyReferenceValues, WeakObjectCache cache) {
		return new InsertObjects(type, metadata, converter, objectsToIds, manyReferenceValues, cache);
	}

	/**
	 * An InsertReferences statement generates an insert statement that inserts
	 * rows into an association table.
	 */
	public InsertStatement createInsertReferences(DatabaseType type, EReference reference, DBMap map, Map objectsToIds, Set proxies) {
		return new InsertReferences(type, reference, map, objectsToIds, proxies);
	}

	/**
	 * An InsertAttributeValues statement generates an insert statement that
	 * inserts rows into an attribute table; this is used for multi-valued
	 * attributes.
	 */
	public InsertStatement createInsertAttributeValues(DBMap map, DatabaseType dbType, Map objectsToIds, EAttribute attribute, int type) {
		return new InsertAttributeValues(map, dbType, objectsToIds, attribute, type);
	}

	/**
	 * A reference count query statement generates a SELECT SQL statement that
	 * counts the referenced EObjects for the given reference.
	 */
	public QueryStatement createReferenceCountQuery(DatabaseType type, DBMap map, Integer id, EReference reference) {
		return new ReferenceCountQuery(type, map, id, reference);
	}

	/**
	 * An index query statement generates a SELECT SQL statement that returns
	 * the minimum position if last is false and the maximum position if last is
	 * true.
	 */
	public QueryStatement createIndexQuery(DatabaseType type, DBMap map, Integer sourceId, EReference reference, Integer targetId) {
		return new IndexQuery(type, map, sourceId, reference, targetId);
	}

	/**
	 * A contains query statement generates a SELECT SQL statement that returns
	 * the given referenceId for the object with the given id and given
	 * reference.
	 */
	public QueryStatement createContainsQuery(DatabaseType type, DBMap map, Integer id, EReference reference, Integer referencedId) {
		return new ContainsQuery(type, map, id, reference, referencedId);
	}

	/**
	 * An attribute query statement generates a SELECT SQL statement that
	 * obtains the values for the given EAttribute.
	 */
	public QueryStatement createAttributeQuery(int[] ids, EAttribute attribute, DBMap map, DatabaseType type) {
		return new AttributeQuery(ids, attribute, map, type);
	}

	/**
	 * An attribute value query statement generates a SELECT SQL statement that
	 * selects rows from the class table according to the given attribute
	 * values. The ids of the matching objects are returned.
	 */
	public QueryStatement createAttributeValueQuery(DatabaseType type, DBMap map, EClass eClass, List values, List ids) {
		return new AttributeValueQuery(type, map, eClass, values, ids);
	}

	/**
	 * A greedy class query obtains the rows whose ids are in the list of ids
	 * from a class table.
	 */
	public QueryStatement createClassQuery(DatabaseType type, EClass eClass, ClassMetadata metadata, int[] ids, DBMap map, boolean getAllObjects, boolean includeContainment) {
		return new ClassQuery(type, eClass, metadata, ids, map, getAllObjects, includeContainment);
	}

	public QueryStatement createGreedyReferenceQuery(int[] ids, EReference reference, DBMap map, DatabaseType type, boolean orderFlag) {
		return new GreedyReferenceQueryStatement(ids, reference, map, type, orderFlag);
	}

	/**
	 * An index statement creates a CREATE INDEX SQL statement.
	 */
	public SQLStatement createIndexStatement(DatabaseType dbType, DBMap dbMap, EClass eClass) {
		return new IndexStatement(dbType, dbMap, eClass);
	}

	/**
	 * An update reference statement generates an UPDATE statement to set a
	 * reference.
	 */
	public SQLStatement createUpdateReferenceStatement(DatabaseType dbType, DBMap dbMap, Integer sourceId, EReference reference, Integer targetId, int position) {
		return new UpdateReferenceStatement(dbType, dbMap, sourceId, reference, targetId, position);
	}

	/**
	 * An insert reference statement generates an INSERT statement to set a
	 * reference.
	 */
	public SQLStatement createInsertReferenceStatement(DatabaseType dbType, DBMap dbMap, Integer sourceId, EReference reference, Integer targetId, int position) {
		return new InsertReferenceStatement(dbType, dbMap, sourceId, reference, targetId, position);
	}

	/**
	 * A set attribute statement generates an UPDATE statement to set an
	 * attribute value.
	 */
	public SQLStatement createSetAttributeStatement(DatabaseType dbType, DBMap dbMap, Integer id, EAttribute attribute, Object value) {
		return new SetAttributeStatement(dbType, dbMap, id, attribute, value);
	}

	/**
	 * Deletes the objects from the table for the given class with the given
	 * ids.
	 */
	public SQLStatement createDeleteStatement(DatabaseType dbType, DBMap dbMap, EClass eClass, List ids, DatabaseType type) {
		return new DeleteStatement(dbMap, type, eClass, ids);
	}

	/**
	 * @param helper
	 * @param dbMap
	 * @param query
	 * @return
	 */
	public SQLStatement createSimpleSearchQueryStatement(JDBCHelper helper, DBMap dbMap, SimpleSearchQuery query) {
		return new SimpleSearchQueryStatement(helper, dbMap, query);
	}

	/**
	 * @return The target ID of the sourceId.reference(index) object
	 */
	public QueryStatement createGetTargetIdStatement(JDBCHelper helper, DBMap dbMap, int sourceId, EReference reference, int index) {
		return new GetTargetIdStatement(dbMap, helper.getDatabaseType(), sourceId, reference, index);
	}
	
	public SQLStatement createTimeBasedCorrelationQueryStatement(JDBCHelper helper, DBMap dbMap, SimpleSearchQuery query) {
		return new SimpleSearchQueryStatement(helper, dbMap, query);
	}
} // StatementFactory
