/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TableStatement.java,v 1.5 2005/02/16 22:21:32 qiyanli Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.resources.database.internal.impl;

import java.sql.Types;
import java.util.List;

import org.eclipse.hyades.resources.database.internal.DBMap;
import org.eclipse.hyades.resources.database.internal.dbmodel.Column;
import org.eclipse.hyades.resources.database.internal.dbmodel.Constraint;
import org.eclipse.hyades.resources.database.internal.dbmodel.SQLType;
import org.eclipse.hyades.resources.database.internal.dbmodel.StringType;
import org.eclipse.hyades.resources.database.internal.dbmodel.Table;
import org.eclipse.hyades.resources.database.internal.extensions.DatabaseType;
/**
 * This class generates an SQL CREATE TABLE statement.
 */
public class TableStatement extends SQLStatement {
	protected Table table;

	/**
	 * Constructor for TableStatement.
	 */
	public TableStatement(DatabaseType dbType, DBMap map, Table table) {
		super(dbType, map);
		this.table = table;
	}

	/**
	 * @see org.eclipse.hyades.resources.database.internal.SQLStatement#getStatement()
	 */
	public String getStatement() {
		if (statement.length() != 0)
			return statement.toString();

		statement.append("CREATE TABLE ");
		statement.append(addQuotes(table.getName()));
		statement.append(" (");//\n");
		RDBHelper rdbHelper = new RDBHelper();
		Column primaryKey = rdbHelper.getPrimaryKey(table);
		Constraint constraint = rdbHelper.getPrimaryKeyConstraint(table);
		List columns = table.getColumns();

		for (int i = 0, length = columns.size(); i < length; i++) {
			Column column = (Column) columns.get(i);
			boolean primaryKeyFlag = column == primaryKey;

			if (i == length - 1)
				writeColumn(column, constraint, primaryKeyFlag, true);
			else
				writeColumn(column, constraint, primaryKeyFlag, false);
		}

		// TODO ms - check if this is valid for other databases
		//    statement.append(");\n");
		statement.append(") ");
		statement.append(dbType.getTableSpaceString()); 
		return statement.toString();
	}

	protected void writeColumn(Column column, Constraint constraint, boolean primaryKey, boolean last) {
		statement.append("  ");
		statement.append(addQuotes(column.getName()));

		statement.append(" ");
		statement.append(getType(column.getType()));

		if (column.isAllowNull() == false)
			statement.append(" NOT NULL");

		addDefaultValue(column);

		if (primaryKey) {
			//      statement.append(" CONSTRAINT ");
			//      statement.append(constraint.getName());
			statement.append(" PRIMARY KEY");
		}

		if (!last)
			statement.append(",");

		//		statement.append("\n");
	}

	/**
	 * @param column
	 */
	protected void addDefaultValue(Column column) {
		if (column.getDefaultValue() != null) {
			statement.append(" ");
			statement.append(dbType.getDefaultString());
			statement.append(" ");
			if (column.getDefaultValue() instanceof String || column.getType().getSqlType()== Types.VARCHAR) {
				statement.append("'");
				statement.append(column.getDefaultValue());
				statement.append("'");
			} else if (column.getDefaultValue() instanceof Boolean) {
				statement.append("'");
				statement.append(((Boolean) column.getDefaultValue()).booleanValue() ? '1' : '0');
				statement.append("'");
			} else
				statement.append(column.getDefaultValue());
		}
	}

	/**
	 * Return the string representation of the type.
	 */
	protected String getType(SQLType type) {
		int jdbcType = type.getSqlType();

		switch (jdbcType) {
			case Types.SMALLINT :
				return "SMALLINT";
			case Types.INTEGER :
				return "INT";
			case Types.BIGINT :
				return "BIGINT";
			case Types.DOUBLE :
				return "DOUBLE PRECISION";
			case Types.CHAR :
				return "CHAR(" + ((StringType) type).getLength() + ")";
			case Types.LONGVARCHAR :
				return "LONG VARCHAR";
			default :
				return "VARCHAR(" + ((StringType) type).getLength() + ")";
		}
	}
} // TableStatement
