/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TypeConverter.java,v 1.4 2005/02/16 22:21:32 qiyanli Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.resources.database.internal.impl;

import java.util.List;

import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EFactory;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EcorePackage;
/**
 * This class converts from EMF values to values used by the databases.
 * Currently it handles boolean values, because they are stored in a database as
 * the single character 'T' or 'F'.
 */
public class TypeConverter {
	protected static final boolean debug = false;
	public static final String TRUE_VALUE = "T";
	public static final String FALSE_VALUE = "F";

	/**
	 * Constructor for TypeConverter.
	 */
	public TypeConverter() {
		super();
	}

	public void setValue(EObject object, EAttribute attr, EFactory factory, String value) {
		setValue(object, attr, factory, value, false);
	}

	public void setValue(EObject object, EAttribute attr, EFactory factory, String value, boolean many) {
		if (value == null)
			return;

		if (isBooleanType(attr))
			setBooleanValue(object, attr, value, many);
		else {
			Object objectValue = null;

			try {
				objectValue = factory.createFromString(attr.getEAttributeType(), value);

				if (!many)
					object.eSet(attr, objectValue);
				else
					((List) object.eGet(attr)).add(objectValue);
			} catch (Exception e) {
				if (debug) {
					e.printStackTrace();
					System.out.println("Error setting attr: " + attr + " for object " + object + " to " + objectValue);
				}
			}
		}
	}

	protected void setBooleanValue(EObject object, EAttribute attribute, String value, boolean many) {
		if ("T".equals(value)) {
			if (!many)
				object.eSet(attribute, Boolean.TRUE);
			else
				((List) object.eGet(attribute)).add(Boolean.TRUE);
		} else {
			if (!many)
				object.eSet(attribute, Boolean.FALSE);
			else
				((List) object.eGet(attribute)).add(Boolean.FALSE);
		}
	}

	public Object getValue(EObject object, EAttribute attribute) throws Exception {
		if (!object.eIsSet(attribute))
			return null;

		Object value = object.eGet(attribute);

		if (isBooleanType(attribute)) {
			if (value == Boolean.TRUE)
				return TRUE_VALUE;
			else
				return FALSE_VALUE;
		}

		return value;
	}

	public static boolean isBooleanType(EAttribute attribute) {
		EDataType type = attribute.getEAttributeType();
		int id = type.getClassifierID();

		if (id == EcorePackage.EBOOLEAN || id == EcorePackage.EBOOLEAN_OBJECT)
			return true;
		else
			return false;
	}
} // TypeConverter
