/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LogQueryBuilder.java,v 1.19 2005/05/09 18:45:49 slavescu Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.models.cbe.util;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.EcoreFactory;
import org.eclipse.emf.ecore.EcorePackage;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.loaders.util.HyadesResourceExtensions;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.models.cbe.CBEPackage;
import org.eclipse.hyades.models.hierarchy.CorrelationContainerProxy;
import org.eclipse.hyades.models.hierarchy.HierarchyPackage;
import org.eclipse.hyades.models.hierarchy.TRCAgent;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.extensions.BinaryExpression;
import org.eclipse.hyades.models.hierarchy.extensions.CorrelationQuery;
import org.eclipse.hyades.models.hierarchy.extensions.ExtensionsFactory;
import org.eclipse.hyades.models.hierarchy.extensions.LogicalExpression;
import org.eclipse.hyades.models.hierarchy.extensions.LogicalOperators;
import org.eclipse.hyades.models.hierarchy.extensions.OrderByElement;
import org.eclipse.hyades.models.hierarchy.extensions.OrderByOperators;
import org.eclipse.hyades.models.hierarchy.extensions.Query;
import org.eclipse.hyades.models.hierarchy.extensions.QueryResult;
import org.eclipse.hyades.models.hierarchy.extensions.RelationalOperators;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleOperand;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleSearchQuery;
import org.eclipse.hyades.models.hierarchy.extensions.TimeBasedCorrelationQuery;
import org.eclipse.hyades.models.hierarchy.extensions.WhereExpression;
import org.eclipse.hyades.models.hierarchy.util.IExtendedQueryService;
import org.eclipse.hyades.models.hierarchy.util.IFilterElement;
import org.eclipse.hyades.models.hierarchy.util.IHyadesResourceExtension;
import org.eclipse.hyades.models.hierarchy.util.ILogFilterCriteria;
import org.eclipse.hyades.models.hierarchy.util.ISortElement;
/**
 * @author apnan
 *  
 */
public class LogQueryBuilder {
	public static Query createQuery(List inputList, ILogFilterCriteria element, EAttribute attribute) {
		SimpleSearchQuery query = addAgents(inputList, element);
		SimpleOperand operand = ExtensionsFactory.eINSTANCE.createSimpleOperand();
		operand.setType(HierarchyPackage.eINSTANCE.getTRCAgent());
		query.getOutputElements().add(operand);
		operand = ExtensionsFactory.eINSTANCE.createSimpleOperand();
		operand.setFeature(attribute);
		query.getOutputElements().add(operand);
		return query;
	}
	/**
	 * Creates a model query based on a list of input EObjects and a filter
	 * criteria. Once a query is created it can be executed using
	 * {@link #executeQuery(Query, ResourceSet)}
	 * 
	 * @param inputList
	 *            list of EObjects
	 * @param element
	 *            filter criteria
	 * @return model query
	 */
	public static Query createQuery(List inputList, ILogFilterCriteria element) {
		Query query = null;
		EObject inputObject = null;
		int s = inputList.size();
		for (int i = 0; i < s; i++) {
			inputObject = (EObject) inputList.get(i);
			if (inputObject instanceof TRCAgentProxy) {
				if (query == null) {
					query = createQueryFromAgentProxy((TRCAgentProxy) inputObject, element);
				} else {
					if (i == 0) {
						addAgentToQuery((SimpleSearchQuery) query, (TRCAgentProxy) inputObject, element);
					} else {
						addAgentToQuery((SimpleSearchQuery) query, (TRCAgentProxy) inputObject, null);
					}
				}
			} else if (inputObject instanceof CorrelationContainerProxy) {
				query = createQueryFromAgents(((CorrelationContainerProxy) inputObject).getCorrelatedAgents(), element);
			}
		}
		if (query != null)
			query.setDistinct(true);
		return query;
	}
	/**
	 * Executes a model query. Delegates the execution call to the first
	 * HyadesResourceExtension found.
	 * 
	 * @see IExtendedQueryService#executeQuery
	 * 
	 * The returned QueryResult object will then be used to read the actual list
	 * of common base events. For every output element a resultEntry will be
	 * returned. For now only Common Base Events are used as output elements.
	 * 
	 * QueryResult queryResult = LogQueryBuilder.executeQuery(query,
	 * agentProxy.eResource().getResourceSet()); ResultEntry resultEntry
	 * =(ResultEntry)queryResult.getResultEntries().get(0); List myCBEs =
	 * resultEntry.getObjects();
	 * 
	 * @param query
	 * @param resourceSet
	 * @return QueryResult
	 */
	public static QueryResult executeQuery(Query query, ResourceSet resourceSet) {
		if (query == null || query.getSources().size() == 0)
			return null;
		QueryResult queryResult = null;
		String url = (String) query.getSources().get(0);
		IHyadesResourceExtension hyadesResourceExtension = (IHyadesResourceExtension) HyadesResourceExtensions.getInstance().get(LoadersUtils.getPostfix(url));
		if (hyadesResourceExtension != null) {
			List notLoadedTypes = new ArrayList();
			notLoadedTypes.add(HierarchyPackage.eINSTANCE.getAbstractDefaultEvent());
			notLoadedTypes.add(HierarchyPackage.eINSTANCE.getTRCAgentProxy());
			queryResult = hyadesResourceExtension.executeQuery(query, resourceSet, notLoadedTypes);
			return queryResult;
		}
		return queryResult;
	}
	/**
	 * @param query
	 */
	public static boolean convertToCreationTimeIfRequired(Query query) {
		boolean res=false;
		for (Iterator iter = query.eAllContents(); iter.hasNext();) {
			EObject element = (EObject) iter.next();
			if(element instanceof SimpleOperand)
			{
				SimpleOperand so = (SimpleOperand)element;
				if(so.getFeature()== CBEPackage.eINSTANCE.getCBECommonBaseEvent_AdjustedCreationTime())
				{
					so.setFeature(CBEPackage.eINSTANCE.getCBECommonBaseEvent_CreationTime());
					res=true;
				}
			}
		}
		return res;
	}
	/**
	 * This method should be used when creating a query for searching records in
	 * the log views based on a search criteria. The method will append to the
	 * existing log filter criteria the search criteria, returning a new query.
	 * 
	 * @param inputList
	 *            list of EObjects
	 * @param element
	 *            filter criteria
	 * @return search query
	 */
	public static Query createSearchQuery(List inputList, ILogFilterCriteria filterElement, ILogFilterCriteria searchElement) {
		Query query = createQuery(inputList, filterElement);
		if (query == null) {
			query = createQuery(inputList, searchElement);
		} else {
			query = getMergedQuery(query, searchElement);
		}
		return query;
	}
	public static Query createSearchQuery(Query query, ILogFilterCriteria searchElement) {
		if (query instanceof CorrelationQuery) {
			CorrelationQuery searchQuery = (CorrelationQuery) EcoreUtil.copy(query);
			SimpleSearchQuery searchSubQuery = simulateOldWhereExpressions();
			List whereCriterias = ((LogicalExpression) searchSubQuery.getWhereExpression()).getArguments();
			IFilterElement[] searchFilters = searchElement.getFilters();
			List attributes = CBEPackage.eINSTANCE.getCBECommonBaseEvent().getEAllAttributes();
			EAttribute attribute = null;
			for (int i = 0; i < searchFilters.length; i++) {
				attribute = findAttributeInList(searchFilters[i].getAttribute(), attributes);
				if (attribute != null && !isFilterInWhereClause(searchFilters[i], whereCriterias)) {
					whereCriterias.add(createBinaryExpression(searchFilters[i], attribute));
				}
			}
			searchQuery.getSubQuery().add(searchSubQuery);
			return searchQuery;
		} else
			return getMergedQuery(query, searchElement);
	}
	/**
	 * This method should be used when a filter needs to be appended to an
	 * existing query. An example would be when creating a query for searching
	 * records in the log view based on a search criteria. The method will
	 * append the search criteria to the existing log query.
	 * 
	 * @param query
	 *            existing filter query
	 * @return merged query
	 */
	public static Query getMergedQuery(Query query, ILogFilterCriteria mergeElement) {
		SimpleSearchQuery searchQuery = (SimpleSearchQuery) EcoreUtil.copy(query);
		addWhereExpression(searchQuery);
		List whereCriteria = ((LogicalExpression) searchQuery.getWhereExpression()).getArguments();
		IFilterElement[] searchFilters = mergeElement.getFilters();
		List attributes = CBEPackage.eINSTANCE.getCBECommonBaseEvent().getEAllAttributes();
		EAttribute attribute = null;
		for (int i = 0; i < searchFilters.length; i++) {
			attribute = findAttributeInList(searchFilters[i].getAttribute(), attributes);
			if (attribute != null && !isFilterInWhereClause(searchFilters[i], whereCriteria)) {
				whereCriteria.add(createBinaryExpression(searchFilters[i], attribute));
			}
		}
		return searchQuery;
	}
	/**
	 * @param searchQuery
	 */
	private static void addWhereExpression(SimpleSearchQuery searchQuery) {
		if(searchQuery.getWhereExpression()==null)
		{
			LogicalExpression l = ExtensionsFactory.eINSTANCE.createLogicalExpression();
			//simulate the old where expressions
			l.setOperator(LogicalOperators.AND_LITERAL);
			searchQuery.setWhereExpression(l);
		}
	}
	public static Query createCorrelation(List agentProxies) {
		TimeBasedCorrelationQuery query = ExtensionsFactory.eINSTANCE.createTimeBasedCorrelationQuery();
		int s = agentProxies.size();
		EObject inputObject = null;
		double delta = 0;
		for (int i = 0; i < s; i++) {
			inputObject = (EObject) agentProxies.get(i);
			if (inputObject instanceof TRCAgentProxy) {
				addAgentToQuery((SimpleSearchQuery) query, (TRCAgentProxy) inputObject, null);
				if(((TRCAgentProxy) inputObject).getDeltaTime()!=0){
					delta = ((TRCAgentProxy) inputObject).getDeltaTime();
				}else
				if (((TRCAgentProxy) inputObject).getProcessProxy() != null && ((TRCAgentProxy) inputObject).getProcessProxy().getNode() != null)
					delta = ((TRCAgentProxy) inputObject).getProcessProxy().getNode().getDeltaTime();
				else
					delta = 0;
				query.getDeltaTime().add(new Double(delta));
			}
		}
		return query;
	}
	public static Query createCorrelatedEventsQuery(List agentProxies) {
		TimeBasedCorrelationQuery query = (TimeBasedCorrelationQuery) createCorrelation(agentProxies);
		SimpleOperand Operand = ExtensionsFactory.eINSTANCE.createSimpleOperand();
		Operand.setType(CBEPackage.eINSTANCE.getCBECommonBaseEvent());
		query.getOutputElements().add(Operand);
		return query;
	}
	public static Query createOutboundCorrelationQuery(List agentProxies, ILogFilterCriteria filter) {
		TimeBasedCorrelationQuery query = (TimeBasedCorrelationQuery) createCorrelation(agentProxies);
		SimpleOperand Operand1 = ExtensionsFactory.eINSTANCE.createSimpleOperand();
		Operand1.setFeature(HierarchyPackage.eINSTANCE.getCorrelationEntry_Key());
		query.getOutputElements().add(Operand1);
		SimpleOperand Operand2 = ExtensionsFactory.eINSTANCE.createSimpleOperand();
		Operand2.setFeature(HierarchyPackage.eINSTANCE.getCorrelationEntry_Value());
		query.getOutputElements().add(Operand2);
		return getMergedQuery(query, filter);
	}
	public static Query createInboundCorrelationQuery(List agentProxies, ILogFilterCriteria filter) {
		TimeBasedCorrelationQuery query = (TimeBasedCorrelationQuery) createCorrelation(agentProxies);
		SimpleOperand Operand1 = ExtensionsFactory.eINSTANCE.createSimpleOperand();
		Operand1.setFeature(HierarchyPackage.eINSTANCE.getCorrelationEntry_Value());
		query.getOutputElements().add(Operand1);
		SimpleOperand Operand2 = ExtensionsFactory.eINSTANCE.createSimpleOperand();
		Operand2.setFeature(HierarchyPackage.eINSTANCE.getCorrelationEntry_Key());
		query.getOutputElements().add(Operand2);
		return getMergedQuery(query, filter);
	}
	private static void addAgentToQuery(SimpleSearchQuery query, TRCAgentProxy inputObject, ILogFilterCriteria element) {
		TRCAgent agent = (TRCAgent) inputObject.eGet(HierarchyPackage.eINSTANCE.getTRCAgentProxy_Agent(), false);
		if (agent != null) {
			query.getSources().add(EcoreUtil.getURI(agent).toString());
			if (element != null) {
				((LogicalExpression) query.getWhereExpression()).getArguments().addAll(createWhereClause(CBEPackage.eINSTANCE.getCBECommonBaseEvent(), element.getFilters()));
				addSeverityFilters(query, element);
				query.getOrderByExpresions().addAll(createOrderByClause(CBEPackage.eINSTANCE.getCBECommonBaseEvent(), element.getSortColumns()));
			}
		}
	}
	private static void addSeverityFilters(SimpleSearchQuery query, ILogFilterCriteria element) {
		Map options = element.getFilterOptions();
		if (options != null) {
			if (options.get(ILogFilterCriteria.OPTION_FILTER_SEV1).equals("1") && options.get(ILogFilterCriteria.OPTION_FILTER_SEV2).equals("1") && options.get(ILogFilterCriteria.OPTION_FILTER_SEV3).equals("1")) {
				return;
			} else {
				if (options.get(ILogFilterCriteria.OPTION_FILTER_SEV1).equals("1") && options.get(ILogFilterCriteria.OPTION_FILTER_SEV2).equals("0") && options.get(ILogFilterCriteria.OPTION_FILTER_SEV3).equals("0")) {
					getANDArguments(query).add(createBinaryExpression(CBEPackage.eINSTANCE.getCBECommonBaseEvent_Severity(), RelationalOperators.get(RelationalOperators.GT), new Integer(49)));
					getANDArguments(query).add(createBinaryExpression(CBEPackage.eINSTANCE.getCBECommonBaseEvent_Severity(), RelationalOperators.get(RelationalOperators.LT), new Integer(71)));
				} else if (options.get(ILogFilterCriteria.OPTION_FILTER_SEV1).equals("1") && options.get(ILogFilterCriteria.OPTION_FILTER_SEV2).equals("1") && options.get(ILogFilterCriteria.OPTION_FILTER_SEV3).equals("0")) {
					getANDArguments(query).add(createBinaryExpression(CBEPackage.eINSTANCE.getCBECommonBaseEvent_Severity(), RelationalOperators.get(RelationalOperators.GT), new Integer(29)));
					getANDArguments(query).add(createBinaryExpression(CBEPackage.eINSTANCE.getCBECommonBaseEvent_Severity(), RelationalOperators.get(RelationalOperators.LT), new Integer(71)));
				} else if (options.get(ILogFilterCriteria.OPTION_FILTER_SEV1).equals("0") && options.get(ILogFilterCriteria.OPTION_FILTER_SEV2).equals("1") && options.get(ILogFilterCriteria.OPTION_FILTER_SEV3).equals("0")) {
					getANDArguments(query).add(createBinaryExpression(CBEPackage.eINSTANCE.getCBECommonBaseEvent_Severity(), RelationalOperators.get(RelationalOperators.GT), new Integer(29)));
					getANDArguments(query).add(createBinaryExpression(CBEPackage.eINSTANCE.getCBECommonBaseEvent_Severity(), RelationalOperators.get(RelationalOperators.LT), new Integer(50)));
				} else if (options.get(ILogFilterCriteria.OPTION_FILTER_SEV1).equals("0") && options.get(ILogFilterCriteria.OPTION_FILTER_SEV2).equals("1") && options.get(ILogFilterCriteria.OPTION_FILTER_SEV3).equals("0")) {
					getANDArguments(query).add(createBinaryExpression(CBEPackage.eINSTANCE.getCBECommonBaseEvent_Severity(), RelationalOperators.get(RelationalOperators.GT), new Integer(29)));
					getANDArguments(query).add(createBinaryExpression(CBEPackage.eINSTANCE.getCBECommonBaseEvent_Severity(), RelationalOperators.get(RelationalOperators.LT), new Integer(50)));
				} else if (options.get(ILogFilterCriteria.OPTION_FILTER_SEV1).equals("0") && options.get(ILogFilterCriteria.OPTION_FILTER_SEV2).equals("1") && options.get(ILogFilterCriteria.OPTION_FILTER_SEV3).equals("1")) {
					getANDArguments(query).add(createBinaryExpression(CBEPackage.eINSTANCE.getCBECommonBaseEvent_Severity(), RelationalOperators.get(RelationalOperators.LT), new Integer(50)));
				} else if (options.get(ILogFilterCriteria.OPTION_FILTER_SEV1).equals("0") && options.get(ILogFilterCriteria.OPTION_FILTER_SEV2).equals("0") && options.get(ILogFilterCriteria.OPTION_FILTER_SEV3).equals("1")) {
					getANDArguments(query).add(createBinaryExpression(CBEPackage.eINSTANCE.getCBECommonBaseEvent_Severity(), RelationalOperators.get(RelationalOperators.LT), new Integer(30)));
				}
			}
		}
	}
	public static List getANDArguments(SimpleSearchQuery query) {
		if(query.getWhereExpression()==null)
			return Collections.EMPTY_LIST;
		return ((LogicalExpression) query.getWhereExpression()).getArguments();
	}
	private static Query createQueryFromAgents(List inputList, ILogFilterCriteria element) {
		SimpleSearchQuery query = addAgents(inputList, element);
		SimpleOperand Operand = ExtensionsFactory.eINSTANCE.createSimpleOperand();
		Operand.setType(CBEPackage.eINSTANCE.getCBECommonBaseEvent());
		query.getOutputElements().add(Operand);
		return query;
	}
	private static SimpleSearchQuery addAgents(List inputList, ILogFilterCriteria element) {
		int s = inputList.size();
		TRCAgentProxy agentProxy = null;
		SimpleSearchQuery query = simulateOldWhereExpressions();
		for (int i = 0; i < s; i++) {
			agentProxy = (TRCAgentProxy) inputList.get(i);
			addAgentToQuery(query, agentProxy, element);
		}
		return query;
	}
	private static SimpleSearchQuery simulateOldWhereExpressions() {
		SimpleSearchQuery query = ExtensionsFactory.eINSTANCE.createSimpleSearchQuery();
		LogicalExpression l = ExtensionsFactory.eINSTANCE.createLogicalExpression();
		//simulate the old where expressions
		l.setOperator(LogicalOperators.AND_LITERAL);
		query.setWhereExpression(l);
		return query;
	}
	private static Query createQueryFromAgentProxy(TRCAgentProxy inputObject, ILogFilterCriteria element) {
		SimpleSearchQuery query = simulateOldWhereExpressions();
		SimpleOperand Operand = ExtensionsFactory.eINSTANCE.createSimpleOperand();
		Operand.setType(CBEPackage.eINSTANCE.getCBECommonBaseEvent());
		query.getOutputElements().add(Operand);
		addAgentToQuery(query, inputObject, element);
		return query;
	}
	private static List createWhereClause(EClass eClass, IFilterElement[] elements) {
		List whereCriteria = new ArrayList();
		List attributes = eClass.getEAllAttributes();
		EAttribute attribute = null;
		for (int i = 0; i < elements.length; i++) {
			attribute = findAttributeInList(elements[i].getAttribute(), attributes);
			if (attribute != null) {
				whereCriteria.add(createBinaryExpression(elements[i], attribute));
			}
		}
		return whereCriteria;
	}
	private static BinaryExpression createBinaryExpression(IFilterElement element, EAttribute attribute) {
		BinaryExpression binaryExpression = ExtensionsFactory.eINSTANCE.createBinaryExpression();
		SimpleOperand Operand = ExtensionsFactory.eINSTANCE.createSimpleOperand();
		Operand.setFeature(attribute);
		binaryExpression.setLeftOperand(Operand);
		String value = element.getValue();
		if (element.getOperator().equals("=")) {
			if (value.indexOf('*') > -1) {
				value = value.replace('*', '%');
				binaryExpression.setOperator(RelationalOperators.LIKE_LITERAL);
			} else {
				binaryExpression.setOperator(RelationalOperators.EQ_LITERAL);
			}
		} else if (element.getOperator().equals("<")) {
			binaryExpression.setOperator(RelationalOperators.LT_LITERAL);
		} else if (element.getOperator().equals(">")) {
			binaryExpression.setOperator(RelationalOperators.GT_LITERAL);
		} else if (element.getOperator().equals("<>")) {
			binaryExpression.setOperator(RelationalOperators.NEQ_LITERAL);
		} else if (element.getOperator().equals("LIKE")) {
			binaryExpression.setOperator(RelationalOperators.LIKE_LITERAL);
		}
		setRightOperandValue(binaryExpression, convertToType(attribute.getEType(), value));
		return binaryExpression;
	}
	/**
	 * @param binaryExpression
	 * @param object
	 */
	public static void setRightOperandValue(BinaryExpression binaryExpression, Object object) {
		SimpleOperand operand = null;
		if (binaryExpression.getRightOperands().isEmpty()) {
			operand = ExtensionsFactory.eINSTANCE.createSimpleOperand();
			binaryExpression.getRightOperands().add(operand);
		} else {
			operand = (SimpleOperand) binaryExpression.getRightOperands().get(0);
		}
		if(object instanceof EStructuralFeature)
		{
			operand.setFeature((EStructuralFeature)object);
			return;
		}
		if(object instanceof EClass)
		{
			operand.setType((EClass)object);
			return;
		}
		if (object instanceof Integer) {
			operand.setValueType(EcorePackage.eINSTANCE.getEIntegerObject());
		} else if (object instanceof Double) {
			operand.setValueType(EcorePackage.eINSTANCE.getEDoubleObject());
		} else if (object instanceof Short) {
			operand.setValueType(EcorePackage.eINSTANCE.getEShortObject());
		} else if (object instanceof Long) {
			operand.setValueType(EcorePackage.eINSTANCE.getELongObject());
		} else if (object instanceof String) {
			operand.setValueType(EcorePackage.eINSTANCE.getEString());
		}
		operand.setValue(object);
	}
	/**
	 * @param attribute
	 * @param value
	 * @return
	 */
	private static Object convertToType(EClassifier attribute, String value) {
		if (attribute instanceof EDataType) {
			return EcoreFactory.eINSTANCE.createFromString((EDataType) attribute, value);
			//			if(object instanceof String){
			//				formatFilterValue((String)object);
			//			}
			//			return object;
		}
		return value;
	}
	/**
	 * Formats strings in order to be used in SQL statements.
	 * 
	 * @param string
	 */
	private static void formatFilterValue(String string) {
		StringBuffer sb = new StringBuffer();
		int l = string.length();
		for (int i = 0; i < l; i++) {
			char ch = string.charAt(i);
			if (ch == '%') {
				sb.append("\\%");
			} else if (ch == '_') {
				sb.append("\\_");
			} else {
				sb.append(ch);
			}
		}
		string = sb.toString();
	}
	private static BinaryExpression createBinaryExpression(EAttribute attribute, RelationalOperators operator, Object value) {
		BinaryExpression binaryExpression = ExtensionsFactory.eINSTANCE.createBinaryExpression();
		SimpleOperand Operand = ExtensionsFactory.eINSTANCE.createSimpleOperand();
		Operand.setFeature(attribute);
		binaryExpression.setLeftOperand(Operand);
		binaryExpression.setOperator(operator);
		setRightOperandValue(binaryExpression, value);
		return binaryExpression;
	}
	public static List createOrderByClause(EClass eClass, ISortElement[] elements) {
		if (elements == null)
			return Collections.EMPTY_LIST;
		List orderByCriteria = new ArrayList();
		List attributes = eClass.getEAllAttributes();
		EAttribute attribute = null;
		for (int i = 0; i < elements.length; i++) {
			attribute = findAttributeInList(elements[i].getSortColumn(), attributes);
			if (attribute != null) {
				orderByCriteria.add(createOrderByElement(elements[i], attribute));
			}
		}
		return orderByCriteria;
	}
	private static OrderByElement createOrderByElement(ISortElement element, EAttribute attribute) {
		OrderByElement orderByElement = ExtensionsFactory.eINSTANCE.createOrderByElement();
		SimpleOperand Operand = ExtensionsFactory.eINSTANCE.createSimpleOperand();
		Operand.setFeature(attribute);
		orderByElement.setOperand(Operand);
		if (element.isAscending())
			orderByElement.setOperator(OrderByOperators.ASC_LITERAL);
		else
			orderByElement.setOperator(OrderByOperators.DESC_LITERAL);
		return orderByElement;
	}
	private static EAttribute findAttributeInList(String name, List attributeList) {
		int s = attributeList.size();
		for (int i = 0; i < s; i++) {
			EAttribute attr = (EAttribute) attributeList.get(i);
			if (attr.getName().equalsIgnoreCase((name)))
				return attr;
		}
		return null;
	}
	private static boolean isFilterInWhereClause(IFilterElement element, List whereClause) {
//		int s = whereClause.size();
//		BinaryExpression where = null;
//		for (int i = 0; i < s; i++) {
//			where = (BinaryExpression) whereClause.get(i);
//			if (where != null && ((SimpleOperand)where.getLeftOperand()).getFeature().getName().equalsIgnoreCase(element.getAttribute()) && where.getOperator() == RelationalOperators.get(element.getOperator()) && element.getValue().equals(where.getRightOperands().get(0)))
//				return true;
//		}
		return false;
	}
	public static boolean compare(Query query1, Query query2) {
	
		List outputElements1 = query1.getOutputElements();
		List outputElements2 = query2.getOutputElements();
		if (outputElements1.size() != outputElements2.size())
			return false;
		int s = outputElements1.size();
		for (int i = 0; i < s; i++) {
			SimpleOperand left1 = (SimpleOperand) outputElements1.get(i);
			SimpleOperand left2 = (SimpleOperand) outputElements2.get(i);
			if ((left1 != null && left2 == null) || (left1 == null && left2 != null))
				return false;
			if (left1 != null && left2 != null && left1.getType() != left2.getType()) {
				return false;
			} else {
				if (left1 != null && left2 != null && left1.getFeature() != left2.getFeature())
					return false;
			}
		}
		List sourceElements1 = query1.getSources();
		List sourceElements2 = query2.getSources();
		if (sourceElements1.size() != sourceElements2.size())
			return false;
		s = sourceElements1.size();
		for (int i = 0; i < s; i++) {
			if (sourceElements1.get(i) != null && !sourceElements1.get(i).equals(sourceElements2.get(i))) {
				return false;
			}
		}
		
		if (query1 instanceof SimpleSearchQuery && query2 instanceof SimpleSearchQuery) {
			WhereExpression w1 = ((SimpleSearchQuery)query1).getWhereExpression();
			WhereExpression w2 = ((SimpleSearchQuery)query2).getWhereExpression();
			Iterator w1i = w1.eAllContents();
			Iterator w2i = w2.eAllContents();
			for (; w1i.hasNext() && w2i.hasNext();) {
				EObject e1 = (EObject) w1i.next();
				EObject e2 = (EObject) w2i.next();
				if(e1==null && e2==null)
					continue;
				if(e1!=null && e2 ==null)
					return false;
				if(e1==null && e2 !=null)
					return false;
					
				if(e1.eClass() != e2.eClass())
					return false;
				if(e1 instanceof LogicalExpression)
				{
					if(((LogicalExpression)e1).getOperator() != ((LogicalExpression)e2).getOperator())
						return false;
					if(((LogicalExpression)e1).getArguments().size() != ((LogicalExpression)e2).getArguments().size())
						return false;
				}
				else
				if(e1 instanceof BinaryExpression)
				{
					BinaryExpression expression1 = (BinaryExpression) e1;
					BinaryExpression expression2 = (BinaryExpression) e2;
					if (((SimpleOperand)expression1.getLeftOperand()).getFeature() !=((SimpleOperand)expression2.getLeftOperand()).getFeature())
						return false;
					if (((SimpleOperand)expression1.getLeftOperand()).getType() !=((SimpleOperand)expression2.getLeftOperand()).getType())
						return false;
					Object v1 = ((SimpleOperand)expression1.getLeftOperand()).getValue();
					Object v2 = ((SimpleOperand)expression2.getLeftOperand()).getValue();
					
					if (v1==null && v2!=null) 
						return false;
					if(v1!=null && !v1.equals(v2))
						return false;
					if (expression1.getOperator() != expression2.getOperator())
						return false;
					if (expression1.getRightOperands().size() != expression2.getRightOperands().size())
						return false;
					for (int i=0;i<expression1.getRightOperands().size(); i++) {
						
						if (((SimpleOperand)expression1.getRightOperands().get(i)).getFeature() !=((SimpleOperand)expression2.getRightOperands().get(i)).getFeature())
							return false;
						if (((SimpleOperand)expression1.getRightOperands().get(i)).getType() !=((SimpleOperand)expression2.getRightOperands().get(i)).getType())
							return false;
						v1 = ((SimpleOperand)expression1.getRightOperands().get(i)).getValue();
						v2 = ((SimpleOperand)expression2.getRightOperands().get(i)).getValue();
						
						if (v1==null && v2!=null) 
							return false;
						if(v1!=null && !v1.equals(v2))
							return false;
					}
				}
				
			}
			
//			List whereElements1 = getANDArguments((SimpleSearchQuery) query1);
//			List whereElements2 = getANDArguments((SimpleSearchQuery) query2);
//			if (whereElements1.size() != whereElements2.size())
//				return false;
//			s = whereElements1.size();
//			for (int i = 0; i < s; i++) {
//				BinaryExpression expression1 = (BinaryExpression) whereElements1.get(i);
//				BinaryExpression expression2 = (BinaryExpression) whereElements2.get(i);
//				if ((expression1 != null && expression2 == null) || (expression1 == null && expression2 != null))
//					return false;
//				if (((SimpleOperand)expression1.getLeftOperand()).getFeature() != ((SimpleOperand)expression2.getLeftOperand()).getFeature())
//					return false;
//				if (expression1.getOperator().getValue() != expression2.getOperator().getValue())
//					return false;
//				if (!expression1.getRightOperands().get(0).equals(expression2.getRightOperands().get(0)))
//					return false;
//			}
		}
		return true;
	}
}