/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: CommonBaseEvent.h,v 1.24 2005/04/29 19:33:41 hleung Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

#ifndef COMMONBASEEVENT_H
#define COMMONBASEEVENT_H

/**********************************************************************
 * File: CommonBaseEvent.h
 *
 * This header file defines the nine Common Base Event data types as 
 * required to implement the Common Base Event v1.0.1 specification.  
 * The nine data types include:
 * 
 *   -hcbe_AssociationEngine 
 *   -hcbe_MsgDataElement 
 *   -hcbe_AssociatedEvent
 *   -hcbe_ComponentIdentification
 *   -hcbe_ContextDataElement
 *   -hcbe_ExtendedDataElement
 *   -hcbe_SituationType
 *   -hcbe_Situation
 * 
 * The declaration section of each of the above data types consists of 
 * the following four sub-sections:
 *     
 *   -Function pointer type declaration.
 *   -Function table structure declaration.
 *   -Data structure declaration.
 *   -Constructor and destructor declaration.
 *
 * The data structure declaration sub-section contains a field named 
 * 'fpRoot' which is the pointer to the defined function table.  This 
 * field is used for accessing the functions defined in the function
 * table.
 *
 * This header also defines several utility functions used for following
 *
 *   -Common Base Event serialization to XML.
 *   -Common Base Event de-serialization from XML.
 *   -Global Unique IDentifier (GUID) generation.
 *   -Time stamp generation in UTC format.
 *
 * See the declaration section for the utility functions for more 
 * details.
 *
 * The nine Common Base Event data types defined in this header file
 * are illustrated in the Common Base Event Sample.  See Sample.c and 
 * Sample.c for more details.
 *
 *
 * Author(s): Mike Hao
 *            Bob Duncan
 *            James McCurdy
 *            Paul Slauenwhite
 * Version:   April 28, 2005
 * Since:     October 14, 2004
 ***********************************************************************/

#include <stdlib.h>


#if ! defined MVS && ! defined __OS400__
#include <malloc.h>
#endif

#include <stdio.h>

#ifndef NULL
	#define NULL (void *)0
#endif

/* These typedefs are required by OS/400 to bypass a compiler defect. Do not remove. */
#ifdef __OS400__
typedef struct hcbe_AssociationEngine hcbe_AssociationEngine;
typedef struct hcbe_MsgDataElement hcbe_MsgDataElement;
typedef struct hcbe_AssociatedEvent hcbe_AssociatedEvent;
typedef struct hcbe_ComponentIdentification hcbe_ComponentIdentification;
typedef struct hcbe_ContextDataElement hcbe_ContextDataElement;
typedef struct hcbe_ExtendedDataElement hcbe_ExtendedDataElement;
typedef struct hcbe_SituationType hcbe_SituationType;
typedef struct hcbe_Situation hcbe_Situation;
typedef struct hcbe_TemplateContentHandler hcbe_TemplateContentHandler;
typedef struct hcbe_EventFactory hcbe_EventFactory;
typedef struct hcbe_CommonBaseEvent hcbe_CommonBaseEvent;
#endif


#ifdef __cplusplus
extern "C" {
#endif

/**********************************************************************
 * AssociationEngine Declarations Section
 **********************************************************************/
 
/* 
 * Sub-Section: AssciationEngine Function Poiner Type Declaration 
 */
typedef char* (*hcbe_AssociationEngineGetFP) (struct hcbe_AssociationEngine*);
typedef void (*hcbe_AssociationEngineSetFP) (struct hcbe_AssociationEngine*, char*);

/* 
 * Sub-Section: AssciationEngine Function Table Declaration 
 */
typedef struct hcbe_AssociationEngineFPTable
{
    hcbe_AssociationEngineGetFP getId;
    hcbe_AssociationEngineGetFP getName;
    hcbe_AssociationEngineGetFP getType;
    hcbe_AssociationEngineSetFP setId;
    hcbe_AssociationEngineSetFP setName;
    hcbe_AssociationEngineSetFP setType;
}hcbe_AssociationEngineFPTable;

/* 
 * Sub-Section: AssciationEngine Data Structure Declaration 
 */
typedef struct hcbe_AssociationEngine
{
	char* id;
    char* name;
	char* type;
    int _deserialized;
	hcbe_AssociationEngineFPTable* fpRoot;

}hcbe_AssociationEngine;

/* 
 * Sub-Section: AssciationEngine Constructor and Destructor Declaration 
 */
hcbe_AssociationEngine* hcbe_associationEngineConstr();
char* hcbe_associationEngineDestr(hcbe_AssociationEngine* associationEngine);


/**********************************************************************
 * MsgDataElement Declarations Section
 **********************************************************************/

/* 
 * Sub-Section: MsgDataElement Function Poiner Type Declaration 
 */
typedef char* (*hcbe_MsgDataElementGetFP) (struct hcbe_MsgDataElement*);
typedef void (*hcbe_MsgDataElementSetFP) (struct hcbe_MsgDataElement*, char*);
typedef char** (*hcbe_MsgDataElementGetCatalogTokensFP) (struct hcbe_MsgDataElement*);
typedef void (*hcbe_MsgDataElementSetCatalogTokensFP) (struct hcbe_MsgDataElement*, char**, int);
typedef int (*hcbe_MsgDataElementArrGetCounterFP)(struct hcbe_MsgDataElement*);
typedef char* (*hcbe_MsgDataElementAddMsgCatalogTokenFP)(struct hcbe_MsgDataElement*, char*);

/* 
 * Sub-Section: MsgDataElement Function Table Declaration 
 */
typedef struct 
{
    hcbe_MsgDataElementGetFP  getMsgId,
		                 getMsgIdType,
		                 getMsgCatalog,
		                 getMsgCatalogId,
		                 getMsgCatalogType,
		                 getMsgLocale;   
    
	hcbe_MsgDataElementSetFP  setMsgId, 	
		                 setMsgIdType, 	
		                 setMsgCatalog, 	
		                 setMsgCatalogId,
				         setMsgCatalogType,    	
		                 setMsgLocale;   

	hcbe_MsgDataElementGetCatalogTokensFP getMsgCatalogTokens; 		
	hcbe_MsgDataElementSetCatalogTokensFP setMsgCatalogTokens;

	hcbe_MsgDataElementArrGetCounterFP getMsgCatalogTokensSize;

    /* 80016 */
    hcbe_MsgDataElementAddMsgCatalogTokenFP addMsgCatalogToken;

}hcbe_MsgDataElementFPTable;

/* 
 * Sub-Section: MsgDataElement Data Structure Declaration 
 */
typedef struct hcbe_MsgDataElement
{
	char* _msgId;
    char* _msgIdType;
	char* _msgCatalog;
	char* _msgCatalogId;
	char** _msgCatalogTokens;
	char* _msgCatalogType;
	char* _msgLocale;
	int _tokensSize;
    int _deserialized;

	hcbe_MsgDataElementFPTable* fpRoot;
}hcbe_MsgDataElement;

/* 
 * Sub-Section: MsgDataElement Constructor and Destructor Declaration 
 */
hcbe_MsgDataElement* hcbe_msgDataElementConstr();
char* hcbe_msgDataElementDestr(hcbe_MsgDataElement* msgDataElement);


/**********************************************************************
 * AssociatedEvent Declarations Section
 **********************************************************************/

/* 
 * Sub-Section: AssociatedEvent Function Poiner Type Declaration 
 */
typedef char* (*hcbe_AssociatedEventGetFP) (struct hcbe_AssociatedEvent*);
typedef void (*hcbe_AssociatedEventSetFP) (struct hcbe_AssociatedEvent*, char*);
typedef hcbe_AssociationEngine* (*hcbe_AssociatedEventGetEngineInfoFP) (struct hcbe_AssociatedEvent*);
typedef void (*hcbe_AssociatedEventSetEngineInfoFP) (struct hcbe_AssociatedEvent*, hcbe_AssociationEngine*);

/* 
 * Sub-Section: AssociatedEvent Function Table Declaration 
 */
typedef struct 
{
    hcbe_AssociatedEventGetFP getAssociationEngine,
							  getResolvedEvents;
     
    hcbe_AssociatedEventSetFP setAssociationEngine,
							  setResolvedEvents;

	hcbe_AssociatedEventGetEngineInfoFP getAssociationEngineInfo;
	hcbe_AssociatedEventSetEngineInfoFP setAssociationEngineInfo;

}hcbe_AssociatedEventFPTable;

/* 
 * Sub-Section: AssociatedEvent Data Structure Declaration 
 */
typedef struct hcbe_AssociatedEvent 
{
	char* _associationEngine;
    hcbe_AssociationEngine* _associationEngineInfo;
	char* _resolvedEvents;  
    int _deserialized;	

	hcbe_AssociatedEventFPTable* fpRoot;
}hcbe_AssociatedEvent;

/* 
 * Sub-Section: AssociatedEvent Constructor and Destructor Declaration
 */
hcbe_AssociatedEvent* hcbe_associatedEventConstr();
char* hcbe_associatedEventDestr(hcbe_AssociatedEvent* associatedEven);


/**********************************************************************
 * ComponentIdentification Declarations Section
 **********************************************************************/

/* 
 * Sub-Section: ComponentIdentification Function Poiner Type Declaration 
 */
typedef char* (*hcbe_ComponentIdentificationGetFP) (struct hcbe_ComponentIdentification*);
typedef void (*hcbe_ComponentIdentificationSetFP) (struct hcbe_ComponentIdentification*, char*);

/* 
 * Sub-Section: ComponentIdentification Function Table Declaration 
 */
typedef struct hcbe_ComponentIdentificationFPTable
{
    hcbe_ComponentIdentificationGetFP getLocation;
	hcbe_ComponentIdentificationGetFP getLocationType;
    hcbe_ComponentIdentificationGetFP getApplication;
    hcbe_ComponentIdentificationGetFP getExecutionEnvironment;
    hcbe_ComponentIdentificationGetFP getComponent;
    hcbe_ComponentIdentificationGetFP getSubComponent;
    hcbe_ComponentIdentificationGetFP getComponentIdType;
    hcbe_ComponentIdentificationGetFP getInstanceId;
    hcbe_ComponentIdentificationGetFP getProcessId;
    hcbe_ComponentIdentificationGetFP getThreadId;
    hcbe_ComponentIdentificationGetFP getComponentType;

    hcbe_ComponentIdentificationSetFP setLocation;
	hcbe_ComponentIdentificationSetFP setLocationType;
	hcbe_ComponentIdentificationSetFP setApplication;
    hcbe_ComponentIdentificationSetFP setExecutionEnvironment;
    hcbe_ComponentIdentificationSetFP setComponent;
    hcbe_ComponentIdentificationSetFP setSubComponent;
    hcbe_ComponentIdentificationSetFP setComponentIdType;
    hcbe_ComponentIdentificationSetFP setInstanceId;
    hcbe_ComponentIdentificationSetFP setProcessId;
    hcbe_ComponentIdentificationSetFP setThreadId;
    hcbe_ComponentIdentificationSetFP setComponentType;

}hcbe_ComponentIdentificationFPTable;

/* 
 * Sub-Section: ComponentIdentification Data Structure Declaration 
 */
typedef struct hcbe_ComponentIdentification
{
	char* location;
    char* locationType;
	char* application;
    char* executionEnvironment;
    char* component;
    char* subComponent;
    char* componentIdType;
    char* instanceId;
    char* processId;
    char* threadId;
    char* componentType;
    int _deserialized;
	hcbe_ComponentIdentificationFPTable* fpRoot;
}hcbe_ComponentIdentification;

/* 
 * Sub-Section: ComponentIdentification Constructor and Destructor Declaration 
 */
hcbe_ComponentIdentification* hcbe_componentIdentificationConstr();
char* hcbe_componentIdentificationDestr(hcbe_ComponentIdentification* componentIdentification);


/**********************************************************************
 * ContextDataElement Declarations Section
 **********************************************************************/

/* 
 * Sub-Section: ContextDataElement Function Poiner Type Declaration 
 */
typedef char* (*hcbe_ContextDataElementGetFP) (struct hcbe_ContextDataElement*);
typedef void (*hcbe_ContextDataElementSetFP) (struct hcbe_ContextDataElement*, char*);

/* 
 * Sub-Section: ContextDataElement Function Table Declaration 
 */
typedef struct hcbe_ContextDataElementFPTable
{
    hcbe_ContextDataElementGetFP getContextId;
	hcbe_ContextDataElementGetFP getType;
	hcbe_ContextDataElementGetFP getName;
    hcbe_ContextDataElementGetFP getContextValue;

    hcbe_ContextDataElementSetFP setContextId;
    hcbe_ContextDataElementSetFP setType;
    hcbe_ContextDataElementSetFP setName;
    hcbe_ContextDataElementSetFP setContextValue;

}hcbe_ContextDataElementFPTable;

/* 
 * Sub-Section: ContextDataElement Data Structure Declaration 
 */
typedef struct hcbe_ContextDataElement
{
	char* contextId;
    char* type;
	char* name;
    char* contextValue;
    int _deserialized;
	hcbe_ContextDataElementFPTable* fpRoot;

}hcbe_ContextDataElement;

/* 
 * Sub-Section: ContextDataElement Constructor and Destructor Declaration 
 */
hcbe_ContextDataElement* hcbe_contextDataElementConstr();
char* hcbe_contextDataElementDestr(hcbe_ContextDataElement* contextDataElement);


/**********************************************************************
 * ExtendedDataElement Declarations Section
 **********************************************************************/

/* 
 * Sub-Section: ExtendedDataElement Constants Declaration 
 */
#define HCBE_TYPE_NO_VALUE_VALUE 0   
#define HCBE_TYPE_BYTE_VALUE 1   
#define HCBE_TYPE_SHORT_VALUE 2
#define HCBE_TYPE_INT_VALUE 3   
#define HCBE_TYPE_LONG_VALUE 4   
#define HCBE_TYPE_FLOAT_VALUE 5   
#define HCBE_TYPE_DOUBLE_VALUE 6    
#define HCBE_TYPE_STRING_VALUE 7   
#define HCBE_TYPE_DATE_TIME_VALUE 8   
#define HCBE_TYPE_BOOLEAN_VALUE 9    
#define HCBE_TYPE_BYTE_ARRAY_VALUE 10   
#define HCBE_TYPE_SHORT_ARRAY_VALUE 11   
#define HCBE_TYPE_INT_ARRAY_VALUE 12   
#define HCBE_TYPE_LONG_ARRAY_VALUE 13   
#define HCBE_TYPE_FLOAT_ARRAY_VALUE 14   
#define HCBE_TYPE_DOUBLE_ARRAY_VALUE 15   
#define HCBE_TYPE_STRING_ARRAY_VALUE 16   
#define HCBE_TYPE_DATE_TIME_ARRAY_VALUE 17   
#define HCBE_TYPE_BOOLEAN_ARRAY_VALUE 18   
#define HCBE_TYPE_HEX_BINARY_VALUE 19

#define HCBE_TYPE_NO_VALUE "noValue"
#define HCBE_TYPE_BYTE "byte"
#define HCBE_TYPE_SHORT "short"  
#define HCBE_TYPE_INT "int"
#define HCBE_TYPE_LONG "long"  
#define HCBE_TYPE_FLOAT "float"   
#define HCBE_TYPE_DOUBLE "double"  
#define HCBE_TYPE_STRING "string"   
#define HCBE_TYPE_DATE_TIME "dateTime"    
#define HCBE_TYPE_BOOLEAN "boolean"   
#define HCBE_TYPE_BYTE_ARRAY "byteArray"   
#define HCBE_TYPE_SHORT_ARRAY "shortArray"
#define HCBE_TYPE_INT_ARRAY "intArray"   
#define HCBE_TYPE_LONG_ARRAY "longArray"    
#define HCBE_TYPE_FLOAT_ARRAY "floatArray"
#define HCBE_TYPE_DOUBLE_ARRAY "doubleArray"    
#define HCBE_TYPE_STRING_ARRAY "stringArray"   
#define HCBE_TYPE_DATE_TIME_ARRAY "dateTimeArray"
#define HCBE_TYPE_BOOLEAN_ARRAY "booleanArray"   
#define HCBE_TYPE_HEX_BINARY "hexBinary"    

/* 
 * Sub-Section: ExtendedDataElement Function Poiner Type Declaration 
 */
typedef	char* (*hcbe_ExtendedDataElementGetFP)(struct hcbe_ExtendedDataElement*);
typedef void (*hcbe_ExtendedDataElementSetFP)(struct hcbe_ExtendedDataElement*, char*);
typedef	char** (*hcbe_ExtendedDataElementGetValuesFP)(struct hcbe_ExtendedDataElement*);
typedef void (*hcbe_ExtendedDataElementSetValuesFP)(struct hcbe_ExtendedDataElement*, char**, int);
typedef struct hcbe_ExtendedDataElement** (*hcbe_ExtendedDataElementGetChildrenFP)(struct hcbe_ExtendedDataElement*);
typedef void (*hcbe_ExtendedDataElementSetChildrenFP)(struct hcbe_ExtendedDataElement*, struct hcbe_ExtendedDataElement**, int);
typedef int (*hcbe_ExtendedDataElementArrGetCounterFP)(struct hcbe_ExtendedDataElement*);
typedef int (*hcbe_ExtendedDataElementGetTypeAsIntFP)(struct hcbe_ExtendedDataElement*);

/* 80016 */
typedef struct hcbe_ExtendedDataElement* (*hcbe_ExtendedDataElement_AddChildFP1)(struct hcbe_ExtendedDataElement*, char*, char*);
typedef struct hcbe_ExtendedDataElement* (*hcbe_ExtendedDataElement_AddChildFP2)(struct hcbe_ExtendedDataElement*, char*, char**, int);
typedef struct hcbe_ExtendedDataElement* (*hcbe_ExtendedDataElement_AddChildFP3)(struct hcbe_ExtendedDataElement*, char*, short);
typedef struct hcbe_ExtendedDataElement* (*hcbe_ExtendedDataElement_AddChildFP4)(struct hcbe_ExtendedDataElement*, char*, short*, int);
typedef struct hcbe_ExtendedDataElement* (*hcbe_ExtendedDataElement_AddChildFP5)(struct hcbe_ExtendedDataElement*, char*, int);
typedef struct hcbe_ExtendedDataElement* (*hcbe_ExtendedDataElement_AddChildFP6)(struct hcbe_ExtendedDataElement*, char*, int*, int);
typedef struct hcbe_ExtendedDataElement* (*hcbe_ExtendedDataElement_AddChildFP7)(struct hcbe_ExtendedDataElement*, char*, long);
typedef struct hcbe_ExtendedDataElement* (*hcbe_ExtendedDataElement_AddChildFP8)(struct hcbe_ExtendedDataElement*, char*, long*, int);
typedef struct hcbe_ExtendedDataElement* (*hcbe_ExtendedDataElement_AddChildFP9)(struct hcbe_ExtendedDataElement*, char*, float);
typedef struct hcbe_ExtendedDataElement* (*hcbe_ExtendedDataElement_AddChildFP10)(struct hcbe_ExtendedDataElement*, char*, float*, int);
typedef struct hcbe_ExtendedDataElement* (*hcbe_ExtendedDataElement_AddChildFP11)(struct hcbe_ExtendedDataElement*, char*, double);
typedef struct hcbe_ExtendedDataElement* (*hcbe_ExtendedDataElement_AddChildFP12)(struct hcbe_ExtendedDataElement*, char*, double*, int);


/* 
 * Sub-Section: ExtendedDataElement Function Table Declaration 
 */
typedef struct {      	
    hcbe_ExtendedDataElementGetFP getName,
							 getType,
							 getHexValue;

    hcbe_ExtendedDataElementSetFP setName,
							 setType,
							 setHexValue;

	hcbe_ExtendedDataElementGetValuesFP getValues;
    hcbe_ExtendedDataElementSetValuesFP setValues;

	hcbe_ExtendedDataElementGetChildrenFP getChildren;
	hcbe_ExtendedDataElementSetChildrenFP setChildren;

	hcbe_ExtendedDataElementArrGetCounterFP getValuesSize,
									   getChildrenSize;

	hcbe_ExtendedDataElementGetTypeAsIntFP getTypeAsInt;

	/* 80016 */
	hcbe_ExtendedDataElement_AddChildFP1 addChildWithStringValue, 
                                         addChildWithDateValue, 
                                         addChildWithBooleanValue, 
                                         addChildWithHexValue;
    hcbe_ExtendedDataElement_AddChildFP2 addChildWithStringArrayValues, 
                                         addChildWithDateArrayValues, 
                                         addChildWithBooleanArrayValues;
    hcbe_ExtendedDataElement_AddChildFP3 addChildWithShortValue;
    hcbe_ExtendedDataElement_AddChildFP4 addChildWithShortArrayValues;
    hcbe_ExtendedDataElement_AddChildFP5 addChildWithIntValue;
    hcbe_ExtendedDataElement_AddChildFP6 addChildWithIntArrayValues;
    hcbe_ExtendedDataElement_AddChildFP7 addChildWithLongValue;
    hcbe_ExtendedDataElement_AddChildFP8 addChildWithLongArrayValues;
    hcbe_ExtendedDataElement_AddChildFP9 addChildWithFloatValue;
    hcbe_ExtendedDataElement_AddChildFP10 addChildWithFloatArrayValues;
    hcbe_ExtendedDataElement_AddChildFP11 addChildWithDoubleValue;
    hcbe_ExtendedDataElement_AddChildFP12 addChildWithDoubleArrayValues;

}hcbe_ExtendedDataElementFPTable;

/* 
 * Sub-Section: ExtendedDataElement Data Structure Declaration 
 */
typedef struct hcbe_ExtendedDataElement
{
	char* _name;
    char* _type;
	char** _values;    	
	char* _hexValue;
	struct hcbe_ExtendedDataElement** _children;

	int _valuesSize;
	int _childrenSize;
    int _deserialized;

	hcbe_ExtendedDataElementFPTable* fpRoot;
}hcbe_ExtendedDataElement;

/* 
 * Sub-Section: ExtendedDataElement Constructor and Destructor Declaration 
 */
hcbe_ExtendedDataElement* hcbe_extendedDataElementConstr();
char* hcbe_extendedDataElementDestr(hcbe_ExtendedDataElement* extendedDataElement);


/**********************************************************************
 * SituationType Declarations Section
 **********************************************************************/

/* 
 * Sub-Section: SituationType Constants Declaration 
 */
#define HCBE_START_SITUATION 0
#define HCBE_STOP_SITUATION 1
#define HCBE_CONNECT_SITUATION 2
#define HCBE_REPORT_SITUATION 3
#define HCBE_FEATURE_SITUATION 4
#define HCBE_CONFIGURE_SITUATION 5
#define HCBE_DEPENDENCY_SITUATION 6
#define HCBE_CREATE_SITUATION 7
#define HCBE_DESTROY_SITUATION 8
#define HCBE_AVAILABLE_SITUATION 9
#define HCBE_REQUEST_SITUATION 10
#define HCBE_OTHER_SITUATION 11

#define HCBE_SITUATION_TYPE_BADTYPE_ERR 1
#define HCBE_SITUATION_TYPE_BADTYPE_ERR_MSG "Invalid type Specified"
#define HCBE_SITUATION_TYPE_NOMEMORY_ERR 2
#define HCBE_SITUATION_TYPE_NOMEMORY_ERR_MSG "Insufficient available memory"

/* 
 * Sub-Section: SituationType Function Pointer Type Declaration 
 */
typedef char* (*hcbe_SituationTypeGetFP) (struct hcbe_SituationType*, char*);
typedef char* (*hcbe_SituationTypeSetFP) (struct hcbe_SituationType*, char*);

/* 
 * Sub-Section: SituationType Function Table Declaration 
 */
typedef struct hcbe_SituationTypeFPTable
{
    hcbe_SituationTypeGetFP getReasoningScope;
    hcbe_SituationTypeGetFP getSuccessDisposition;
    hcbe_SituationTypeGetFP getSituationQualifier;
    hcbe_SituationTypeGetFP getSituationDisposition;
    hcbe_SituationTypeGetFP getReportCategory;
    hcbe_SituationTypeGetFP getFeatureDisposition;
    hcbe_SituationTypeGetFP getDependencyDisposition;
    hcbe_SituationTypeGetFP getOperationDisposition;
    hcbe_SituationTypeGetFP getProcessingDisposition;
    hcbe_SituationTypeGetFP getAvailabilityDisposition;
    hcbe_SituationTypeGetFP getAnyData;

    hcbe_SituationTypeSetFP setReasoningScope;
    hcbe_SituationTypeSetFP setSuccessDisposition;
    hcbe_SituationTypeSetFP setSituationQualifier;
    hcbe_SituationTypeSetFP setSituationDisposition;
    hcbe_SituationTypeSetFP setReportCategory;
    hcbe_SituationTypeSetFP setFeatureDisposition;
    hcbe_SituationTypeSetFP setDependencyDisposition;
    hcbe_SituationTypeSetFP setOperationDisposition;
    hcbe_SituationTypeSetFP setProcessingDisposition;
    hcbe_SituationTypeSetFP setAvailabilityDisposition;
    hcbe_SituationTypeSetFP setAnyData;
}hcbe_SituationTypeFPTable;

/* 
 * Sub-Section: SituationType Data Structure Declaration 
 */
typedef struct hcbe_SituationType
{
    int type;
	char* reasoningScope;
    char* successDisposition;
	char* situationQualifier;
    char* situationDisposition;
    char* reportCategory;
    char* featureDisposition;
    char* dependencyDisposition;
    char* operationDisposition;
    char* processingDisposition;
    char* availabilityDisposition;
    char* anyData;
    int _deserialized;
	hcbe_SituationTypeFPTable* fpRoot;

}hcbe_SituationType;

/* 
 * Sub-Section: SituationType Constructor and Destructor Declaration 
 */
hcbe_SituationType* hcbe_situationTypeConstr(int type, int* err);
char* hcbe_situationTypeDestr(hcbe_SituationType* situationType);


/**********************************************************************
 * Situation Declarations Section
 **********************************************************************/

/* 
 * Sub-Section: Situation Function Pointer Type Declaration 
 */
typedef char* (*hcbe_SituationGetFP) (struct hcbe_Situation*);
typedef void (*hcbe_SituationSetFP) (struct hcbe_Situation*, char*);
typedef hcbe_SituationType* (*hcbe_SituationGetSituationTypeFP) (struct hcbe_Situation*);
typedef void (*hcbe_SituationSetSituationTypeFP) (struct hcbe_Situation*, hcbe_SituationType*);

/* 
 * Sub-Section: Situation Function Table Declaration 
 */
typedef struct hcbe_SituationFPTable
{
    hcbe_SituationGetFP getCategoryName;
    hcbe_SituationGetSituationTypeFP getSituationType;
    hcbe_SituationSetFP setCategoryName;
    hcbe_SituationSetSituationTypeFP setSituationType;
}hcbe_SituationFPTable;

/* 
 * Sub-Section: Situation Data Structure Declaration 
 */
typedef struct hcbe_Situation
{
	char* categoryName;
    hcbe_SituationType* situationType;
    int _deserialized;
	hcbe_SituationFPTable* fpRoot;
}hcbe_Situation;

/* 
 * Sub-Section: Situation Constructor and Destructor Declaration 
 */
hcbe_Situation* hcbe_situationConstr();
char* hcbe_situationDestr(hcbe_Situation* situation);


/**********************************************************************
 * TemplateContentHandler Declarations Section  - defect 80018
 **********************************************************************/
 
/* 
 * Sub-Section: TemplateContentHandler Function Poiner Type Declaration 
 */
typedef void (*hcbe_TemplateContentHandler_EventSetFP)(struct hcbe_TemplateContentHandler*, struct hcbe_CommonBaseEvent*);
typedef struct hcbe_CommonBaseEvent* (*hcbe_TemplateContentHandler_EventGetFP)(struct hcbe_TemplateContentHandler*);
typedef int (*hcbe_TemplateContentHandler_EventCompleteFP)(struct hcbe_TemplateContentHandler*, struct hcbe_CommonBaseEvent*);

/* 
 * Sub-Section: TemplateContentHandler Function Table Declaration 
 */
typedef struct hcbe_TemplateContentHandlerFPTable{
    hcbe_TemplateContentHandler_EventSetFP setTemplateEvent;
    hcbe_TemplateContentHandler_EventGetFP getTemplateEvent;  
    hcbe_TemplateContentHandler_EventCompleteFP completeEvent;
}hcbe_TemplateContentHandlerFPTable;

/* 
 * Sub-Section: TemplateContentHandler Data Structure Declaration 
 */
typedef struct hcbe_TemplateContentHandler{
    struct hcbe_CommonBaseEvent *_templateEvent;
    int _isTemplateEventSet;
    hcbe_TemplateContentHandlerFPTable* fpRoot;

}hcbe_TemplateContentHandler;

/* 
 * Sub-Section: TemplateContentHandler Constructor and Destructor Declaration 
 */
hcbe_TemplateContentHandler* hcbe_templateContentHandlerConstr();
char* hcbe_templateContentHandlerDestr(hcbe_TemplateContentHandler*);


/**********************************************************************
 * EventFactory Declarations Section
 **********************************************************************/
 
/* 
 * Sub-Section: EventFactory Function Poiner Type Declaration 
 */
typedef hcbe_AssociatedEvent* (*hcbe_EventFactory_CreateAssociatedEventFP)();
typedef hcbe_AssociationEngine* (*hcbe_EventFactory_CreateAssociationEngineFP)();
typedef hcbe_Situation* (*hcbe_EventFactory_CreateSituationFP)();
typedef hcbe_Situation* (*hcbe_EventFactory_CreateSituationFP1)(char*, char*, char*, char*);
typedef hcbe_Situation* (*hcbe_EventFactory_CreateSituationFP2)(char*, char*, char*);
typedef hcbe_Situation* (*hcbe_EventFactory_CreateSituationFP3)(char*, char*);
typedef hcbe_Situation* (*hcbe_EventFactory_CreateSituationFP4)(char*);

typedef hcbe_ComponentIdentification* (*hcbe_EventFactory_CreateComponentIdentificationFP)();
typedef hcbe_ContextDataElement*  (*hcbe_EventFactory_CreateContextDataElementFP)();
typedef hcbe_MsgDataElement* (*hcbe_EventFactory_CreateMsgDataElementFP)();
typedef hcbe_ExtendedDataElement* (*hcbe_EventFactory_CreateExtendedDataElementFP)();
typedef struct hcbe_CommonBaseEvent* (*hcbe_EventFactory_CreateCommonBaseEventGenFP)();
typedef struct hcbe_CommonBaseEvent* (*hcbe_EventFactory_CreateCommonBaseEventFP)(struct hcbe_EventFactory*);
typedef struct hcbe_CommonBaseEvent* (*hcbe_EventFactory_CreateCommonBaseEventWithExtensionNameFP)(char* );
typedef struct hcbe_CommonBaseEvent* (*hcbe_EventFactory_CreateCommonBaseEventWithExtensionNameCreationTimeFP)(char*, char*);
typedef struct hcbe_CommonBaseEvent* (*hcbe_EventFactory_CreateCommonBaseEventWithSeveritySituationMsgFP)(struct hcbe_EventFactory*, short, hcbe_Situation*, char*);
typedef struct hcbe_CommonBaseEvent* (*hcbe_EventFactory_CreateCommonBaseEventWithCompIdsSeveritySituationMsgFP)(struct hcbe_EventFactory*, short, hcbe_ComponentIdentification*, hcbe_ComponentIdentification*, hcbe_Situation*, char*);
typedef struct hcbe_CommonBaseEvent* (*hcbe_EventFactory_CreateCommonBaseEventWithMsgDataSeveritySituationMsgFP)(struct hcbe_EventFactory*, short, hcbe_Situation*, char*, char**, int, char*, char*);
typedef struct hcbe_CommonBaseEvent* (*hcbe_EventFactory_CreateCommonBaseEventWithCompIdsMsgDataSeveritySituationMsgFP)(struct hcbe_EventFactory*, short, hcbe_ComponentIdentification*, hcbe_ComponentIdentification*, hcbe_Situation*, char*, char**, int, char*, char*);
typedef void (*hcbe_EventFactory_SetContentHandlerFP)(struct hcbe_EventFactory*, hcbe_TemplateContentHandler*);
typedef hcbe_TemplateContentHandler* (*hcbe_EventFactory_GetContentHandlerFP)(struct hcbe_EventFactory*);
typedef void (*hcbe_EventFactory_SetCompleteEventFP)(struct hcbe_EventFactory*, int);
typedef int (*hcbe_EventFactory_GetCompleteEventFP)(struct hcbe_EventFactory*);

/* 
 * Sub-Section: EventFactory Function Table Declaration 
 */
typedef struct hcbe_EventFactoryFPTable{
    hcbe_EventFactory_CreateAssociatedEventFP createAssociatedEvent;
    hcbe_EventFactory_CreateAssociationEngineFP createAssociationEngine;  
    hcbe_EventFactory_CreateSituationFP1 createAvailableSituation;
    hcbe_EventFactory_CreateSituationFP2 createConnectSituation,
                                         createRequestSituation,
                                         createStartSituation,
                                         createStopSituation;
    hcbe_EventFactory_CreateSituationFP3 createCreateSituation,
                                         createDependencySituation,
                                         createDestroySituation,
                                         createFeatureSituation,
                                         createConfigureSituation,
                                         createReportSituation;
    hcbe_EventFactory_CreateSituationFP4 createOtherSituation;
                                        
    hcbe_EventFactory_CreateComponentIdentificationFP createComponentIdentification;
    hcbe_EventFactory_CreateContextDataElementFP createContextDataElement;
    hcbe_EventFactory_CreateMsgDataElementFP createMsgDataElement;
    hcbe_EventFactory_CreateExtendedDataElementFP createExtendedDataElement;
    hcbe_EventFactory_CreateCommonBaseEventGenFP createCommonBaseEventGen;
    hcbe_EventFactory_CreateCommonBaseEventFP createCommonBaseEvent;
    hcbe_EventFactory_CreateCommonBaseEventWithExtensionNameFP createCommonBaseEventWithExtensionName;
    hcbe_EventFactory_CreateCommonBaseEventWithExtensionNameCreationTimeFP createCommonBaseEventWithExtensionNameCreationTime;
    hcbe_EventFactory_CreateCommonBaseEventWithSeveritySituationMsgFP createCommonBaseEventWithSeveritySituationMsg;                                                                             
    hcbe_EventFactory_CreateCommonBaseEventWithCompIdsSeveritySituationMsgFP createCommonBaseEventWithCompIdsSeveritySituationMsg;
    hcbe_EventFactory_CreateCommonBaseEventWithMsgDataSeveritySituationMsgFP createCommonBaseEventWithMsgDataSeveritySituationMsg;
    hcbe_EventFactory_CreateCommonBaseEventWithCompIdsMsgDataSeveritySituationMsgFP createCommonBaseEventWithCompIdsMsgDataSeveritySituationMsg;
                                                                                   
    hcbe_EventFactory_SetContentHandlerFP setContentHandler;
    hcbe_EventFactory_GetContentHandlerFP getContentHandler;
    hcbe_EventFactory_SetCompleteEventFP setCompleteEvent;
    hcbe_EventFactory_GetCompleteEventFP getCompleteEvent;
}hcbe_EventFactoryFPTable;

/* 
 * Sub-Section: EventFactory Data Structure Declaration 
 */
typedef struct hcbe_EventFactory{
    hcbe_TemplateContentHandler* _templateContentHandler;
    int _completeEvent;
    hcbe_EventFactoryFPTable* fpRoot;
}hcbe_EventFactory;

/* 
 * Sub-Section: EventFactory Constructor and Destructor Declaration 
 */
hcbe_EventFactory* hcbe_eventFactoryConstr();
char* hcbe_eventFactoryDestr(hcbe_EventFactory* eventFactory);

/* 80018 end */

/**********************************************************************
 * CommonBaseEvent Declarations Section
 **********************************************************************/

/* 
 * Sub-Section: CommonBaseEvent Function Pointer Type Declaration 
 */
typedef char* (*hcbe_CommonBaseEventGetStringFP) (struct hcbe_CommonBaseEvent*);
typedef void (*hcbe_CommonBaseEventSetStringFP) (struct hcbe_CommonBaseEvent*, char*);
typedef short (*hcbe_CommonBaseEventGetShortFP) (struct hcbe_CommonBaseEvent*);
typedef void (*hcbe_CommonBaseEventSetShortFP) (struct hcbe_CommonBaseEvent*, short);
typedef long (*hcbe_CommonBaseEventGetLongFP) (struct hcbe_CommonBaseEvent*);
typedef void (*hcbe_CommonBaseEventSetLongFP) (struct hcbe_CommonBaseEvent*, long);

typedef hcbe_ComponentIdentification* (*hcbe_CommonBaseEventGetCompIdFP)(struct hcbe_CommonBaseEvent*);
typedef void (*hcbe_CommonBaseEventSetCompIdFP)(struct hcbe_CommonBaseEvent*, hcbe_ComponentIdentification*);
typedef hcbe_Situation* (*hcbe_CommonBaseEventGetSituationFP)(struct hcbe_CommonBaseEvent*);
typedef void (*hcbe_CommonBaseEventSetSituationFP)(struct hcbe_CommonBaseEvent*, hcbe_Situation*);
typedef hcbe_ContextDataElement** (*hcbe_CommonBaseEventGetContextDataElementsFP)(struct hcbe_CommonBaseEvent*);
typedef void (*hcbe_CommonBaseEventSetContextDataElementsFP)(struct hcbe_CommonBaseEvent*, hcbe_ContextDataElement**, int);  
typedef hcbe_MsgDataElement* (*hcbe_CommonBaseEventGetMsgDataElementFP)(struct hcbe_CommonBaseEvent*);
typedef void (*hcbe_CommonBaseEventSetMsgDataElementFP)(struct hcbe_CommonBaseEvent*, hcbe_MsgDataElement*);
typedef hcbe_AssociatedEvent** (*hcbe_CommonBaseEventGetAssociatedEventsFP)(struct hcbe_CommonBaseEvent*);
typedef void (*hcbe_CommonBaseEventSetAssociatedEventsFP)(struct hcbe_CommonBaseEvent*, hcbe_AssociatedEvent**, int);
typedef hcbe_ExtendedDataElement** (*hcbe_CommonBaseEventGetExtendedDataElementsFP)(struct hcbe_CommonBaseEvent*);
typedef void (*hcbe_CommonBaseEventSetExtendedDataElementsFP)(struct hcbe_CommonBaseEvent*, hcbe_ExtendedDataElement**, int);
typedef char** (*hcbe_CommonBaseEventGetAnyFP)(struct hcbe_CommonBaseEvent*);
typedef void (*hcbe_CommonBaseEventSetAnyFP)(struct hcbe_CommonBaseEvent*, char**, int);
typedef int (*hcbe_CommonBaseEventArrGetCounterFP)(struct hcbe_CommonBaseEvent*);
typedef int (*hcbe_CommonBaseEventAnyGetCounterFP)(struct hcbe_CommonBaseEvent*);

/* 80016 */
typedef hcbe_MsgDataElement* (*hcbe_CommonBaseEventSetMsgDataElementFP2)(struct hcbe_CommonBaseEvent*, char*, char*, char**, int, char*, char*, char*, char*);                                                                        
typedef void (*hcbe_CommonBaseEventSetCompIdFP2)(struct hcbe_CommonBaseEvent*, char*, char*, char*, char*, char*, char*, char*, char*, char*, char*, char*);
typedef void (*hcbe_CommonBaseEventSetCompIdFP3)(struct hcbe_CommonBaseEvent*, char*, char*, char*, char*, char*, char*, char*);

typedef hcbe_ContextDataElement* (*hcbe_CommonBaseEventAddContextDataElementFP)(struct hcbe_CommonBaseEvent*, struct hcbe_ContextDataElement*);
typedef hcbe_ContextDataElement* (*hcbe_CommonBaseEventAddContextDataElementFP2)(struct hcbe_CommonBaseEvent*, char*, char*, char*);
typedef void (*hcbe_CommonBaseEventRemoveContextDataElementsFP)(struct hcbe_CommonBaseEvent*, char*);
typedef void (*hcbe_CommonBaseEventClearContextDataElementFP)(struct hcbe_CommonBaseEvent*);

typedef hcbe_AssociatedEvent* (*hcbe_CommonBaseEventAddAssociatedEventFP)(struct hcbe_CommonBaseEvent*, struct hcbe_AssociatedEvent*);
typedef hcbe_AssociatedEvent* (*hcbe_CommonBaseEventAddAssociatedEventFP2)(struct hcbe_CommonBaseEvent*, char*, char*, char*, char*);
typedef void (*hcbe_CommonBaseEventClearAssociatedEventFP)(struct hcbe_CommonBaseEvent*);

typedef hcbe_ExtendedDataElement* (*hcbe_CommonBaseEventAddExtendedDataElementFP)(struct hcbe_CommonBaseEvent*, struct hcbe_ExtendedDataElement*);
typedef hcbe_ExtendedDataElement* (*hcbe_CommonBaseEventAddExtendedDataElementFP1)(struct hcbe_CommonBaseEvent*, char*, char*);
typedef hcbe_ExtendedDataElement* (*hcbe_CommonBaseEventAddExtendedDataElementFP2)(struct hcbe_CommonBaseEvent*, char*, char**, int);
typedef hcbe_ExtendedDataElement* (*hcbe_CommonBaseEventAddExtendedDataElementFP3)(struct hcbe_CommonBaseEvent*, char*, short);
typedef hcbe_ExtendedDataElement* (*hcbe_CommonBaseEventAddExtendedDataElementFP4)(struct hcbe_CommonBaseEvent*, char*, short*, int);
typedef hcbe_ExtendedDataElement* (*hcbe_CommonBaseEventAddExtendedDataElementFP5)(struct hcbe_CommonBaseEvent*, char*, int);
typedef hcbe_ExtendedDataElement* (*hcbe_CommonBaseEventAddExtendedDataElementFP6)(struct hcbe_CommonBaseEvent*, char*, int*, int);
typedef hcbe_ExtendedDataElement* (*hcbe_CommonBaseEventAddExtendedDataElementFP7)(struct hcbe_CommonBaseEvent*, char*, long);
typedef hcbe_ExtendedDataElement* (*hcbe_CommonBaseEventAddExtendedDataElementFP8)(struct hcbe_CommonBaseEvent*, char*, long*, int);
typedef hcbe_ExtendedDataElement* (*hcbe_CommonBaseEventAddExtendedDataElementFP9)(struct hcbe_CommonBaseEvent*, char*, float);
typedef hcbe_ExtendedDataElement* (*hcbe_CommonBaseEventAddExtendedDataElementFP10)(struct hcbe_CommonBaseEvent*, char*, float*, int);
typedef hcbe_ExtendedDataElement* (*hcbe_CommonBaseEventAddExtendedDataElementFP11)(struct hcbe_CommonBaseEvent*, char*, double);
typedef hcbe_ExtendedDataElement* (*hcbe_CommonBaseEventAddExtendedDataElementFP12)(struct hcbe_CommonBaseEvent*, char*, double*, int);
typedef void (*hcbe_CommonBaseEventRemoveExtendedDataElementsFP)(struct hcbe_CommonBaseEvent*, char*);
typedef void (*hcbe_CommonBaseEventClearExtendedDataElementFP)(struct hcbe_CommonBaseEvent*);

typedef char* (*hcbe_CommonBaseEvent_addAnyFP)(struct hcbe_CommonBaseEvent*, char*);
typedef void (*hcbe_CommonBaseEventClearAnyFP)(struct hcbe_CommonBaseEvent*);

/* 80018 */
typedef void (*hcbe_CommonBaseEventSetContentHandlerFP)(struct hcbe_CommonBaseEvent*, struct hcbe_TemplateContentHandler*);
typedef hcbe_TemplateContentHandler* (*hcbe_CommonBaseEventGetContentHandlerFP)(struct hcbe_CommonBaseEvent*);
typedef int (*hcbe_CommonBaseEventIsCompleteFP)(struct hcbe_CommonBaseEvent*);
typedef void (*hcbe_CommonBaseEventCompleteFP) (struct hcbe_CommonBaseEvent*);

/* 
 * Sub-Section: CommonBaseEvent Function Table Declaration 
 */
typedef struct
{
    hcbe_CommonBaseEventGetStringFP getVersion,
			       getLocalInstanceId,
			       getGlobalInstanceId,
			       getCreationTime,
			       getMsg,
			       getExtensionName;


    hcbe_CommonBaseEventSetStringFP setVersion,
                               setLocalInstanceId,
                               setGlobalInstanceId,
                               setCreationTime,
                               setMsg,
                               setExtensionName;

    hcbe_CommonBaseEventGetShortFP getSeverity,
			                  getPriority,
			                  getRepeatCount;

    hcbe_CommonBaseEventSetShortFP setSeverity,
                              setPriority,
                              setRepeatCount;

    hcbe_CommonBaseEventGetLongFP getElapsedTime,
                             getSequenceNumber;

    hcbe_CommonBaseEventSetLongFP setElapsedTime,
                             setSequenceNumber;

    hcbe_CommonBaseEventGetCompIdFP getReporterComponentIdentification,
                               getSourceComponentIdentification;

    hcbe_CommonBaseEventSetCompIdFP setReporterComponentIdentification,
                               setSourceComponentIdentification;

    hcbe_CommonBaseEventGetSituationFP getSituation;
    hcbe_CommonBaseEventSetSituationFP setSituation;

    hcbe_CommonBaseEventGetContextDataElementsFP getContextDataElements;
    hcbe_CommonBaseEventSetContextDataElementsFP setContextDataElements;

    hcbe_CommonBaseEventGetMsgDataElementFP getMsgDataElement;
    hcbe_CommonBaseEventSetMsgDataElementFP setMsgDataElement;

    hcbe_CommonBaseEventGetAssociatedEventsFP getAssociatedEvents;
    hcbe_CommonBaseEventSetAssociatedEventsFP setAssociatedEvents;
    hcbe_CommonBaseEventClearAssociatedEventFP clearAssociatedEvents;

    hcbe_CommonBaseEventGetExtendedDataElementsFP getExtendedDataElements;
    hcbe_CommonBaseEventSetExtendedDataElementsFP setExtendedDataElements;

	hcbe_CommonBaseEventGetAnyFP getAny;
	hcbe_CommonBaseEventSetAnyFP setAny;

	hcbe_CommonBaseEventArrGetCounterFP getContextDataElementsSize,
								   getAssociatedEventsSize,
	                               getExtendedDataElementsSize,
	    						   getAnySize;

    /* 80016 */
    hcbe_CommonBaseEventSetMsgDataElementFP2 setMsgDataElementWithFullProperties;
                                             
    hcbe_CommonBaseEventSetCompIdFP2 setReporterComponentIdWithFullProperties,
                                     setSourceComponentIdWithFullProperties;
    hcbe_CommonBaseEventSetCompIdFP3 setReporterComponentIdWith7Properties,
                                     setSourceComponentIdWith7Properties;

    hcbe_CommonBaseEventAddContextDataElementFP addContextDataElement;
 
    hcbe_CommonBaseEventAddContextDataElementFP2 addContextDataElementWithId,
                                                 addContextDataElementWithValue;
    hcbe_CommonBaseEventRemoveContextDataElementsFP removeContextDataElements;
    hcbe_CommonBaseEventClearContextDataElementFP clearContextDataElements;
    
    hcbe_CommonBaseEventAddAssociatedEventFP addAssociatedEvent;
    hcbe_CommonBaseEventAddAssociatedEventFP2 addAssociatedEventGen;

    hcbe_CommonBaseEventAddExtendedDataElementFP addExtendedDataElement;
    hcbe_CommonBaseEventAddExtendedDataElementFP1 addExtendedDataElementWithStringValue, 
                                                  addExtendedDataElementWithDateValue, 
                                                  addExtendedDataElementWithBooleanValue, 
                                                  addExtendedDataElementWithHexValue;
    hcbe_CommonBaseEventAddExtendedDataElementFP2 addExtendedDataElementWithStringArrayValues, 
                                                  addExtendedDataElementWithDateArrayValues, 
                                                  addExtendedDataElementWithBooleanArrayValues;
    hcbe_CommonBaseEventAddExtendedDataElementFP3 addExtendedDataElementWithShortValue;
    hcbe_CommonBaseEventAddExtendedDataElementFP4 addExtendedDataElementWithShortArrayValues;
    hcbe_CommonBaseEventAddExtendedDataElementFP5 addExtendedDataElementWithIntValue;
    hcbe_CommonBaseEventAddExtendedDataElementFP6 addExtendedDataElementWithIntArrayValues;
    hcbe_CommonBaseEventAddExtendedDataElementFP7 addExtendedDataElementWithLongValue;
    hcbe_CommonBaseEventAddExtendedDataElementFP8 addExtendedDataElementWithLongArrayValues;
    hcbe_CommonBaseEventAddExtendedDataElementFP9 addExtendedDataElementWithFloatValue;
    hcbe_CommonBaseEventAddExtendedDataElementFP10 addExtendedDataElementWithFloatArrayValues;
    hcbe_CommonBaseEventAddExtendedDataElementFP11 addExtendedDataElementWithDoubleValue;
    hcbe_CommonBaseEventAddExtendedDataElementFP12 addExtendedDataElementWithDoubleArrayValues;
    hcbe_CommonBaseEventRemoveExtendedDataElementsFP removeExtendedDataElements;
    hcbe_CommonBaseEventClearExtendedDataElementFP clearExtendedDataElements;

    hcbe_CommonBaseEvent_addAnyFP addAny;
    hcbe_CommonBaseEventClearAnyFP clearAny;

    /* 80018 */
    hcbe_CommonBaseEventSetContentHandlerFP setContentHandler;
    hcbe_CommonBaseEventGetContentHandlerFP getContentHandler;
    hcbe_CommonBaseEventIsCompleteFP isComplete;
    hcbe_CommonBaseEventCompleteFP complete;

}hcbe_CommonBaseEventFPTable;

/* 
 * Sub-Section: CommonBaseEvent Data Structure Declaration 
 */
typedef struct hcbe_CommonBaseEvent
{
	char* _version;
    char* _localInstanceId;
    char* _globalInstanceId;
    char* _creationTime;
    short _severity;
    short _priority;
    hcbe_ComponentIdentification* _reporterComponentId;
    hcbe_ComponentIdentification* _sourceComponentId;
    hcbe_Situation* _situation;
    hcbe_ContextDataElement** _contextDataElements;
    hcbe_MsgDataElement* _msgDataElement;
    char* _msg;
    short _repeatCount;
    long _elapsedTime;
    hcbe_AssociatedEvent** _associatedEvents;
    char* _extensionName;
    hcbe_ExtendedDataElement** _extendedDataElements;
    long _sequenceNumber;
	char** _any;
    hcbe_TemplateContentHandler* _templateContentHandler;
    int _isComplete;
	 
    int _isSetCreationTime;
    int _isSetElapsedTime;
    int _isSetPriority;
    int _isSetRepeatCount;
    int _isSetSequenceNumber; 
    int _isSetSeverity;

	int _contextDataElementsSize;
	int _associatedEventsSize;
	int _extendedDataElementsSize;
	int _anySize;
    int _deserialized;

	hcbe_CommonBaseEventFPTable* fpRoot;

}hcbe_CommonBaseEvent;

/* 
 * Sub-Section: CommonBaseEvent Constructor and Destructor Declaration 
 */
hcbe_CommonBaseEvent* hcbe_commonBaseEventConstr();
char* hcbe_commonBaseEventDestr(hcbe_CommonBaseEvent*);


/**********************************************************************
 * Utility Functions Declarations Section
 **********************************************************************/

int hcbe_associationEngineToCanonicalXMLString(hcbe_AssociationEngine*, char*, int);

int hcbe_associationEngineToCanonicalXMLDocString(hcbe_AssociationEngine*, char*, int);

int hcbe_commonBaseEventToCanonicalXMLString(hcbe_CommonBaseEvent*, char*, int, int); 

int hcbe_commonBaseEventToCanonicalXMLDocString(hcbe_CommonBaseEvent*, char*, int, int);

hcbe_AssociationEngine* hcbe_associationEngineFromCanonicalXMLString(char* xmlString);

hcbe_AssociationEngine* hcbe_associationEngineFromCanonicalXMLDocString(char* aXMLDocString);

hcbe_CommonBaseEvent* hcbe_commonBaseEventFromCanonicalXMLString(char* xmlString);

hcbe_CommonBaseEvent* hcbe_commonBaseEventFromCanonicalXMLDocString(char* aXMLDocString);

int hcbe_getCommonBaseEventXMLStringSize(hcbe_CommonBaseEvent*, int);

int hcbe_xmlUtilityNormalize(char*, const char*, int);

int hcbe_xmlUtilityDenormalize(char*, const char*, int);

int hcbe_serialize(char* fileName, char*);

int hcbe_generateGUID(char* guid);

int hcbe_getCurrentTimeUTC(char* timestamp);


#ifdef __cplusplus
}
#endif

#endif