/**********************************************************************
* Copyright (c) 2005 IBM Corporation and others.
* All rights reserved.   This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
* $Id: Sample.c,v 1.18 2005/04/29 20:29:18 paules Exp $
*
* Contributors:
* IBM - Initial API and implementation
**********************************************************************/

#include <stdio.h>
#include <string.h>
#include <time.h>

#include "CommonBaseEvent.h"
#include "XmlUtility.h"

/**********************************************************************
 * File: Sample.c
 *
 * This source file illustrates the Common Base Event v1.0.1 
 * implementation.    
 * 
 * The sample demonstrates creating and configuring a Common Base Event
 * structure and serializing the Common Base Event to Common Base Event 
 * XML.  The  Common Base Event XML fragment is written to standard
 * output.
 *
 * The following nine Common Base Event data types and their properties
 * based on the Common Base Event v1.0.1 specification are illustrated 
 * in this sample:
 * 
 *   -hcbe_AssociationEngine 
 *   -hcbe_MsgDataElement 
 *   -hcbe_AssociatedEvent
 *   -hcbe_ComponentIdentification
 *   -hcbe_ContextDataElement
 *   -hcbe_ExtendedDataElement
 *   -hcbe_SituationType
 *   -hcbe_Situation
 *
 *
 * Author(s): Paul Slauenwhite
 * Version:   April 29, 2005
 * Since:     October 14, 2004
 ***********************************************************************/

int main(int argc, char* argv[]){

    /* Step 1:  Define the required structures and variables: */

	/* Define a parameter error flag: */
	int error = 0;

	/* Create an event factory structure: */
	hcbe_EventFactory* eventFactory = hcbe_eventFactoryConstr();

	/* Create a template Common Base Event structure: */
	hcbe_CommonBaseEvent* templateCommonBaseEvent = hcbe_commonBaseEventFromCanonicalXMLString("<CommonBaseEvent localInstanceId=\"Sample Event\"><sourceComponentId location=\"127.0.0.1\" locationType=\"IPV4\" componentIdType=\"Application\" componentType=\"Eclipse_TPTP\"/><situation categoryName=\"ReportSituation\"><situationType xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:type=\"ReportSituation\" reasoningScope=\"INTERNAL\" reportCategory=\"LOG\"/></situation></CommonBaseEvent>");      	

	/* Create a template content hander structure: */
    hcbe_TemplateContentHandler* templateContentHandler = hcbe_templateContentHandlerConstr();

  	/* Define a Common Base Event structure: */
	hcbe_CommonBaseEvent* commonBaseEvent;

	/* Define a source component identification structure: */
	hcbe_ComponentIdentification* sourceComponentIdentification;

	/* Create an extended data element structure: */
	hcbe_ExtendedDataElement* extendedDataElement = hcbe_extendedDataElementConstr();

	/* Define an array buffer for the extended data element's values: */
    char* extendedDataElementValues[1];

	/* Define an array buffer for the extended data elements: */
    hcbe_ExtendedDataElement* extendedDataElements[1];

	/* Define the size (8K) of the character buffer for the serialized Common Base Event XML fragment: */
    int commonBaseEventXMLFragmentSize = 8192;

	/* Create a character buffer for the serialized Common Base Event XML fragment: */
    char* commonBaseEventXMLFragmentBuffer = ((char*)(malloc(commonBaseEventXMLFragmentSize)));
	
	/* Define a character buffer for the Common Base Event's Globally Unique IDentifier (GUID): */	
	char guidBuffer[64];

	/* Define a character buffer for the Common Base Event's creation time: */	
    char creationTimeBuffer[32];


	/* Step 2:  Configure the event factory with the template event and its content handler: */
    templateContentHandler->fpRoot->setTemplateEvent(templateContentHandler, templateCommonBaseEvent);    
   
	/* Set the template event and its content handler on the event factory: */
	eventFactory->fpRoot->setContentHandler(eventFactory, templateContentHandler);  

    /* Set the event factory to auto-complete newly created Common Base Events: */
    eventFactory->fpRoot->setCompleteEvent(eventFactory,1);


	/* Step 3:  Create the Common Base Event structure and set its properties: */
    commonBaseEvent = eventFactory->fpRoot->createCommonBaseEvent(eventFactory);

	/* Set the Common Base Event's Globally Unique IDentifier (GUID): */
	hcbe_generateGUID(guidBuffer);
	commonBaseEvent->fpRoot->setGlobalInstanceId(commonBaseEvent, guidBuffer);

    /* Set the Common Base Event's message: */
    commonBaseEvent->fpRoot->setMsg(commonBaseEvent, "Hyades Common Base Event v1.0.1 Sample log message.");

	/* Set the Common Base Event's severity: */
    commonBaseEvent->fpRoot->setSeverity(commonBaseEvent, 60);

	/* Set the Common Base Event's creation time: */
    /* NOTE:  XSD:dateTime value is in Universal Coordinated Time (UTC) (e.g. yyyy-MM-ddTHH:mm:ss.SSSSSSZ\0). */
    hcbe_getCurrentTimeUTC(creationTimeBuffer);
    commonBaseEvent->fpRoot->setCreationTime(commonBaseEvent, creationTimeBuffer);

	/* Retrieve and set the source component identification's properties: */
    sourceComponentIdentification = commonBaseEvent->fpRoot->getSourceComponentIdentification(commonBaseEvent);
	
	sourceComponentIdentification->fpRoot->setComponent(sourceComponentIdentification,"Sample");
	sourceComponentIdentification->fpRoot->setSubComponent(sourceComponentIdentification,"main");

	/* Set the Common Base Event's source component identification: */
	commonBaseEvent->fpRoot->setSourceComponentIdentification(commonBaseEvent,sourceComponentIdentification);
	
	/* Set the extended data element's properties: */
    extendedDataElement->fpRoot->setName(extendedDataElement,"Sample extended data element name");
	extendedDataElement->fpRoot->setType(extendedDataElement,HCBE_TYPE_STRING);
	extendedDataElementValues[0] = "Sample extended data element value";
	extendedDataElement->fpRoot->setValues(extendedDataElement,extendedDataElementValues,1);

	extendedDataElements[0] = extendedDataElement;

	/* Set the Common Base Event's extended data element: */
	commonBaseEvent->fpRoot->setExtendedDataElements(commonBaseEvent,extendedDataElements,1);	
		

	/* Step 4:  Serialize the Common Base Event structure to a Common Base Event XML fragment: */
	while (hcbe_commonBaseEventToCanonicalXMLString(commonBaseEvent, commonBaseEventXMLFragmentBuffer, commonBaseEventXMLFragmentSize, 1) < 0) {
            
		free(commonBaseEventXMLFragmentBuffer);
    
        commonBaseEventXMLFragmentSize += 1024;

		commonBaseEventXMLFragmentBuffer = ((char*)(malloc(commonBaseEventXMLFragmentSize)));
	}


	/* Step 5:  Write the serialized Common Base Event XML fragment to standard output: */
	printf(commonBaseEventXMLFragmentBuffer, 1, strlen(commonBaseEventXMLFragmentBuffer));

	
	/* Step 6:  Free memory: */
	free(commonBaseEventXMLFragmentBuffer);
    hcbe_commonBaseEventDestr(commonBaseEvent);
    hcbe_eventFactoryDestr(eventFactory);
    hcbe_templateContentHandlerDestr(templateContentHandler);

	printf("\nSample successfully ended!\n");

	return(EXIT_SUCCESS);
}






