/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: InsertObjects.java,v 1.4 2005/02/16 22:21:29 qiyanli Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.resources.database.internal.extensions;
import java.sql.PreparedStatement;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.hyades.resources.database.internal.dbmodel.Column;
import org.eclipse.hyades.resources.database.internal.dbmodel.Table;
import org.eclipse.hyades.resources.database.internal.impl.ClassMetadata;
import org.eclipse.hyades.resources.database.internal.impl.InsertStatement;
import org.eclipse.hyades.resources.database.internal.impl.ReferenceInfo;
import org.eclipse.hyades.resources.database.internal.impl.TypeConverter;
import org.eclipse.hyades.resources.database.internal.impl.WeakObjectCache;
/**
 * This class generates one or more insert statements for adding objects to
 * class tables.
 */
public class InsertObjects extends InsertStatement {
	protected ClassMetadata metadata;
	protected TypeConverter converter;
	protected Map manyReferenceValues;
	protected WeakObjectCache cache;
	public InsertObjects(DatabaseType type, ClassMetadata metadata, TypeConverter converter, Map objectsToIds, Map manyReferenceValues, WeakObjectCache cache) {
		super(null, type, objectsToIds);
		this.cache = cache;
		this.metadata = metadata;
		this.converter = converter;
		this.manyReferenceValues = manyReferenceValues;
	}
	/**
	 * Create a row for the class table.
	 */
	protected StringBuffer createRow(EObject object) throws Exception {
		EAttribute[] attribs = metadata.getSingleAttributes();
		int[] sqlTypes = metadata.getClassTableSQLTypes();
		StringBuffer row = new StringBuffer();
		row.append("(");
		boolean valueAppended = false;
		for (int i = 0; i < attribs.length; i++) {
			if (valueAppended)
				row.append(", ");
			Object value = converter.getValue(object, attribs[i]);
			appendValue(row, value, sqlTypes[i]);
			valueAppended = true;
		}
		if (metadata.isComputeId()) {
			if (valueAppended)
				row.append(", ");
			valueAppended = true;

			appendValue(row, objectsToIds.get(object), Types.INTEGER);
		}

		if (valueAppended)
			row.append(", ");
		appendValue(row, getParentPath(object), Types.VARCHAR);

		EReference[] references = metadata.getClassTableReferences();
		for (int i = 0; i < references.length; i++) {
			EReference reference = references[i];
			if (reference.isMany()) {
				Map referenceMap = (Map) manyReferenceValues.get(object);
				if (referenceMap == null)
					row.append(", NULL, NULL");
				else {
					ReferenceInfo info = (ReferenceInfo) referenceMap.get(reference);
					if (info == null)
						row.append(", NULL, NULL");
					else {
						Object id = objectsToIds.get(info.getSource());
						if (id != null)
							row.append(", " + id.toString());
						else
							row.append(", NULL");
						row.append(", " + info.getPosition());
					}
				}
			} else {
				EObject value = (EObject) object.eGet(reference, false);
				row.append(", ");
				if (value == null)
					row.append("NULL");
				else {
					Object id = objectsToIds.get(value);
					if (id != null)
						row.append(id.toString());
					else
						row.append("NULL");
				}
			}
		}
		row.append(")");
		return row;
	}
	protected Table getTable() {
		return metadata.getTable();
	}
	protected List getColumns() {
		List columns = new ArrayList();
		columns.addAll(metadata.getClassTableColumns());
		if (metadata.isComputeId())
			columns.add(metadata.getPrimaryKey());
		columns.add(getColumn("p_p"));
		List referenceColumns = metadata.getReferenceClassTableColumns();
		columns.addAll(referenceColumns);
		return columns;
	}
	/**
	 * @param string
	 * @return
	 */
	private Column getColumn(String columnName) {
		for (Iterator iter = getTable().getColumns().iterator(); iter.hasNext();) {
			Column element = (Column) iter.next();
			if (element.getName().equals(columnName))
				return element;
		}
		return null;
	}

	/**
	 * @param object
	 * @return
	 */
//	protected Table getTableName(EObject object) {
//		ClassMetadata classMetadata = metadata.getDbMap().getClassMetadata(object.eClass());
//
//		return classMetadata.getTable();
//	}
	/**
	 * @param object
	 * @return
	 */
	protected String getParentPath(EObject object) {
		String parentPath = "/";
		boolean first = true;
		object = object.eContainer();
		while (object != null) {
			if (first) {
				first = false;
			} else {
				parentPath = "/" + parentPath;
			}
			Integer id = (Integer) objectsToIds.get(object);
			if (id == null) {
				id = cache.getId(object);
			}
//			parentPath = getTableName(object).getName().hashCode() + ":" + id + parentPath;
			parentPath = id + parentPath;
			object = object.eContainer();
		}
		return parentPath;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.resources.database.internal.impl.InsertStatement#addBatchedRow(java.sql.PreparedStatement)
	 */
	protected void addBatchedRow(PreparedStatement ps,EObject object) throws Exception{
		EAttribute[] attribs = metadata.getSingleAttributes();
		int[] sqlTypes = metadata.getClassTableSQLTypes();
		int colPos=1;
		for (int i = 0; i < attribs.length; i++) {
			Object value = converter.getValue(object, attribs[i]);
			appendBatchedValue(ps,colPos++,value, sqlTypes[i]);
		}
		if (metadata.isComputeId()) {
			appendBatchedValue(ps,colPos++,objectsToIds.get(object), Types.INTEGER);
		}
		appendBatchedValue(ps,colPos++, getParentPath(object), Types.VARCHAR);

		EReference[] references = metadata.getClassTableReferences();
		for (int i = 0; i < references.length; i++) {
			EReference reference = references[i];
			if (reference.isMany()) {
				Map referenceMap = (Map) manyReferenceValues.get(object);
				if (referenceMap == null)
				{
					appendBatchedValue(ps,colPos++, null,Types.INTEGER);
					appendBatchedValue(ps,colPos++, null,Types.INTEGER);
				}
				else {
					ReferenceInfo info = (ReferenceInfo) referenceMap.get(reference);
					if (info == null)
					{
						appendBatchedValue(ps,colPos++, null,Types.INTEGER);
						appendBatchedValue(ps,colPos++, null,Types.INTEGER);
					}
					else {
						Object id = objectsToIds.get(info.getSource());
						if (id != null)
							appendBatchedValue(ps,colPos++,id, Types.INTEGER);
						else
						{
							appendBatchedValue(ps,colPos++, null,Types.INTEGER);
						}
						appendBatchedValue(ps,colPos++, new Integer(info.getPosition()),Types.INTEGER);
					}
				}
			} else {
				EObject value = (EObject) object.eGet(reference, false);
				if (value == null)
					appendBatchedValue(ps,colPos++, null,Types.INTEGER);
				else {
					Object id = objectsToIds.get(value);
					if (id != null)
						appendBatchedValue(ps,colPos++,id, Types.INTEGER);
					else
						appendBatchedValue(ps,colPos++, null,Types.INTEGER);
				}
			}
		}
		ps.addBatch();
	}

} // InsertObjects
