/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: AttributeValueQuery.java,v 1.4 2005/02/16 22:21:29 qiyanli Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.resources.database.internal.impl;

import java.sql.Types;
import java.util.List;

import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.hyades.resources.database.internal.DBMap;
import org.eclipse.hyades.resources.database.internal.dbmodel.Column;
import org.eclipse.hyades.resources.database.internal.extensions.DatabaseType;
/**
 * This class generates an SQL SELECT statement that retrieves the ids from rows
 * in the class table where the attribute values match the given values and the
 * given ids.
 */
public class AttributeValueQuery extends QueryStatement {
	protected EClass eClass;
	protected List values;
	protected List ids;
	protected String primaryKeyName;

	/**
	 * Constructor for AttributeValueQuery.
	 * 
	 * @param map
	 * @param eClass
	 * @param values
	 * @param ids
	 */
	public AttributeValueQuery(DatabaseType type, DBMap map, EClass eClass, List values, List ids) {
		super(type, map, eClass);
		this.eClass = eClass;
		this.values = values;
		this.ids = ids;
	}

	/**
	 * Select the id column from the class table.
	 */
	protected void createSelectFrom() {
		RDBHelper rdbHelper = new RDBHelper();
		primaryKeyName = rdbHelper.getPrimaryKey(table).getName();
		statement.append("SELECT ");
		statement.append(addQuotes(primaryKeyName));
		statement.append(" FROM ");
		statement.append(addQuotes(table.getName()));
	}

	/**
	 * Create the where clause.
	 */
	protected void createWhereClause(int begin) {
		if (values.isEmpty() && ids.isEmpty())
			return;

		statement.append(" WHERE");
		createAttributeTests();
		createIdTest();
	}

	/**
	 * Create the attribute equality tests.
	 */
	protected void createAttributeTests() {
		if (values.isEmpty())
			return;

		for (int i = 0, l = values.size(); i < l - 1; i += 2) {
			EAttribute attribute = (EAttribute) values.get(i);
			Object value = values.get(i + 1);
			createAttributeTest(attribute, value, i);
		}
	}

	/**
	 * Create a test to see if the attribute column value is equal to the given
	 * value.
	 */
	protected void createAttributeTest(EAttribute attribute, Object value, int i) {
		String name = getColumnName(attribute);
		int type = getSQLType(attribute);

		if (i == 0)
			statement.append(" ");
		else
			statement.append(" AND ");

		if (isString(type) && value == null) {
			statement.append(addQuotes(name));
			statement.append(" IS NULL");
		} else {
			statement.append(addQuotes(name));
			statement.append(" = ");

			if (TypeConverter.isBooleanType(attribute))
				appendBooleanValue(value);
			else
				appendValue(value, type);
		}
	}

	/**
	 * Append the value to the where clause.
	 */
	protected void appendValue(Object value, int type) {
		if (isString(type)) {
			statement.append("'");
			statement.append(value.toString());
			statement.append("'");
		} else
			statement.append(value.toString());
	}

	/**
	 * Appends a boolean value to the statement.
	 */
	protected void appendBooleanValue(Object value) {
		statement.append("'");

		if (value == Boolean.TRUE)
			statement.append(TypeConverter.TRUE_VALUE);
		else if (value == Boolean.FALSE)
			statement.append(TypeConverter.FALSE_VALUE);

		statement.append("'");
	}

	/**
	 * Returns TRUE if the SQL type is a string type.
	 */
	protected boolean isString(int type) {
		return type == Types.VARCHAR || type == Types.CHAR;
	}

	/**
	 * Get the column name for the given attribute.
	 */
	protected String getColumnName(EAttribute attribute) {
		Column column = getColumn(attribute);
		return column.getName();
	}

	/**
	 * Return the column for the given attribute.
	 */
	protected Column getColumn(EAttribute attribute) {
		DBMap.AttributeData data = (DBMap.AttributeData) dbMap.getDBRepresentation(attribute);
		return data.getValueColumn();
	}

	/**
	 * Get the SQL type for the column for the given attribute.
	 */
	protected int getSQLType(EAttribute attribute) {
		Column column = getColumn(attribute);
		return column.getType().getSqlType();
	}

	/**
	 * Create the id test to test for an id that is one of the ids.
	 */
	protected void createIdTest() {
		if (ids.isEmpty())
			return;

		statement.append(" AND ");
		statement.append(addQuotes(primaryKeyName));
		statement.append(" IN ( ");

		boolean first = true;

		for (int i = 0, l = ids.size(); i < l; i++) {
			Integer id = (Integer) ids.get(i);

			if (first) {
				first = false;
				statement.append(id.toString());
			} else {
				statement.append(", ");
				statement.append(id.toString());
			}
		}

		statement.append(")");
	}
} // AttributeValueQuery
