/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DBMapBuilder.java,v 1.4 2005/02/16 22:21:29 qiyanli Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.resources.database.internal.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EModelElement;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.hyades.resources.database.internal.DBMap;
import org.eclipse.hyades.resources.database.internal.TypeMap;
import org.eclipse.hyades.resources.database.internal.dbmodel.Column;
import org.eclipse.hyades.resources.database.internal.dbmodel.Constraint;
import org.eclipse.hyades.resources.database.internal.dbmodel.Database;
import org.eclipse.hyades.resources.database.internal.dbmodel.DbmodelFactory;
import org.eclipse.hyades.resources.database.internal.dbmodel.Table;
/**
 * This class makes a DB map from Ecore objects to RDB objects.
 * <p>
 * The created mapping from Ecore to the RDB model is as follows: EPackage -->
 * RDBDatabase EClass --> ClassData EAttribute --> AttributeData EReference -->
 * ReferenceData
 */
public class DBMapBuilder {
	protected boolean debug = false;

	public static final int CONSTRAINT_NAME_LENGTH = 18;
	public static final String PRIMARY_KEY_NAME = "PrimaryKey_";
	public static final String INDEX_NAME = "Index_";
	public static final String SOURCE_ID_COLUMN_NAME = "Source_Id";
	public static final String TARGET_ID_COLUMN_NAME = "Target_Id";
	public static final String TARGET_ORDER_COLUMN_NAME = "Target_Order";
	public static final String SOURCE_ORDER_COLUMN_NAME = "Source_Order";
	public static final String VALUE_COLUMN_NAME = "Value";
	public static final String ID_COLUMN_NAME = "Id";
	public static final String ORDER_COLUMN_NAME = "Order";
	public static final String RESOURCE_TABLE_NAME = "Resource_Table";
	public static final String URI_COLUMN_NAME = "URI";
	public static final String TABLE_COLUMN_NAME = "Table_Name";
	public static final String IS_PROXY_COLUMN_NAME = "Is_EMF_Proxy";
	public static final String PROXY_URI_COLUMN_NAME = "Proxy_URI";
	public static final String PROXY_TABLE_NAME = "Proxy_Table";
	public static final String ID_TABLE_NAME = "Id_Table";

	protected String databaseName;
	protected DbmodelFactory dbFactory;
	protected DBMap map;
	protected Map classesToNames;
	protected Map classesToReferences;
	protected Database database;
	protected List allPackages;
	protected List allClasses;
	protected List allReferences;
	protected List referencesForTables;
	protected int primaryKeyCount = 0;
	protected int indexCount = 0;
	protected TypeMap typeMap;
	protected RDBHelper rdbHelper;

	/**
	 * Constructor for DBMapBuilder.
	 */
	public DBMapBuilder(String databaseName, TypeMap typeMap) {
		super();
		this.databaseName = databaseName;
		dbFactory = DbmodelFactory.eINSTANCE;
		this.typeMap = typeMap;
		rdbHelper = new RDBHelper();
	}

	public DBMap getMap(EPackage pkg) {
		List pkgs = new ArrayList();
		pkgs.add(pkg);
		return getMap(pkgs);
	}

	public DBMap getMap(List packages) {
		if (map != null)
			return map;

		map = new DBMapImpl();
		allPackages = getAllPackages(packages);

		database = dbFactory.createDatabase();
		database.setName(databaseName);

		for (int i = 0, l = allPackages.size(); i < l; i++)
			map.add((EModelElement) allPackages.get(i), database);

		processClassifiers();
		processReferences();
		createResourceTable();
		createProxyTable();
		createIdTable();

		return map;
	}

	protected List getAllPackages(List packages) {
		List pkgs = new ArrayList();

		for (int i = 0, l = packages.size(); i < l; i++)
			pkgs.addAll(getPackagesFromPackage((EPackage) packages.get(i)));

		return pkgs;
	}

	protected List getPackagesFromPackage(EPackage pkg) {
		List packages = new ArrayList();
		packages.add(pkg);
		List subpackages = pkg.getESubpackages();

		for (int i = 0, l = subpackages.size(); i < l; i++)
			packages.addAll(getPackagesFromPackage((EPackage) subpackages.get(i)));

		return packages;
	}

	/**
	 * Process all of the classes in each package.
	 */
	protected void processClassifiers() {
		getAllClasses();
		assignClassNames();
		getAllReferences();

		for (int i = 0, l = allClasses.size(); i < l; i++)
			processClass((EClass) allClasses.get(i));
	}

	/**
	 * Get all of the classes in the model and put them in the allClasses list.
	 */
	protected void getAllClasses() {
		allClasses = new ArrayList();

		for (int i = 0, l = allPackages.size(); i < l; i++) {
			EPackage pkg = (EPackage) allPackages.get(i);
			List classifiers = pkg.getEClassifiers();

			for (int j = 0, l2 = classifiers.size(); j < l2; j++) {
				EClassifier classifier = (EClassifier) classifiers.get(j);

				if (classifier instanceof EClass)
					allClasses.add(classifier);
			}
		}
	}

	/**
	 * Assign a unique name to each class and put them in the classesToNames
	 * map. If there are duplicate names, the names for those classes have the
	 * package names added to them.
	 */
	protected void assignClassNames() {
		classesToNames = new HashMap();
		assignNames(allClasses, 0);
	}

	/**
	 * Assign a name to each class in the classes list, including the number of
	 * package names indicated. If the names are unique, put them in the Map;
	 * otherwise, continue processing the names.
	 */
	protected void assignNames(List classes, int packageNumber) {
		Set uniqueNames = new HashSet();
		Map namesToClasses = new HashMap();
		List duplicates = new ArrayList();

		for (int i = 0, l = classes.size(); i < l; i++) {
			EClass eClass = (EClass) classes.get(i);
			assignNameToClass(eClass, uniqueNames, namesToClasses, duplicates, packageNumber);
		}

		if (duplicates.size() > 0) {
			assignNames(duplicates, ++packageNumber);
		}
	}

	/**
	 * Assing a name to the given class, updating the uniqueNames,
	 * namesToClasses, and duplicates collections appropriately.
	 */
	protected void assignNameToClass(EClass eClass, Set uniqueNames, Map namesToClasses, List duplicates, int packageNumber) {
		String name = getName(eClass, packageNumber);

		if (!uniqueNames.contains(name)) {
			classesToNames.put(eClass, name);
			uniqueNames.add(name);
			namesToClasses.put(name, eClass);
		} else {
			duplicates.add(eClass);
			EClass original = (EClass) namesToClasses.get(name);

			if (original != null) {
				classesToNames.remove(original);
				namesToClasses.remove(name);
				duplicates.add(original);
			}
		}
	}

	/**
	 * Compute a name for the class, including the appropriate number of package
	 * names.
	 */
	protected String getName(EClass eClass, int packageNumber) {
		String name = eClass.getName();
		EPackage ePackage = eClass.getEPackage();

		while (ePackage != null && packageNumber > 0) {
			String pkgName = ePackage.getName();
			pkgName = upperCaseFirst(pkgName);
			name = pkgName + "_" + name;
			--packageNumber;
			ePackage = ePackage.getESuperPackage();
		}

		return name;
	}

	protected String upperCaseFirst(String name) {
		return name.substring(0, 1).toUpperCase() + name.substring(1);
	}

	/**
	 * Gets a name for a class.
	 */
	protected String getName(EClass eClass) {
		return (String) classesToNames.get(eClass);
	}

	// Get all of the references in the model.
	protected void getAllReferences() {
		allReferences = new ArrayList();

		for (int i = 0, l = allClasses.size(); i < l; i++) {
			EClass eClass = (EClass) allClasses.get(i);
			List references = eClass.getEReferences();
			allReferences.addAll(references);
		}
	}

	// Create an Table for each class.
	protected void processClass(EClass cls) {
		Table table = rdbHelper.createTable(database, getName(cls));
		DBMap.ClassData data = new DBMap.ClassData(table, true, null);
		map.add(cls, data);

		// add "p_p" column, the format is like a hierarchical path:
		// TableID:ObjectID/TableID:ObjectID/...
		// TableID would be the hashCode() of table name (class name)

		Column parentPath = rdbHelper.addColumnToTable(table, "p_p");
		rdbHelper.setColumnType(parentPath, "Blah", typeMap);

		processFeatures(table, cls);
		processMultiValuedAttributes(cls);
	}

	// Add columns for single-valued, non-transient attributes as
	// well as references. The references may not belong to the class,
	// but have the class as their type.
	protected void processFeatures(Table table, EClass cls) {
		Column primaryKey = null;
		Set columnNames = new HashSet();
		List attributes = cls.getEAttributes();

		for (int i = 0, l = attributes.size(); i < l; i++) {
			EAttribute attrib = (EAttribute) attributes.get(i);
			Column column = addAttributeToClassTable(attrib, table, columnNames);

			if (column != null && isKeyAttribute(attrib))
				primaryKey = column;
		}

		if (classesToReferences == null)
			computeClassTableReferences();

		if (primaryKey == null) {
			primaryKey = rdbHelper.addColumnToTable(table, createPrimaryKeyName(columnNames));
			rdbHelper.setColumnType(primaryKey, "EInt", typeMap);
		}

		List references = (List) classesToReferences.get(cls);

		if (references != null)
			for (int i = 0, l = references.size(); i < l; i++) {
				EReference reference = (EReference) references.get(i);
				addReferenceToClassTable(reference, table, primaryKey);
			}

		primaryKey.setAllowNull(false);
		rdbHelper.addPrimaryKeyToTable(table, primaryKey, PRIMARY_KEY_NAME + (++primaryKeyCount));
		Column proxyURI = rdbHelper.addColumnToTable(table, IS_PROXY_COLUMN_NAME);
		rdbHelper.setColumnType(proxyURI, "EBoolean", typeMap);
	}

	// Create a column in the class table for the attribute, if
	// necessary, and return the column. Add the attribute to the
	// DBMap.
	protected Column addAttributeToClassTable(EAttribute attribute, Table table, Set columnNames) {
		if (attribute.isTransient() || attribute.isMany())
			return null;

		Column column = rdbHelper.addColumnToTable(table, attribute.getName());
		column.setDefaultValue(attribute.getDefaultValue());
		rdbHelper.setColumnType(column, attribute, typeMap);
		DBMap.AttributeData data = new DBMap.AttributeData(column);
		map.add(attribute, data);
		columnNames.add(column.getName());

		if (attribute.isID()) {
			Constraint index = rdbHelper.addIndexToTable(table, INDEX_NAME + (++indexCount));
			column.getConstraints().add(index);
		}

		return column;
	}

	protected void addReferenceToClassTable(EReference reference, Table table, Column primaryKey) {
		Column column = rdbHelper.addColumnToTable(table, getReferenceColumnName(reference));
		column.setDefaultValue(reference.getDefaultValue());

		rdbHelper.setColumnType(column, "EInt", typeMap);
		Column order = null;

		if (reference.isMany()) {
			String name = column.getName() + "_Order";
			order = rdbHelper.addColumnToTable(table, name);
			rdbHelper.setColumnType(order, "EInt", typeMap);
		}

		Column source;
		Column target;

		if (reference.isMany()) {
			source = column;
			target = primaryKey;
		} else {
			source = primaryKey;
			target = column;
		}

		DBMap.ReferenceData data = new DBMap.ReferenceData(table, source, target, order);
		map.add(reference, data);
		EReference opposite = reference.getEOpposite();

		if (opposite != null) {
			data = new DBMap.ReferenceData(table, target, source, null);
			map.add(opposite, data);
		}
	}

	protected String getReferenceColumnName(EReference reference) {
		if (!reference.isMany())
			return reference.getName();
		else {
			EReference opposite = reference.getEOpposite();

			if (opposite == null) {
				if (reference.getEContainingClass() != reference.getEReferenceType()) {
					String name = reference.getEContainingClass().getName();
					return name.substring(0, 1).toLowerCase() + name.substring(1);
				} else
					return reference.getName();
			} else
				return opposite.getName();
		}
	}

	protected String createPrimaryKeyName(Set names) {
		if (!names.contains("id"))
			return "id";

		if (!names.contains("databaseId"))
			return "databaseId";

		return "emfDatabaseId";
	}

	// Determine the references that need to be put in the class
	// table of each class.
	protected void computeClassTableReferences() {
		referencesForTables = new ArrayList();
		classesToReferences = new HashMap();
		Set processedReferences = new HashSet();

		for (int i = 0, l = allReferences.size(); i < l; i++) {
			EReference reference = (EReference) allReferences.get(i);

			if (processedReferences.contains(reference))
				continue;

			EReference opposite = reference.getEOpposite();

			if (!ignoreReference(reference, opposite))
				processClassForReference(reference, opposite);

			processedReferences.add(reference);

			if (opposite != null)
				processedReferences.add(opposite);
		}
	}

	// For many-to-many relationships, add the reference to the
	// referencesForTables
	// list. For many references without an opposite reference, add the
	// reference to the
	// referencesForTables list. For one-to-one relationships, add
	// the reference to its owning class. For one-to-many
	// relationships, add the many reference to the class that
	// is the reference's type.
	protected void processClassForReference(EReference reference, EReference opposite) {
		EClass eClass;
		EReference referenceToAdd;

		if (opposite == null) {
			if (!reference.isMany()) {
				eClass = getClass(reference);
				referenceToAdd = reference;
			} else {
				referencesForTables.add(reference);
				return;
			}
		} else {
			// Change reference to be the non-transient reference,
			// if reference is transient and opposite is not transient
			if (reference.isTransient() && !opposite.isTransient()) {
				EReference temp = opposite;
				opposite = reference;
				reference = temp;
			}

			if (reference.isMany() && opposite.isMany()) {
				referencesForTables.add(reference);
				return;
			} else if (!reference.isMany() && !opposite.isMany()) {
				eClass = reference.getEContainingClass();
				referenceToAdd = reference;
			} else {
				if (reference.isMany())
					referenceToAdd = reference;
				else
					referenceToAdd = opposite;

				if (!referenceToAdd.isUnique()) {
					referencesForTables.add(referenceToAdd);
					return;
				}

				eClass = referenceToAdd.getEReferenceType();
			}
		}

		addReferenceToMap(eClass, referenceToAdd);
	}

	// Ignore a reference if it is has no opposite and is transient,
	// or if both the reference and its opposite are transient.
	protected boolean ignoreReference(EReference reference, EReference opposite) {
		if (reference.isTransient())
			if (opposite == null || opposite.isTransient())
				return true;

		return false;
	}

	// Return the class whose class table will contain the reference.
	// The reference does not have an opposite reference, so this
	// method returns the class that owns the reference if the
	// reference is single-valued, otherwise it returns the class
	// that is the type of the reference if the reference is
	// multi-valued.
	protected EClass getClass(EReference reference) {
		if (!reference.isMany())
			return reference.getEContainingClass();
		else
			return reference.getEReferenceType();
	}

	// Add the reference to the classesToReferences map.
	protected void addReferenceToMap(EClass eClass, EReference reference) {
		List references = (List) classesToReferences.get(eClass);

		if (references == null) {
			references = new ArrayList();
			classesToReferences.put(eClass, references);
		}

		references.add(reference);
	}

	protected void processMultiValuedAttributes(EClass cls) {
		List attributes = cls.getEAttributes();

		for (int i = 0, l = attributes.size(); i < l; i++) {
			EAttribute attribute = (EAttribute) attributes.get(i);

			if (attribute.isMany())
				createAttributeTable(cls, attribute);
		}
	}

	protected void createAttributeTable(EClass cls, EAttribute attribute) {
		String tableName = getName(cls) + "_" + attribute.getName();
		Table table = rdbHelper.createTable(database, tableName);
		Column id = addIdToAttributeTable(table, cls);
		Column value = addValueColumn(table, attribute);
		Column order = addOrderColumn(table);
		DBMap.AttributeData data = new DBMap.AttributeData(table, id, value, order);
		map.add(attribute, data);
	}

	protected Column addIdToAttributeTable(Table table, EClass cls) {
		Column id = rdbHelper.addColumnToTable(table, ID_COLUMN_NAME);
		rdbHelper.setColumnType(id, "EInt", typeMap);
		return id;
	}

	protected Column addValueColumn(Table table, EAttribute attribute) {
		Column value = rdbHelper.addColumnToTable(table, VALUE_COLUMN_NAME);
		rdbHelper.setColumnType(value, attribute, typeMap);
		return value;
	}

	protected Column addOrderColumn(Table table) {
		Column order = rdbHelper.addColumnToTable(table, ORDER_COLUMN_NAME);
		rdbHelper.setColumnType(order, "EInt", typeMap);
		return order;
	}

	// Do something smarter here!
	protected boolean isKeyAttribute(EAttribute attribute) {
		return false;
	}

	// Do this last so reference types will be in the classesToKeyNames
	// map and classes are in the classes list.
	protected void processReferences() {
		for (int i = 0, l = referencesForTables.size(); i < l; i++) {
			EReference reference = (EReference) referencesForTables.get(i);
			createReferenceTable(reference);
		}
	}

	protected void createReferenceTable(EReference reference) {
		EClass cls = reference.getEContainingClass();
		String tableName = getName(cls) + "_" + reference.getName();
		Table table = rdbHelper.createTable(database, tableName);

		Column source = rdbHelper.addColumnToTable(table, SOURCE_ID_COLUMN_NAME);
		rdbHelper.setColumnType(source, "EInt", typeMap);

		Column targetOrder = rdbHelper.addColumnToTable(table, "");
		targetOrder.setName(TARGET_ORDER_COLUMN_NAME);
		rdbHelper.setColumnType(targetOrder, "EInt", typeMap);

		Column target = rdbHelper.addColumnToTable(table, TARGET_ID_COLUMN_NAME);
		rdbHelper.setColumnType(target, "EInt", typeMap);

		EReference opposite = reference.getEOpposite();
		Column sourceOrder = null;

		if (opposite != null && opposite.isMany()) {
			sourceOrder = rdbHelper.addColumnToTable(table, SOURCE_ORDER_COLUMN_NAME);
			rdbHelper.setColumnType(sourceOrder, "EInt", typeMap);
		}

		addReferenceToMap(reference, table, source, target, targetOrder);

		if (opposite != null)
			addReferenceToMap(opposite, table, target, source, sourceOrder);
	}

	protected void addReferenceToMap(EReference reference, Table table, Column source, Column target, Column targetOrder) {
		DBMap.ReferenceData data = new DBMap.ReferenceData(table, source, target, targetOrder);
		map.add(reference, data);
	}

	protected void createResourceTable() {
		Table table = rdbHelper.createTable(database, RESOURCE_TABLE_NAME);
		Column uri = rdbHelper.addColumnToTable(table, URI_COLUMN_NAME);
		rdbHelper.setColumnType(uri, "Blah", typeMap);
		Column tableColumn = rdbHelper.addColumnToTable(table, TABLE_COLUMN_NAME);
		rdbHelper.setColumnType(tableColumn, "Blah", typeMap);
		Column id = rdbHelper.addColumnToTable(table, ID_COLUMN_NAME);
		rdbHelper.setColumnType(id, "EInt", typeMap);
		map.setResourceTable(table);
	}

	protected void createProxyTable() {
		Table table = rdbHelper.createTable(database, PROXY_TABLE_NAME);
		Column uri = rdbHelper.addColumnToTable(table, URI_COLUMN_NAME);
		rdbHelper.setColumnType(uri, "Blah", typeMap);
		Column tableColumn = rdbHelper.addColumnToTable(table, TABLE_COLUMN_NAME);
		rdbHelper.setColumnType(tableColumn, "Blah", typeMap);
		Column id = rdbHelper.addColumnToTable(table, ID_COLUMN_NAME);
		rdbHelper.setColumnType(id, "EInt", typeMap);
		Column proxyURIColumn = rdbHelper.addColumnToTable(table, PROXY_URI_COLUMN_NAME);
		rdbHelper.setColumnType(proxyURIColumn, "Blah", typeMap);
		map.setProxyTable(table);
	}

	protected void createIdTable() {
		Table table = rdbHelper.createTable(database, ID_TABLE_NAME);
		Column id = rdbHelper.addColumnToTable(table, ID_COLUMN_NAME);
		rdbHelper.setColumnType(id, "EInt", typeMap);
		map.setIdTable(table);
	}
} // DBMapBuilder
