/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DBResourceImpl.java,v 1.5 2005/02/16 22:21:31 qiyanli Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.resources.database.internal.impl;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.impl.ResourceImpl;
import org.eclipse.emf.ecore.util.EContentAdapter;
import org.eclipse.hyades.loaders.hierarchy.Constants;
import org.eclipse.hyades.loaders.util.AgentsContext;
import org.eclipse.hyades.loaders.util.HierarchyContext;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.loaders.util.LookupServiceExtensions;
import org.eclipse.hyades.models.hierarchy.TRCAgent;
import org.eclipse.hyades.models.hierarchy.UnresolvedCorrelation;
import org.eclipse.hyades.models.hierarchy.extensions.Query;
import org.eclipse.hyades.models.hierarchy.extensions.QueryResult;
import org.eclipse.hyades.models.hierarchy.util.PerfUtil;
import org.eclipse.hyades.resources.database.internal.DBCollectedExceptions;
import org.eclipse.hyades.resources.database.internal.DBResource;
import org.eclipse.hyades.resources.database.internal.Database;
import org.eclipse.hyades.resources.database.internal.ObjectQuery;
import org.eclipse.hyades.resources.database.internal.QueryFactory;
import org.eclipse.hyades.resources.database.internal.ReferenceQuery;
import org.eclipse.hyades.resources.database.internal.ResourceAddedException;
/**
 * A DBResource uses a database object to load objects from a database and to
 * save objects into a database.
 */
public class DBResourceImpl extends ResourceImpl implements DBResource {
	protected Database database;
	protected Map defaultLoadOptions;
	protected boolean inDatabase;
	protected DBResourceContentAdapter contentAdapter;
	protected HierarchyContext context = null;
	private boolean register;
	public DBResourceImpl() {
		super();
		addAdapter();
	}
	public DBResourceImpl(URI uri) {
		super(uri);
		addAdapter();
	}
	public DBResourceImpl(URI uri, Database database) {
		this(uri);
		this.database = database;
	}
	protected void addAdapter() {
		contentAdapter = new DBResourceContentAdapter();
		eAdapters().add(contentAdapter);
	}
	/**
	 * @see org.eclipse.hyades.resources.database.internal.DBResource#getDatabase()
	 */
	public Database getDatabase() {
		return database;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.resources.database.internal.DBResource#setDatabase(org.eclipse.hyades.resources.database.internal.Database)
	 */
	public void setDatabase(Database database) {
		this.database = database;
	}
	/**
	 * @see org.eclipse.emf.ecore.resource.Resource#save(java.util.Map)
	 */
	public void save(Map options) throws IOException {
		PerfUtil p = new PerfUtil("DBResourceImpl.save() uri="+getURI(),true);
		try {
			doSave(null, options);
			inDatabase = true;
		} catch (IOException e) {
			p.stopAndPrintStatus(e.getLocalizedMessage());
			throw e;
		}
		p.stopAndPrintStatus();
	}
	/**
	 * @see org.eclipse.emf.ecore.resource.impl.ResourceImpl#doSave(java.io.OutputStream,
	 *      java.util.Map)
	 */
	protected void doSave(OutputStream outputStream, Map options) throws IOException {
		try {
			database.add(this);
		} catch (Exception e) {
			// TODO: MS replace this fix for save existing resource
			if (e instanceof ResourceAddedException)
				return;
//			e.fillInStackTrace();
//			e.printStackTrace();
			throw new DBCollectedExceptions(e);
		}
	}
	/**
	 * @see org.eclipse.emf.ecore.resource.Resource#load(java.util.Map)
	 */
	public void load(Map options) throws IOException {
		PerfUtil p = new PerfUtil("DBResourceImpl.load() uri="+getURI(),true);
		try {
			contentAdapter.setIsLoading(true);
			doLoad(null, options);
			inDatabase = true;
			contentAdapter.setIsLoading(false);
		} catch (IOException e) {
			p.stopAndPrintStatus(e.getLocalizedMessage());
			throw e;
		}
		p.stopAndPrintStatus();
	}
	/**
	 * @see org.eclipse.emf.ecore.resource.impl.ResourceImpl#doLoad(java.io.InputStream,
	 *      java.util.Map)
	 */
	protected void doLoad(InputStream inputStream, Map options) throws IOException {

		boolean allObjects = true;
		if (options == null)
			options = defaultLoadOptions;
		if (options != null && options.get(OPTION_TOP_LEVEL) == Boolean.TRUE)
			allObjects = false;
		if (allObjects && defaultLoadOptions != null && defaultLoadOptions.get(OPTION_TOP_LEVEL) == Boolean.TRUE)
			allObjects = false;
		Collection notLoadedClasses = null;
		if (options != null && options.get(OPTION_NOT_LOADED_CLASSES) != null) {
			notLoadedClasses = (Collection) options.get(OPTION_NOT_LOADED_CLASSES);
		} else if (defaultLoadOptions != null) {
			notLoadedClasses = (Collection) defaultLoadOptions.get(OPTION_NOT_LOADED_CLASSES);
		}
		try {
			database.load(this, allObjects, notLoadedClasses);
		} catch (Exception e) {
			e.printStackTrace();
			throw new Resource.IOWrappedException(e);
		}
	}
	public Map getDefaultLoadOptions() {
		if (defaultLoadOptions == null)
			defaultLoadOptions = new HashMap();
		return defaultLoadOptions;
	}
	public EObject getEObject(EClass eClass, EAttribute attribute, Object value, EObject related, EReference reference) throws Exception {
		ObjectQuery query = QueryFactory.INSTANCE.createObjectQuery();
		query.setEClass(eClass);
		query.setURI(uri);
		Map attribValues = query.getAttributeValues();
		if (value != null) {
			if (attribute != null)
				attribValues.put(attribute, value);
			else
				attribValues.put(eClass.getEIDAttribute(), value);
		}
		EObject[] objects = database.getObjects(query);
		if (objects == null)
			return null;
		if (reference != null && related != null) {
			if (!objects[0].eIsSet(reference))
				return testRelatedWithReferenceQuery(objects, related, reference);
			else
				return testRelatedWithReference(objects, related, reference);
		}
		return objects[0];
	}
	protected EObject testRelatedWithReference(EObject[] objects, EObject related, EReference reference) {
		for (int i = 0; i < objects.length; i++) {
			Object referenced = objects[i].eGet(reference);
			if (in(related, referenced))
				return objects[i];
		}
		return null;
	}
	protected boolean in(EObject related, Object referenced) {
		if (related == referenced)
			return true;
		if (referenced instanceof List) {
			List values = (List) referenced;
			for (int i = 0, l = values.size(); i < l; i++)
				if (related == values.get(i))
					return true;
		}
		return false;
	}
	protected EObject testRelatedWithReferenceQuery(EObject[] objects, EObject related, EReference reference) throws Exception {
		for (int i = 0; i < objects.length; i++) {
			ReferenceQuery referenceQuery = QueryFactory.INSTANCE.createReferenceQuery();
			referenceQuery.setEObject(objects[i]);
			referenceQuery.setEReference(reference);
			EObject[] referenced = database.getObjects(referenceQuery);
			if (referenced != null)
				for (int j = 0; j < referenced.length; j++)
					if (referenced[j] == related)
						return objects[i];
		}
		return null;
	}
	public boolean isInDatabase() {
		return inDatabase;
	}
	public EContentAdapter getContentAdapter() {
		return contentAdapter;
	}

	protected void doUnload() {
		// TODO MS improve this for Database Resources - disabled for now
//		EMFUtil.unload(this);
		unloadLookupContext();
	}
	/**
	 *  
	 */
	private void unloadLookupContext() {
		// TODO Auto-generated method stub
		LookupServiceExtensions.getInstance().deregister(context);
		LookupServiceExtensions.getInstance().deregister(null, HierarchyContext.class, getURI().toString());
		context = null;
	}

	/**
	 * @return
	 */
	protected HierarchyContext getContext(EObject object) {
		//        context = (HierarchyContext)
		// LookupServiceExtensions.getInstance().locate(null,
		// HierarchyContext.class, getURI().toString());
		if ((context == null) && object instanceof TRCAgent) {
			context = (HierarchyContext) LookupServiceExtensions.getInstance().locate(null, HierarchyContext.class, getURI().toString());
			if (context == null) {
				context = new HierarchyContext();
			}
			TRCAgent agent = (TRCAgent) object;
			AgentsContext agentsContext = (AgentsContext) LookupServiceExtensions.getInstance().locate(null, AgentsContext.class, LoadersUtils.getLookUpKey(agent.getRuntimeId()));
			if (agentsContext == null) {
				agentsContext = new AgentsContext(agent.getRuntimeId());
				LookupServiceExtensions.getInstance().register(null, agentsContext);
			}
			agentsContext.registerAgent(agent);
			context.setAgent(agent);
			context.setContextURI(getURI().toString());
			LookupServiceExtensions.getInstance().register(null, context);
			return null;
		} else if (object instanceof UnresolvedCorrelation) {
			return null; //this should be added to the global context
		}
		return context;
	}
	/**
	 * @see org.eclipse.emf.ecore.resource.Resource.Internal#attached(org.eclipse.emf.ecore.EObject)
	 */
	public void attached(EObject value) {
		if (register) {
			LookupServiceExtensions.getInstance().register(getContext(value), value);
		}
	}
	/**
	 * @see org.eclipse.emf.ecore.resource.Resource.Internal#detached(org.eclipse.emf.ecore.EObject)
	 */
	public void detached(EObject value) {
		if (register) 
			LookupServiceExtensions.getInstance().deregister(getContext(value), value);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.models.hierarchy.extensions.HyadesResource#delete()
	 */
	public boolean delete() {
		try {
			getDatabase().deleteResource(this.getURI());
			return true;
		} catch (Exception e) {
			// TODO: MS add better handling
			e.printStackTrace();
		}
		return false;
	}
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.models.hierarchy.extensions.HyadesResource#deleteObjects(org.eclipse.emf.common.util.EList)
	 */
	public boolean deleteObjects(EList uris) {
//		throw new UnsupportedOperationException(Constants.NOT_IMPLEMENTED_YET);
		return false;
	}
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.models.hierarchy.extensions.HyadesResource#executeQuery(org.eclipse.hyades.models.hierarchy.extensions.Query)
	 */
	public QueryResult executeQuery(Query query) {
		throw new UnsupportedOperationException(Constants.NOT_IMPLEMENTED_YET);
	}
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.models.hierarchy.extensions.HyadesResource#executeQuery(java.lang.String)
	 */
	public QueryResult executeQuery(String queryName) {
		throw new UnsupportedOperationException(Constants.NOT_IMPLEMENTED_YET);
	}
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.models.hierarchy.extensions.HyadesResource#storeQuery(org.eclipse.hyades.models.hierarchy.extensions.Query)
	 */
	public boolean storeQuery(Query query) {
		throw new UnsupportedOperationException(Constants.NOT_IMPLEMENTED_YET);
	}
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.models.hierarchy.extensions.HyadesResource#unloadObjects(org.eclipse.emf.common.util.EList)
	 */
	public boolean unloadObjects(EList uris) {
//		throw new UnsupportedOperationException(Constants.NOT_IMPLEMENTED_YET);
		return false;
	}
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.models.hierarchy.extensions.HyadesResource#validateQuery(org.eclipse.hyades.models.hierarchy.extensions.Query)
	 */
	public boolean validateQuery(Query query) {
		throw new UnsupportedOperationException(Constants.NOT_IMPLEMENTED_YET);
	}
} // DBResourceImpl
