/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: GetObjectQuery.java,v 1.4 2005/02/16 22:21:31 qiyanli Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.resources.database.internal.impl;

import java.sql.ResultSet;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.resources.database.internal.DBMap;
import org.eclipse.hyades.resources.database.internal.ObjectQuery;
import org.eclipse.hyades.resources.database.internal.dbmodel.Column;
import org.eclipse.hyades.resources.database.internal.dbmodel.Table;
import org.eclipse.hyades.resources.database.internal.extensions.DatabaseType;
import org.eclipse.hyades.resources.database.internal.extensions.GetURICommand;
import org.eclipse.hyades.resources.database.internal.extensions.JDBCHelper;
/**
 * This class gets objects based on an object query.
 */
public class GetObjectQuery extends GetByQueryCommand {
	protected ObjectQuery query;

	public GetObjectQuery(JDBCHelper helper, DBMap map, DatabaseType type, ObjectQuery query, WeakObjectCache cache) {
		super(helper, map, type, cache);
		this.query = query;
	}

	/**
	 * Return an array of EObjects, or null.
	 */
	public Object execute() throws Exception {
		if (query.getEClass() == null)
			return null;

		List ids = getIdsForMatchingObjects();
		EObject[] objects = getObjects(ids);

		if (query.getURI() == null)
			return objects;
		else
			return getObjectsInResource(objects, ids);
	}

	protected EObject[] getObjectsInResource(EObject[] objects, List ids) throws Exception {
		List classesAndIds = new ArrayList();
		classesAndIds.add(query.getEClass());
		classesAndIds.add(ids);
		GetURICommand getURIs = new GetURICommand(helper, dbMap, type, classesAndIds);
		String[] uris = (String[]) getURIs.execute();
		List objectsInResource = new ArrayList();

		for (int i = 0; i < uris.length; i++)
			if (query.getURI().toString().equals(uris[i]))
				objectsInResource.add(objects[i]);

		return toArray(objectsInResource);
	}

	/**
	 * Get the ids of all instances of the given EClass.
	 */
	protected List getIdsForAllObjects() throws Exception {
		List ids = new ArrayList();
		String statement = createQueryForAllObjects();
		Statement jdbcStatement = helper.createStatement();
		ResultSet result = helper.executeQuery(jdbcStatement, statement);

		while (result.next()) {
			int id = result.getInt(1);
			ids.add(new Integer(id));
		}

		result.close();
		jdbcStatement.close();

		return ids;
	}

	/**
	 * Returns a SQL statement that gets the id column for all rows in a class
	 * table.
	 */
	protected String createQueryForAllObjects() {
		DBMap.ClassData data = (DBMap.ClassData) dbMap.getDBRepresentation(query.getEClass());
		Table table = data.getTable();
		RDBHelper rdbHelper = new RDBHelper();
		Column id = rdbHelper.getPrimaryKey(table);

		StringBuffer s = new StringBuffer();
		s.append("SELECT ");
		s.append(addQuotes(id.getName()));
		s.append(" FROM ");
		s.append(addQuotes(table.getName()));

		return s.toString();
	}

	/**
	 * Get the ids of instances whose attribute values match the ones in the
	 * query.
	 */
	protected List getIdsForMatchingObjects() throws Exception {
		Map values = query.getAttributeValues();

		if (values.isEmpty())
			return getIdsForAllObjects();

		DBCommand getIds = factory.createGetIdsOfMatches(helper, dbMap, type, query.getEClass(), values);
		return (List) getIds.execute();
	}

	/**
	 * Create get commands to retrieve the objects from the given class with the
	 * given database ids.
	 */
	protected EObject[] getObjects(List ids) throws Exception {
		List objects = getObjects(query.getEClass(), ids, query.isSetReferences(), query.getNotLoadedClasses());
		return toArray(objects);
	}
} // GetObjectQuery
