/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: GetReferenceQuery.java,v 1.4 2005/02/16 22:21:31 qiyanli Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.resources.database.internal.impl;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.hyades.models.hierarchy.util.PerfUtil;
import org.eclipse.hyades.resources.database.internal.DBMap;
import org.eclipse.hyades.resources.database.internal.ReferenceQuery;
import org.eclipse.hyades.resources.database.internal.extensions.DBCommandFactory;
import org.eclipse.hyades.resources.database.internal.extensions.DatabaseType;
import org.eclipse.hyades.resources.database.internal.extensions.JDBCHelper;

/**
 * This class gets objects based on a reference query.
 */
public class GetReferenceQuery extends GetByQueryCommand {
	protected ReferenceQuery query;
	protected Integer id;

	public GetReferenceQuery(JDBCHelper helper, DBMap map, DatabaseType type, ReferenceQuery query, Integer id, WeakObjectCache cache) {
		super(helper, map, type, cache);
		this.query = query;
		this.id = id;
	}

	/**
	 * Return an array of EObjects, or null.
	 */
	public Object execute() throws Exception {
		if (query.getEObject() == null || query.getEReference() == null)
			return null;

		List referenced = getReferencedObjects(id);
		return toArray(referenced);
	}

	/**
	 * Return all of the referenced objects.
	 */
	protected List getReferencedObjects(Integer id) throws Exception {
		List ids = getIdsOfReferencedObjects(id);
		return getObjects(ids);
	}

	/**
	 * Returns the ids of each referenced object.
	 */
	protected List getIdsOfReferencedObjects(Integer id) throws Exception {
		int[] ids = new int[1];
		ids[0] = id.intValue();
		PerfUtil p = new PerfUtil("GetReferenceQuery.getIdsOfReferencedObjects()",true);
		DBCommand getIds = DBCommandFactory.INSTANCE.createGetIdsOfReferencedObjects(helper, dbMap, type, ids, query.getEReference(), true);
		int[][] referencedIds = (int[][]) getIds.execute();
		List idsList = new ArrayList();

		if (referencedIds[0] == null)
			return idsList;

		for (int i = 0; i < referencedIds[0].length; i++) {
			idsList.add(new Integer(referencedIds[0][i]));
		}
		p.stopAndPrintStatus("id="+id+", size="+idsList.size());

		return idsList;
	}

	/**
	 * Prune the list of ids based on the upper and lower bounds, then invoke
	 * the superclass to obtain the actual objects.
	 */
	protected List getObjects(List ids) throws Exception {
		return getObjects(query.getEReference().getEReferenceType(), prune(ids), query.isSetReferences(), query.getNotLoadedClasses());
	}

	/**
	 * Returns a list based on the lower and upper bounds.
	 */
	protected List prune(List ids) throws Exception {
		int lower = query.getLower();
		int upper = query.getUpper();
		int lastIndex = ids.size();

		if (upper == -1 || upper >= lastIndex)
			upper = lastIndex;
		else
			++upper;

		if (lower >= lastIndex)
			return new ArrayList();
		else
			return ids.subList(lower, upper);
	}
} // GetReferenceQuery
