/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SimpleSearchQueryStatement.java,v 1.13 2005/04/07 20:10:56 slavescu Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.resources.database.internal.impl;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EModelElement;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.hyades.models.hierarchy.extensions.ArithmeticExpression;
import org.eclipse.hyades.models.hierarchy.extensions.BinaryExpression;
import org.eclipse.hyades.models.hierarchy.extensions.LogicalExpression;
import org.eclipse.hyades.models.hierarchy.extensions.LogicalOperators;
import org.eclipse.hyades.models.hierarchy.extensions.NumericFunction;
import org.eclipse.hyades.models.hierarchy.extensions.Operand;
import org.eclipse.hyades.models.hierarchy.extensions.OrderByElement;
import org.eclipse.hyades.models.hierarchy.extensions.RelationalOperators;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleOperand;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleSearchQuery;
import org.eclipse.hyades.models.hierarchy.extensions.WhereExpression;
import org.eclipse.hyades.models.hierarchy.util.PerfUtil;
import org.eclipse.hyades.models.hierarchy.util.internal.QueryUtils;
import org.eclipse.hyades.resources.database.internal.DBCollectedExceptions;
import org.eclipse.hyades.resources.database.internal.DBMap;
import org.eclipse.hyades.resources.database.internal.extensions.DBCommandFactory;
import org.eclipse.hyades.resources.database.internal.extensions.JDBCHelper;
/**
 * @author slavescu
 */
public class SimpleSearchQueryStatement extends QueryStatement {
	protected JDBCHelper helper;
	protected SimpleSearchQuery query;
	protected Set processedAlready = new HashSet();
	protected boolean first = true;
	protected boolean all = false;
	protected Set fromSet = new HashSet();
	public class SimpleSearchQueryParser {
		protected Map namesIndex;
		protected StringBuffer result;
		public SimpleSearchQueryParser() {
		}
		protected WhereExpression getExpression(String string) {
			return (WhereExpression) namesIndex.get(string);
		}
		protected Operand getOperand(String string) {
			return (Operand) namesIndex.get(string);
		}
		protected StringBuffer internalParse() {
			//			this.node = node;
			if (query.getWhereExpression() == null)
				return getInternalBuffer();
			parseWhereExpression(query.getWhereExpression());
			return getInternalBuffer();
		}
		protected void parseArithmeticExpression(ArithmeticExpression expression) {
			if (expression.getName() != null && expression.getName().startsWith(QueryUtils.DOLLAR_SIGN)) {
				parseOperand(getOperand(expression.getName().substring(1)));
				return;
			}
			for (Iterator iter = expression.getArguments().iterator(); iter.hasNext();) {
				Operand element = (Operand) iter.next();
				parseOperand(element);
			}
		}
		protected void parseBinaryExpression(BinaryExpression expression) {
			if (expression.getName() != null && expression.getName().startsWith(QueryUtils.DOLLAR_SIGN)) {
				parseWhereExpression(getExpression(expression.getName().substring(1)));
				return;
			}
			Operand operand = expression.getLeftOperand();
			parseOperand(operand);
			getInternalBuffer().append(" " + RelationalOperators.getString(expression.getOperator().getValue()) + " ");
			if (expression.getOperator() == RelationalOperators.BETWEEN_LITERAL) {
				// use only the first two right operands, more/less then two are not supported here
				getInternalBuffer().append(" ");
				operand = (Operand) expression.getRightOperands().get(0);
				parseOperand(operand);
				getInternalBuffer().append(" AND ");
				operand = (Operand) expression.getRightOperands().get(1);
				parseOperand(operand);
			} else if (expression.getOperator() == RelationalOperators.IN_LITERAL) {
				boolean first = true;
				getInternalBuffer().append("(");
				for (Iterator iter = expression.getRightOperands().iterator(); iter.hasNext();) {
					if (first)
						first = false;
					else
						getInternalBuffer().append(",");
					operand = (Operand) iter.next();
					parseOperand(operand);
				}
				getInternalBuffer().append(") ");
			} else {
				// use only the first right operand, multiple operands not supported here
				operand = (Operand) expression.getRightOperands().get(0);
				parseOperand(operand);
			}
		}
		protected void parseLogicalExpression(LogicalExpression expression) {
			if (expression.getName() != null && expression.getName().startsWith(QueryUtils.DOLLAR_SIGN)) {
				parseWhereExpression(getExpression(expression.getName().substring(1)));
				return;
			}
			if (expression.getOperator() == LogicalOperators.NOT_LITERAL) {
				if (!expression.getArguments().isEmpty()) {
					getInternalBuffer().append(expression.getOperator().getName() + "(");
					//use only the first argument, ignore the rest
					parseWhereExpression((WhereExpression) expression.getArguments().get(0));
					getInternalBuffer().append(") ");
				}
			} else {
				boolean first = true;
				for (Iterator iter = expression.getArguments().iterator(); iter.hasNext();) {
					WhereExpression ex = (WhereExpression) iter.next();
					if (first) {
						getInternalBuffer().append("(");
						first = false;
					} else {
						getInternalBuffer().append(" " + expression.getOperator().getName() + " ");
					}
					parseWhereExpression(ex);
				}
				if (!first)
					getInternalBuffer().append(") ");
			}
		}
		protected void parseNumericFunction(NumericFunction function) {
			if (function.getName() != null && function.getName().startsWith(QueryUtils.DOLLAR_SIGN)) {
				parseOperand(getOperand(function.getName().substring(1)));
				return;
			}
			for (Iterator iter = function.getArguments().iterator(); iter.hasNext();) {
				Operand element = (Operand) iter.next();
				parseOperand(element);
			}
		}
		protected void parseOperand(Operand operand) {
			if (operand instanceof ArithmeticExpression) {
				parseArithmeticExpression((ArithmeticExpression) operand);//.newNumeric();
			} else if (operand instanceof NumericFunction) {
				parseNumericFunction((NumericFunction) operand);//.newNumeric();
			} else
				parseSimpleOperand((SimpleOperand) operand);
		}
		protected void parseSimpleOperand(SimpleOperand simpleOperand) {
			if (simpleOperand.getName() != null && simpleOperand.getName().startsWith(QueryUtils.DOLLAR_SIGN)) {
				parseOperand(getOperand(simpleOperand.getName().substring(1)));
			}
			if (simpleOperand.getType() != null) {
				fromSet.add(addOperandName(simpleOperand));
			} else if (simpleOperand.getFeature() != null) {
				fromSet.add(addOperandName(simpleOperand));
			} else {
				Object v = simpleOperand.getValue();
				if (v instanceof String) {
					getInternalBuffer().append("'");
					if (simpleOperand.eContainer() instanceof BinaryExpression && ((BinaryExpression) simpleOperand.eContainer()).getOperator() == RelationalOperators.LIKE_LITERAL)
						getInternalBuffer().append(prepareLikeValue((String) v));
					else
						getInternalBuffer().append(v);
					getInternalBuffer().append("'");
				} else if (v instanceof Boolean) {
					getInternalBuffer().append("'");
					getInternalBuffer().append(((Boolean) v).booleanValue() ? '1' : '0');
					getInternalBuffer().append("'");
				} else
					getInternalBuffer().append(v);
			}
		}
		/**
		 * @param string
		 * @return
		 */
		protected StringBuffer prepareLikeValue(String value) {
			StringBuffer r = new StringBuffer();
			char previous = ' ';
			for (int i = 0; i < value.length(); i++) {
				if (value.charAt(i) == '*' || value.charAt(i) == '%') {
					if (previous != '%') {
						previous = '%';
						r.append('%');
					}
				} else if (value.charAt(i) == '?') {
					previous = '_';
					r.append('_');
				} else
				{
					previous = value.charAt(i);
					r.append(previous);
				}
			}
			return r;
		}
		protected void parseWhereExpression(WhereExpression expression) {
			if (expression instanceof LogicalExpression)
				parseLogicalExpression((LogicalExpression) expression);
			else
				parseBinaryExpression((BinaryExpression) expression);
		}
		public StringBuffer getResult() {
			return result;
		}
		public StringBuffer parse() {
			return internalParse();
		}
//		protected void processRightOperand(BinaryExpression sbe) {
//			SimpleOperand ro = getRightOperand(sbe);
//			if (ro.getFeature() != null) {
//				addRightOperand(ro.getFeature());
//			} else if (ro.getType() != null) {
//				addRightOperand(ro.getType());
//			} else if (getRightOperandValue(sbe) instanceof String) {
//				getInternalBuffer().append("'");
//				getInternalBuffer().append(ro.toString());
//				getInternalBuffer().append("'");
//			} else if (getRightOperandValue(sbe) instanceof Boolean) {
//				getInternalBuffer().append("'");
//				getInternalBuffer().append(((Boolean) getRightOperandValue(sbe)).booleanValue() ? '1' : '0');
//				getInternalBuffer().append("'");
//			} else
//				getInternalBuffer().append(getRightOperandValue(sbe).toString());
//		}
		protected String addOperandName(SimpleOperand oe) {
			String tableName = "";
			String columnName = "";
			if (oe.getFeature() != null) {
				processedAlready.add(oe.getFeature());
				if(oe.getFeature() instanceof EAttribute)
				{
					DBMap.AttributeData data = (DBMap.AttributeData) dbMap.getDBRepresentation(oe.getFeature());
					tableName = data.getValueColumn().getTable().getName();
					columnName = data.getValueColumn().getName();
				}
				else
				{
					DBMap.ReferenceData data = (DBMap.ReferenceData) dbMap.getDBRepresentation(oe.getFeature());
					tableName = data.getTable().getName();
					columnName = data.getTargetColumn().getName();
				}
			} else if (oe.getType() != null) {
				processedAlready.add(oe.getType());
				DBMap.ClassData data = (DBMap.ClassData) dbMap.getDBRepresentation(oe.getType());
				tableName = data.getTable().getName();
				columnName = dbMap.getClassMetadata(oe.getType()).getPrimaryKey().getName();
			}
			if (query.isCount()) {// || oe.isCount() ) {
				getInternalBuffer().append("COUNT(");
			}
			getInternalBuffer().append(addQuotes(tableName));
			getInternalBuffer().append('.');
			getInternalBuffer().append(addQuotes(columnName));
			if (query.isCount()) {// || oe.isCount()) {
				getInternalBuffer().append(")");
			}
			return tableName;
		}
//		protected void addRightOperand(EModelElement classifier) {
//			String tableName = "";
//			String columnName = "";
//			if (classifier instanceof EStructuralFeature) {
//				if (classifier instanceof EAttribute) {
//					DBMap.AttributeData data = (DBMap.AttributeData) dbMap.getDBRepresentation(classifier);
//					tableName = data.getValueColumn().getTable().getName();
//					columnName = data.getValueColumn().getName();
//				} else {
//					DBMap.ReferenceData data = (DBMap.ReferenceData) dbMap.getDBRepresentation(classifier);
//					tableName = data.getTargetColumn().getTable().getName();
//					columnName = data.getTargetColumn().getName();
//				}
//			} else if (classifier instanceof EClass) {
//				DBMap.ClassData data = (DBMap.ClassData) dbMap.getDBRepresentation(classifier);
//				tableName = data.getTable().getName();
//				columnName = dbMap.getClassMetadata((EClass) classifier).getPrimaryKey().getName();
//			}
//			if (tableName.length() > 0 && columnName.length() > 0) {
//				getInternalBuffer().append(addQuotes(tableName));
//				getInternalBuffer().append('.');
//				getInternalBuffer().append(addQuotes(columnName));
//			}
//		}
		protected StringBuffer getInternalBuffer() {
			if (result == null)
				result = new StringBuffer();
			return result;
		}
		protected EObject getOperandType(SimpleOperand oe) {
			if (oe.getFeature() != null) {
				return oe.getFeature();
			} else if (oe.getType() != null) {
				return oe.getType();
			}
			return null;
		}
		public void setResultBuffer(StringBuffer result) {
			this.result = result;
		}
	}
	/**
	 * @param helper
	 * @param dbMap
	 * @param query
	 */
	public SimpleSearchQueryStatement(JDBCHelper helper, DBMap dbMap, SimpleSearchQuery query) {
		super(helper.getDatabaseType(), dbMap, null);
		this.helper = helper;
		this.query = query;
	}
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.resources.database.internal.impl.QueryStatement#createOrderClause()
	 */
	protected void createOrderClause() {
		first = true;
		processedAlready.clear();
		processOrderByExpressionsInOrderByClause();
		processedAlready.clear();
	}
	/**
	 *  
	 */
	protected void processOrderByExpressionsInOrderByClause() {
		for (Iterator iter = query.getOrderByExpresions().iterator(); iter.hasNext();) {
			OrderByElement oe = (OrderByElement) iter.next();
			SimpleOperand op = (SimpleOperand) oe.getOperand();
			if (op == null || op.getFeature() == null || oe.getOperator() == null)
				continue;
			if (processedAlready.contains(op.getFeature()))
				continue;
			processedAlready.add(op.getFeature());
			if (first) {
				statement.append(" ORDER BY ");
				first = false;
			} else
				statement.append(", ");
			DBMap.AttributeData data = (DBMap.AttributeData) dbMap.getDBRepresentation(op.getFeature());
			statement.append(addQuotes(data.getValueColumn().getName()));
			statement.append(" ");
			statement.append(oe.getOperator().getName());
		}
	}
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.resources.database.internal.impl.QueryStatement#createSelectFrom()
	 */
	protected void createSelectFrom() {
		first = true;
		all = false;
		fromSet.clear();
		processedAlready.clear();
		processOutputElementsInSelectFrom();
		processOrderByInSelectFrom();
		processWhereClauseInSelectFrom();
		//	  	if (dbType.isOrderRequired()) {
		//	  	  statement.append(", ");
		//	  	  statement.append(addQuotes(order));
		//	    }
		//	  
		processedAlready.clear();
		first = true;
		processFromSet();
		processedAlready.clear();
	}
	protected void processWhereClauseInSelectFrom() {
		String tableName;
		for (Iterator iter = QueryUtils.getClassPredicatesNoSubClassesIndex(query).keySet().iterator(); iter.hasNext();) {
			EClass type = (EClass) iter.next();
			tableName = null;
			DBMap.ClassData data = (DBMap.ClassData) dbMap.getDBRepresentation(type);
			tableName = data.getTable().getName();
			if (tableName != null)
				fromSet.add(tableName);
		}
	}
//	/**
//	 *  
//	 */
//	protected void processWhereClauseInSelectFromOld() {
//		String tableName;
//		for (Iterator iter = LogQueryBuilder.getANDArguments(query).iterator(); iter.hasNext();) {
//			BinaryExpression sbe = (BinaryExpression) iter.next();
//			SimpleOperand oe = (SimpleOperand) sbe.getLeftOperand();
//			tableName = null;
//			if (oe.getFeature() != null) {
//				DBMap.AttributeData data = (DBMap.AttributeData) dbMap.getDBRepresentation(oe.getFeature());
//				tableName = data.getValueColumn().getTable().getName();
//			} else if (oe.getType() != null) {
//				DBMap.ClassData data = (DBMap.ClassData) dbMap.getDBRepresentation(oe.getType());
//				tableName = data.getTable().getName();
//			}
//			if (tableName != null)
//				fromSet.add(tableName);
//			tableName = null;
//			if (getRightOperand(sbe).getFeature() != null) {
//				element = getRightOperand(sbe).getFeature();
//				if (element instanceof EAttribute) {
//					DBMap.AttributeData data = (DBMap.AttributeData) dbMap.getDBRepresentation(element);
//					tableName = data.getValueColumn().getTable().getName();
//				} else {
//					DBMap.ReferenceData data = (DBMap.ReferenceData) dbMap.getDBRepresentation(element);
//					tableName = data.getTargetColumn().getTable().getName();
//				}
//			} else if (getRightOperand(sbe).getType() != null) {
//				element = getRightOperand(sbe).getType();
//				DBMap.ClassData data = (DBMap.ClassData) dbMap.getDBRepresentation(element);
//				tableName = data.getTable().getName();
//			}
//			if (tableName != null)
//				fromSet.add(tableName);
//		}
//	}
	/**
	 *  
	 */
	protected void processFromSet() {
		for (Iterator iter = fromSet.iterator(); iter.hasNext();) {
			String tableName = (String) iter.next();
			if (processedAlready.contains(tableName))
				continue;
			if (first) {
				statement.append(" FROM ");
				first = false;
			} else
				statement.append(", ");
			statement.append(addQuotes(tableName));
		}
	}
	protected void processOutputElementsInSelectFrom() {
		for (Iterator iter = query.getOutputElements().iterator(); iter.hasNext();) {
			SimpleOperand oe = (SimpleOperand) iter.next();
			if (processedAlready.contains(getOperandType(oe))) {
				continue;
			}
			if (first) {
				statement.append("SELECT ");
				if (query.isDistinct()) {
					statement.append("DISTINCT ");
				}
				first = false;
			} else
				statement.append(", ");
			fromSet.add(addOperandName(oe));
		}
		if (first && query.isCount()) {
			statement.append("SELECT ");
			statement.append("COUNT(*) ");
			all = true;
			first = false;
		} else if (first) {
			statement.append("SELECT ");
			if (query.isDistinct()) {
				statement.append("DISTINCT ");
			}
			statement.append("* ");
			all = true;
			first = false;
		}
	}
	protected void processOrderByInSelectFrom() {
		if (!all && query.isDistinct()) {
			for (Iterator iter = query.getOrderByExpresions().iterator(); iter.hasNext();) {
				OrderByElement oe = (OrderByElement) iter.next();
				if (processedAlready.contains(getOperandType((SimpleOperand) oe.getOperand()))) {
					continue;
				}
				if (first) {
					statement.append("SELECT ");
					if (query.isDistinct()) {
						statement.append("DISTINCT ");
					}
					first = false;
				} else
					statement.append(", ");
				fromSet.add(addOperandName((SimpleOperand) oe.getOperand()));
			}
		}
	}
	/**
	 * @param columnName
	 * @param fromSet
	 * @param oe
	 * @return
	 */
	protected String addOperandName(SimpleOperand oe) {
		String tableName = "";
		String columnName = "";
		if (oe.getFeature() != null) {
			processedAlready.add(oe.getFeature());
			if(oe.getFeature() instanceof EAttribute)
			{
				DBMap.AttributeData data = (DBMap.AttributeData) dbMap.getDBRepresentation(oe.getFeature());
				tableName = data.getValueColumn().getTable().getName();
				columnName = data.getValueColumn().getName();
			}
			else
			{
				DBMap.ReferenceData data = (DBMap.ReferenceData) dbMap.getDBRepresentation(oe.getFeature());
				tableName = data.getTable().getName();
				columnName = data.getTargetColumn().getName();
			}
		} else if (oe.getType() != null) {
			processedAlready.add(oe.getType());
			DBMap.ClassData data = (DBMap.ClassData) dbMap.getDBRepresentation(oe.getType());
			tableName = data.getTable().getName();
			columnName = dbMap.getClassMetadata(oe.getType()).getPrimaryKey().getName();
		}
		if (query.isCount()) {// || oe.isCount() ) {
			statement.append("COUNT(");
		}
		statement.append(addQuotes(tableName));
		statement.append('.');
		statement.append(addQuotes(columnName));
		if (query.isCount()) {// || oe.isCount()) {
			statement.append(")");
		}
		return tableName;
	}
	protected EObject getOperandType(SimpleOperand oe) {
		if (oe.getFeature() != null) {
			return oe.getFeature();
		} else if (oe.getType() != null) {
			return oe.getType();
		}
		return null;
	}
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.resources.database.internal.impl.QueryStatement#createWhereClause(int)
	 */
	protected void createWhereClause(int begin) {
		first = true;
		processedAlready.clear();
		processWhereExpressions();
		appendWhereClauseForSources();
		processedAlready.clear();
	}
//	protected void processBinaryExpressionsOld() {
//		for (Iterator iter = LogQueryBuilder.getANDArguments(query).iterator(); iter.hasNext();) {
//			BinaryExpression sbe = (BinaryExpression) iter.next();
//			if (processedAlready.contains(sbe))
//				continue;
//			if (first) {
//				statement.append(" WHERE ");
//				first = false;
//			} else
//				statement.append(" AND ");
//			addOperandName((SimpleOperand) sbe.getLeftOperand());
//			statement.append(" ");
//			statement.append(RelationalOperators.getString(sbe.getOperator().getValue()));
//			statement.append(" ");
//			processRightOperand(sbe);
//		}
//	}
	protected void processWhereExpressions() {
		SimpleSearchQueryParser parser = new SimpleSearchQueryParser();
		StringBuffer whereExprString = parser.parse();
		if (whereExprString.length() > 0) {
			if (first) {
				statement.append(" WHERE ");
				first = false;
			}
			statement.append(whereExprString);
		}
	}
	protected void processRightOperand(BinaryExpression sbe) {
		SimpleOperand ro = getRightOperand(sbe);
		if (ro.getFeature() != null) {
			addRightOperand(ro.getFeature());
		} else if (ro.getType() != null) {
			addRightOperand(ro.getType());
		} else if (getRightOperandValue(sbe) instanceof String) {
			statement.append("'");
			statement.append(ro.toString());
			statement.append("'");
		} else if (getRightOperandValue(sbe) instanceof Boolean) {
			statement.append("'");
			statement.append(((Boolean) getRightOperandValue(sbe)).booleanValue() ? '1' : '0');
			statement.append("'");
		} else
			statement.append(getRightOperandValue(sbe).toString());
	}
	/**
	 * @param sbe
	 * @return
	 */
	protected Object getRightOperandValue(BinaryExpression sbe) {
		if (!sbe.getRightOperands().isEmpty() && sbe.getRightOperands().get(0) != null) {
			return ((SimpleOperand) sbe.getRightOperands().get(0)).getValue();
		}
		return null;
	}
	protected SimpleOperand getRightOperand(BinaryExpression sbe) {
		if (!sbe.getRightOperands().isEmpty() && sbe.getRightOperands().get(0) != null) {
			return ((SimpleOperand) sbe.getRightOperands().get(0));
		}
		return null;
	}
	protected void addRightOperand(EModelElement classifier) {
		String tableName = "";
		String columnName = "";
		if (classifier instanceof EStructuralFeature) {
			if (classifier instanceof EAttribute) {
				DBMap.AttributeData data = (DBMap.AttributeData) dbMap.getDBRepresentation(classifier);
				tableName = data.getValueColumn().getTable().getName();
				columnName = data.getValueColumn().getName();
			} else {
				DBMap.ReferenceData data = (DBMap.ReferenceData) dbMap.getDBRepresentation(classifier);
				tableName = data.getTargetColumn().getTable().getName();
				columnName = data.getTargetColumn().getName();
			}
		} else if (classifier instanceof EClass) {
			DBMap.ClassData data = (DBMap.ClassData) dbMap.getDBRepresentation(classifier);
			tableName = data.getTable().getName();
			columnName = dbMap.getClassMetadata((EClass) classifier).getPrimaryKey().getName();
		}
		if (tableName.length() > 0 && columnName.length() > 0) {
			statement.append(addQuotes(tableName));
			statement.append('.');
			statement.append(addQuotes(columnName));
		}
	}
	protected void appendWhereClauseForSources() {
		boolean firstSource = true;
		boolean andBracket = false;
		for (Iterator iter = query.getSources().iterator(); iter.hasNext();) {
			String source = (String) iter.next();
			if (first) {
				statement.append(" WHERE (");
				first = false;
				firstSource = false;
			} else if (firstSource) {
				statement.append(" AND (");
				if (query.getSources().size() > 1) {
					statement.append(" (");
					andBracket = true;
				}
				firstSource = false;
			} else
				statement.append(") OR (");
			String containerPath = getObjectCompressedURIFragment(source);
			addContainedIn(containerPath, isRootObject(source));
		}
		if (query.getSources().size() > 0)
			statement.append(") ");
		if (andBracket)
			statement.append(") ");
	}
	/**
	 * @param source
	 * @return
	 */
	protected boolean isRootObject(String source) {
		int i = source.indexOf('#');
		if (i != -1) {
			if (source.indexOf(i + 1, '/') == -1) {
				return true;
			}
		}
		return false;
	}
	/**
	 * @param containerPath
	 */
	protected void addContainedIn(String containerPath, boolean rootObject) {
		boolean first = true;
		for (Iterator iter = query.getOutputElements().iterator(); iter.hasNext();) {
			SimpleOperand output = (SimpleOperand) iter.next();
			EClass eClass = null;
			if (output.getFeature() != null)
				eClass = output.getFeature().getEContainingClass();
			else
				eClass = output.getType();
			String tableName = dbMap.getClassMetadata(eClass).getTable().getName();
			if (first) {
				first = false;
			} else
				statement.append(" AND ");
			statement.append(addQuotes(tableName));
			statement.append(".");
			// TODO MS - fix this for any root object, not only for TRCAgent
			if (rootObject && tableName.equals("TRCAgent")) {
				statement.append(addQuotes(dbMap.getClassMetadata(eClass).getPrimaryKey().getName()));
				statement.append(" = ");
				statement.append(containerPath.substring(0, containerPath.indexOf('/')));
			} else {
				statement.append(addQuotes("p_p"));
				statement.append(" LIKE ");
				statement.append("'" + containerPath + "%'");
			}
		}
	}
	/**
	 * @param source
	 * @return
	 */
	protected String getObjectCompressedURIFragment(String source) {
		PerfUtil p = new PerfUtil("SimpleSearchQueryStatement.getObjectCompressedURIFragment()", true);
		DBCommand command = DBCommandFactory.INSTANCE.createGetCommpressedPathByURICommand(helper, dbMap, source);
		String s = "";
		try {
			s = (String) command.execute();
		} catch (Exception e) {
			p.stopAndPrintStatus(e.getLocalizedMessage());
			throw new DBCollectedExceptions(e);
		}
		p.stopAndPrintStatus(s);
		return s;
	}
}