/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.statistical.ui.editor.internal;

import java.util.Vector;

import org.eclipse.hyades.statistical.ui.EditorPlugin;
import org.eclipse.hyades.statistical.ui.variableloader.internal.SDModifiableVariableRepresentation;
import org.eclipse.hyades.statistical.ui.widgets.table.internal.InvalidTableValueException;
import org.eclipse.hyades.statistical.ui.widgets.table.internal.TableAdapter;
import org.eclipse.hyades.statistical.ui.widgets.zoomslider.internal.ZoomControlBar;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;

public class ControlTable extends TableAdapter implements SelectionListener, FocusListener {

	public static final int VAL_CONTROL = 0;
	public static final int VAL_NAME = 1;
	public static final int VAL_DESC = 2;
	public static final int VAL_IMAGE = 3;
	public static final int VAL_SLIDER = 4;
	public static final int VAL_MODIFY = 5;

	public static final int COL_NAME = 0;
	public static final int COL_DESC = 1;
	public static final int COL_IMAGE = 2;
	public static final int COL_SLIDER = 3;
	public static final int COL_MODIFY = 4;
	
	private Vector datas = new Vector();
	private Vector listeners = new Vector();
	
	private Integer[] sliderChoices = new Integer[] {
			new Integer(1),
		};
	private String[] sliderChoicesString = new String[] {
			"1",
		};

	public void addListener(ControlTableListener newListener) {
		listeners.add(newListener);
	}

	public void notifyListenersEdit(ZoomControlBar b) {
		if (b != null) {
			for (int v = 0; v < listeners.size(); v++)
			{
				ControlTableListener TCL = (ControlTableListener) (listeners.elementAt(v));
				TCL.barEdited(b);
			}
		}
	}
	public void notifyListenersSelect(SDModifiableVariableRepresentation g) {
		for (int v = 0; v < listeners.size(); v++)
		{
			ControlTableListener TCL = (ControlTableListener) (listeners.elementAt(v));
			TCL.barSelected(g);
		}
	}
	
	public ControlTable(Composite parent) {
		super(parent);

		String[] names = new String[] {
			EditorPlugin.getString("TABLE_COLUMN_NAME"),
			EditorPlugin.getString("TABLE_COLUMN_DESCRIPTION"),
			EditorPlugin.getString("TABLE_COLUMN_IMAGE"),
			EditorPlugin.getString("TABLE_COLUMN_SLIDER"),
			EditorPlugin.getString("TABLE_COLUMN_VALUE"),
		};

		int[] types = new int[] {
			LABEL,
			LABEL,
			LABEL,
			CHOICE,
			TEXT	
		};
		
		setColumnNames(names);			
		setColumnTypes(types);

		autoSizeOnContents = new boolean[]{true, false, false, true, true};
		columnExpands = new boolean[]{true, false, false, true, true};
				
		configure();

		getViewer().getTable().addFocusListener(this);
		getViewer().getTable().addSelectionListener(this);
	}	

	public void ensureControlsValid() {

		for (int i = 0; i < datas.size(); i++) {
			BarElement element = (BarElement)datas.get(i);

			if (element.slider.intValue() >= sliderChoices.length) {
				element.slider = new Integer(0);	
			}
		}
			
		configure();
	}

	public void setSliderChoices(Integer[] values, String[] names) {
		sliderChoices = values;
		sliderChoicesString = names;

		ensureControlsValid();
		
		redraw();
	}
	
	public Object[] getElements(Object inputObject) {
		return datas.toArray();
	}
	
	public void setSelectedIndex(int i) {
//		getViewer().setSelection(new StructuredSelection(datas.get(i)));
		getViewer().getTable().setFocus();
		getViewer().getTable().setSelection(i);
		getViewer().getTable().showSelection();
		getViewer().getTable().redraw();
	}
	
	public void setControlValues(Object[][] values)
	{
		for (int i = 0; i < datas.size(); i++) {
			try {
				BarElement ge = (BarElement)datas.get(i);
//				if (ge != null) ge.dispose();	
			} catch (Throwable e) {}
		}

		datas.clear();
		for (int i = 0; i < values.length; i++)
			datas.add(new BarElement((ZoomControlBar)values[i][0], (String)values[i][1], (String)values[i][2], (Image)values[i][3], (Integer)values[i][4], (SDModifiableVariableRepresentation)values[i][5]));

		configure();
	}

	public Object[][] getControlValues()
	{
		Object[][] result = new Object[datas.size()][6];
		for (int v = 0; v < datas.size(); v++)
		{
			result[v][0] = ((BarElement) datas.get(v)).bar;
			result[v][1] = ((BarElement) datas.get(v)).name;
			result[v][2] = ((BarElement) datas.get(v)).description;
			result[v][3] = ((BarElement) datas.get(v)).image;
			result[v][4] = ((BarElement) datas.get(v)).slider;
			result[v][5] = ((BarElement) datas.get(v)).rep;
		}
		return result;
	}
	
	public String [] getComboChoices(Object element, int columnIndex)
	{
		BarElement el = (BarElement)element;

		if (columnIndex == COL_SLIDER) {
			return sliderChoicesString;
		} 
		
		if (element == null) {
			return new String[] {"this is big enough"};
		} else if (el.rep.getEnumerationValues() != null) {
			return el.rep.getEnumerationValues();
		}
		return new String[0];
	}

	public int getColumnType(Object element, int column)
	{
		BarElement el = (BarElement)element;
		if (column != COL_MODIFY) {
			return super.getColumnType(element,column);
		} else {
			if (el.rep.getModifierType() == SDModifiableVariableRepresentation.STRING) {
				return CHOICE;
			} else {
				return TEXT;
			}
		}
	}

	public Image getColumnImage(Object element, int columnIndex)
	{
		BarElement el = (BarElement) element;
	
		if (columnIndex == COL_IMAGE) {
			return el.image;
		} else {
			return null;	
		}
	}
		
	public String getColumnText(Object element, int columnIndex)
	{
		BarElement el = (BarElement) element;
		
		if (columnIndex == COL_NAME) {
			return el.name;	
		} else if (columnIndex == COL_DESC) {
			return el.description;
		} else if (columnIndex == COL_IMAGE) {
			return "";
		} else if (columnIndex == COL_SLIDER) {
			int index = el.slider.intValue();
			if (index >= 0 && index < sliderChoicesString.length) {
				return sliderChoicesString[index];
			} else {
				return "error";
			}
		} else if (columnIndex == COL_MODIFY) {
			return ""+el.rep.getLastRequestedValue();
		}
		
		return "";
	}
	
	public void valueChanged(Object element, int column, String newValue) throws InvalidTableValueException
	{
		try {
			BarElement el = (BarElement) element;
			
			double d = Double.parseDouble(newValue);
			d = Math.min(d,el.rep.getMaxBound());
			d = Math.max(d,el.rep.getMinBound());

			el.rep.setRequestedValue(System.currentTimeMillis(),d);

			notifyListenersEdit(el.bar);
		} catch (Throwable e) {
			EditorPlugin.DBG.warning("invalid table value entered "+newValue);
			throw new InvalidTableValueException(e.toString());	
		}
		
	}
	public void valueChanged(Object element, int column, Integer newValue) throws InvalidTableValueException
	{
//System.err.println("VALUE = "+newValue);		
		if (column == COL_SLIDER) {
			try {
				BarElement el = (BarElement) element;
				el.slider = sliderChoices[newValue.intValue()];
				
				notifyListenersEdit(el.bar);
			} catch (Throwable e) {
				EditorPlugin.DBG.warning("invalid table value entered "+newValue);
				throw new InvalidTableValueException(e.toString());	
			}
		} else if (column == COL_MODIFY) {
			try {
				BarElement el = (BarElement) element;
			
				String s = el.rep.getEnumerationValues()[newValue.intValue()];
				el.rep.setRequestedValue(System.currentTimeMillis(),s);

//System.err.println("STRING = "+s);		
	
				notifyListenersEdit(el.bar);
			} catch (Throwable e) {
				EditorPlugin.DBG.warning("invalid table value entered "+newValue);
				throw new InvalidTableValueException(e.toString());	
			}
		}
		
		configure();
	}

	
	public class BarElement
	{
		public ZoomControlBar bar;
		public String name;
		public String description;
		public Integer slider;
		public Image image;
		public SDModifiableVariableRepresentation rep;
		
		public BarElement(ZoomControlBar bar, String name, String description, Image image, Integer slider, SDModifiableVariableRepresentation rep)
		{
			this.bar = bar;
			this.name = name;
			this.description = description;
			this.image = image;
			this.slider = slider;
			this.rep = rep;

		}

		public void update() {
		}
	}
	
boolean has_focus = false;		
	public SDModifiableVariableRepresentation getSelectedRep() {
		if (!has_focus) return null;
		int n = getViewer().getTable().getSelectionIndex();
		if (n == -1) return null;
		BarElement element = (BarElement)datas.get(n);
		return element.rep;
	}	

	public void widgetDefaultSelected(SelectionEvent e) {
		widgetSelected(e);
	}
	public void widgetSelected(SelectionEvent e) {
		SDModifiableVariableRepresentation g = getSelectedRep();
		if (g != null) {
			notifyListenersSelect(g);
		}
	}
	public void focusLost(FocusEvent e) {
		has_focus = false;
	}
	public void focusGained(FocusEvent e) {
		has_focus = true;
	}
	
}