/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.statistical.ui.widgets.zoomslider.internal;

import java.util.Date;
import java.util.GregorianCalendar;

import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Composite;

public class TimeZoomSlider extends ZoomSlider
{
    private static final double [] LINEAR_PROGRESSION = {1, 2, 5};
    private static final double [] OOM4 = {1, 2, 3, 6}; // 10 secs > 1 min
    private static final double [] OOM5 = {1.2, 3, 6}; // 2 mins > 10 mins
    private static final double [] OOM6 = {1.2, 1.8, 3.6, 7.2}; // 20 mins > 2 hrs
    private static final double [] OOM7 = {1.8, 3.6, 7.2, 8.64}; // 5 hrs > 24 hrs
    
    ZoomControlBar bar;
    
    double start;
    
    double reconfigurePeriod = 10;
    
    double length;
    
    public GregorianCalendar calendar = new GregorianCalendar();

    public long startTime = System.currentTimeMillis()/1000;
    
    boolean wallTime = true;
    
    public TimeZoomSlider(Composite parent)
    {
    	super(parent);
    }
    
    // CREATE TICKS
    /** Create the vector of ticks */
    protected void createTicks()
    {
        calculateIncrement();
        
        // Empty the tick vector
        tickVector.removeAllElements();
        
        // Refill the tick vector
        double tickValue = ZoomSliderUtilities.round(visible[MIN] - increment, increment);
        
        //EditorPlugin.DBG.info(tickValue + " + " + increment + " = " + (tickValue + increment));
        
        while (tickValue <= visible[MAX] + increment)
        {
            //EditorPlugin.DBG.info((long)tickValue);
            
            tickVector.addElement(new TimeZoomSliderTick(this, tickValue));

            tickValue += increment;
            
            tickValue = ZoomSliderUtilities.round(tickValue, increment);
        }
        
        // Depending on the pixel spacing between ticks find a reasonable number of 
        // intermediate selecteable points (5 or 10)
		int selectablePoints = 10;
    	int tickSpacing = (int)(getPixelRange()/(getVisibleValueRange()/increment));
		if (tickSpacing < 20)
		{
		    selectablePoints = 5;
		}
		
        // Set the unit increment based on the intermediate selectable points
		unitIncrement = increment/selectablePoints;
		if (unitIncrement < resolution)
		{
		    unitIncrement = resolution;
		}
    }


    public void setWallTime(boolean wall)
    {
        wallTime = wall;
    }
    
    public boolean isWallTime()
    {
        return wallTime;
    }
    
    public void findDecimalPointPosition()
    {
    	GC gcTemp = new GC(this);
    	
        int tail = gcTemp.stringExtent(".000").x;    	
                
        decimalPointPosition = getBounds().width - 10;
        
        gcTemp.dispose();
    }
    
    /** The preferred size of the component is dependent on the maximum possible string
        length out of all the ticks */
	public Point computeSize(int wHint, int hHint, boolean changed)
	{
		Point standard = super.computeSize(wHint, hHint, changed);
		Point size = new Point(standard.x, standard.y); 
		
   	    if (orientation == VERTICAL)
	    {
			if (wHint == SWT.DEFAULT)
			{
				GC gcTemp = new GC(this);				
				size.x = gcTemp.stringExtent("99:99:99").x + 20;
				gcTemp.dispose();				
			}
			else
			{
				size.x = wHint;
			}
			
			if (hHint != SWT.DEFAULT)
			{
				size.y = hHint;
			}
	    }
        else
	    {
	    	if (hHint == SWT.DEFAULT)
	    	{
	    		size.y = getFontHeight() + 20;
	    	}
	    	else
	    	{
	    		size.y = hHint;
	    	}
	    	
	    	if (wHint != SWT.DEFAULT)
	    	{
	    		size.x = wHint;
	    	}
	    }
	    
	    return size;
	}
	
	// CALCULATE INCREMENT
    /** Calculate the value spacing between consecutive ticks */
    protected void calculateIncrement()
    {
        TickValueRange tvr = new TickValueRange();
        
        if (tvr.outside())
        {
            int maxTickCount = getPixelRange() / MIN_TICKWIDTH;

            increment = getVisibleValueRange() / maxTickCount;

            int orderOfMagnitude = ZoomSliderUtilities.calculateOOM(increment);
            
            double [] progression = getProgression(orderOfMagnitude);
            
            increment = progression[0] * Math.pow(10, orderOfMagnitude);
            
            while (increment < tvr.min)
            {
                for (int i=0; i<progression.length; i++)
                {
                    increment = progression[i] * Math.pow(10, orderOfMagnitude);
                    
                    if (increment >= tvr.min)
                    {
                        increment = (double)((long)increment);
                        
                        break;
                    }
                }
                orderOfMagnitude++;
                
                progression = getProgression(orderOfMagnitude);
            }
            
            incrementOOM = ZoomSliderUtilities.calculateOOM(increment);
        }        
    }   
    
    public double [] getProgression(int oom)
    {
        switch (oom)
        {
            case 4:
                return OOM4;
            case 5:
                return OOM5;
            case 6:
                return OOM6;
            case 7:
                return OOM7;
            default:
                return LINEAR_PROGRESSION;
        }
    }    
    

    public double getZero()
    {
        GregorianCalendar calendar = new GregorianCalendar();
        
        calendar.setTime(new Date());
        
        double hrs = 1 + (double)calendar.get(GregorianCalendar.HOUR_OF_DAY);
        
        double mins = calendar.get(GregorianCalendar.MINUTE);
        
        double secs = calendar.get(GregorianCalendar.SECOND);
        
        return (hrs * 60 * 60) + (mins * 60) + secs;
    }    
}        
        
