/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: HierarchyURIConverterImpl.java,v 1.10 2005/04/18 21:02:33 slavescu Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.models.hierarchy.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.net.URLConnection;
import java.util.Map;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.plugin.EcorePlugin;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.URIConverter;
import org.eclipse.emf.ecore.resource.impl.ResourceImpl;
import org.eclipse.emf.ecore.resource.impl.URIConverterImpl;


/**
 * A highly functional and extensible URI converter implementation.
 * <p>
 * This implementation provides seamless transparent Eclipse integration
 * by supporting the <code>platform:/resource</code> mechanism both inside of Eclipse and outside of Eclipse.
 * Furthermore, although the implementation imports
 * both {@link org.eclipse.core.runtime} and {@link org.eclipse.core.resources},
 * and hence requires the Eclipse libraries at development time,
 * the implementation does <b>not</b> require them at runtime.
 * Clients of this implementation must be cautious if they wish to maintain this platform neutral behaviour.
 * </p>
 */
public class HierarchyURIConverterImpl extends URIConverterImpl {
    //~ Static fields/initializers -----------------------------------------------------------------

    /**
     * The cached Eclipse workspace.
     */
    protected static IWorkspaceRoot workspaceRoot = EcorePlugin.getWorkspaceRoot();

    //~ Instance fields ----------------------------------------------------------------------------

    /**
     * The URI map.
     */
    protected URIMap uriMap;

    //~ Constructors -------------------------------------------------------------------------------

    /**
     * Creates an instance.
     */
    public HierarchyURIConverterImpl() {
        // This is a hacky way to test stand-alone platform:/resource support in what's really an headless environment.
        //
        // org.eclipse.core.resources.IProject [] projects = workspaceRoot.getProjects();
        // for (int i = 0; i < projects.length; ++i)
        // {
        //   String rootContainerName = projects[i].getName();
        //   URI rootContainerLocation = URI.createFileURI(projects[i].getLocation().toString() + "/");
        //   platformResourceMap.put(rootContainerName, rootContainerLocation);
        // }
        //
        // workspaceRoot = null;
    }

    //~ Methods ------------------------------------------------------------------------------------

    /*
     * Javadoc copied from interface.
     */
    public Map getURIMap() {
        return getInternalURIMap();
    }

    /**
     * Creates an input stream for the URI and returns it.
     * <p>
     * This implementation {@link #normalize normalizes} the URI and uses that as the basis for further processing.
     * It factors out the URI schemes <code>file</code> and <code>platform</code> (with leading <code>resource</code> segment)
     * for special processing by
     * {@link #createFileInputStream createFileInputStream} and
     * {@link #createPlatformResourceInputStream createPlatformResourceInputStream}, respectively.
     * The file-based URI is {@link URI#toFileString converted} to a file path, e.g.,
     *<pre>
     *  file:///C:/directory/file
     *    ->
     *   C:/directory/file
     *</pre>
     * and the platform-based URI is converted to a platform path by trimming the leading <code>platform:/resource</code>, e.g.,
     *<pre>
     *  platform:/resource/project/directory/file
     *    ->
     *  /project/directory/file
     *</pre>
     * All other cases are handled as standard URLs by {@link #createURLInputStream createURLInputStream}.
     * </p>
     * @return an open Input stream.
     * @exception IOException if there is a problem obtaining an open input stream.
     */
    public InputStream createInputStream(URI uri) throws IOException {
        URI converted = normalize(uri);
        String scheme = converted.scheme();

        if ("file".equals(scheme)) {
            String filePath = converted.toFileString();

            return createFileInputStream(filePath);
        } else if ("platform".equals(scheme) && (converted.segmentCount() > 1) && "resource".equals(converted.segment(0))) {
            StringBuffer platformResourcePath = new StringBuffer();

            for (int i = 1, size = converted.segmentCount(); i < size; ++i) {
                platformResourcePath.append('/');
                platformResourcePath.append(converted.segment(i));
            }

            return createPlatformResourceInputStream(platformResourcePath.toString());
        } else if (scheme == null) {
            String filePath = converted.toString();

            return createFileInputStream(filePath);
        } else {
            return createURLInputStream(converted);
        }
    }

    /**
     * Creates an output stream for the URI and returns it.
     * <p>
     * This implementation {@link #normalize normalizes} the URI and uses that as the basis for further processing.
     * It factors out the URI schemes <code>file</code> and <code>platform</code> (with leading <code>resource</code> segment)
     * for special processing by
     * {@link #createFileOutputStream createFileOutputStream} and
     * {@link #createPlatformResourceOutputStream createPlatformResourceOutputStream}, respectively.
     * The file-based URI is {@link URI#toFileString converted} to a file path, e.g.,
     *<pre>
     *  file:///C:/directory/file
     *    ->
     *   C:/directory/file
     *</pre>
     * and the platform-based URI is converted to a platform path by trimming the leading <code>platform:/resource</code>, e.g.,
     *<pre>
     *  platform:/resource/project/directory/file
     *    ->
     *  /project/directory/file
     *</pre>
     * All other cases are handled as standard URLs by {@link #createURLOutputStream createURLOutputStream}.
     * </p>
     * @return an open output stream.
     * @exception IOException if there is a problem obtaining an open output stream.
     */
    public OutputStream createOutputStream(URI uri) throws IOException {
        URI converted = normalize(uri);
        String scheme = converted.scheme();

        if ("file".equals(scheme)) {
            String filePath = converted.toFileString();

            return createFileOutputStream(filePath);
        } else if ("platform".equals(scheme) && (converted.segmentCount() > 1) && "resource".equals(converted.segment(0))) {
            StringBuffer platformResourcePath = new StringBuffer();

            for (int i = 1, size = converted.segmentCount(); i < size; ++i) {
                platformResourcePath.append('/');
                platformResourcePath.append(converted.segment(i));
            }

            return createPlatformResourceOutputStream(platformResourcePath.toString());
        } else if (scheme == null) {
            String filePath = converted.toString();

            return createFileOutputStream(filePath);
        } else {
            return createURLOutputStream(converted);
        }
    }
	public String toString(URI uri) {
		URI converted = normalize(uri);
		String scheme = converted.scheme();
		if ("file".equals(scheme)) {
			return converted.toFileString();
		} else if ("platform".equals(scheme) && (converted.segmentCount() > 1) && "resource".equals(converted.segment(0))) {
			StringBuffer platformResourcePath = new StringBuffer();
			for (int i = 1, size = converted.segmentCount(); i < size; ++i) {
				platformResourcePath.append('/');
				platformResourcePath.append(converted.segment(i));
			}
			return platformResourcePath.toString();
		} else {
			return converted.toString();
		}
	}
    /**
     * Returns the normalized form of the URI.
     * <p>
     * This implementation does precisely and only the {@link URIConverter#normalize typical} thing.
     * It calls itself recursively so that mapped chains are followed.
     * </p>
     * @param uri the URI to normalize.
     * @return the normalized form.
     * @see org.eclipse.emf.ecore.plugin.EcorePlugin#getPlatformResourceMap
     */
    public URI normalize(URI uri) {
        String fragment = uri.fragment();
        URI result = (fragment == null) ? getInternalURIMap().getURI(uri) : getInternalURIMap().getURI(uri.trimFragment()).appendFragment(fragment);
        String scheme = result.scheme();

        if (scheme == null) {
            // ECLIPSE-DEPEND-BEGIN
            if (workspaceRoot != null) {
                if (result.hasAbsolutePath()) {
                    result = URI.createPlatformResourceURI(result.toString());
                }
            } else
            // ECLIPSE-DEPEND-END
             {
                result = URI.createURI("file:" + result);
            }
        }

        if (result.equals(uri)) {
            return uri;
        } else {
            return normalize(result);
        }
    }

    /**
     * Creates an input stream for the file path and returns it.
     * <p>
     * This implementation allocates a {@link FileInputStream}.
     * </p>
     * @return an open input stream.
     * @exception IOException if there is a problem obtaining an open input stream.
     */
    protected InputStream createFileInputStream(String filePath) throws IOException {
        File file = new File(filePath);
        InputStream inputStream = new FileInputStream(file);

        return inputStream;
    }

    /**
     * Creates an output stream for the file path and returns it.
     * <p>
     * This implementation allocates a {@link FileOutputStream} and creates subdirectories as necessary.
     * </p>
     * @return an open output stream.
     * @exception IOException if there is a problem obtaining an open output stream.
     */
    protected OutputStream createFileOutputStream(String filePath) throws IOException {
        File file = new File(filePath);
        String parent = file.getParent();

        if (parent != null) {
            new File(parent).mkdirs();
        }

        OutputStream outputStream = new FileOutputStream(file);

        return outputStream;
    }

    /**
     * Creates an input stream for the platform resource path and returns it.
     * <p>
     * This implementation does one of two things, depending on the runtime environment.
     * If there is an Eclipse workspace, it delegates to
     * {@link WorkbenchHelper#createPlatformResourceInputStream WorkbenchHelper.createPlatformResourceInputStream},
     * which gives the expected Eclipse behaviour.
     * Otherwise, the {@link EcorePlugin#resolvePlatformResourcePath resolved} URI
     * is delegated to {@link #createInputStream createInputStream}
     * for recursive processing.
     * @return an open input stream.
     * @exception IOException if there is a problem obtaining an open input stream or a valid interpretation of the path.
     * @see EcorePlugin#resolvePlatformResourcePath(String)
     */
    protected InputStream createPlatformResourceInputStream(String platformResourcePath) throws IOException {
        // ECLIPSE-DEPEND-BEGIN
        if (workspaceRoot != null) {
            return WorkbenchHelper.createPlatformResourceInputStream(platformResourcePath);
        } else
        // ECLIPSE-DEPEND-END
         {
            URI resolvedLocation = EcorePlugin.resolvePlatformResourcePath(platformResourcePath);

            if (resolvedLocation != null) {
                return createInputStream(resolvedLocation);
            }

            throw new IOException("The path '" + platformResourcePath + "' is unmapped");
        }
    }

    /**
     * Creates an output stream for the platform resource path and returns it.
     * <p>
     * This implementation does one of two things, depending on the runtime environment.
     * If there is an Eclipse workspace, it delegates to
     * {@link WorkbenchHelper#createPlatformResourceOutputStream WorkbenchHelper.createPlatformResourceOutputStream},
     * which gives the expected Eclipse behaviour.
     * Otherwise, the {@link EcorePlugin#resolvePlatformResourcePath resolved} URI
     * is delegated to {@link #createOutputStream createOutputStream}
     * for recursive processing.
     * @return an open output stream.
     * @exception IOException if there is a problem obtaining an open output stream or a valid interpretation of the path.
     * @see EcorePlugin#resolvePlatformResourcePath(String)
     */
    protected OutputStream createPlatformResourceOutputStream(String platformResourcePath) throws IOException {
        // ECLIPSE-DEPEND-BEGIN
        if (workspaceRoot != null) {
            return WorkbenchHelper.createPlatformResourceOutputStream(platformResourcePath);
        } else
        // ECLIPSE-DEPEND-END
         {
            URI resolvedLocation = EcorePlugin.resolvePlatformResourcePath(platformResourcePath);

            if (resolvedLocation != null) {
                return createOutputStream(resolvedLocation);
            }

            throw new IOException("The path '" + platformResourcePath + "' is unmapped");
        }
    }

    /**
     * Creates an input stream for the URI, assuming it's a URL, and returns it.
     * @return an open input stream.
     * @exception IOException if there is a problem obtaining an open input stream.
     */
    protected InputStream createURLInputStream(URI uri) throws IOException {
        URL url = new URL(uri.toString());
        URLConnection urlConnection = url.openConnection();

        return urlConnection.getInputStream();
    }

    /**
     * Creates an output stream for the URI, assuming it's a URL, and returns it.
     * @return an open output stream.
     * @exception IOException if there is a problem obtaining an open output stream.
     */
    protected OutputStream createURLOutputStream(URI uri) throws IOException {
        URL url = new URL(uri.toString());
        URLConnection urlConnection = url.openConnection();

        urlConnection.setDoOutput(true);

        return urlConnection.getOutputStream();
    }

    //~ Inner Interfaces ---------------------------------------------------------------------------

    // ECLIPSE-DEPEND-END

    /**
     * A map that remaps URIs.
     */
    public interface URIMap extends Map {
        /**
         * Returns the remapped URI, or the URI itself.
         * @param uri the URI to remap.
         * @return the remapped URI, or the URI itself.
         */
        URI getURI(URI uri);
    }

    //~ Inner Classes ------------------------------------------------------------------------------

    public static class HierarchyPipedInputStream extends InputStream {
        private byte[] inBuf;
        private int inCount;
        private int inPos;
        private long inTotalLength;

        /**
         * Constructor MyPipedInputStream.
         * @param buf
         */
        public HierarchyPipedInputStream(byte[] buf, int length) {
            inBuf = buf;
            inCount = length;
            inTotalLength = length;
            inPos = 0;
        }

        /**
         * Sets the inBuf.
         * @param inBuf The inBuf to set
         */
        public void setInBuf(byte[] buf, int length) {
            inBuf = buf;
            inCount = length;
            inTotalLength = length;
            inPos = 0;
        }

        /**
         * @see java.io.InputStream#available()
         */
        public int available() throws IOException {
            return inCount;
        }

        /**
         * @see java.io.InputStream#markSupported()
         */
        public boolean markSupported() {
            return false;
        }

        /**
         * @see java.io.InputStream#read()
         */
        public int read() throws IOException {
            if (available() == 0) {
                return -1;
            } else {
                inCount--;

                return inBuf[inPos++];
            }
        }

        /**
         * @see java.io.InputStream#read(byte[], int, int)
         */
        public int read(byte[] outBuf, int offset, int length) throws IOException {
            if (available() == 0) {
                return -1;
            }

            //            int readBytes = 0;
            //            while ((length > 0) && (inPos < inLastPos)) {
            //                outBuf[offset++] = inBuf[inPos++];
            //                length--;
            //                readBytes++;
            //            }
            int readBytes = min(inCount, length);

            System.arraycopy(inBuf, inPos, outBuf, offset, readBytes);

            inPos = inPos + readBytes;
            inCount = inCount - readBytes;

            return readBytes;
        }

        /**
         * @see java.io.InputStream#read(byte[])
         */
        public int read(byte[] arg0) throws IOException {
            return read(arg0, 0, arg0.length);
        }

        /**
         * @see java.io.InputStream#skip(long)
         */
        public long skip(long arg0) throws IOException {
            arg0 = arg0 + inPos;

            if (arg0 < inTotalLength) {
                inPos = (int) arg0;

                return arg0;
            } else {
                return -1;
            }
        }

        protected int min(int a, int b) {
            if (a < b) {
                return a;
            }

            return b;
        }
    }

    public static class HierarchyPlatformResourceOutputStream extends OutputStream {
        protected HierarchyPipedInputStream pipedInputStream;
        protected IFile file;
        protected IProgressMonitor progressMonitor;
        protected boolean firstCall = true;
        protected boolean force;
        protected boolean keepHistory;
        protected int SAVE_TRESHOLD = 1024 * 256;
        protected byte[] outBuf = new byte[SAVE_TRESHOLD];
        protected int outPos;

        public HierarchyPlatformResourceOutputStream(IFile file, boolean force, boolean keepHistory, IProgressMonitor monitor) throws IOException {
            this.file = file;
            this.force = force;
            this.keepHistory = keepHistory;
            this.progressMonitor = monitor;

            //	  if (file.exists())
            //	  {
            //		try {
            //			file.delete(force,keepHistory,progressMonitor);
            //		} catch (CoreException e) {
            //			throw new ResourceImpl.IOWrappedException(e);
            //		}
            //	  }
            //	  else
            createContainer(file.getParent());
            pipedInputStream = new HierarchyPipedInputStream(outBuf, 0);
        }

        /**
         * @see java.io.OutputStream#close()
         */
        public void close() throws IOException {
            flush();
        }

        /**
         * @see java.io.OutputStream#flush()
         */
        public void flush() throws IOException {
            if (outPos > 0) {
                writeToWorkspaceResource();
                outPos = 0;
            }
        }

        /**
         * @see java.io.OutputStream#write(byte[], int, int)
         */
        public void write(byte[] data, int offset, int length) throws IOException {
            //            while (length > 0) {
            //                while ((outPos < SAVE_TRESHHOLD) && (length > 0)) {
            //                    outBuf[outPos++] = data[offset++];
            //                    length--;
            //                }
            //
            //                if (outPos == SAVE_TRESHHOLD) {
            //                    flush();
            //                }
            //            }
            while (length > 0) {
                int writeBytes = min(SAVE_TRESHOLD - outPos, length);

                System.arraycopy(data, offset, outBuf, outPos, writeBytes);
                outPos = outPos + writeBytes;
                offset = offset + writeBytes;
                length = length - writeBytes;

                if (outPos == SAVE_TRESHOLD) {
                    flush();
                }
            }
        }

        /**
         * @see java.io.OutputStream#write(byte[])
         */
        public void write(byte[] arg0) throws IOException {
            write(arg0, 0, arg0.length);
        }

        /**
         * @see java.io.OutputStream#write(int)
         */
        public void write(int arg0) throws IOException {
            outBuf[outPos++] = (byte) arg0;

            if (outPos == SAVE_TRESHOLD) {
                flush();
            }
        }

        protected void createContainer(IContainer container) throws IOException {
            if (!container.exists()) {
                if (container.getType() == IResource.FOLDER) {
                    createContainer(container.getParent());

                    try {
                        ((IFolder) container).create(force, keepHistory, progressMonitor);
                    } catch (CoreException exception) {
                        throw new ResourceImpl.IOWrappedException(exception);
                    }
                }
            }
        }

        protected int min(int a, int b) {
            if (a < b) {
                return a;
            }

            return b;
        }

        /**
         * Method writeToResource.
         */
        private void writeToWorkspaceResource() throws IOException {
            pipedInputStream.setInBuf(outBuf, outPos);

            try {
                if (!file.exists()) {
                    file.create(pipedInputStream, force, progressMonitor);
                    firstCall = false;
                } else {
                    if (!file.isLocal(IResource.DEPTH_ONE) || !file.isSynchronized(IResource.DEPTH_ONE)) {
                        file.refreshLocal(IResource.DEPTH_ONE, progressMonitor);
                    }

                    if (firstCall) {
                        file.setContents(pipedInputStream, force, keepHistory, progressMonitor);
                        firstCall = false;
                    } else {
                        file.appendContents(pipedInputStream, force, keepHistory, progressMonitor);
                    }
                }
            } catch (CoreException exception) {
                throw new Resource.IOWrappedException(exception);
            }
        }
    }

    /**
     * Isolated Eclipse workbench utilities.
     */
    public static class WorkbenchHelper {
        /**
         * Creates an input stream for the given {@link IFile} path.
         * <p>
         * This implementation uses {@link IFile#getContents IFile.getContents}.
         * </p>
         * @return an open input stream.
         * @see IWorkspaceRoot#getFile(IPath)
         * @see IFile#getContents
         * @exception IOException if there is a problem obtaining an open input stream.
         */
        public static InputStream createPlatformResourceInputStream(String platformResourcePath) throws IOException {
            IFile file = workspaceRoot.getFile(new Path(platformResourcePath));

            try {
                if (!file.isLocal(IResource.DEPTH_ONE) || !file.isSynchronized(IResource.DEPTH_ONE)) {
                    file.refreshLocal(IResource.DEPTH_ONE, null);
                }

                return file.getContents();
            } catch (CoreException exception) {
                throw new Resource.IOWrappedException(exception);
            }
        }

        /**
         * Creates an output stream for the given {@link IFile} path.
         * <p>
         * This implementation uses a {@link PerftraceURIConverterImpl.PlatformResourceOutputStream}.
         * </p>
         * @return an open output stream.
         * @exception IOException if there is a problem obtaining an open output stream.
         * @see IWorkspaceRoot#getFile(IPath)
         * @see PerftraceURIConverterImpl.PlatformResourceOutputStream
         * @see IFile#setContents(InputStream, boolean, boolean, IProgressMonitor)
         */
        public static OutputStream createPlatformResourceOutputStream(String platformResourcePath) throws IOException {
            IFile file = workspaceRoot.getFile(new Path(platformResourcePath));

            return new HierarchyPlatformResourceOutputStream(file, false, false, null);
        }
    }
}
