/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 *
 * $Id: SimpleSearchQueryEngine.java,v 1.33 2005/05/05 21:53:57 slavescu Exp $
 **********************************************************************/
package org.eclipse.hyades.models.hierarchy.util.internal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.ConcurrentModificationException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.hyades.models.hierarchy.extensions.ArithmeticExpression;
import org.eclipse.hyades.models.hierarchy.extensions.ArithmeticOperators;
import org.eclipse.hyades.models.hierarchy.extensions.BinaryExpression;
import org.eclipse.hyades.models.hierarchy.extensions.ExtensionsFactory;
import org.eclipse.hyades.models.hierarchy.extensions.LogicalExpression;
import org.eclipse.hyades.models.hierarchy.extensions.LogicalOperators;
import org.eclipse.hyades.models.hierarchy.extensions.NumericFunction;
import org.eclipse.hyades.models.hierarchy.extensions.Operand;
import org.eclipse.hyades.models.hierarchy.extensions.OrderByElement;
import org.eclipse.hyades.models.hierarchy.extensions.OrderByOperators;
import org.eclipse.hyades.models.hierarchy.extensions.QueryResult;
import org.eclipse.hyades.models.hierarchy.extensions.RelationalOperators;
import org.eclipse.hyades.models.hierarchy.extensions.ResultEntry;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleOperand;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleSearchQuery;
import org.eclipse.hyades.models.hierarchy.extensions.WhereExpression;
import org.eclipse.hyades.models.hierarchy.extensions.impl.SimpleOperandImpl;
import org.eclipse.hyades.models.hierarchy.util.ContainmentTraverser;
import org.eclipse.hyades.models.hierarchy.util.FastList;
import org.eclipse.hyades.models.hierarchy.util.PerfUtil;
/**
 * @author Marius Slavescu (slavescu@ca.ibm.com)
 * @since 3.3
 *  
 */
public class SimpleSearchQueryEngine {
	// protected Set currentPath = new HashSet(20);
	protected Set alreadyProcessed = new HashSet();
	protected Map classPredicatesIndex = new HashMap();
	protected Map featurePredicatesIndex = new HashMap();
	// protected Map instancePredicatesIndex = new HashMap();
	protected Map namesIndex = new HashMap();
	protected Set outputClasses = new HashSet();
	protected Set outputClassesSuperTypes = new HashSet();
	protected Set outputClassesSubTypes = new HashSet();
	protected SimpleSearchQuery query;
	protected QueryResult queryResult;
	protected Set requiredPaths = new HashSet();
	protected Collection rootClasses = new HashSet();
	protected Collection rootNodes = new HashSet();
	protected ResourceSet targetResourceSet;
	protected Map whereExpressionValuesIndex = new HashMap();
	protected SimpleSearchQueryEvaluator queryEvaluator = new SimpleSearchQueryEvaluator();
	protected EvalResult result;
	protected List resetStack = new FastList();
	protected List resetParentsStack = new FastList();
	public static final int IS_COMPLETE = 0x01;
	public static final int IS_NOT_COMPLETE = 0x02;
	public static final int IS_TRUE = 0x04;
	public static final int IS_FALSE = 0x08;
	public static final int IS_APPLICABLE = 0x10;
	/**
	 * @author Marius Slavescu (slavescu@ca.ibm.com)
	 * @since 3.3
	 *  
	 */
	public class EvalResult {
		public static final int BOOLEAN = 4;
		public static final int NUMERIC = 2;
		public static final int OBJECT = 5;
		public static final int SIMPLE_OPERAND = 1;
		public static final int STRING = 3;
		protected Boolean booleanValue;
		protected int compareResult;
		protected boolean complete;
		protected EObject context;
		protected EClass contextType;
		protected WhereExpression expression;
		protected Number numericValue;
		protected Object objectValue;
		protected Operand operand;
		protected List partialResults = new ArrayList();
		protected String stringValue;
		protected int valueType = -1;
		public EvalResult() {
		}
		public EvalResult(Operand operand) {
			this.operand = operand;
			contextType = getContextType();
		}
		public EvalResult(WhereExpression expression) {
			this.expression = expression;
		}
		public String toString() {
			StringBuffer result = new StringBuffer("\n" + getLevelSpaces().toString());
			// result.append(super.toString());
			result.append(" (complete: ");
			result.append(this.complete);
			// result.append(", objectValue: ");
			// result.append(this.objectValue);
			result.append(", valueType: ");
			result.append(this.valueType);
			result.append(", booleanValue: ");
			result.append(this.booleanValue);
			// result.append(", context: ");
			// result.append(this.context);
			result.append(", contextType: ");
			result.append(this.contextType);
			// result.append(", expression: ");
			// result.append(this.expression);
			result.append(", numericValue: ");
			result.append(this.numericValue);
			// result.append(", operand: ");
			// result.append(this.operand);
			result.append(", stringValue: ");
			result.append(this.stringValue);
			result.append(", partialResults: ");
			for (Iterator iter = this.partialResults.iterator(); iter.hasNext();) {
				EvalResult element = (EvalResult) iter.next();
				result.append(element.toString());
			}
			result.append(')');
			return result.toString();
		}
		private StringBuffer getLevelSpaces() {
			StringBuffer s = new StringBuffer();
			if (operand != null)
				addLevelSpaces(operand, s);
			else
				addLevelSpaces(expression, s);
			return s;
		}
		private void addLevelSpaces(EObject o, StringBuffer s) {
			if (o == null)
				return;
			s.append(" ");
			addLevelSpaces(o.eContainer(), s);
		}
		protected Number add() {
			Number result = null;
			for (int i = 0; i < partialResults.size() && i < 2; i++) {
				if (result == null)
					result = (((EvalResult) partialResults.get(i)).numericValue());
				else {
					if (result instanceof Double) {
						result = new Double(result.doubleValue() + ((EvalResult) partialResults.get(i)).numericValue().doubleValue());
					} else if (result instanceof Integer) {
						result = new Integer(result.intValue() + ((EvalResult) partialResults.get(i)).numericValue().intValue());
					} else if (result instanceof Float) {
						result = new Float(result.floatValue() + ((EvalResult) partialResults.get(i)).numericValue().floatValue());
					} else if (result instanceof Short) {
						result = new Short((short) (result.shortValue() + ((EvalResult) partialResults.get(i)).numericValue().shortValue()));
					} else if (result instanceof Long) {
						result = new Long(result.longValue() + ((EvalResult) partialResults.get(i)).numericValue().longValue());
					} else if (result instanceof Byte) {
						result = new Long(result.byteValue() + ((EvalResult) partialResults.get(i)).numericValue().byteValue());
					}
				}
			}
			return result;
		}
		public void addPartialResult(EvalResult result) {
			partialResults.add(result);
		}
		public Boolean booleanValue() {
			return booleanValue;
		}
		protected int compare(EvalResult lhs, EvalResult rhs) {
			if (lhs.isNumeric() && rhs.isNumeric() && lhs.numericValue() instanceof Comparable) {
				return ((Comparable) lhs.numericValue()).compareTo(rhs.numericValue());
			} else if (lhs.isBoolean() && lhs.isBoolean()) {
				if (lhs.booleanValue() == Boolean.TRUE) {
					if (rhs.booleanValue() == Boolean.TRUE)
						return 0;
					else
						return 1;
				} else {
					if (rhs.booleanValue() == Boolean.FALSE)
						return 0;
					else
						return -1;
				}
			} else if (lhs.isObject() && rhs.isObject()) {
				if (rhs.objectValue() instanceof List)
					return ((List) rhs.objectValue()).contains(lhs.objectValue()) ? 0 : -1;
				else
					return lhs.objectValue().equals(rhs.objectValue()) ? 0 : 1;
			} else if (((BinaryExpression) expression).isCaseInsensitive())
				return lhs.stringValue().toLowerCase().compareTo(rhs.stringValue().toLowerCase());
			else
				return lhs.stringValue().compareTo(rhs.stringValue());
		}
		public EvalResult computePartialResult() {
			if (isComplete())
				return this;
			if (operand != null) {
				if (operand instanceof NumericFunction)
					updateNumericFunctionValue();
				else if (operand instanceof ArithmeticExpression)
					updateArithmeticExpressionValue();
				else
					updateSimpleOperand();
			} else if (expression != null) {
				if (expression instanceof LogicalExpression)
					updateLogicalExpressionValue();
				else
					updateBinaryExpressionValue();
			}
			return this;
		}
		protected Number divide() {
			Number result = null;
			for (int i = 0; i < partialResults.size() && i < 2; i++) {
				if (result == null)
					result = (((EvalResult) partialResults.get(i)).numericValue());
				else {
					if (result instanceof Double) {
						result = new Double(result.doubleValue() / ((EvalResult) partialResults.get(i)).numericValue().doubleValue());
					} else if (result instanceof Integer) {
						result = new Integer(result.intValue() / ((EvalResult) partialResults.get(i)).numericValue().intValue());
					} else if (result instanceof Float) {
						result = new Float(result.floatValue() / ((EvalResult) partialResults.get(i)).numericValue().floatValue());
					} else if (result instanceof Short) {
						result = new Short((short) (result.shortValue() / ((EvalResult) partialResults.get(i)).numericValue().shortValue()));
					} else if (result instanceof Long) {
						result = new Long(result.longValue() / ((EvalResult) partialResults.get(i)).numericValue().longValue());
					} else if (result instanceof Byte) {
						result = new Long(result.byteValue() / ((EvalResult) partialResults.get(i)).numericValue().byteValue());
					}
				}
			}
			return result;
		}
		public EvalResult eval(EObject element) {
			for (int i = 0; i < partialResults.size(); i++) {
				EvalResult r = ((EvalResult) partialResults.get(i));
				r.eval(element);
			}
			if (expression != null || !(operand instanceof SimpleOperand)) {
				resetValues();
				complete = false;
			} else if (isValidContext(element)) {
				context = element;
				resetValues();
				complete = false;
			}
			computePartialResult();
			return this;
		}
		protected EClass getContextType() {
			if (operand instanceof SimpleOperandImpl) {
				if (((SimpleOperand) operand).getType() != null)
					return ((SimpleOperand) operand).getType();
				if (((SimpleOperand) operand).getFeature() != null)
					return ((SimpleOperand) operand).getFeature().getEContainingClass();
			}
			return null;
		}
		protected Number integerDevide() {
			int result = 0;
			boolean first = true;
			for (int i = 0; i < partialResults.size() && i < 2; i++) {
				if (first) {
					result = (((EvalResult) partialResults.get(i)).numericValue()).intValue();
					first = false;
				} else {
					result = result / ((EvalResult) partialResults.get(i)).numericValue().intValue();
				}
			}
			return new Integer(result);
		}
		protected boolean isArithmeticExpressionComplete(ArithmeticExpression arithmeticExpression) {
			int i = 0;
			for (; i < partialResults.size(); i++) {
				if (!((EvalResult) partialResults.get(i)).isComplete()) {
					return false;
				}
			}
			return i == (arithmeticExpression.getArguments().size() - 1);
		}
		protected boolean isBetweenComplete() {
			if (partialResults.size() == 3) {
				EvalResult lhs = (EvalResult) partialResults.get(0);
				EvalResult rhs1 = (EvalResult) partialResults.get(1);
				EvalResult rhs2 = (EvalResult) partialResults.get(2);
				if (lhs.isComplete() && rhs1.isComplete() && rhs2.isComplete()) {
					int c1 = compare(rhs1, lhs);
					if (c1 > 0)
						compareResult = -1;
					else {
						int c2 = compare(lhs, rhs2);
						if (c2 > 0)
							compareResult = 1;
						else
							compareResult = 0;
					}
					complete = true;
					return true;
				}
			}
			return false;
		}
		protected boolean isBinaryComplete() {
			if (partialResults.size() == 2) {
				EvalResult lhs = (EvalResult) partialResults.get(0);
				EvalResult rhs = (EvalResult) partialResults.get(1);
				if (lhs.isComplete() && rhs.isComplete()) {
					compareResult = compare(lhs, rhs);
					complete = true;
					return true;
				}
			}
			return false;
		}
		protected boolean isBoolean() {
			return valueType == BOOLEAN;
		}
		public boolean isComplete() {
			if (complete) {
				return true;
			}
			if (objectValue != null || stringValue != null || booleanValue != null || numericValue != null) {
				complete = true;
				return true;
			} else
				return false;
		}
		protected boolean isInComplete() {
			BinaryExpression binaryExpression = (BinaryExpression) expression;
			if (partialResults.isEmpty())
				return false;
			EvalResult lhs = (EvalResult) partialResults.get(0);
			if (!lhs.isComplete())
				return false;
			int i = 1, j = 1;
			compareResult = -1;
			for (; i < partialResults.size(); i++) {
				EvalResult rhs = (EvalResult) partialResults.get(1);
				if (rhs.isComplete()) {
					j++;
					compareResult = compare(lhs, rhs);
					if (compareResult == 0) {
						complete = true;
						return true;
					}
				}
			}
			if (i == j) {
				complete = true;
				booleanValue = Boolean.FALSE;
			}
			return false;
		}
		protected boolean isLikeComplete() {
			if (partialResults.size() == 2) {
				EvalResult lhs = (EvalResult) partialResults.get(0);
				EvalResult rhs = (EvalResult) partialResults.get(1);
				if (lhs.isComplete() && rhs.isComplete()) {
					compareResult = like(lhs, rhs);
					complete = true;
					return true;
				}
			}
			return false;
		}
		protected boolean isNumeric() {
			return valueType == NUMERIC;
		}
		protected boolean isObject() {
			return valueType == OBJECT;
		}
		protected boolean isString() {
			return valueType == STRING;
		}
		protected boolean isValidContext(EObject element) {
			if (contextType != null) {
				if (element != null && (element.eClass() == contextType || QueryUtils.getAllSubTypes(contextType).contains(element.eClass())))
					return true;
			}
			return false;
		}
		protected int like(EvalResult lhs, EvalResult rhs) {
			return like(lhs.stringValue(), rhs.stringValue());
		}
		protected int like(String lhs, String rhs) {
			if (((BinaryExpression) expression).isCaseInsensitive()) {
				lhs = lhs.toLowerCase();
				rhs = rhs.toLowerCase();
			}
			int lhs_pos = 0;
			int rhs_pos = 0;
			int lhs_len = lhs.length();
			int rhs_len = rhs.length();
			int wildcard = -1;
			int curent_lhs_pos = -1;
			while (true) {
				if (rhs_pos < rhs_len && rhs.charAt(rhs_pos) == '*') {
					wildcard = ++rhs_pos;
					curent_lhs_pos = lhs_pos;
				} else if (lhs_pos == lhs_len) {
					return rhs_pos == rhs_len ? 0 : 1;
				} else if (rhs_pos < rhs_len && (lhs.charAt(lhs_pos) == rhs.charAt(rhs_pos) || rhs.charAt(rhs_pos) == '?')) {
					lhs_pos = lhs_pos + 1;
					rhs_pos = rhs_pos + 1;
				} else if (wildcard >= 0) {
					lhs_pos = ++curent_lhs_pos;
					rhs_pos = wildcard;
				} else {
					return -1;
				}
			}
		}
		protected Number mod() {
			Number result = null;
			// will evaluate only the first two arguments like lsh % rhs,
			// returns the the lhs if rhs is missing
			for (int i = 0; i < partialResults.size() && i < 2; i++) {
				if (result == null)
					result = (((EvalResult) partialResults.get(i)).numericValue());
				else {
					if (result instanceof Double) {
						result = new Double(result.doubleValue() % ((EvalResult) partialResults.get(i)).numericValue().doubleValue());
					} else if (result instanceof Integer) {
						result = new Integer(result.intValue() % ((EvalResult) partialResults.get(i)).numericValue().intValue());
					} else if (result instanceof Float) {
						result = new Float(result.floatValue() % ((EvalResult) partialResults.get(i)).numericValue().floatValue());
					} else if (result instanceof Short) {
						result = new Short((short) (result.shortValue() % ((EvalResult) partialResults.get(i)).numericValue().shortValue()));
					} else if (result instanceof Long) {
						result = new Long(result.longValue() % ((EvalResult) partialResults.get(i)).numericValue().longValue());
					} else if (result instanceof Byte) {
						result = new Long(result.byteValue() % ((EvalResult) partialResults.get(i)).numericValue().byteValue());
					}
				}
			}
			return result;
		}
		protected Number multiply() {
			Number result = null;
			for (int i = 0; i < partialResults.size(); i++) {
				if (result == null)
					result = ((EvalResult) partialResults.get(i)).numericValue();
				else {
					if (result instanceof Double) {
						result = new Double(result.doubleValue() * ((EvalResult) partialResults.get(i)).numericValue().doubleValue());
					} else if (result instanceof Integer) {
						result = new Integer(result.intValue() * ((EvalResult) partialResults.get(i)).numericValue().intValue());
					} else if (result instanceof Float) {
						result = new Float(result.floatValue() * ((EvalResult) partialResults.get(i)).numericValue().floatValue());
					} else if (result instanceof Short) {
						result = new Short((short) (result.shortValue() * ((EvalResult) partialResults.get(i)).numericValue().shortValue()));
					} else if (result instanceof Long) {
						result = new Long(result.longValue() * ((EvalResult) partialResults.get(i)).numericValue().longValue());
					} else if (result instanceof Byte) {
						result = new Long(result.byteValue() * ((EvalResult) partialResults.get(i)).numericValue().byteValue());
					}
				}
			}
			return result;
		}
		public Number numericValue() {
			return numericValue;
		}
		public Object objectValue() {
			return objectValue;
		}
		public boolean reset(EObject element) {
			if (isValidContext(element)) {
				context = null;
				complete = false;
			}
			for (int i = 0; i < partialResults.size(); i++) {
				EvalResult r = ((EvalResult) partialResults.get(i));
				if (!r.reset(element)) {
					complete = false;
				}
			}
			if (!complete) {
				resetValues();
			}
			return isComplete();
		}
		protected void resetValues() {
			numericValue = null;
			stringValue = null;
			objectValue = null;
			booleanValue = null;
		}
		public String stringValue() {
			if (stringValue != null)
				return stringValue;
			if (numericValue != null)
				return numericValue.toString();
			if (booleanValue != null)
				return booleanValue.toString();
			if (objectValue != null)
				return objectValue.toString();
			return "";
		}
		protected Number substract() {
			Number result = null;
			for (int i = 0; i < partialResults.size(); i++) {
				if (result == null)
					result = ((EvalResult) partialResults.get(i)).numericValue();
				else {
					if (result instanceof Double) {
						result = new Double(result.doubleValue() - ((EvalResult) partialResults.get(i)).numericValue().doubleValue());
					} else if (result instanceof Integer) {
						result = new Integer(result.intValue() - ((EvalResult) partialResults.get(i)).numericValue().intValue());
					} else if (result instanceof Float) {
						result = new Float(result.floatValue() - ((EvalResult) partialResults.get(i)).numericValue().floatValue());
					} else if (result instanceof Short) {
						result = new Short((short) (result.shortValue() - ((EvalResult) partialResults.get(i)).numericValue().shortValue()));
					} else if (result instanceof Long) {
						result = new Long(result.longValue() - ((EvalResult) partialResults.get(i)).numericValue().longValue());
					} else if (result instanceof Byte) {
						result = new Long(result.byteValue() - ((EvalResult) partialResults.get(i)).numericValue().byteValue());
					}
				}
			}
			return result;
		}
		protected void updateArithmeticExpressionValue() {
			ArithmeticExpression arithmeticExpression = (ArithmeticExpression) expression;
			if (isArithmeticExpressionComplete(arithmeticExpression)) {
				switch (arithmeticExpression.getOperator().getValue()) {
					case ArithmeticOperators.ADD :
						numericValue = add();
						break;
					case ArithmeticOperators.DIVIDE :
						numericValue = divide();
						break;
					case ArithmeticOperators.INTEGER_DIVIDE :
						numericValue = integerDevide();
						break;
					case ArithmeticOperators.MOD :
						numericValue = mod();
						break;
					case ArithmeticOperators.MULTIPLY :
						numericValue = multiply();
						break;
					case ArithmeticOperators.SUBSTRACT :
						numericValue = substract();
						break;
				}
			}
		}
		protected void updateBinaryExpressionValue() {
			BinaryExpression binaryExpression = (BinaryExpression) expression;
			if (binaryExpression.getOperator() == RelationalOperators.BETWEEN_LITERAL) {
				if (isBetweenComplete())
					booleanValue = Boolean.valueOf(compareResult == 0);
			} else if (binaryExpression.getOperator() == RelationalOperators.IN_LITERAL) {
				if (isInComplete())
					booleanValue = Boolean.valueOf(compareResult == 0);
			} else if (binaryExpression.getOperator() == RelationalOperators.LIKE_LITERAL) {
				if (isLikeComplete())
					booleanValue = Boolean.valueOf(compareResult == 0);
			} else if (isBinaryComplete()) {
				switch (binaryExpression.getOperator().getValue()) {
					case RelationalOperators.EQ :
						booleanValue = Boolean.valueOf(compareResult == 0);
						break;
					case RelationalOperators.GE :
						booleanValue = Boolean.valueOf(compareResult >= 0);
						break;
					case RelationalOperators.GT :
						booleanValue = Boolean.valueOf(compareResult > 0);
						break;
					case RelationalOperators.LE :
						booleanValue = Boolean.valueOf(compareResult <= 0);
						break;
					case RelationalOperators.LT :
						booleanValue = Boolean.valueOf(compareResult < 0);
						break;
					case RelationalOperators.NEQ :
						booleanValue = Boolean.valueOf(compareResult != 0);
						break;
				}
			}
		}
		protected void updateLogicalExpressionValue() {
			LogicalExpression logicalExpression = (LogicalExpression) expression;
			int processed = 0;
			for (int i = 0; i < partialResults.size(); i++) {
				EvalResult result = (EvalResult) partialResults.get(i);
				if (result.isComplete()) {
					if (i == 0 && logicalExpression.getOperator().getValue() == LogicalOperators.NOT) {
						complete = true;
						booleanValue = Boolean.valueOf(!result.booleanValue().booleanValue());
					} else if (logicalExpression.getOperator().getValue() == LogicalOperators.AND) {
						processed++;
						if (!result.booleanValue().booleanValue()) {
							complete = true;
							booleanValue = Boolean.FALSE;
						}
					} else if (logicalExpression.getOperator().getValue() == LogicalOperators.OR) {
						processed++;
						if (result.booleanValue().booleanValue()) {
							complete = true;
							booleanValue = Boolean.TRUE;
						}
					}
				}
				if (complete)
					break;
			}
			if (!complete) {
				if (logicalExpression.getArguments().size() == processed) {
					complete = true;
					if (logicalExpression.getOperator().getValue() == LogicalOperators.AND)
						booleanValue = Boolean.TRUE;
					else if (logicalExpression.getOperator().getValue() == LogicalOperators.OR) {
						if (logicalExpression.getArguments().size() == 0)
							booleanValue = Boolean.TRUE;
						else
							booleanValue = Boolean.FALSE;
					}
				}
			}
		}
		protected void updateNumericFunctionValue() {
			// TODO implement COUNT, MAX and MIN
			// NumericFunction numericFunction = (NumericFunction) operand;
			// if (numericFunction.getFunction() ==
			// NumericFunctions.COUNT_LITERAL) {
			// }
		}
		protected void updateSimpleOperand() {
			if (isComplete())
				return;
			SimpleOperand simpleOperand = ((SimpleOperand) operand);
			if (simpleOperand.getType() != null) {
				if (context == null)
					return;
				valueType = OBJECT;
				objectValue = context;
				complete = true;
			} else if (simpleOperand.getFeature() != null) {
				if (context == null)
					return;
				try {
					Object v = context.eGet(simpleOperand.getFeature(), false);
					updateSimpleOperandValue(v);
				} catch (Exception e) {
					// ignore features that are not applicable to the current
					// context
				}
			} else {
				Object v = simpleOperand.getValue();
				updateSimpleOperandValue(v);
			}
		}
		protected void updateSimpleOperandValue(Object v) {
			if (v instanceof Number) {
				numericValue = (Number) v;
				valueType = NUMERIC;
			} else if (v instanceof Boolean) {
				booleanValue = (Boolean) v;
				valueType = BOOLEAN;
			} else if (v instanceof String) {
				stringValue = v.toString();
				valueType = STRING;
			} else {
				objectValue = v;
				valueType = OBJECT;
			}
			complete = true;
		}
		public EvalResult evalOnly(EStructuralFeature sf, Object value) {
			EClass c = sf.getEContainingClass();
			if (!classPredicatesIndex.containsKey(c)) {
				complete = true;
				booleanValue = Boolean.TRUE;
				return this;
			}
			internalEvalOnly(sf, value);
			return this;
		}
		protected EvalResult internalEvalOnly(EStructuralFeature sf, Object value) {
			if (operand instanceof SimpleOperand && ((SimpleOperand) operand).getFeature() == sf) {
				objectValue = value;
				complete = true;
			} else {
				for (int i = 0; i < partialResults.size(); i++) {
					EvalResult r = ((EvalResult) partialResults.get(i));
					r.internalEvalOnly(sf, value);
				}
				computePartialResult();
			}
			return this;
		}
		public int partialEvalOnly(EClass value) {
			if (!classPredicatesIndex.containsKey(value)) {
				return IS_COMPLETE & IS_TRUE;
			}
			return internalEvalOnly(value);
		}
		protected int internalEvalOnly(EClass value) {
			if (operand != null) {
				if (operand instanceof SimpleOperand) {
					if (contextType == null || contextType == value) {
						return (isComplete() ? IS_COMPLETE : IS_NOT_COMPLETE) | IS_APPLICABLE;
					} else {
						return (isComplete() ? IS_COMPLETE : IS_NOT_COMPLETE);
					}
				}
			} else {
				//				if (isComplete())
				//					if (booleanValue().booleanValue())
				//						return IS_COMPLETE | IS_TRUE | IS_APPLICABLE;
				//					else if (partialResults.size() == 0)
				//						return (IS_COMPLETE | IS_FALSE);
				int i = 0, j = 0, k = 0, f = 0, t = 0;
				for (; i < partialResults.size(); i++) {
					EvalResult r = ((EvalResult) partialResults.get(i));
					int res = r.internalEvalOnly(value);
					if ((res & IS_APPLICABLE) != 0) {
						k++;
						if ((res & IS_COMPLETE) != 0) {
							if ((res & IS_TRUE) != 0)
								t++;
							else if ((res & IS_FALSE) != 0)
								f++;
						} else
							return IS_NOT_COMPLETE | IS_APPLICABLE;
					}
				}
				if (k == 0)
					return (isComplete() ? IS_COMPLETE : IS_NOT_COMPLETE);
				if (expression instanceof BinaryExpression) {
					if (isComplete()) {
						if (booleanValue().booleanValue())
							return IS_COMPLETE | IS_APPLICABLE | IS_TRUE;
						else
							return IS_COMPLETE | IS_APPLICABLE | IS_FALSE;
					} else
						return IS_NOT_COMPLETE | IS_APPLICABLE;
				} else if (expression instanceof LogicalExpression) {
					LogicalExpression le = (LogicalExpression) expression;
					if (le.getOperator() == LogicalOperators.AND_LITERAL) {
						if (k == t)
							return IS_COMPLETE | IS_APPLICABLE | IS_TRUE;
						else
							return IS_COMPLETE | IS_APPLICABLE | IS_FALSE;
					}
					if (le.getOperator() == LogicalOperators.OR_LITERAL) {
						if (t > 0)
							return IS_COMPLETE | IS_APPLICABLE | IS_TRUE;
						else if (f > 0)
							return IS_COMPLETE | IS_APPLICABLE | IS_FALSE;
						else
							return IS_NOT_COMPLETE | IS_APPLICABLE;
					}
					if (le.getOperator() == LogicalOperators.NOT_LITERAL) {
						if (t > 0)
							return IS_COMPLETE | IS_APPLICABLE | IS_FALSE;
						else if (f > 0)
							return IS_COMPLETE | IS_APPLICABLE | IS_TRUE;
						else
							return IS_NOT_COMPLETE | IS_APPLICABLE;
					}
				}
			}
			return 0;
		}
		public boolean resetOnly(EStructuralFeature sf) {
			EClass c = sf.getEContainingClass();
			if (!classPredicatesIndex.containsKey(c)) {
				complete = false;
				booleanValue = null;
				return false;
			}
			return internalResetOnly(sf);
		}
		private boolean internalResetOnly(EStructuralFeature sf) {
			if (operand instanceof SimpleOperand && ((SimpleOperand) operand).getFeature() == sf) {
				objectValue = null;
				complete = false;
			} else {
				for (int i = 0; i < partialResults.size(); i++) {
					EvalResult r = ((EvalResult) partialResults.get(i));
					if (!r.internalResetOnly(sf)) {
						complete = false;
					}
				}
				if (!complete) {
					resetValues();
				}
			}
			return isComplete();
		}
	}
	/**
	 * @author Marius Slavescu (slavescu@ca.ibm.com)
	 * @since 3.3
	 *  
	 */
	public class SimpleSearchQueryEvaluator {
		public SimpleSearchQueryEvaluator() {
		}
		protected WhereExpression getExpression(String string) {
			return (WhereExpression) namesIndex.get(string);
		}
		protected Operand getOperand(String string) {
			return (Operand) namesIndex.get(string);
		}
		protected EvalResult internalPrepareEvalResult() {
			if (query.getWhereExpression() == null)
				return new EvalResult();
			if (whereExpressionValuesIndex.get(query.getWhereExpression()) != null)
				return (EvalResult) whereExpressionValuesIndex.get(query.getWhereExpression());
			return parseWhereExpression(query.getWhereExpression());
		}
		protected EvalResult parseArithmeticExpression(ArithmeticExpression expression) {
			if (whereExpressionValuesIndex.get(expression) != null)
				return (EvalResult) whereExpressionValuesIndex.get(expression);
			if (expression.getName() != null && expression.getName().startsWith(QueryUtils.DOLLAR_SIGN))
				return parseOperand(getOperand(expression.getName().substring(1)));
			EvalResult result = new EvalResult(expression);
			whereExpressionValuesIndex.put(expression, result);
			for (Iterator iter = expression.getArguments().iterator(); iter.hasNext();) {
				Operand element = (Operand) iter.next();
				result.addPartialResult(parseOperand(element));
			}
			return result;
		}
		protected EvalResult parseBinaryExpression(BinaryExpression expression) {
			if (whereExpressionValuesIndex.get(expression) != null)
				return (EvalResult) whereExpressionValuesIndex.get(expression);
			if (expression.getName() != null && expression.getName().startsWith(QueryUtils.DOLLAR_SIGN))
				return parseWhereExpression(getExpression(expression.getName().substring(1)));
			EvalResult result = new EvalResult(expression);
			whereExpressionValuesIndex.put(expression, result);
			Operand operand = expression.getLeftOperand();
			result.addPartialResult(parseOperand(operand));
			for (Iterator iter = expression.getRightOperands().iterator(); iter.hasNext();) {
				Operand element = (Operand) iter.next();
				result.addPartialResult(parseOperand(element));
			}
			return result;
		}
		protected EvalResult parseLogicalExpression(LogicalExpression expression) {
			if (whereExpressionValuesIndex.get(expression) != null)
				return (EvalResult) whereExpressionValuesIndex.get(expression);
			if (expression.getName() != null && expression.getName().startsWith(QueryUtils.DOLLAR_SIGN))
				return parseWhereExpression(getExpression(expression.getName().substring(1)));
			EvalResult result = new EvalResult(expression);
			whereExpressionValuesIndex.put(expression, result);
			for (Iterator iter = expression.getArguments().iterator(); iter.hasNext();) {
				WhereExpression element = (WhereExpression) iter.next();
				result.addPartialResult(parseWhereExpression(element));
			}
			return result;
		}
		protected EvalResult parseNumericFunction(NumericFunction function) {
			if (whereExpressionValuesIndex.get(function) != null)
				return (EvalResult) whereExpressionValuesIndex.get(function);
			if (function.getName() != null && function.getName().startsWith(QueryUtils.DOLLAR_SIGN))
				return parseOperand(getOperand(function.getName().substring(1)));
			EvalResult result = new EvalResult(function);
			whereExpressionValuesIndex.put(function, result);
			for (Iterator iter = function.getArguments().iterator(); iter.hasNext();) {
				Operand element = (Operand) iter.next();
				result.addPartialResult(parseOperand(element));
			}
			return result;
		}
		protected EvalResult parseOperand(Operand operand) {
			if (operand instanceof ArithmeticExpression) {
				return parseArithmeticExpression((ArithmeticExpression) operand);
			} else if (operand instanceof NumericFunction) {
				return parseNumericFunction((NumericFunction) operand);
			}
			return parseSimpleOperand(operand);
		}
		protected EvalResult parseSimpleOperand(Operand operand) {
			if (whereExpressionValuesIndex.get(operand) != null)
				return (EvalResult) whereExpressionValuesIndex.get(operand);
			if (operand.getName() != null && operand.getName().startsWith(QueryUtils.DOLLAR_SIGN))
				return parseSimpleOperand(getOperand(operand.getName().substring(1)));
			EvalResult result = new EvalResult(operand);
			whereExpressionValuesIndex.put(operand, result);
			return result;
		}
		protected EvalResult parseWhereExpression(WhereExpression expression) {
			if (expression instanceof LogicalExpression)
				return parseLogicalExpression((LogicalExpression) expression);
			else
				return parseBinaryExpression((BinaryExpression) expression);
		}
		public EvalResult prepareEvalResult() {
			return internalPrepareEvalResult();
		}
	}
	public SimpleSearchQueryEngine(SimpleSearchQuery query, ResourceSet targetResourceSet) {
		this.query = query;
		this.targetResourceSet = targetResourceSet;
	}
	protected void addResultValue(EObject element) {
		// TODO add support for multiple output elements
		if (queryResult.getResultEntries().size() == 1) {
			List l = (List) ((ResultEntry) queryResult.getResultEntries().get(0)).getValue();
			if (l == null) {
				l = new ArrayList();
				((ResultEntry) queryResult.getResultEntries().get(0)).setValue(l);
			}
			l.add(element);
			return;
		}
		throw new RuntimeException("Not implemented yet !");
	}
	protected void buildRequiredTraversalPaths() {
		alreadyProcessed.clear();
		for (Iterator iter = rootClasses.iterator(); iter.hasNext();) {
			EClass element = (EClass) iter.next();
			processContainmentPath(element);
		}
	}
	protected void evalSelfAndParents(EObject element) {
		if (element == null || alreadyProcessed.contains(element))
			return;
		if (classPredicatesIndex.containsKey(element.eClass())) {
			alreadyProcessed.add(element);
			result.eval(element);
			resetParentsStack.add(element);
			//			traverseFeatures(element, resetParentsStack);
		}
		if (element.eContainer() != null)
			evalSelfAndParents(element.eContainer());
	}
	protected void resetSelfAndParents(EObject element, EvalResult result) {
		if (element == null || alreadyProcessed.contains(element))
			return;
		if (classPredicatesIndex.containsKey(element.eClass()))
			result.reset(element);
		alreadyProcessed.add(element);
		if (element.eContainer() != null)
			resetSelfAndParents(element.eContainer(), result);
	}
	public QueryResult execute() {
		PerfUtil p = new PerfUtil(null, false);
//		p.setDebug(true);
		p.setMessageAndStart("SimpleSearchQueryEngine.execute()");
		prepareResult();
		populateRootNodesAndClasses();
		buildRequiredTraversalPaths();
		indexWhereExpression();
		populateResult();
		sortResult();
		limitResult();
		p.stopAndPrintStatus();
		return queryResult;
	}
	protected void indexWhereExpression() {
		List roots = new ArrayList();
		if (query.getWhereExpression() == null)
			return;
		roots.add(query.getWhereExpression());
		ContainmentTraverser containmentTraverser = new ContainmentTraverser(roots) {
			protected boolean beforeChildren(EObject element) {
				String name = QueryUtils.getElementName(element);
				if (name != null && name.startsWith(QueryUtils.DOLLAR_SIGN))
					return super.beforeChildren(element);
				if (whereExpressionValuesIndex.containsKey(element))
					return super.beforeChildren(element);
				// initialize evaluation result
				whereExpressionValuesIndex.put(element, null);
				if (element instanceof SimpleOperand) {
					SimpleOperand o = (SimpleOperand) element;
					EClass c = o.getType();
					if (c != null) {
						Collection l = (Collection) classPredicatesIndex.get(c);
						if (l == null) {
							l = new HashSet();
							classPredicatesIndex.put(c, l);
							QueryUtils.addSubTypes(c, classPredicatesIndex, l);
						}
						if (!l.contains(element))
							l.add(element);
					} else if (o.getFeature() != null) {
						if (!requiredPaths.contains(o.getFeature())) {
							c = o.getFeature().getEContainingClass();
							//							if(o.getFeature().getName().equals("isA"))
							//								System.out.println("SimpleSearchQueryEngine.indexWhereExpression()");
							//							if(o.getFeature().getName().equals("refTarget"))
							//								System.out.println("SimpleSearchQueryEngine.indexWhereExpression()");
							//							if(c.getName().equals("TRCHeapObject"))
							//								System.out.println("SimpleSearchQueryEngine.indexWhereExpression()");
							Collection l = (Collection) classPredicatesIndex.get(c);
							if (l == null) {
								l = new HashSet();
								classPredicatesIndex.put(c, l);
								QueryUtils.addSubTypes(c, classPredicatesIndex, l);
							}
							if (!l.contains(element)) {
								l.add(element);
								QueryUtils.addSubTypes(c, classPredicatesIndex, l);
							}
							l = (Collection) featurePredicatesIndex.get(c);
							if (l == null) {
								l = new HashSet();
								featurePredicatesIndex.put(c, l);
								QueryUtils.addSubTypes(c, featurePredicatesIndex, l);
							}
							if (!l.contains(o.getFeature())) {
								l.add(o.getFeature());
								QueryUtils.addSubTypes(c, featurePredicatesIndex, l);
							}
						}
					}
				}
				if (name != null)
					namesIndex.put(name, element);
				return super.beforeChildren(element);
			}
		};
		containmentTraverser.traverse();
	}
	protected boolean isOutputElement(EObject element) {
		EClass c = element.eClass();
		return outputClasses.contains(c) || outputClassesSubTypes.contains(c);
	}
	protected void limitResult() {
		if (query.getMaxElements() == 0)
			return;
		// support only for one output element
		ResultEntry tempResult = (ResultEntry) queryResult.getResultEntries().get(0);
		List initialList = ((List) ((ResultEntry) queryResult.getResultEntries().get(0)).getValue());
		List resultList = new ArrayList();
		int start, end = 0;
		if (query.getStartWith() >= 0) {
			start = query.getStartWith();
			end = Math.min(start + query.getMaxElements() - 1, initialList.size() - 1);
		} else {
			end = Math.max(0, initialList.size() + query.getStartWith());
			start = Math.max(0, end - query.getMaxElements());
		}
		for (int i = start; i <= end; i++) {
			resultList.add(initialList.get(i));
		}
		tempResult.setValue(resultList);
	}
	protected void populateResult() {
		result = queryEvaluator.prepareEvalResult();
		alreadyProcessed.clear();
		ContainmentTraverser containmentTraverser = new ContainmentTraverser(rootNodes) {
			int depth = 0;
			protected boolean afterChildren(EObject element) {
				depth--;
				if (classPredicatesIndex.containsKey(element.eClass())) {
					if (depth == 0) {
						//						alreadyProcessed.clear();
						//						resetSelfAndParents(element.eContainer(), result);
						if (!resetParentsStack.isEmpty()) {
							processResetList(resetParentsStack, 0);
						}
					}
					int ei = resetStack.lastIndexOf(element);
					if (ei >= 0) {
						processResetList(resetStack, ei);
					}
				}
				return true;
			}
			protected boolean beforeChildren(EObject element) {
				depth++;
				if (classPredicatesIndex.containsKey(element.eClass())) {
					if (depth == 1) {
						alreadyProcessed.clear();
						evalSelfAndParents(element.eContainer());
					}
					result.eval(element);
					resetStack.add(element);
					if (isOutputElement(element)) {
						if (result.isComplete()) {
							if (result.booleanValue().booleanValue())
								addResultValue(element);
						} else {
							if (traverseFeatures(element, resetStack)) {
								if (result.isComplete())
									if (result.booleanValue().booleanValue())
										addResultValue(element);
							}
						}
						pruneSubtree = true;
					} else {
						traverseFeatures(element, resetStack);
					}
				}
				return true;
			}
			protected boolean isRequiredFeature(EStructuralFeature sf) {
				return requiredPaths.contains(sf);
			}
		};
		try {
			containmentTraverser.traverse();
		} catch (ConcurrentModificationException e) {
			// ignore this exception and return;
		}
	}
	protected boolean traverseFeatures(EObject element, List resetList) {
		Collection fl = (Collection) featurePredicatesIndex.get(element.eClass());
		boolean ret = false;
		if (fl != null) {
			for (Iterator iter = fl.iterator(); iter.hasNext();) {
				if (ret)
					break;
				EStructuralFeature sf = (EStructuralFeature) iter.next();
				if (requiredPaths.contains(sf))
					continue;
				if (sf.isMany()) {
					try {
						List rl = (List) element.eGet(sf);
						for (Iterator iterator = rl.iterator(); iterator.hasNext();) {
							if (ret)
								break;
							EObject le = (EObject) iterator.next();
							if (classPredicatesIndex.get(le.eClass()) != null) {
								resetList.add(le);
								int pos = resetList.size() - 1;
								ret = processEObject(resetList, le, pos);
								if (!ret && processResetListIsRequired(le)) {
									processResetList(resetList, pos);
								}
							}
						}
					} catch (Exception e) {
						// TODO: handle exception
					}
				} else {
					if (sf instanceof EReference) {
						try {
							EObject ro = (EObject) element.eGet(sf);
							if (classPredicatesIndex.get(ro.eClass()) != null) {
								resetList.add(ro);
								int pos = resetList.size() - 1;
								ret = processEObject(resetList, ro, pos);
								if (!ret && processResetListIsRequired(ro)) {
									processResetList(resetList, pos);
								}
							}
						} catch (Exception e) {
							// TODO: handle exception
						}
					}
				}
			}
		}
		return ret;
	}
	protected boolean processResetListIsRequired(EObject eObject) {
		if ((result.partialEvalOnly(eObject.eClass()) & (IS_COMPLETE | IS_TRUE | IS_APPLICABLE)) == (IS_COMPLETE | IS_TRUE | IS_APPLICABLE)) {
			return false;
		}
		return true;
	}
	protected boolean processEObject(List resetList, EObject eObject, int pos) {
		boolean ret;
		result.eval(eObject);
		if (result.isComplete()) {
			ret = addResultValueIfRequired(eObject, pos);
		} else {
			ret = traverseFeatures(eObject, resetList);
			if (result.isComplete()) {
				ret = addResultValueIfRequired(eObject, pos);
			}
		}
		return ret;
	}
	protected boolean addResultValueIfRequired(EObject eObject, int pos) {
		if (result.booleanValue().booleanValue()) {
			if (isOutputElement(eObject)) {
				addResultValue(eObject);
				return false;
			} else
				return true;
		}
		return false;
	}
	protected void populateRootNodesAndClasses() {
		rootNodes.clear();
		rootClasses.clear();
		for (Iterator iter = query.getSources().iterator(); iter.hasNext();) {
			String element = (String) iter.next();
			URI uri = URI.createURI(element);
			EObject n = targetResourceSet.getEObject(uri, false);
			EClass c = n.eClass();
			if (!rootNodes.contains(n))
				rootNodes.add(n);
			if (!rootClasses.contains(c))
				rootClasses.add(c);
		}
	}
	protected void prepareResult() {
		queryResult = ExtensionsFactory.eINSTANCE.createQueryResult();
		queryResult.setQuery(query);
		for (Iterator iter = query.getOutputElements().iterator(); iter.hasNext();) {
			// support only simple operand
			SimpleOperand outputElement = (SimpleOperand) iter.next();
			EClass c = outputElement.getType();
			if (c == null && outputElement.getFeature() != null) {
				c = outputElement.getFeature().getEContainingClass();
			}
			if (c != null) {
				outputClasses.add(c);
				QueryUtils.addSubTypes(c, outputClassesSubTypes);
				QueryUtils.addSuperTypes(c, outputClassesSuperTypes);
				classPredicatesIndex.put(c, null);
				QueryUtils.addSubTypes(c, classPredicatesIndex,null);
			}
			ResultEntry resultEntry = ExtensionsFactory.eINSTANCE.createResultEntry();
			queryResult.getResultEntries().add(resultEntry);
			List res = new ArrayList();
			resultEntry.setValue(res);
		}
	}
	protected boolean processContainmentPath(EClass element) {
		if (alreadyProcessed.contains(element))
			return outputClasses.contains(element) || outputClassesSuperTypes.contains(element);
		alreadyProcessed.add(element);
		boolean result = false;
		boolean ret = false;
		if (outputClasses.contains(element) || outputClassesSuperTypes.contains(element))
			result = true;
		for (Iterator iter = element.getEAllContainments().iterator(); iter.hasNext();) {
			EReference ref = (EReference) iter.next();
			ret = processContainmentPath(ref.getEReferenceType());
			result = result || ret;
			if (ret)
				requiredPaths.add(ref);
		}
		return result;
	}
	protected void sortResult() {
		// support only one order by element
		if (query.getOrderByExpresions().size() != 1)
			return;
		final EAttribute compareAttribute = (EAttribute) ((SimpleOperand) ((OrderByElement) query.getOrderByExpresions().get(0)).getOperand()).getFeature();
		final OrderByOperators compareOperator = ((OrderByElement) query.getOrderByExpresions().get(0)).getOperator();
		final boolean reversOrder = compareOperator.getValue() == OrderByOperators.DESC;
		Comparator comparator = new Comparator() {
			public int compare(Object o1, Object o2) {
				try {
					Object o1Value = ((EObject) o1).eGet(compareAttribute);
					Object o2Value = ((EObject) o2).eGet(compareAttribute);
					try {
						return reversOrder ? -((Comparable) o1Value).compareTo(o2Value) : ((Comparable) o1Value).compareTo(o2Value);
					} catch (Exception e) {
						// ignore and compare the toString values
					}
					return reversOrder ? -o1Value.toString().compareTo(o2Value.toString()) : o1Value.toString().compareTo(o2Value.toString());
				} catch (Exception e) {
					// if feature doesn't exist, don't change the order
					return 0;
				}
			}
		};
		ResultEntry tempResult = (ResultEntry) queryResult.getResultEntries().get(0);
		List resultList = (List) tempResult.getValue();
		Collections.sort(resultList, comparator);
		tempResult.setValue(resultList);
	}
	protected void processResetList(List list, int startPosition) {
		alreadyProcessed.clear();
		for (int i = list.size() - 1; i >= startPosition; i--) {
			EObject resetElement = (EObject) list.get(i);
			if (alreadyProcessed.contains(resetElement.eClass()))
				continue;
			result.reset(resetElement);
			alreadyProcessed.add(resetElement.eClass());
			list.remove(i);
		}
	}
}